!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module CorotatingBinary is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir CorotatingBinary
!! @brief Contains files necessary for the CorotatingBinary Calculation

!> @file CorotatingBinary/problem.f90
!! @brief Main file for module CorotatingBinary

!> @defgroup CorotatingBinary Corotating Binary Module
!! @brief Module for calculating collapse of a uniform cloud
!! @ingroup Modules

!> Corotating Binary Module 
!! @ingroup CorotatingBinary
MODULE Problem
  USE DataDeclarations
  USE Ambients
  USE Winds
  USE PhysicsDeclarations
  USE CommonFunctions
  USE ParticleDeclarations
  USE PointGravitySrc
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  REAL(KIND=qPREC) :: wind_density, wind_temp, r_p, v_w, omega_star, star_phi, r_bondi, xloc(3), iAccretion=KRUMHOLZ_ACCRETION
  LOGICAL :: lCreateParticle=.false.
CONTAINS


  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
     TYPE(ParticleDef), POINTER :: Particle
     TYPE(WindDef), POINTER :: Wind
     INTEGER :: i
     NAMELIST /AmbientData/ wind_density, wind_temp, r_p, v_w, Omega_star, star_phi, xloc, r_bondi, lCreateParticle, iAccretion
     OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
     READ(PROBLEM_DATA_HANDLE,NML=AmbientData)    
     IF (lCreateParticle) THEN
        NULLIFY(Particle)
        CALL CreateParticle(Particle)
        Particle%q(1)=v_w**2*r_bondi/ScaleGrav
        Particle%xloc=xloc
        Particle%iAccrete=iAccretion
        CALL CreatePointGravityObject(Particle%PointGravityObj)
        Particle%lFixed=.true.
        Particle%PointGravityObj%soft_length=4d0*sink_dx
        Particle%PointGravityObj%Mass=Particle%q(1)
        Particle%PointGravityObj%x0=Particle%xloc
        CALL AddSinkParticle(Particle)
     END IF
     IF (nDim == 3) THEN
        DO i=1,2
           CALL CreateWind(Wind)
           Wind%dir=3
           Wind%edge=i
           Wind%Type=OUTFLOW_ONLY
           Wind%density=wind_density
           Wind%temperature=wind_temp
        END DO
     END IF
   END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
    INTEGER, DIMENSION(3,2) :: ip
    ip=1
    ip(1:nDim,2)=Info%mX(1:nDim)
!    write(*,*) 'setting wind params for', ip
    CALL SetWind(Info, ip)  
!    DO i=1,48
!       IF (ANY(Info%q(i,ip(2,1):
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: l, ip(3,2)
    INTEGER, PARAMETER, DIMENSION(2) :: winddir=(/1,2/), windedge=(/1,2/)
!    ip=1
!    ip(1:nDim,1)=60
!    ip(1:nDim,2)=68
    DO l=1,2
       IF (GhostOverlap(Info, winddir(l), windedge(l),ip)) THEN
          CALL SetWind(Info, ip)
       END IF
    END DO
  END SUBROUTINE ProblemBeforeStep

  SUBROUTINE SetWind(Info, ip)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k, ip(3,2)
    REAL(KIND=qPREC) :: dx, dz,pos(3), t
    dx=levels(Info%level)%dx
    dz=merge(dx, 0d0, nDim==3)
    t=levels(Info%level)%tnow
    DO i=ip(1,1), ip(1,2)
       pos(1)=Info%xBounds(1,1)+(REAL(i)-half)*dx
       DO j=ip(2,1), ip(2,2)
          pos(2)=Info%xBounds(2,1)+(REAL(j)-half)*dx
          DO k=ip(3,1), ip(3,2)
             pos(3)=Info%xBounds(3,1)+(REAL(k)-half)*dz
             CALL SetWindq(Info%q(i,j,k,:),pos,t)
          END DO
       END DO
    END DO
  END SUBROUTINE SetWind


  SUBROUTINE SetWindq(q, pos,t)
    REAL(KIND=qPREC), DIMENSION(:) :: q
    REAL(KIND=qPREC), DIMENSION(3) :: pos
    INTEGER, PARAMETER :: nIters=8
    INTEGER :: i
    REAL(KIND=qPREC) :: t_r, r, d(3), xp(3),v(3), dmag,t,vp(3),n(3)
    r=sqrt(sum(pos**2))
    t_r=t-r/v_w    
!    IF (r > v_w/Omega_star) THEN
!       write(*,*) 'WARNING ... multiple solutions exist... - consider shrinking grid'
       !STOP
!    END IF
!    write(*,*) pos
    DO i=1,nIters
       xp=r_p*(/cos(star_phi+Omega_star*t_r-OmegaRot*t), sin(star_phi+Omega_star*t_r-OmegaRot*t), 0d0/)
       d=(pos-xP)
       dmag=sqrt(sum(d**2))
       vp=r_p*Omega_star*(/-sin(star_phi+Omega_star*t_r-OmegaRot*t), cos(star_phi+Omega_star*t_r-OmegaRot*t), 0d0/)
       n=SolveCrossEq(d, -Cross3D(vp, d)/v_w)
       v=v_w*n+vp
       t_r=t-dmag/sqrt(sum(v**2))
!       write(*,*) i, t_r
    END DO
    v=v+Omegarot*(/pos(2), -pos(1),0d0/)
    q(1)=wind_density/dmag**(REAL(nDim)-1d0)
!    q(1)=100d0+omega_star*t_r
    q(imom(1:nDim))=q(1)*v(1:nDim)

    IF (iE /= 0) q(iE)=half*sum(q(imom(1:nDim))**2)/q(1)+gamma7*q(1)*wind_temp
!    write(*,*) 'setting wind q', pos, q(1), wind_density
  END SUBROUTINE SetWindq

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

