!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module FieldLoopRestart is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir FieldLoopRestart
!! @brief Contains files necessary for module FieldLoopRestart

!> @file problem.f90
!! @brief Main file for module FieldLoopRestart

!> @defgroup FieldLoopRestart Field Loop Restart Module
!! @brief Module for setting up the advection of a field loop across the grid
!! @ingroup Modules

!> Module for setting up the advection of a field loop across the grid
!! @ingroup FieldLoopRestart

MODULE Problem

  USE DataDeclarations
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE Ambients
  IMPLICIT NONE
  SAVE
  PRIVATE
  
  PUBLIC ProblemModuleInit, ProblemGridInit, &
         ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep

  REAL(KIND=qPrec) :: rho,p,v(3),Ao,thickness,R,phi,theta
  REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: infoq
  INTEGER :: sample_res=16

  LOGICAL :: lCooling
  LOGICAL :: lResolutionTest=.false.
  INTEGER :: iCooling
  

CONTAINS

  !> Initializes module variables
  SUBROUTINE ProblemModuleInit
    INTEGER :: iErr
    TYPE(AmbientDef), POINTER :: Ambient
    NAMELIST /ProblemData/ rho,p,v,Ao,R,thickness,phi,theta,lCooling,iCooling, lResolutionTest

    OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data')
       
    READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
    
    CLOSE(PROBLEM_DATA_HANDLE, IOSTAT=iErr)
    
    
    IF (nDim == 2 .AND. ((phi /= zero) .OR. (theta /= zero))) THEN
       PRINT *, "CalcConstantProblem() error: phi and theta need to be 0 in a 2D problem."
       STOP
    END IF
    NULLIFY(Ambient)
    CALL CreateAmbient(Ambient)
    Ambient%density=rho
    Ambient%pressure=p
    Ambient%velocity=v
END SUBROUTINE ProblemModuleInit
 
  !> Initial Conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
	!! @brief Initializes the grid data according to the requirements of the problem.
	!! @param Info A grid structure.	
    TYPE (InfoDef) :: Info
    INTEGER :: i,j,k,l,m,ii,jj,kk
	INTEGER :: rmbc,zrmbc
	INTEGER :: mx, my, mz
    REAL(KIND=qPrec) :: pos(3), mini_pos(3), x_rotated(3), s, temp(3)
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), ALLOCATABLE :: A
	INTEGER :: iErr
	
    
!    IF (ALLOCATED(A))  DEALLOCATE(A)

	! Calculating the number of ghost cells on each side of the grid.
	rmbc = levels(Info%level)%gmbc(1)!CoarsenRatio(Info%level-1) * mbc
    SELECT CASE(nDim)
    CASE(2)
       zrmbc=0
    CASE(3)
       zrmbc=rmbc
    END SELECT

	mx = Info%mX(1)
	my = Info%mX(2)
	mz = Info%mX(3)
!	write(*,*) Info%mX, Info%xBounds
!        write(*,*) Info%level
!        write(*,*) levels(Info%level)%dx
    CALL ConvertTotalToInternalEnergy(Info%q(1:Info%mX(1), 1:Info%mX(2), 1:Info%mX(3), :))

    IF (nDim == 3) THEN
	   
       ALLOCATE(A(1-rmbc:mx+1+rmbc,1-rmbc:my+1+rmbc, 1-zrmbc:mz+1+zrmbc,3), STAT=iErr)
	   
	   IF (iErr /= 0) THEN
	       PRINT *, "DomainInitProblem() error: u unable to allocate 3D array."
		   STOP
	   END IF
	   
	   
       A=0.0	! A = Magnetic potential (B = curl(A)).
	   
       DO k=1-zrmbc, mz+1+zrmbc
          DO j=1-rmbc, my+1+rmbc
             DO i=1-rmbc, mx+1+rmbc
                DO ii=0,0
                   DO jj=0,0
                      DO kk=0,0
                         IF (maxval(abs(A(i,j,k,:))) == 0) THEN
!                            write(*,*) shape(pos)
!                            write(*,*) shape(Info%xBounds)
!                            write(*,*) pos
!                            write(*,*) Info%xBounds(1:3,1)
                            pos=Info%xBounds(1:3,1)
                            pos=levels(Info%level)%dx
                            pos=(/REAL(i,KIND=qPREC)-1d0, REAL(j,KIND=qPREC)-1d0, REAL(k,KIND=qPREC)-1d0/) * levels(Info%level)%dx
                            pos=(/REAL(i,8)-1, REAL(j,8)-1, REAL(k,8)-1/) * levels(Info%level)%dx

						 
                            pos=Info%xBounds(1:3,1)+(/REAL(i,8)-1, REAL(j,8)-1, REAL(k,8)-1/) * levels(Info%level)%dx
                            pos(1)=pos(1)+(GxBounds(1,2)-GxBounds(1,1))*(modulo(ii+1,3)-1)
                            pos(2)=pos(2)+(GxBounds(2,2)-GxBounds(2,1))*(modulo(jj+1,3)-1)
                            pos(3)=pos(3)+(GxBounds(3,2)-GxBounds(3,1))*(modulo(kk+1,3)-1)
							
                            DO l=1,3
                               mini_pos=pos
                               DO m=1,sample_res
                                  mini_pos(l) = pos(l) + levels(Info%level)%dx*(REAL(m,8)-half)/REAL(sample_res,8) !sub sample each edge
                                  x_rotated=rotate_y(rotate_z(mini_pos,-phi),-theta) !rotate coordinates again so that vector along jet-axis is along x-axis
                                  s=sqrt(x_rotated(1)**2+x_rotated(2)**2)
                                  IF (abs(x_rotated(3)) < half*thickness .AND. s <= R) THEN !inside of loop region
                                     temp=emf_source_3D(s)
                                     A(i,j,k,l)=A(i,j,k,l)+temp(l)
                                  END IF
                               END DO
                            END DO
                         END IF
                      END DO
                   END DO
                END DO
             END DO
          END DO
       END DO
	   
       A=A/REAL(sample_res, 8)

	   ! Calculate magnetic fluxes from A (B = curl(A)).

IF (MaintainAuxArrays) THEN
       Info%aux=0.0

       Info%aux(1-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc, 1-zrmbc:mz+zrmbc, 1)=&
             (A(1-rmbc:mx+rmbc+1, 2-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 3) &
            - A(1-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc, 3))/levels(Info%level)%dX &
            - (A(1-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc,2-zrmbc:mz+zrmbc+1, 2)&
            - A(1-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc, 2))/levels(Info%level)%dx

       Info%aux(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1, 1-zrmbc:mz+zrmbc, 2)=&
              (A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1,2-zrmbc:mz+zrmbc+1, 1)&
            - A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 1))/levels(Info%level)%dx&
            - (A(2-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 3)&
            - A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 3))/levels(Info%level)%dx

       Info%aux(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc, 1-zrmbc:mz+zrmbc+1, 3)=&
            (A(2-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc+1, 2)&
            - A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc+1, 2))/levels(Info%level)%dx&
            - (A(1-rmbc:mx+rmbc, 2-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc+1, 1)&
            - A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc+1, 1))/levels(Info%level)%dx
END IF
!       Info%q=0d0

	   ! Calculate B-fields
IF (MaintainAuxArrays) THEN
       forall(i=1-rmbc:mx+rmbc,j=1-rmbc:my+rmbc, k=1-zrmbc:mz+zrmbc)
          Info%q(i,j,k,iBx)=half*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
          Info%q(i,j,k,iBy)=half*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
          Info%q(i,j,k,iBz)=half*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
       end forall
END IF

!       v = rotate_z(rotate_y(v,theta),phi)
    ELSE
	
		! 2D case.

       ALLOCATE(A(1-rmbc:mx+rmbc+1,1-rmbc:my+rmbc+1, 1-zrmbc:mz+zrmbc,1), STAT=iErr)
	   
	   IF (iErr /= 0) THEN
	    PRINT *, "DomainInitProblem() error: unable to allocate 2D array."
		   STOP
	   END IF
	   
       A=0.0		! A = magnetic potential.
	   
	   ! Initialize A.
       DO j=1-rmbc, my+rmbc+1
          DO i=1-rmbc, mx+rmbc+1
             DO ii=0,2 
                DO jj=0,2
                   IF (maxval(abs(A(i,j,1,:))) == 0) THEN
                      pos=Info%xBounds(1:3,1)+(/REAL(i,8)-1, REAL(j,8)-1, 0d0/) * levels(Info%level)%dx
					  pos(1)=pos(1)+(GxBounds(1,2)-GxBounds(1,1))*(modulo(ii+1,3)-1)
					  pos(2)=pos(2)+(GxBounds(2,2)-GxBounds(2,1))*(modulo(jj+1,3)-1)
                      mini_pos=pos
                      x_rotated=mini_pos
                      s=sqrt(x_rotated(1)**2+x_rotated(2)**2)
                      IF (s <= R) THEN !inside of loop region
                         temp=emf_source_3D(s)
                         A(i,j,1,1)=temp(3)
                      END IF
                   END IF
                END DO
             END DO
          END DO
       END DO

		! Calculate magnetic fluxes (B = curl(A)).
IF (MaintainAuxArrays) THEN
       Info%aux=0.0

       Info%aux(1-rmbc:mx+1+rmbc, 1-rmbc:my+rmbc, 1-zrmbc:mz+zrmbc, 1)=&
             (A(1-rmbc:mx+rmbc+1, 2-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 1) &
            -A(1-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc,1-zrmbc:mz+zrmbc, 1))/levels(Info%level)%dx

       Info%aux(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1, 1-zrmbc:mz+zrmbc, 2)=&
            -(A(2-rmbc:mx+rmbc+1, 1-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 1)&
            -A(1-rmbc:mx+rmbc, 1-rmbc:my+rmbc+1,1-zrmbc:mz+zrmbc, 1))/levels(Info%level)%dx
END IF

!       Info%q=0d0

       ! Calculate magnetic field values from fluxes.
IF (MaintainAuxArrays) THEN
       forall(i=1-rmbc:mx+rmbc,j=1-rmbc:my+rmbc, k=1-zrmbc:mz+zrmbc)
          Info%q(i,j,k,iBx)=half*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
          Info%q(i,j,k,iBy)=half*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
       end forall
END IF

    END IF

    ! Set other variables.

!    Info%q(:,:,:,1)=rho
!    Info%q(:,:,:,2)=rho*v(1)
!    Info%q(:,:,:,3)=rho*v(2)
!    Info%q(:,:,:,4)=rho*v(3)
	
	! Calculate the total energy (E_total = E_thermal + E_kinetic + E_magnetic).

    CALL ConvertInternalToTotalEnergy(Info%q(1:Info%mX(1), 1:Info%mX(2), 1:Info%mX(3), :))

    DEALLOCATE(A)

 END SUBROUTINE ProblemGridInit

  !> Does nothing
  !! @param Info Info object
 SUBROUTINE ProblemBeforeStep(Info)
    !! @brief Performs any tasks required before the advance step.
    !! @param Info A grid structure.	
    TYPE (InfoDef) :: Info
    INTEGER, DIMENSION(0:2) :: Steps = (/1,4,11/)
    LOGICAL, DIMENSION(0:2) :: RestartTriggered=(/.false.,.false.,.false./)
    INTEGER :: i
    IF (Info%level > 2) RETURN
    i = levels(Info%level)%CurrentLevelStep

    IF (steps(Info%level) == i .AND. MPI_ID == 0 .AND. .NOT. RestartTriggered(Info%level)) THEN
       write(*,*) 'Processor ', MPI_ID, ' purposely requesting restart on level ', Info%level, 'step ', i, 'to test code'
       lRequestRestart=.true.
       RestartTriggered(Info%level)=.true.
    END IF
 END SUBROUTINE ProblemBeforeStep

  !> Does nothing
  !! @param Info Info object
  SUBROUTINE ProblemAfterStep(Info)
     !! @brief Performs any post-step corrections that are required.
     !! @param Info A grid structure.	
     TYPE (InfoDef) :: Info
     REAL(KIND=qPREC) :: dx, t, x, y, s, Bx, By, err
     INTEGER :: i,j
     IF (lResolutionTest) THEN
        t=levels(Info%level)%tnow+levels(Info%level)%dt
        IF (abs(t-.1d0) < 1e-6) THEN
           write(*,*) 'saving at t=', t
           ALLOCATE(infoq(1:Info%mX(1),1:Info%mX(2),2))
           infoq=Info%q(1:Info%mX(1),1:Info%mX(2),1,iBx:iBy)
        ELSE IF (abs(t-1.1) < 1e-6) THEN
           write(*,*) 'checking at t=', t
           dx=levels(Info%level)%dx
           err=0
           DO i=1, Info%mX(1)
              x = GxBounds(1,1)+modulo(Info%xBounds(1,1)+(REAL(i,8)+half)*dx - v(1)*(t-.1) - GxBounds(1,1), GxBounds(1,2)-GxBounds(1,1))
              DO j=1, Info%mX(2)
                 y = GxBounds(2,1)+modulo(Info%xBounds(2,1)+(REAL(j,8)+half)*dx - v(2)*(t-.1) - GxBounds(2,1), GxBounds(2,2)-GxBounds(2,1))
                 s=sqrt(x**2+y**2)
!                 IF (s <= R) THEN 
!                    Bx=-Ao*y/s
!                    By=Ao*x/s
!                 ELSE
!                    Bx=0
!                    By=0
!                 END IF
                 if (r > R/2d0) THEN
                    err=err+sum(abs(infoq(i,j,:)-info%q(i,j,1,iBx:iBy)))*dx**2
                 END if
              END DO
           END DO
           write(*,*) 'resolution, error= ', dx, err
        END IF
     END IF
  END SUBROUTINE ProblemAfterStep

  !> Does nothing
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
  	!! @brief Sets error flags according to problem-specific conditions..
	!! @param Info A grid structure.	
    TYPE (InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep


  !> Calculates the vector potential at a radius s
  !! @param s radius in cylindrical coordinates
  FUNCTION emf_source_3D(s)
    !! @param s A double-precision number.
    REAL(KIND=qPrec) :: s,w(3),emf_source_3D(3)
    w=(/0d0,0d0,1d0/)*Ao*(R-s)    
    emf_source_3D(1:3)=rotate_z(rotate_y(w,theta),phi) !rotate coordinates again so that vector along jet-axis is along x-axis
  END FUNCTION emf_source_3D


END MODULE Problem
