!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module FillingFraction is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiClumps
!! @brief Contains files necessary for the Shape Tester problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiClumps Shape Tester Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> MultiClump Module
!! @ingroup MultiClumps
MODULE Problem
   USE DataDeclarations
   USE SplitRegions
   USE Shapes
   USE EOS
   USE RiemannSolvers
   USE Ambients
   USE TreeDeclarations
   IMPLICIT NONE
   SAVE

   PUBLIC ProblemModuleInit, ProblemGridInit, &
        ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
   PRIVATE
   REAL(KIND=qPREC), DIMENSION(0:MaxDepth) :: filling_fractions
   REAL(KIND=qPREC) :: vel(3), L
   REAL(8) :: ProbtStart
   REAL(8) :: ProbTFinal
   INTEGER(8) :: nRootSteps, iRootSteps, StartStep, CellUpdatesByLevel(0:MaxDepth), TotalCellUpdatesByLevel(0:MaxDepth)
   INTEGER :: branching_ratio(0:MaxDepth)=1
   REAL(KIND=qPREC) :: CellCostByLevel(0:MaxDepth), TotalCellCostByLevel(0:MaxDepth), dy, dz
   REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:,:) :: GxByLevel
   !  REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:) :: WorkLoadByStepByLevel
   INTEGER, ALLOCATABLE, DIMENSION(:,:) :: CellsByStepByLevel
   REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: WorkLoadByRootStep
   INTEGER, ALLOCATABLE, DIMENSION(:) :: CellsByRootStep, indices
   LOGICAL :: lPlaceCentered=.true.
CONTAINS

   !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      INTEGER :: i
      REAL(KIND=qPREC) :: rs_old, rs_new
      NAMELIST /ProblemData/ filling_fractions, vel, branching_ratio, lPlaceCentered
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CLOSE(PROBLEM_DATA_HANDLE)
      filling_fractions=filling_fractions(0)

      L=.5*(GxBounds(1,2)-GxBounds(1,1))
      levels(0)%qTolerance=filling_fractions(0)**(-1d0/REAL(nDim))/L*levels(0)%dx
      DO i=1, MaxLevel
         levels(i)%qTolerance=.5d0*levels(i-1)%qTolerance*filling_fractions(i)**(-1d0/REAL(nDim))
      END DO
      !      write(*,*) levels(:)%qTolerance
      iRootSteps=0
      dy=(GxBounds(2,2)-GxBounds(2,1))/(GxBounds(2,2)-GxBounds(2,1))
      IF (nDim == 2) THEN
         dz=0
      ELSE 
         dz=(GxBounds(1,2)-GxBounds(1,1))/(GxBounds(3,2)-GxBounds(3,1))
      END IF

      ALLOCATE(GxByLevel(0:MaxLevel,product(branching_ratio(0:MaxLevel-1)),3,2))
      GxByLevel=1
      ALLOCATE(indices(0:MaxLevel))
      indices=0
     
      CALL CreateClumpLets(half*SUM(GxBounds,2), 0, half*(GxBounds(1,2)-GxBounds(1,1)))
   END SUBROUTINE ProblemModuleInit

   RECURSIVE SUBROUTINE CreateClumplets(pos,n,hw)
      REAL(KIND=qPREC), DIMENSION(3) :: pos
      INTEGER :: n,i
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: childpos
      REAL(KIND=qPREC) :: hw, hwc
      indices(n)=indices(n)+1
      GxByLevel(n,indices(n),1:nDim,1)=pos(1:nDim)-hw
      GxByLevel(n,indices(n),1:nDim,2)=pos(1:nDim)+hw
      IF (MPI_ID == 0) write(*,'(6F12.5)') GxByLevel(n,indices(n),1:nDim,1:2), hw
      IF (n < MaxLevel) THEN
         ALLOCATE(childpos(branching_ratio(n),3))
         hwc=hw*(filling_fractions(n)/real(branching_ratio(n)))**(1d0/REAL(nDim))
         IF (branching_ratio(n) == 1 .AND. lPlaceCentered) THEN
            childpos(1,:)=pos
         ELSE
            CALL GetChildPos(branching_ratio(n),childpos,hwc/hw*1.1) !REAL(nDim)))
!         write(*,*) childpos, hwc
         END IF
         DO i=1,branching_ratio(n)
            CALL CreateClumpLets(pos+hw*childpos(i,:), n+1, hwc)
         END DO
         DEALLOCATE(childpos)
      END IF
   END SUBROUTINE CreateClumplets


   SUBROUTINE GetChildPos(n_children,childpos,r)
      INTEGER :: n_children
      REAL(KIND=qPREC), DIMENSION(:,:) :: childpos
      REAL(KIND=qPREC) :: r
      INTEGER, PARAMETER :: nIterations=100000
      LOGICAL :: lSuccess
      INTEGER :: j,k,l
      REAL :: a
      !create childpositions where each child is at least 2r apart and within (1-r) of the center...
      childpos=0
      DO j=1,nIterations
         lSuccess=.true.
         DO k=1,n_children
            CALL Random(a)
            childpos(k,1)=a
            CALL Random(a)
            childpos(k,2)=a
            IF (nDim == 3) THEN
               CALL Random(a)
               childpos(k,3)=a
            END IF
            Childpos(k,1:nDim)=2d0*(childpos(k,1:nDim)-.5d0)*(1d0-r) !rescale to be within a random cube with half width (1-r)                      
            DO l=1,k-1
               IF (ALL(childpos(k,1:nDim)-childpos(l,1:nDim) < 2d0*r)) THEN
                  lSuccess=.false.
                  EXIT
               END IF
            END DO
            IF (.Not. lSuccess) EXIT
         END DO
         IF (lSuccess) EXIT
      END DO
      IF (.NOT. lSuccess) THEN
         IF (MPI_ID == 0) write(*,*) 'failed to place', n_children, 'clumplets of radius ',r

         STOP
      END IF

   END SUBROUTINE GetChildPos

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j,k, rmbc, zrmbc
      REAL(KIND=qPREC) :: x, y, z, dx, rho, r
      Info%q=0
      Info%q(:,:,:,1)=Info%level+1
      DO i=1,nDim
         Info%q(:,:,:,imom(i))=(Info%level+1)*vel(i)
      END DO
      Info%q(:,:,:,iE)=1d0+(Info%level+1)*sum(vel**2)/2d0
   END SUBROUTINE ProblemGridInit


   SUBROUTINE ProblemBeforeGlobalStep(n)
      USE Timing
      REAL(8) :: temp
      INTEGER :: n, i, LocalNodes, ExtNodes, TotalNodes, MyCells, TotalCells, TotalExtNodes, iErr
      REAL(8) :: AdvanceTime(9)
      INTEGER, ALLOCATABLE, DIMENSION(:,:) :: TempMeanCellsByStepByLevel, MaxCellsByStepByLevel, CellsByRootStep
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:) :: MaxWorkLoadByStepByLevel, MeanWorkLoadByStepByLevel, MeanCellsByStepByLevel, sigmacellsbystepbylevel, sigmaworkbystepbylevel, workbyrootstep
      INTEGER :: nFinestSteps
      REAL(KIND=qPREC) :: mystats(3), totalstats(3)

      !      write(*,*) n, iRootSteps, StartStep, nRootSteps        
!      IF (n == 0) THEN
         !         write(*,*) levels(n)%CurrentLevelStep
!         IF (levels(0)%CurrentLevelStep==StartStep) THEN
!            ProbtStart= MPI_WTime()
!            CellUpdatesByLevel=0
!            AdvanceTimer%Accumulator(:)=0d0
!            Timers(iBarrier)%Accumulator(:)=0d0
!            Timers(iAmr)%Accumulator(:)=0d0
!         END IF
!      END IF
   END SUBROUTINE ProblemBeforeGlobalStep

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: mx(3)
      REAL(KIND=qPREC) :: advcost
 !     IF (levels(0)%CurrentLevelStep < StartStep) RETURN
!!      mx=Info%mX
!!      CellUpdatesByLevel(Info%level)=CellUpdatesByLevel(Info%level)+PRODUCT(Info%mX)

   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: sample_res(3), nOverlaps,n,i
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      DO i=1,indices(Info%level+1)
         CALL CalcPhysicalOverlaps(Info, GxByLevel(Info%level+1,i,:,:)+spread(vel,2,2)*levels(Info%level)%tnow, mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic,0)            
         IF (nOverlaps > 0) THEN
            DO n=1,nOverlaps
               mS=mSs(n,:,:)
               Info%ErrFlag(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2))=1
            END DO
            DEALLOCATE(mSs, offsets)
         END IF
      END DO
   END SUBROUTINE ProblemSetErrFlag

END MODULE Problem

