!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module GravitationalCascade is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir GravitationalCascade
!! @brief Contains files necessary for the Gravitational Cascade problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup GravitationalCascade Gravitational Cascade Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> Gravitational Cascade Module
!! @ingroup GravitationalCascade
MODULE Problem
  USE DataDeclarations
  USE Clumps
  USE ProcessingDeclarations  
  USE CoolingSrc
  USE Winds  
  USE Ambients
  USE Histograms
  USE Fields
  USE Totals
  USE PDFs
  USE Projections
  USE Refinements
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  
  REAL(KIND=qPREC) :: r_children(0:MaxDepth), rho_children(0:MaxDepth), fill_fraction(0:MaxDepth), xi(0:MaxDepth)
  INTEGER :: ClumpTreeDepth=1, n_children
!   TYPE ClumpTreeDef
!      TYPE(Clumpdef), POINTER :: Clump
!      TYPE(ClumpTreeListDef), POINTER :: Children      
!   END type ClumpTreeDef!

!   TYPE ClumpTreeListDef
!      TYPE(ClumpTreeDef), POINTER :: self
!      TYPE(ClumpTreeDef), POINTER :: next
!   END type ClumpTreeListDef

CONTAINS


  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      TYPE(CoolingDef),POINTER :: coolingobj
      TYPE(AmbientDef), POINTER :: Ambient
      TYPE(ClumpDef), POINTER :: Clump
      TYPE(HistogramDef), POINTER :: HISTOGRAM
      TYPE(PDFDef), POINTER :: PDF
      TYPE(ProjectionDef), POINTER :: Projection
      TYPE(WindDef), POINTER :: Wind
      REAL(KIND=qPREC) :: A,B,ValidRoots(3),r_cloud,r_ambient,chi
      INTEGER :: i,j,nroots,  edge, dim, n
      COMPLEX(8) :: MyRoots(3)
      Logical :: lUseExisting = .False.     

      NAMELIST /ProblemData/ &
           r_cloud, &              !Radius of single cloud
           r_ambient, &            !Effective radius of ambient cloud
           n_children, &           !number of child clumps
           chi, &                  !desired density contrast between child and parent
           ClumpTreeDepth, &          !Depth of nested structures  !0 is no nesting
           lUseExisting            ! flag for checking if clumplets.data exist
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)

      ! Begin with initial size and recursively use the number of child clumps and the filling fractions to generate children of sizes...  Then set the densities of the finest children and work backwards to set the densities of the parents to give the desired mean densities of the parent.  This does however set a limit on the filling fraction?  For example imagine 7 clumps of r=r_parent/3 for a filling fraction of 7/27?  Since the Jeans length scales like rho^-1/2, the "jeans density" scales like r^{-2} so densities of these child clumps would need to be 9 times that of the parent jeans density.  However for the parent cloud to have a mean density 1/9th of the child densities, the child filling fraction must be less then 1/9th so only 3 of these clouds could exist for the parent 'cloud' to be appropriately bound...  (albeit with a density of 0).


      xi(ClumpTreeDepth)=1d0
      DO n=ClumpTreeDepth-1,0, -1
         B=1d0/(xi(n+1)*chi)
         A=1d0-B
         MyRoots=GetRoots(n_children**2*A**3,3d0*n_children**2*A**2*B-1d0,3*n_children**2*A*B**2,n_children**2*B**3,nroots)
         j=0
         DO i=1,nroots 
            IF (AIMAG(MyRoots(i)) < tiny(1d0) .AND. REAL(MyRoots(i)) > tiny(1d0) .AND. REAL(1d0-MyRoots(i)) > tiny(1d0)) THEN
               j=j+1
               ValidRoots(j)=REAL(MyRoots(i))
            END IF
         END DO
         write(*,*) myroots(1:nroots)
         !         write(*,*) n_children**2*A**3*myroots**3+(3d0*n_children**2*A**2*B-1d0)*myroots**2+3*n_children**2*A*B**2*myroots+n_children**2*B**3
         !      stop
         IF (j == 0) THEN
            IF (MPI_ID == 0) WRITE(*,*) 'No Real Roots Found.  Just ', MyRoots
            STOP
         ELSEIF (j == 1) THEN
            IF (MPI_ID == 0) WRITE(*,*) 'Found 1 Real Root', ValidRoots(1)
         ELSE
            IF (MPI_ID == 0) WRITE(*,*) 'Found several real roots... using largest of ValidRoots(1:j)'
            ValidRoots(1)=minval(ValidRoots(1:j))
         END IF
         fill_fraction(n)=ValidRoots(1)      
         xi(n)=chi*xi(n+1)*fill_fraction(n)+(1d0-fill_fraction(n)) !over density of level n-1 due to clumplets
      END DO
      r_children(0)=r_cloud
      DO n=1, ClumpTreeDepth
         r_children(n)=r_children(n-1)*fill_fraction(n-1)**(1d0/3d0) !REAL(nDim))
      END DO
      DO n=0,ClumpTreeDepth
         rho_children(n)=JeansDensity(r_children(n), MinTemp)/xi(n)
      END DO

!      write(*,*) rho_children(0)*(1d0-fill_fraction(0))+fill_fraction(0)*((1d0-fill_fraction(1))*chi+fill_fraction(1)*chi*chi)/xi(0)
      write(*,*) 'xi=', xi(0:ClumpTreeDepth)
      write(*,*) 'r_children=', r_children(0:ClumpTreeDepth)
      write(*,*) 'rho_children=', rho_children(0:ClumpTreeDepth)
      write(*,*) 'filling_fraction', fill_fraction(0:ClumpTreeDepth)
      CALL CreateAmbient(Ambient)
      Ambient%density=JeansDensity(r_ambient,MinTemp)

      CLOSE(PROBLEM_DATA_HANDLE)      
      
     if(lUseExisting) then    
        OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='clumplets.data', STATUS="OLD")
        write(*,*) sum(n_children**(/(n, n=0,ClumpTreeDepth)/))
        do i=1,sum(n_children**(/(n, n=0,ClumpTreeDepth)/))
           CALL CreateClump(Clump)
           read(PROBLEM_DATA_HANDLE,'(5e25.16)') Clump%density, Clump%position, Clump%radius
           CALL UpdateClump(Clump)
        end do 
     else
        OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='clumplets.data', STATUS="UNKNOWN")
        CALL CreateClumplets(half*Sum(GxBounds(:,:),2), 0) !puts it in middle
     end if

      CLOSE(PROBLEM_DATA_HANDLE)      

      DO i=1,nDim
         DO edge=1,2
            IF (Gmthbc(i,edge) == 1) THEN 
               CALL CreateWind(Wind)
               Wind%dir=i
               Wind%edge=edge
               Wind%type=OUTFLOW_ONLY
               Wind%density=Ambient%density
               Wind%temperature=Ambient%Pressure/Ambient%Density
               Wind%B=Ambient%B
            END IF
         END DO
      END DO


!      CALL AddAllTotals(GASCOMP)
!      CALL AddAllTotals(PARTICLECOMP)
!      CALL AddAllTotals(BOTHCOMP)

!      CALL CreateHistogram(Histogram)
!      Histogram%Field%iD=1
!      Histogram%Field%component=GASCOMP
!      Histogram%minvalue=.1d0
!      Histogram%maxvalue=1d8
      !      Histogram%nbins=nbins
!      Histogram%scale=LOGSCALE

!      CALL CreateProjection(projection)
!      Projection%Field%iD=Mass_Field
!      Projection%Field%component=BOTHCOMP
!      Projection%dim=3
!      ALLOCATE(Projection%Image)
!      Projection%Image%Scaling=LOGSCALE

!      CALL CreateProjection(projection)
!      Projection%Field%iD=Mass_Field
!      Projection%Field%component=BOTHCOMP
!      Projection%dim=1
!      ALLOCATE(Projection%Image)
!      Projection%Image%Scaling=LOGSCALE

!      CALL CreateProjection(projection)
!      Projection%Field%iD=Mass_Field
!      Projection%Field%component=BOTHCOMP
!      Projection%pow=1d0
!      Projection%dim=2
!      ALLOCATE(Projection%Image)
!      Projection%Image%Scaling=LOGSCALE

      CALL ClearAllRefinements()
      CALL AddRefinementThreshold(JeansLength_Field, LESSTHAN, (/(16d0*levels(i)%dx,i=0,MaxLevel)/))

   END SUBROUTINE ProblemModuleInit

   RECURSIVE SUBROUTINE CreateClumplets(pos,n)
      REAL(KIND=qPREC), DIMENSION(3) :: pos
      INTEGER :: n,i
      TYPE(ClumpDef), POINTER :: Clump
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: childpos

      CALL CreateClump(Clump)
      Clump%density=rho_children(n)
      Clump%position=pos
      Clump%radius=r_children(n)
      CALL UpdateClump(Clump)

      if (MPI_ID == 0) then
         write(PROBLEM_DATA_HANDLE,'(5e25.16)') Clump%density, Clump%position, Clump%radius
      end if

      IF (n < ClumpTreeDepth) THEN
         ALLOCATE(childpos(n_children,3))
         CALL GetChildPos(n_children,childpos,1.0*fill_fraction(n)**(1d0/3d0)) !REAL(nDim)))
         DO i=1,n_children
            CALL CreateClumpLets(pos+r_children(n)*childpos(i,:), n+1)
         END DO
         DEALLOCATE(childpos)
      END IF
   END SUBROUTINE CreateClumplets


  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Could be used to update grids pre-step
  !! @param Info Info Object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
 END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info

  END SUBROUTINE ProblemAfterStep


  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    USE CoolingSrc
    TYPE(InfoDef) :: Info
    
 END SUBROUTINE ProblemSetErrFlag


  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep


  FUNCTION GetRoots(a,b,c,d,nroot)
     COMPLEX(8) :: GetRoots(3)
     REAL(8) :: a,b,c,d,DD,p,q,u,v,temp1,temp2,phi
     REAL(8) :: y1,y2,y3,y2r,y2i
     INTEGER :: nroot
     ! Variables used:
     !   a, b, c, d  ... coefficients (input)
     !   y1, y2, y3  ... three transformed solutions
     !   y2r, y2i    ... real and imaginary parts of a pair of complex roots
     !   nroot       ... number of roots
     !
     ! Formula used are given in Tuma, "Engineering Mathematics Handbook", p7
     !   (McGraw Hill, 1978).
     !   Step 0: If a is 0. use the quadrati! formula to avoid dividing by 0.
     !   Step 1: Calculate p and q
     !           p = ( 3*c/a - (b/a)**2 ) / 3
     !           q = ( 2*(b/a)**3 - 9*b*c/a/a + 27*d/a ) / 27
     !   Step 2: Calculate discriminant D
     !           D = (p/3)**3 + (q/2)**2
     !   Step 3: Depending on the sign of D, we follow different strategy.
     !           If D<0, three distinct real roots.
     !           If D=0, three real roots of which at least two are equal.
     !           If D>0, one real and two complex roots.
     !   Step 3a: For D>0 and D=0,
     !           Calculate u and v
     !           u = cubic_root(-q/2 + sqrt(D))
     !           v = cubic_root(-q/2 - sqrt(D))
     !           Find the three transformed roots
     !           y1 = u + v
     !           y2 = -(u+v)/2 + i (u-v)*sqrt(3)/2
     !           y3 = -(u+v)/2 - i (u-v)*sqrt(3)/2
     !   Step 3b Alternately, for D<0, a trigonometri! formulation is more convenient
     !           y1 =  2 * sqrt(|p|/3) * cos(phi/3)
     !           y2 = -2 * sqrt(|p|/3) * cos((phi+pi)/3)
     !           y3 = -2 * sqrt(|p|/3) * cos((phi-pi)/3)
     !           where phi = acos(-q/2/sqrt(|p|**3/27))
     !                 pi  = 3.141592654...
     !   Step 4  Finally, find the three roots
     !           x = y - b/a/3
     !
     ! ----------------------------------------------------------------------
     ! Instructor: Nam Sun Wang
     ! ----------------------------------------------------------------------

     ! Declare variables

     ! Step 0: If a is 0 use the quadrati! formula. -------------------------
     IF(a .eq. 0d0)THEN
        if(b .eq. 0d0)then
           if(c .eq. 0d0)then
              !         We have a non-equation; therefore, we have no valid solution
              nroot = 0
           else
              !         We have a linear equation with 1 root.
              nroot = 1
              Getroots(1) = cmplx(-d/c, 0d0)
           endif
        else
           !     We have a true quadrati! equation.  Apply the quadrati! formula to find two roots.
           nroot = 2
           DD = c*c-4d0*b*d
           if(DD .ge. 0d0)then
              Getroots(1) = cmplx((-c+sqrt(DD))/2d0/b, 0d0)
              Getroots(2) = cmplx((-c-sqrt(DD))/2d0/b, 0d0)
           else
              Getroots(1) = cmplx(-c/2d0/b, +sqrt(-DD)/2d0/b)
              Getroots(2) = cmplx(-c/2d0/b, -sqrt(-DD)/2d0/b)
           endif
        endif

     ELSE

        ! Cubi! equation with 3 roots
        nroot = 3

        ! Step 1: Calculate p and q --------------------------------------------
        p  = c/a - b*b/a/a/3d0
        q  = (2d0*b*b*b/a/a/a - 9d0*b*c/a/a + 27d0*d/a) / 27d0

        ! Step 2: Calculate DD (discriminant) ----------------------------------
        DD = p*p*p/27d0 + q*q/4d0

        ! Step 3: Branch to different algorithms based on DD ------------------

        if(DD .lt. 0d0)then
           !       Step 3b:
           !       3 real unequal roots -- use the trigonometri! formulation
           phi = acos(-q/2d0/sqrt(abs(p*p*p)/27d0))
           temp1=2d0*sqrt(abs(p)/3d0)
           y1 =  temp1*cos(phi/3d0)
           y2 = -temp1*cos((phi+pi)/3d0)
           y3 = -temp1*cos((phi-pi)/3d0)
           temp1 = b/a/3d0
           y2 = y2-temp1
           y3 = y3-temp1

        else
           !       Step 3a:
           !       1 real root & 2 conjugate complex roots OR 3 real roots (some are equal)
           temp1 = -q/2d0 + sqrt(DD)
           temp2 = -q/2d0 - sqrt(DD)
           u = abs(temp1)**(1d0/3d0)
           v = abs(temp2)**(1d0/3d0)
           if(temp1 .lt. 0d0) u=-u
           if(temp2 .lt. 0d0) v=-v
           y1  = u + v
           y2r = -(u+v)/2d0
           y2i =  (u-v)*sqrt(3d0)/2d0
           temp1 = b/a/3d0
           y2r=y2r-temp1


        endif

        ! Step 4: Final transformation -----------------------------------------

        y1 = y1-temp1

        ! Assign answers -------------------------------------------------------
        if(DD .lt. 0d0)then
           Getroots(1) = cmplx( y1,  0d0)
           Getroots(2) = cmplx( y2,  0d0)
           Getroots(3) = cmplx( y3,  0d0)
        elseif(DD .eq. 0d0)then
           Getroots(1) = cmplx( y1,  0d0)
           Getroots(2) = cmplx(y2r,  0d0)
           Getroots(3) = cmplx(y2r,  0d0)
        else
           Getroots(1) = cmplx( y1,  0d0)
           Getroots(2) = cmplx(y2r, y2i)
           Getroots(3) = cmplx(y2r,-y2i)
        endif

     ENDIF

  end FUNCTION GetRoots



  FUNCTION JeansDensity(r, temp)
     REAL(KIND=qPREC) :: r, temp, JeansDensity
     JeansDensity=gamma*temp/r**2*pi/ScaleGrav
  END FUNCTION JeansDensity


  SUBROUTINE GetChildPos(n_children,childpos,r)
     INTEGER :: n_children
     REAL(KIND=qPREC), DIMENSION(:,:) :: childpos
     REAL(KIND=qPREC) :: r
     INTEGER, PARAMETER :: nIterations=10000
     LOGICAL :: lSuccess
     INTEGER :: j,k,l
     REAL :: a
     !create childpositions where each child is at least 2r apart and within (1-r) of the center...
     childpos=0
     DO j=1,nIterations
        lSuccess=.true.
        DO k=1,n_children
           CALL Random(a)
           childpos(k,1)=a
           CALL Random(a)
           childpos(k,2)=a
           IF (nDim == 3) THEN
              CALL Random(a)
              childpos(k,3)=a
           END IF
!           write(*,*) 'try', Childpos(k,:)
           Childpos(k,1:nDim)=2d0*(childpos(k,1:nDim)-.5d0)*(1d0-r) !rescale to be within a random cube with half width (1-r)                      
!           write(*,*) 'trying', Childpos(k,:)
           IF (sqrt(sum(childpos(k,1:nDim)**2)) > (1d0-r)) THEN
!              write(*,*) 'too close to boudnary'
              lSuccess=.false.
           ELSE
              DO l=1,k-1
                 IF (sqrt(sum((childpos(k,1:nDim)-childpos(l,1:nDim))**2)) < 2d0*r) THEN
                    lSuccess=.false.
!                    write(*,*) 'too close to sibling'
                    EXIT
                 END IF
              END DO
           END IF
           IF (.Not. lSuccess) EXIT
        END DO
        IF (lSuccess) EXIT
     END DO
     IF (.NOT. lSuccess) THEN
        IF (MPI_ID == 0) write(*,*) 'failed to place', n_children, 'clumplets of radius ',r
        
        STOP
     END IF

  END SUBROUTINE GetChildPos




END MODULE Problem

