!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module GravoTurbulence is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir GravoTurbulence
!! @brief Contains files necessary for module GravoTurbulence

!> @file problem.f90
!! @brief Main file for module GravoTurbulence

!> @defgroup GravoTurbulence Gravitational Turbulence
!! @brief Module for setting up the advection of a field loop across the grid
!! @ingroup Modules

!> Module for setting up the advection of a field loop across the grid
!! @ingroup GravoTurbulence

MODULE Problem

   USE DataDeclarations
   USE GlobalDeclarations
   USE PhysicsDeclarations
   USE Ambients
   USE Perturbation
   USE CoolingSrc
   USE Clumps
   USE Totals
   USE PDFs
   USE Projections
   USE Fields
   USE Histograms
   USE ParticleDeclarations
   USE Shapes
   USE EOS
   USE LayoutDeclarations
   USE LayoutControl
   USE ObjectControl
   USE ProcessingDeclarations
   IMPLICIT NONE
   SAVE
   PRIVATE

   PUBLIC ProblemModuleInit, ProblemGridInit, &
        ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
   REAL(KIND=qPREC), DIMENSION(0:MaxDepth) :: cells_per_cooling_length=0
   TYPE(CoolingDef),POINTER :: coolingobj
   REAL(KIND=qPrec), PARAMETER :: Y=2d-26
   !   REAL(KIND=qPREC), DIMENSION(3), PARAMETER :: offset=(/4d0,-4d0,0d0/)/sqrt(2d0)
   TYPE(ClumpDef), POINTER :: clump
   TYPE(AmbientDef), POINTER :: Ambient
   TYPE(LayoutDef), POINTER :: Layout
   REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: data
   INTEGER :: RemapLevel
CONTAINS

   !> Initializes module variables
   SUBROUTINE ProblemModuleInit
      REAL(KIND=qPREC) :: rho, p
      INTEGER :: i, nDwaves, nVwaves, kmin, kmax, nclumps
      REAL(KIND=qPREC) :: wavevector(3), amplitude, phase, amplitudes(3), phases(3), JeansLength, temp, density_amp, velocity_amp, beta,chi,pos(3),radius, pmass, scale, t_ff, t_c, velocity_factor, mass_factor, GE, KE, TE, mass_clump, alpha_virial=0d0, thickness
      LOGICAL :: lCooling
      TYPE(particledef), POINTER :: particle
      TYPE(HistogramDef), POINTER :: HISTOGRAM
      TYPE(PDFDef), POINTER :: PDF
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: nbins=100
      INTEGER :: nParticleCells=0
      NAMELIST /ProblemData/ rho, temp, lCooling, kmin,kmax,beta, density_amp, velocity_amp, cells_per_cooling_length,chi,pos,radius, pmass, scale, nParticleCells, nbins, velocity_factor, mass_factor, alpha_virial, CellsPerJeansLength, RemapLevel, thickness
      NAMELIST /WaveData/ wavevector, amplitude, phase
      NAMELIST /vWaveData/ wavevector, amplitudes, phases


      !      CALL AddTotalVar(TotalMass)
      !           CALL AddTotalVar(1, 'mymass')

      CALL AddAllTotals(GASCOMP)
      CALL AddAllTotals(PARTICLECOMP)
      CALL AddAllTotals(BOTHCOMP)

      CALL AddDiagnosticVar(MPI_ID_FIELD)
      CALL AddDiagnosticVar(ErrFlag_FIELD)

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data')     
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)    

      IF (nParticleCells > 0) THEN
         DefaultParticleRadius=nParticleCells
      END IF

      CALL CreateHistogram(Histogram)
      Histogram%Field%iD=1
      !    Histogram%Field%name=''
      Histogram%Field%component=GASCOMP
      Histogram%minvalue=.1d0
      Histogram%maxvalue=1d8
      Histogram%nbins=nbins
      Histogram%scale=LOGSCALE

      CALL CreatePDF(PDF)
      PDF%Field(1)%iD=Mass_Field
      PDF%Field(1)%name='density'
      PDF%Field(1)%component=GASCOMP
      PDF%Field(2)%iD=P_Field
      PDF%Field(2)%name='pressure'
      PDF%Field(2)%component=GASCOMP
      PDF%minvalue=(/.01d0,100d0/)
      PDF%maxvalue=(/1d7,1d6/)
      PDF%nbins=(/400,400/)
      PDF%Scale=(/LOGSCALE,LOGSCALE/)
      PDF%WeightField=BINBYVOLUME

      CALL CreatePDF(PDF)
      PDF%Field(:)%iD=(/Mass_Field, P_Field/)
      PDF%Field(1)%name='density'
      PDF%Field(2)%name='pressure'
      PDF%Field(:)%component=GASCOMP
      !    PDF%Field(2)%name='pressure'
      PDF%minvalue=(/.01d0,100d0/)
      PDF%maxvalue=(/1d7,1d6/)
      PDF%nbins=(/400,400/)
      PDF%Scale=(/LOGSCALE,LOGSCALE/)
      PDF%WeightField=BINBYMASS


      CALL CreateProjection(projection)
      Projection%Field%iD=Mass_Field
      Projection%Field%component=BOTHCOMP
      Projection%dim=3
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE

      CALL CreateProjection(projection)
      Projection%Field%iD=Mass_Field
      Projection%Field%component=BOTHCOMP
      Projection%dim=1
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE

      CALL CreateProjection(projection)
      Projection%Field%iD=Mass_Field
      Projection%Field%component=BOTHCOMP
      Projection%pow=1d0
      Projection%dim=2
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE

      !!    CALL CreateProjection(projection)
      !    Projection%Field%iD=CoolingStrength_Field
      !    Projection%Field%component=GASCOMP
      !    Projection%pow=1d0
      !    Projection%dim=3
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE

      !    CALL CreateProjection(projection)
      !    Projection%Field%iD=CoolingStrength_Field
      !    Projection%Field%component=GASCOMP
      !    Projection%pow=1d0
      !    Projection%dim=2
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE

      !    CALL CreateProjection(projection)
      !    Projection%Field%iD=CoolingStrength_Field
      !    Projection%Field%component=GASCOMP
      !    Projection%pow=1d0
      !    Projection%dim=1
      !    ALLOCATE(Projection%Image)
      !    Projection%Image%Scaling=LOGSCALE




      NULLIFY(Ambient)
      CALL CreateAmbient(Ambient)
      IF (lCooling) THEN
         IF (.NOT. lRestart) THEN
            CALL CreateCoolingObject(coolingobj)
         ELSE
            coolingobj => firstcoolingobj
         END IF
         coolingobj%iCooling=IICOOL
         coolingobj%floortemp=1d0
         coolingobj%mintemp=0.001
      END IF
      IF (iE /= 0 .AND. temp == 0) THEN
         IF (lCooling) THEN
            CALL InitIICool(coolingobj)
            temp=GetIICoolEqTemp(rho)
            !         Ambient%Pressure=Ambient%density*temp/TempScale
            IF (MPI_ID ==0) write(*,*) 'Equilibrium temp (K)=', temp
         ELSE
            IF (MPI_ID == 0) write(*,*)  'Temperature must be specified if cooling is not used'
            STOP
         END IF
      END IF
      IF (iE == 0) temp=Iso_Speed2*TempScale
      Ambient%density=rho/chi
      IF (iE /= 0) Ambient%pressure=rho*temp/TempScale !Ambient%density*GetIICoolEqTemp(Ambient%density)/TempScale
      IF (MPI_ID == 0) THEN
         !       IF (iE /= 0) write(*,*) 'Ambient temp=', GetIICoolEqTemp(Ambient%density)
         IF (iE /= 0) write(*,*) 'clump temp=', temp
      END IF

      IF (lSelfGravity) THEN
         JeansLength=sqrt(gamma*temp/TempScale*Pi/ScaleGrav/rho)
         mass_clump=4d0/3d0*mass_factor*pi*radius**3*rho
         GE=3d0/5d0*mass_clump**2*ScaleGrav/radius
         TE=mass_clump*temp/TempScale*3/2
         IF (alpha_virial /= 0) THEN
            KE=alpha_virial*GE/2-TE
            IF (KE < 0) THEN
               write(*,*) 'virial parameter too low given thermal energy'
               STOP
            END IF
            velocity_amp=sqrt(2d0*KE/mass_clump)/velocity_factor
            !          KE=.5*mass_clump*(velocity_amp*velocity_factor)**2
            !          alpha_virial=2*(KE+TE)/GE         
            IF (MPI_ID == 0) write(*,*) 'velocity_amp=', velocity_amp
         ELSE
            KE=.5*mass_clump*(velocity_amp*velocity_factor)**2
            alpha_virial=2*(KE+TE)/GE

         END IF
         IF (MPI_ID == 0) THEN
            write(*,'(A30,2A25)') 'Quantity: ', 'Computationl Units', 'Astro Units'
            write(*,'(A30,2E25.16)') 'Temperature of sphere', temp/TempScale, temp
            write(*,'(A30,2E25.16)') 'sound speed', sqrt(gamma*temp/TempScale), sqrt(gamma*temp/TempScale)*VelScale/1e5
            write(*,'(A30,2E25.16)') 'free fall time', sqrt(3*pi/32/ScaleGrav/rho), sqrt(3*pi/32/ScaleGrav/rho)*TimeScale/yr/1e6
            write(*,'(A30,2E25.16)') 'Radius of sphere', radius, radius*lScale/pc
            write(*,'(A30,2E25.16)') 'Mass of sphere', mass_clump, mass_clump*mScale/msolar
            WRITE(*,'(A30,2E25.16)') 'Jeans Length ', JeansLength, JeansLength*lScale/pc
            WRITE(*,'(A30,2E25.16)') 'Jeans Mass ', 4d0/3d0*pi*(JeansLength/2d0)**3*rho, 4d0/3d0*pi*(JeansLength/2d0)**3*rho*mScale/mSolar
            WRITE(*,'(A30,2E25.16)') 'Gravitational Energy',  GE, GE*pScale*lScale**3
            write(*,'(A30,2E25.16)') 'Kinetic Energy of sphere = ', KE, KE*pScale*lScale**3
            write(*,'(A30,2E25.16)') 'Thermal Energy of sphere = ', TE, TE*pScale*lScale**3
            write(*,'(A30,2E25.16)') 'Velocity dispersion= ', (KE+TE)/mass_clump, (KE+TE)/mass_clump*velScale/1e5
            write(*,*) 'Virial parameter = ', 2*(KE+TE)/GE
         END IF

      END IF


      DO i=0,-1
         rho=10d0**(.25*REAL(i)-1d0)
         IF (iE /= 0) THEN
            temp=GetIICoolEqTemp(rho)
         ELSE
            temp=Iso_Speed2*TempScale
         END IF
         JeansLength=sqrt(gamma*temp/TempScale*Pi/ScaleGrav/rho)  !pc
         t_ff=sqrt(3d0*pi/32d0/ScaleGrav/rho)*TimeScale/yr/1e6 !myr
         t_c=rho*temp/max(abs(rho*nScale*Y * (rho*nScale*IICoolingRate(temp*TempScale) ) * coolingobj%ScaleCool * gamma1 ),1d-6)*TimeScale/yr/1e6

         write(*,'(8E24.15)') rho, temp, JeansLength, t_ff, t_c
      END DO
      !      STOP
      IF (pMass > 0) THEN
         CALL CreateParticle(particle)
         Particle%Q(1)=pMass
         Particle%xloc=pos !+offset
         Particle%radius=1
         CALL AddSinkParticle(particle)
         CALL CreatePointGravityObject(particle%PointGravityObj)
         particle%PointGravityObj%mass=pMass
         particle%PointGravityObj%x0(1:nDim)=particle%xloc
         particle%PointGravityObj%soft_length=4d0*sink_dx !particle%radius*sink_dx
         particle%PointGravityObj%soft_function=SPLINESOFT
         particle%iAccrete=FEDERRATH_ACCRETION
         !         NULLIFY(particle)
         !         CALL CreateParticle(particle)
         !         Particle%Q(1)=pMass
         !         Particle%xloc=pos-offset
         !         Particle%radius=1
         !         CALL AddSinkParticle(particle)

      ELSE
         CALL CreateClump(clump)
         clump%radius=radius
         clump%density=rho
         !         clump%pressure=rho*temp/TempScale
         clump%temperature=temp/TempScale
         clump%position=pos!+offset
         clump%subsample=0
         clump%thickness=thickness
         CALL UpdateClump(clump)
         !          NULLIFY(Clump)
         !          CALL CreateClump(clump)
         !          clump%radius=radius
         !          clump%density=rho
         !         clump%pressure=rho*temp/TempScale
         !          clump%temperature=temp/TempScale
         !          clump%position=pos-offset
         !          CALL UpdateClump(clump)
      END IF

      !      IF (nDwaves > 0) THEN
      !         CALL CreatePerturbationObj(Ambient%DensityPerturbation)
      !         Ambient%DensityPerturbation%type=COSINESERIES
      !         CALL InitPerturbationWaves(Ambient%DensityPerturbation, nDwaves)
      !         DO i=1,nDwaves
      !            READ(PROBLEM_DATA_HANDLE, NML=WaveData)
      !            CALL AddPerturbationWave(Ambient%DensityPerturbation, wavevector*2d0*Pi/max(GxBounds(:,2)-GxBounds(:,1),1!e-6), phase, amplitude)
      !         END DO
      !      END IF
      !    IF (density_amp > 0d0) THEN
      !       CALL CreatePerturbationObj(Clump%DensityPerturbation)        
      !       CALL CreateSpectra(Clump%DensityPerturbation, kmin, kmax, beta, density_amp, scale)
      !    END IF
      !    IF (velocity_amp > 0) THEN
      !       CALL CreateVectorPerturbationObj(Clump%VelocityPerturbation)
      !       CALL CreateSolenoidalSpectra(Clump%VelocityPerturbation, kmin, kmax, beta, velocity_amp, scale)
      !    END IF

      !      IF (nVwaves > 0) THEN
      !         CALL CreateVectorPerturbationObj(Ambient%VelocityPerturbation)
      !         Ambient%VelocityPerturbation%type=COSINESERIES
      !         CALL InitVectorPerturbationWaves(Ambient%VelocityPerturbation, nVwaves)
      !         DO i=1,nVwaves
      !            READ(PROBLEM_DATA_HANDLE, NML=VWaveData)
      !            CALL AddVectorPerturbationWave(Ambient%VelocityPerturbation, wavevector*2d0*Pi/max(GxBounds(:,2)-GxBounds!(:,1),1e-6), phases, amplitudes)
      !         END DO
      !      END IF
      !      write(*,*) VectorPerturbationValue(Ambient%VelocityPerturbation,(/.1d0,.1d0,.1d0/))

      CLOSE(PROBLEM_DATA_HANDLE)

   END SUBROUTINE ProblemModuleInit

   !> Initial Conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      !! @brief Initializes the grid data according to the requirements of the problem.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
      REAL(KIND=qPREC) :: p, f, rho, vel(3), pos(3)
      INTEGER :: i,j,k
      IF (levels(Info%level)%tnow == start_Time .AND. lRestart) THEN
         IF (Info%level < RemapLevel) THEN
!            CALL AmbientGridInit(Info)
!            CALL ClumpGridInit(Info)          
            CALL ObjectsGridInit(Info,AMBIENTOBJ)
            CALL ObjectsGridInit(Info,CLUMPOBJ)
         ELSE
            DO i=1,Info%mX(1)
               DO j=1,Info%mx(2)
                  DO k=1,Info%mx(3)
                     IF (IsInShape(Clump%Shape, CellPos(Info, i, j, k), pos)) THEN
                        f=smooth_tanh(sqrt(sum(pos**2))/Clump%Shape%size_param(1),Clump%thickness)
                        p=Press(Info%q(i,j,k,:))*f+Ambient%pressure*(1d0-f)
                        rho=Info%q(i,j,k,1)*f+Ambient%density*(1d0-f)
                        vel(1:nDim)=Info%q(i,j,k,imom(1:nDim))/Info%q(i,j,k,1)*f
                     ELSE
                        rho=Ambient%density
                        p=Ambient%pressure
                        vel(1:nDim)=0d0
                     END IF

                     Info%q(i,j,k,1)=rho
                     Info%q(i,j,k,imom(1:nDim))=rho*vel(1:nDim)
                     IF (iE /= 0d0) Info%q(i,j,k,iE)=gamma7*p+half*sum(Info%q(i,j,k,imom(1:nDim)**2))/Info%q(i,j,k,1)
                  END DO
               END DO
            END DO
         END IF
      END IF
   END SUBROUTINE ProblemGridInit

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      !! @brief Performs any tasks required before the advance step.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
   END SUBROUTINE ProblemBeforeStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemAfterStep(Info)
      !! @brief Performs any post-step corrections that are required.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      !! @brief Sets error flags according to problem-specific conditions..
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
      REAL(KIND=qPREC) :: P, T, cooling_length, temp, pos(3), dx, dz, duh(3), cool_rate, x, dist_from_boundary
      INTEGER :: i,j,k, mO(3,2)

      IF (cells_per_cooling_length(Info%level) > 0d0) THEN
         temp=levels(Info%level)%dx*cells_per_cooling_length(Info%level)
         DO i=1,Info%mX(1)
            DO j=1,Info%mX(2)
               DO k=1,Info%mX(3)
                  P=Press(Info%q(i,j,k,:))
                  T=TempScale*P/Info%q(i,j,k,1)
                  cool_rate=max(abs(Info%q(i,j,k,1)*nScale*Y * (Info%q(i,j,k,1)*nScale*IICoolingRate(T) ) * coolingobj%ScaleCool * gamma1 ),1d-6)
                  cooling_length=sqrt(gamma*P/Info%q(i,j,k,1))*(P/cool_rate)
                  IF (cooling_length < temp) Info%ErrFlag(i,j,k)=1
               END DO
            END DO
         END DO
      END IF
      
      IF (levels(Info%level)%tnow == start_time .AND. lRestart) THEN !remap cloud...
         mO=MapBoxToInfo(Clump%Shape%xBounds, Info, 0)
         Info%ErrFlag(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=1
      END IF
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n,i
      INTEGER :: mB(3,2)
      INTEGER, DIMENSION(:), ALLOCATABLE :: FieldID
      TYPE(NodeDefList), POINTER :: nodelist
      LOGICAL, SAVE :: lFirstTime=.true.
      IF (lFirstTime) THEN
         IF (iE /= 0) THEN
            ALLOCATE(FieldID(5))
            FieldID=(/(i,i=1,5)/)
         ELSE
            ALLOCATE(FieldID(4))
            FieldID=(/(i,i=1,4)/)
         END IF

         IF (levels(n)%tnow == start_Time .AND. lRestart .AND. n == 0) THEN
            CALL CreateLayout(GmGlobal, layout)
            layout%level = 0
            mB=layout%mB(MPI_ID,:,:)
            ALLOCATE(data(mB(1,1):mB(1,2), mB(2,1):mB(2,2),mB(3,1):mB(3,2),4))
            CALL LoadFieldIntoLayout(layout, data, FieldID)
            CALL StoreLayout(layout, data, 'ISO512')
            lFirstTime=.false.
         ELSEIF (levels(n)%tnow == start_time .AND. lRestart .AND. n == RemapLevel) THEN
            layout%level = RemapLevel
            layout%mB(:,1:nDim,:)=layout%mB(:,1:nDim,:)+spread(spread(levels(RemapLevel)%mX(1:nDim)/2-GmX(1:nDim)/2,1,MPI_NP), 3, 2) !shift layout to center of level 1 grid
            CALL UnloadFieldFromLayout(layout, data, FieldID, lHydroPeriodic, levels(n)%gmbc(1))
            CALL DestroyLayout(layout)
            DEALLOCATE(data)
            lFirstTime=.false.
         END IF

         nodelist=>Nodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            CALL ProblemGridInit(nodelist%self%info)
            nodelist=>nodelist%next
         END DO
         DEALLOCATE(FieldID)
      END IF
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem
