!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module HydroStaticStar is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir HydroStaticStar 
!! @brief Contains files necessary for the Hydrostatic Equilibrium calculations

!> @file problem.f90
!! @brief Main file for module Problem
MODULE Problem
  USE DataDeclarations
  USE Ambients
  USE PointGravitySrc
  USE ParticleDeclarations !n
  USE Profiles
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(AmbientDef), POINTER :: Ambient       
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
      INTEGER :: i, soft_function, nPoints
      TYPE(ParticleDef), POINTER :: Particle => null()
      TYPE(PointGravityDef), POINTER :: PointGravityObj => null()
      REAL(KIND=qPREC) :: mass=1d0
      REAL(KIND=qPREC) :: pressure_inf, soft_radius       
      REAL(KIND=qPREC), DIMENSION(3) :: xloc, velocity= (/0,0,0/)  
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: temp
      NAMELIST /ProblemData/ velocity, xloc, mass, soft_radius, soft_function, pressure_inf

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CALL CreateAmbient(Ambient,1d0,1d0)
      Ambient%PersistInBoundaries=.true.

      IF (.NOT. lRestart) THEN
         CALL CreateParticle(Particle) !n
         Particle%q(1)=mass*mSolar/mScale !n
         Particle%xloc=xloc !n
         CALL CreatePointGravityObject(Particle%PointGravityObj)
         Particle%lFixed=.true. !n
         PointGravityObj=>Particle%PointGravityObj
      ELSE 
         ALLOCATE(PointGravityObj)
      ENDIF

      PointGravityObj%mass=mass*mSolar/mScale
      PointGravityObj%x0=xloc
      PointGravityObj%v0=velocity
      PointGravityObj%t0=levels(0)%tnow
      PointGravityObj%soft_length=soft_radius
      PointGravityObj%soft_function=soft_function

      CLOSE(PROBLEM_DATA_HANDLE)

      !Open file containing density profile
      open(UNIT=PROBLEM_DATA_HANDLE, FILE='amb.data', STATUS='old', FORM='FORMATTED')     
      READ(PROBLEM_DATA_HANDLE, *) nPoints
      
      !Profile object is allocated
      CALL CreateProfile(Ambient%profile, nPoints, (/Mass_Field, P_Field/), RADIAL)
      DO i=1,nPoints
         READ(PROBLEM_DATA_HANDLE, *) Ambient%profile%data(i,:) ! position, density, pressure in cgs
         Ambient%profile%data(i,:)=Ambient%profile%data(i,:) / (/lScale, rScale, pScale/) !rescale to cu
      END DO

      !Now soften the density profile using Plummer type softening with a soft radius that is 80% of the pointgravity soft_length
      ALLOCATE(temp(NPoints))
      DO i=1,nPoints
         temp(i) = getProfileValue(sqrt(Ambient%profile%data(i,1)**2 & 
                 + (PointGravityObj%soft_length*0.8d0)**2),Mass_Field,Ambient%profile)
      END DO
      Ambient%profile%data(:,2)=temp
      DEALLOCATE(temp)

      CLOSE(PROBLEM_DATA_HANDLE)
 
      ! Density profile is now populated
      ! Calculate pressure profile needed for hydrostatic equilibrium
      CALL Profile_PointGravityHSE(Ambient%profile, PointGravityObj, pressure_inf)


  END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
 END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
      INTEGER :: i,j,k
      Info%ErrFlag(:,:,:)=1  ! refines over entire grid
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

