!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module IsoMHDWaves is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir IsoMHDWaves
!! @brief Contains files necessary for the IsoMHDWaves module

!> @file problem.f90
!! @brief Main file for module IsoMHDWaves

!> @defgroup IsoMHDWaves Isothermal MHD Waves 
!! @brief Module for setting up hydro waves
!! @ingroup Modules

!> IsoMHDWaves Module
!! @ingroup IsoMHDWaves
MODULE Problem
  USE DataDeclarations
  USE SplitRegions
  USE Shapes
  USE EOS
  USE Ambients
  USE RiemannSolvers
  USE Totals
  USE Fields
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  PRIVATE
  INTEGER, DIMENSIOn(:), ALLOCATABLE :: map
  REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: dq, q, w, dw
  REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: lambda, A, Lamb, Am, An, qExact
  REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: l, r
  REAL(KIND=qPREC) :: wave_speed
  REAL(KIND=qPREC) :: rho, p, vx, vy, vz, Bx, By, Bz, WaveStrength, wavespeed
  INTEGER :: WaveFamily
  
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      NAMELIST /ProblemData/ rho, p, vx, vy, vz, Bx, By, Bz, WaveFamily, WaveStrength
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)

   END SUBROUTINE ProblemModuleInit

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j
      LOGICAL :: lFirstTime=.true.
      LOGICAL :: request_eigens
      INTEGER :: n(3)
      REAL(KIND=qPREC) :: left_fact, right_fact
      SAVE
      IF (lFirstTime) THEN
         lFirstTime=.false.
         IF (MPI_ID == 0) write(*,*) NrWaves, NrHydroVars
         ALLOCATE(w(1:NrHydroVars), dw(1:NrHydroVars), q(1:NrHydroVars),dq(1:NrHydroVars))
         ALLOCATE (r(nDim,NrWaves,NrWaves),l(nDim,NrWaves,NrWaves),lambda(nDim,NrWaves))
         ALLOCATE (A(NrHydroVars, NrHydroVars), Lamb(NrWaves, NrWaves), Am(NrWaves, NrWaves),An(NrWaves, NrWaves))
         w(1)=rho
         w(ivx)=vx
         IF (ivy /= 0) w(ivy)=vy
         IF (ivz /= 0) w(ivz)=vz
         IF (iE /= 0) w(iE)=P
         IF (iBx /= 0) w(iBx)=Bx
         IF (iBy /= 0) w(iBy)=By
         IF (iBz /= 0) w(iBz)=Bz
         write(*,*) 'sounds speed = ', sqrt(gamma*P/rho)
         write(*,*) w
         CALL calc_eigens(request_eigens, w, (/.true.,.true.,.true./),lambda, n, l, r,1,1,1,0)

         ALLOCATE(map(NrWaves))
         DO i=1,NrWaves
            map(i)=onedx_i(i)
            IF (lMHD .and. onedx_i(i) > iBx) map(i)=map(i)-1
         END DO
         write(*,*) 'map = ', map
         wavespeed=lambda(1,WaveFamily)
         write(*,*) 'wavespeeds', lambda(1,:)
         write(*,*) 'wave speed=', wavespeed
         write(*,*) 'loop should take ', (GxBounds(1,2)-GxBounds(1,1))/wavespeed
!         left_fact=rho/sqrt(2d0*gamma*P/rho) / 0.1118033988749895
!         right_fact=1d0/left_fact
!         l=l*left_fact
!         r=r*right_fact
         Lamb=0
         DO i=1,n(1)
            lamb(i,i)=lambda(1,i)
         END DO

         IF (MPI_ID == 0) THEN
            write(*,*) 'found ', n(1), 'waves'
           
            write(*,*) 'left eigen vectors are rows of'
            CALL OutputDoubleArray(transpose(l(1,1:n(1),map(:))))
            
            
            write(*,*) 'right eigen vectors are columns of'
            CALL OutputDoubleArray(r(1,1:n(1),map(:)))
            
            write(*,*) 'Check that L x R = I'
            CALL OutputDoubleArray(matmul(transpose(l(1,1:n(1),:)), r(1,1:n(1),:)))
            


            ! L x A x R = Lam
            ! L(-1) x Lam x R(-1) = R x Lam x L
            Am=matmul(matmul(transpose(l(1,1:n(1),:)), lamb(1:n(1),1:n(1))),r(1,1:n(1),:))
            write(*,*) 'Reconstruct matrix A'
            CALL OutputDoubleArray(Am)
            An=Am
            DO i=1,size(OneDx_i)
               An(:,map(i))=Am(:,i)
            END DO
            DO i=1,size(OneDx_i)
               Am(map(i),:)=An(i,:)
            END DO
            write(*,*) 'Reconstruct matrix A in normal index'
            CALL OutputDoubleArray(Am)
         END IF
!         STOP
      END IF
      dw=0d0
      DO j=1,NrWaves
         dw(oneDx_i(j))=r(1,WaveFamily, j)
      END DO
      write(*,*) 'dw=', dw
!      STOP
      dq=w+dw*1d-6
      CALL prim_to_cons(dq)
      q=w
      CALL prim_to_cons(q)
      write(*,*) 'dq=', (dq-q)/1d-6
!      STOP
      DO i=1, Info%mX(1)
         Info%q(i,1,1,1:size(w)) = w+dw*WaveStrength*sin(2d0*Pi*real(i)/real(Info%mX(1)))
         CALL prim_to_cons(Info%q(i,1,1,:))
      END DO
      IF (MaintainAuxArrays) THEN
         IF (Info%mx(1) > Info%mX(2)) Info%aux(1:Info%mX(1),1:Info%mx(2)+1,1,2) = SPREAD(Info%q(1:Info%mX(1),1,1,iBy),2,Info%mX(2)+1)
         IF (Info%mX(2) > Info%mX(1)) Info%aux(1:Info%mX(1)+1,1:Info%mx(2),1,1) = SPREAD(Info%q(1,1:Info%mX(2),1,iBx),1,Info%mX(1)+1)
      END IF
!      STOP
   END SUBROUTINE ProblemGridInit

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: rmbc,i,j
      REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: tempaux
      rmbc=levels(Info%level)%gmbc(1)
      IF (lMHD .AND. nDim == 2 .AND. Info%mX(1) == Info%mX(2)) THEN !Angled aux fields -  need to calculate potential
         ALLOCATE(tempaux(1:Info%mX(1)+1,1:Info%mX(2)+1,2))
         tempaux=Info%aux(1:Info%mX(1)+1,1:Info%mX(2)+1,1,1:2)

         DO j=-rmbc, Info%mX(2)+rmbc
            Info%aux(1-rmbc:j,Info%mx(2)+1-j,1,1)=Info%q(1,1,1,iBx)
            Info%aux(j+1:Info%mX(1)+1+rmbc,Info%mx(2)+1-j,1,1)=Info%q(Info%mX(1),Info%mX(2),1,iBx)
         END DO
         DO i=1-rmbc, Info%mX(1)+rmbc
            Info%aux(info%mx(1)+1-i,1-rmbc:i,1,2)=Info%q(1,1,1,iBy)
            Info%aux(info%mx(1)+1-i,i+1:Info%mX(2)+1+rmbc,1,2)=Info%q(Info%mX(1),Info%mX(2),1,iBy)
         END DO
         Info%aux(1:Info%mX(1)+1,1:Info%mX(2),1,1)=tempaux(1:Info%mX(1)+1,1:Info%mX(2),1)
         Info%aux(1:Info%mX(1),1:Info%mX(2)+1,1,2)=tempaux(1:Info%mX(1),1:Info%mX(2)+1,2)

         Info%q(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,iBx)=half*(Info%aux(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,1)+Info%aux(2-rmbc:Info%mX(1)+1+rmbc,1-rmbc:Info%mX(2)+rmbc,1,1))
         Info%q(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,iBy)=half*(Info%aux(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,2)+Info%aux(1-rmbc:Info%mX(1)+rmbc,2-rmbc:Info%mX(2)+1+rmbc,1,2))
         DEALLOCATE(tempaux)
      END IF

   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: err
      REAL(KIND=qPREC) :: phase
      ALLOCATE(err(1:NrHydroVars))
      IF (levels(info%level)%tnow+levels(info%level)%dt == final_time) THEN
         phase=-(2d0*Pi*(final_time*wavespeed)/(GxBounds(1,2)-GxBounds(1,1)))
         write(*,*) 'phase = ', phase
         OPEN(UNIT=11, FILE='out/data.curve', status='unknown')
         write(11,*) '# rho'         
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,1)
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,1)
         END DO
         write(11,*) 
         write(11,*) 
         write(11,*) '# rho_Exact'         
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(1)+dw(1)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(1)+dw(1)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
         END DO
         write(11,*) 
         write(11,*) 


         write(11,*) '# vx'
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivx)/Info%q(i,1,1,1)
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivx)/Info%q(i,i,1,1)
         END DO
         write(11,*) 
         write(11,*) 

         write(11,*) '# vx_Exact'         
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(ivx)+dw(ivx)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(ivx)+dw(ivx)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
         END DO
         write(11,*) 
         write(11,*) 

         write(11,*) '# P'
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, (Press(Info%q(i,1,1,:)))
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), (Press(Info%q(i,i,1,:)))
         END DO
         write(11,*) 
         write(11,*) 
         write(11,*) '# P_Exact'         
         IF (iE /= 0) THEN
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(iE)+dw(iE)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(iE)+dw(iE)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
         ELSE
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Iso_Speed2*(w(1)+dw(1)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1))))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Iso_Speed2*(w(1)+dw(1)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1))))
            END DO
         END IF
         write(11,*) 
         write(11,*) 


         IF (ivy /= 0) THEN
            write(11,*) '# vy'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivy)/Info%q(i,1,1,1)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivy)/Info%q(i,i,1,1)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# vy_Exact'         
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(ivy)+dw(ivy)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(ivy)+dw(ivy)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
            write(11,*) 
            write(11,*) 
         END IF
         IF (ivz /= 0) THEN
            write(11,*) '# vz'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivz)/Info%q(i,1,1,1)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivz)/Info%q(i,i,1,1)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# vz_Exact'         
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(ivz)+dw(ivz)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(ivz)+dw(ivz)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
            write(11,*) 
            write(11,*) 

         END IF
         IF (iBx /= 0) THEN
            write(11,*) '# Bx'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBx)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBx)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# Bx_Exact'         
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(iBx)+dw(iBx)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(iBx)+dw(iBx)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
            write(11,*) 
            write(11,*) 

         END IF
         IF (iBy /= 0) THEN
            write(11,*) '# By'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBy)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBy)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# iBy_Exact'         
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(iBy)+dw(iBy)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(iBy)+dw(iBy)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
            write(11,*) 
            write(11,*) 

         END IF
         IF (iBz /= 0) THEN
            write(11,*) '# Bz'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBz)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBz)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# iBz_Exact'         
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, w(iBz)+dw(iBz)*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1)))
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), w(iBz)+dw(iBz)*WaveStrength*sin(phase+2d0*Pi*sqrt(2d0)*real(i)/real(Info%mX(1)))
            END DO
            write(11,*) 
            write(11,*) 

         END IF
         err=0d0
         DO i=1,Info%mX(1)
            CALL cons_to_prim(Info%q(i,1,1,:))
            err=err+abs(w+dw*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1))) - Info%q(i,1,1,1:NrHydroVars))
!            write(*,*) i,abs(w+dw*WaveStrength*sin(phase+2d0*Pi*real(i)/real(Info%mX(1))) - Info%q(i,1,1,1:NrHydroVars))
         END DO
         err=err/real(Info%mX(1))
         write(*,*) 'dx, error=', levels(Info%level)%dx, sum(err)
      END IF
   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

