!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module IsotropicTurbulence is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir IsotropicTurbulence
!! @brief Contains files necessary for the IsotropicTurbulence Calculation

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup IsotropicTurbulence Isotropic Turbulence Module
!! @brief Module for calculating collapse of a uniform cloud
!! @ingroup Modules

!> Isotropic Turbulence Module 
!! @ingroup IsotropicTurbulence
MODULE Problem
   USE DataDeclarations
   USE Ambients
   USE PFFT
   USE Clumps
   USE LayoutDeclarations
   USE Spectras
   USE Histograms
   USE PDFs
   USE Totals
   USE Projections
   USE CoolingSrc

   IMPLICIT NONE
   SAVE

   PUBLIC ProblemModuleInit, ProblemGridInit, &
        ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep

   INTEGER iReal1D, iImag1D, iRhoBack
   TYPE(PFFTPlanDef), POINTER :: Plan
   REAL(KIND=qPREC) :: beta=-1.5d0, Mach_turb=5d0, kmax=2d0, alpha_const=0d0, alpha, alpha_est, KE_Target, KE_avg, relaxation_time, KE=0d0
   TYPE(LayoutDef), POINTER :: layout
   REAL(8), DIMENSION(:,:,:,:), POINTER :: data
   INTEGER, DIMENSION(:), ALLOCATABLE :: FieldID
   TYPE(AmbientDef), POINTER :: Ambient
   REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: forcing, tempforcing
   TYPE(TotalDef), POINTER :: Total, PxTotal, PyTotal, PzTotal
   REAL(KIND=qPREC) :: density, pressure
   REAL(KIND=qPREC) :: t_start_avg, t_final_avg, t_cross, dv(3)
   TYPE(ProjectionDef), POINTER :: Projection
   LOGICAL :: lCooling
   TYPE(CoolingDef),POINTER :: coolingobj
CONTAINS

   !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
      REAL(KIND=qPREC) :: A(3),kvec(3),k2, v_turb
      COMPLEX(8) :: B(3)
      REAL :: rand
      INTEGER :: i,j,k,l
      TYPE(ClumpDef), POINTER :: Clump
      INTEGER, DIMENSION(:,:), ALLOCATABLE :: pos
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: val
      TYPE(SpectraDef), POINTER :: Spectra
      LOGICAL :: lUseExisting=.false.
      CHARACTER(len=5) :: str
      COMPLEX(8) :: data(3)
      INTEGER :: nwaves
      TYPE(HistogramDef), POINTER :: HISTOGRAM
      TYPE(PDFDef), POINTER :: PDF
      TYPE(ProjectionDef), POINTER :: Projection
      NAMELIST /ProblemData/ density, pressure, kmax, Mach_turb, alpha_const, lUseExisting, lCooling, IICoolPar

      IF (lRegrid) RETURN
      IF (.NOT. lPostProcess) THEN
         CALL CreateTotal(Total)
         Total%Field%Component=GASCOMP
         Total%Field%iD=KE_Field
         
         CALL CreateTotal(PxTotal)
         PxTotal%Field%Component=GASCOMP
         PxTotal%Field%iD=Px_Field
         
         CALL CreateTotal(PyTotal)
         PyTotal%Field%Component=GASCOMP
         PyTotal%Field%iD=Py_Field
         
         IF (nDim == 3) THEN 
            CALL CreateTotal(PzTotal)
            PzTotal%Field%Component=GASCOMP
            PzTotal%Field%iD=Pz_Field
         END IF
         
         CALL CreateSpectra(Spectra)         
         ALLOCATE(Spectra%Fields(nDim))
         IF (nDim == 3) THEN
            Spectra%Fields(:)%id=(/vx_Field, vy_Field, vz_Field/)
         ELSE
            Spectra%Fields(:)%id=(/vx_Field, vy_Field/)
         END IF
         Spectra%type=VECTOR_SPECT
         CALL CreateSpectra(Spectra)
         ALLOCATE(Spectra%Fields(1))
         Spectra%Fields(:)%id=(/Mass_Field/)
         Spectra%type=SCALAR_SPECT
         
         CALL CreateProjection(projection)
         Projection%Field%iD=Mass_Field
         Projection%Field%component=BOTHCOMP
         Projection%dim=3
      END IF


      CALL CreatePDF(PDF)
      PDF%Field(1)%iD=Mass_Field
      PDF%Field(1)%name='density'
      PDF%Field(1)%component=GASCOMP
      PDF%Field(2)%iD=VMag_Field
      PDF%Field(2)%name='velocity'
      PDF%Field(2)%component=GASCOMP
      PDF%minvalue=(/1e-4,1/)
      PDF%maxvalue=(/1e6,1e3/)
      PDF%nbins=(/400,400/)
      PDF%Scale=(/LOGSCALE,LOGSCALE/)
      PDF%WeightField=BINBYVOLUME


      CALL CreateHistogram(Histogram)
      Histogram%Field%iD=1
      !    Histogram%Field%name=''
      Histogram%Field%component=GASCOMP
      Histogram%minvalue=1e-4
      Histogram%maxvalue=1d6
      Histogram%nbins=1000
      Histogram%scale=LOGSCALE


      IF (iEOS /= EOS_ISOTHERMAL) THEN
         CALL CreatePDF(PDF)
         PDF%Field(1)%iD=Mass_Field
         PDF%Field(1)%name='density'
         PDF%Field(1)%component=GASCOMP
         PDF%Field(2)%iD=P_Field
         PDF%Field(2)%name='pressure'
         PDF%Field(2)%component=GASCOMP
         PDF%minvalue=(/1e-4,100/)
         PDF%maxvalue=(/1e6,1e6/)
         PDF%nbins=(/400,400/)
         PDF%Scale=(/LOGSCALE,LOGSCALE/)
         PDF%WeightField=BINBYVOLUME
      END IF

      CALL AddAllTotals(GASCOMP)

      IF (lPostProcess) RETURN

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CLOSE(PROBLEM_DATA_HANDLE)
      CALL CreateAmbient(Ambient)
      Ambient%density=density
      Ambient%pressure=pressure
      v_turb=Mach_turb*sqrt(gamma*pressure/density)
      KE_target=half*product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))*density*v_turb**2
      !     relaxation_time=(GxBounds(1,2)-GxBounds(1,1))/v_turb
      t_cross=(GxBounds(1,2)-GxBounds(1,1))/v_turb
      IF (alpha_const /= 0d0) THEN
         alpha=alpha_const
      ELSE
         alpha=v_turb/(2d0*t_cross)
      END IF
      IF (MPI_ID == 0) write(*,*) 'alpha=', alpha, v_turb, t_cross
      !     alpha=alpha_est
      !      IF (.NOT. lRestart) THEN
      t_start_avg=-1d0
      t_final_avg=-1d0
      KE_avg=0d0
      ALLOCATE(pos(3,3))
      ALLOCATE(val(3))
      pos(1,:)=(/1,2,1/)
      pos(2,:)=(/3,1,1/)
      pos(3,:)=(/1,8,4/)
      val(1:3)=(/1.0,.5,.4/)
      CALL CreatePlan(plan, 0, GmGlobal, nDim)
     
      IF (lUseExisting) THEN
         plan%data=0d0
         OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='waves.data', STATUS="OLD")
         READ(PROBLEM_DATA_HANDLE, *) nWaves
         IF (MPI_ID == 0) write(*,*) 'found ', nWaves, ' waves'
         DO l=1,nWaves
            read(PROBLEM_DATA_HANDLE,'(A5,3I4,20E25.16)') str, i,j,k,data
            IF (ALL((/i,j,k/) >= plan%lmB(:,1)) .AND. ALL((/i,j,k/) <= plan%lmB(:,2))) THEN
               plan%data(i,j,k,1:nDim)=data(1:nDim)
               write(*,'(A5,3I4,20E25.16)') 'Wave ', i,j,k,data
            END IF
         END DO
      ELSE
         A=0
         DO i=plan%lmB(1,1),plan%lmB(1,2)
            DO j=plan%lmB(2,1),plan%lmB(2,2)
               DO k=plan%lmB(3,1),plan%lmB(3,2)
                  kvec=SpectraK((/i,j,k/),plan%mB)
                  k2=sum(kvec**2)
                  IF (k2 > 0d0 .AND. k2 <= kmax+1e-6) THEN
                     ! want to choose random complex components for vk(:)
                     ! 3 real angles - 3 real amplitudes
                     ! want ampiltudes to be evenly distributed on a unit sphere  
                     IF (nDim == 2) THEN
                        CALL random_circle(A(1:2))
                     ELSE
                        CALL random_sphere(A)
                     END IF
                     DO l=1,nDim
                        CALL random_number(rand)
                        B(l)=A(l)*exp(2d0*pi*rand*cmplx(0,1))
                     END DO
                     B=B-DOT_PRODUCT(Kvec,B)*Kvec/k2 !subtract off dilational component
                     plan%data(i,j,k,1:nDim)=B(1:nDim)*k2**(beta/2d0)
                     write(*,'(A5,3I4,20E25.16)') 'Wave ', i,j,k,B*k2**(beta/2d0)
                  ELSE
                     plan%data(i,j,k,:)=0d0
                  END IF
               END DO
            END DO
         END DO
      END IF
      CALL ExecutePlan(plan, BACKWARD)
      !     END IF

      IF (lCooling) THEN
         IF (.NOT. lRestart) THEN
            ! see sources/cooling.f90::CreateCoolingObject for
            ! default values of a cooling source term
            CALL CreateCoolingObject(coolingobj)
         ELSE
            coolingobj => firstcoolingobj
         END IF
         coolingobj%iCooling=IICOOL
         coolingobj%floortemp=1d0
         coolingobj%mintemp=0.001
      END IF

   END SUBROUTINE ProblemModuleInit



   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: ip
      INTEGER :: rmbc
   END SUBROUTINE ProblemGridInit

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: rmbc, ip(3,2), i,j,k
      rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)
      ip=1
      ip(1:nDim,1)=1-rmbc
      ip(1:nDim,2)=Info%mX(1:nDim)+rmbc   
      IF (ASSOCIATED(forcing)) THEN
         !         IF (ALL(Info%mGlobal(:,1)==1)) write(*,*) 'forcing(1,1,1)=', .125d0*(/(sum(forcing(1:2,1:2,1:2,i)),i=1,3)/)
         !         IF (ALL(Info%mGlobal(:,1)==1)) write(*,*) 'forcing(1,1,1)=', forcing(1,1,1,:)
         IF (alpha > 0d0 .OR. ANY(dv > 0d0)) THEN
            IF (iE /= 0d0) THEN
               FORALL(i=ip(1,1):ip(1,2), j=ip(2,1):ip(2,2), k=ip(3,1):ip(3,2))
                  Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-half*sum(Info%q(i,j,k,imom(1:nDim))**2)/Info%q(i,j,k,1)
               END FORALL
            END IF
            DO i=ip(1,1), ip(1,2)
               DO j=ip(2,1), ip(2,2)
                  DO k=ip(3,1), ip(3,2)
                     Info%q(i,j,k,imom(1:nDim))=Info%q(i,j,k,imom(1:nDim))+(alpha*forcing(i,j,k,1:nDim)*levels(Info%level)%dt+dv(1:nDim))*Info%q(i,j,k,1)
                  END DO
               END DO
            END DO
            dv=0d0
            IF (iE /= 0d0) THEN
               FORALL(i=ip(1,1):ip(1,2), j=ip(2,1):ip(2,2), k=ip(3,1):ip(3,2))
                  Info%q(i,j,k,iE)=Info%q(i,j,k,iE)+half*sum(Info%q(i,j,k,imom(1:nDim))**2)/Info%q(i,j,k,1)
               END FORALL
            END IF

         END IF
      END IF
      !    END IF
      !       Info%q(:,:,:,5)=gamma7*Ambient%pressure
   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mB, mO
      !    IF (lRestart) THEN
      !       mB(:,1)=33
      !       mB(:,2)=96
      !       mO(:,1)=max(mB(:,1), Info%mGlobal(:,1))
      !       mO(:,2)=min(mB(:,2), Info%mGlobal(:,2))
      !       mO=mO-spread(Info%mGlobal(:,1)-1,2,2)
      !       IF (ALL(mO(:,2) >= mO(:,1))) THEN
      !          Info%ErrFlag(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=1
      !       END IF
      !    END IF
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      TYPE(InfoDef), POINTER :: Info
      INTEGER :: n
      INTEGER :: ip(3,2), i, rmbc
      INTEGER, DIMENSION(:,:), ALLOCATABLE :: FieldID2
      REAL(KIND=qPREC) :: dalpha
      LOGICAL :: lFirstTime=.true.
      IF (lRegrid .OR. lPostProcess) RETURN
      IF (n == 0 .AND. levels(n)%tnow == start_time .AND. lFirstTime) THEN
         rmbc=levels(n)%gmbc(1)
         IF (nDim == 3) THEN
            CALL UnloadFieldFromPFFT(plan, reshape((/ivx,ivy,ivz,0,0,0/),(/3,2/)), lHydroPeriodic, rmbc)     
         ELSE
            CALL UnloadFieldFromPFFT(plan, reshape((/ivx,ivy,0,0/),(/2,2/)), lHydroPeriodic, rmbc)     
         END IF
         CALL ProcessTotal(Total)
         Info=>Nodes(n)%p%self%info
         ip=1
         ip(1:nDim,1)=1-rmbc
         ip(1:nDim,2)=Info%mX(1:nDim)+rmbc
         ALLOCATE(forcing(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nDim))
         forcing=Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),imom(1:nDim))/sqrt(2d0*Total%CurrentValue/Product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))/density)
         Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),imom(1:nDim))=0d0
         Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1)=density
         IF (iE /= 0d0) Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),iE)=gamma7*pressure
         lFirstTime=.true.
      ELSE IF (n == 0 .AND. lRestart .AND. lFirstTime) THEN
         Info=>Nodes(n)%p%self%info
         rmbc=levels(n)%gmbc(1)
         ip=1
         ip(1:nDim,1)=1-rmbc
         ip(1:nDim,2)=Info%mX(1:nDim)+rmbc
         ALLOCATE(tempforcing(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:imom(nDim)))
         tempforcing=Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:imom(nDim))
         Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), 1)=density
         IF (nDim == 3) THEN
            CALL UnloadFieldFromPFFT(plan, reshape((/ivx,ivy,ivz,0,0,0/),(/3,2/)), lHydroPeriodic, rmbc)              
         ELSE
            CALL UnloadFieldFromPFFT(plan, reshape((/ivx,ivy,0,0/),(/2,2/)), lHydroPeriodic, rmbc)              
         END IF
         CALL ProcessTotal(Total)
         ALLOCATE(forcing(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nDim))
         forcing=Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),imom(1:nDim))/sqrt(2d0*Total%CurrentValue/Product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))/density)
         Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:imom(nDim))=tempforcing
         DEALLOCATE(tempforcing)
         lFirstTime=.true.
      END IF
      IF (n == 0) THEN
         CALL ProcessTotal(Total)
         IF (MPI_ID == 0) WRITE(*,*) 'ABC, Total kinetic energy = ', KE
         CALL ProcessTotal(PxTotal)
         CALL ProcessTotal(PyTotal)
         IF (nDim == 3) THEN
            CALL ProcessTotal(PzTotal)
            dv=-(/PxTotal%CurrentValue, PyTotal%CurrentValue, PzTotal%CurrentValue/)/Product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))/density
         ELSE
            dv(1:nDim)=-(/PxTotal%CurrentValue, PyTotal%CurrentValue/)/Product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))/density
         END IF
         IF (MPI_ID == 0) WRITE(*,*) 'ABC, velocity shift = ', dv(1:nDim)
         IF (t_final_avg == -1d0) THEN
            IF (Total%CurrentValue < KE) THEN !finally reached maximum velocity
               IF (MPI_ID == 0) write(*,*) 'ABC, Kinetic energy peaked at', KE
               t_start_avg = levels(n)%tnow
               t_final_avg=t_start_avg+half*t_cross
               KE_avg=0d0
            END IF
         END IF
         KE = Total%CurrentValue
         IF (t_final_avg /= -1d0 .AND. levels(n)%tnow > t_start_avg) THEN
            IF (levels(n)%tnow < t_final_avg) THEN
               !             CALL ProcessTotal(Total)
               KE_avg=KE_avg+KE*levels(n)%dt
            ELSE
               IF (levels(n)%tnow > t_final_avg) THEN
                  t_final_avg=levels(n)%tnow
                  KE_avg=KE_avg/(t_final_avg-t_start_avg)
                  IF (MPI_ID == 0) WRITE(*,*) 'ABC, time averaged kinetic energy = ', KE_avg
                  IF (alpha_const /= 0d0) THEN
                     alpha=alpha*KE_target/KE_avg
                     IF (MPI_ID == 0) WRITE(*,*) 'ABC, adjusting alpha to ', alpha
                  END IF
                  KE_avg=0d0
                  t_start_avg=t_final_avg+t_cross
                  t_final_avg=t_start_avg+.5*t_cross
               END IF
            END IF
         END IF
      END IF

      !     IF (lFirstTime .AND. lRestart .AND. n == 0) THEN
      !        CALL CreateLayout(GmGlobal, layout)
      !        layout%level = 0
      !        mB=layout%mB(MPI_ID,:,:)
      !        ALLOCATE(data(mB(1,1):mB(1,2), mB(2,1):mB(2,2),mB(3,1):mB(3,2),4))
      !        ALLOCATE(FieldID(4))
      !        FieldID=(/(i,i=1,4)/)
      !        CALL LoadFieldIntoLayout(layout, data, FieldID)
      !     ELSEIF (lFirstTime .AND. lRestart .AND. n == 1) THEN
      !        layout%level = 1
      !        layout%mB(:,:,:)=layout%mB(:,:,:)+64 
      !        CALL UnloadFieldFromLayout(layout, data, FieldID)
      !        CALL DestroyLayout(layout)
      !        DEALLOCATE(data)
      !        lFirstTime=.false.
      !     END IF

   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

