!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module Marquee is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir Marquee
!! @brief Contains files necessary for module Marquee

!> @file marqui.f90
!! @brief Main file for module Marquee

!> @defgroup Marquee Marquee Module
!! @brief Module for displaying a marqui across the grid
!! @ingroup Modules

!> Module for displaying a marqui across the grid
!! @ingroup Marquee
MODULE Problem
   USE DataDeclarations
   USE GlobalDeclarations
   USE PhysicsDeclarations
   USE Ambients
   IMPLICIT NONE    ! It's safer to require explicit declarations
   SAVE             ! Save module information
   PRIVATE          ! Everything is implicitly private, i.e. accessible only
   ! to this module.
   PUBLIC :: ProblemModuleInit,ProblemGridInit,ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
   REAL(KIND=qprec) :: ybounds(2), textspeed, backtextspeed, indexspeed, tstart, contrast, messagedx, rhoBackground, &
        EnergyBackground, drho, dp, de, rhoMessage, EnergyMessage, period, dv, yperiod, ydv
   INTEGER(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: message
   NAMELIST/ProblemData/ybounds, textspeed, backtextspeed, tstart, contrast, rhoBackGround, period, dv, yperiod, ydv
   INTEGER :: msize(2)
   REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut

CONTAINS

   !> Initializes marqui module variables
   SUBROUTINE ProblemModuleInit
      INTEGER :: iErr, i
      character(50) :: x
      TYPE(AmbientDef), POINTER :: Ambient
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut

      OPEN(UNIT=PROBLEM_DATA_HANDLE,FILE='problem.data',STATUS='old',      &
           FORM='formatted')
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CALL CreateAmbient(Ambient)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)

      CLOSE(PROBLEM_DATA_HANDLE)
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='message.data', STATUS='old')
      READ(PROBLEM_DATA_HANDLE,*) msize
      !     write(*,*) "Message size=", msize
      ALLOCATE(message(PROBLEM_DATA_HANDLE+msize(1),msize(2)))
      message(1:10,:)=0
      write(x,'(A1,I10,A3)') "(",size(message,1),"I1)"
      write(*,*) x
      DO i=1,msize(2)
         READ(PROBLEM_DATA_HANDLE,x) message(11:,msize(2)+1-i)
      END DO
      CLOSE(PROBLEM_DATA_HANDLE)
      messagedx=(ybounds(2)-ybounds(1))/msize(2)
      indexspeed=textspeed/messagedx      
      !      rhoBackground=rhoOut
      EnergyBackground=gamma7*pOut+half*rhoBackGround*backtextspeed**2
      rhoMessage=rhoBackground*contrast

      EnergyMessage=gamma7*pOut+half*rhoMessage*textspeed**2

      drho=rhoMessage-rhoBackground
      dp=-(rhoMessage*textspeed)+(rhoBackground*backtextspeed)
      write(*,'(5E13.2)') rhoMessage, textspeed, rhoBackground, backtextspeed, dp
      dE=EnergyMessage-EnergyBackground
      write(*,*) drho, dp
      write(*,*) 
      write(*,*) (-rhoBackground*backtextspeed+dp)/(rhoBackGround+drho)

   END SUBROUTINE ProblemModuleInit

   !> Initializes grid
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      ! Interface declarations
      TYPE (InfoDef) :: Info  ! Data associated with this grid
      CALL ProblemBeforeStep(Info)
   END SUBROUTINE ProblemGridInit

   !> Sets boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      ! Interface declarations
      TYPE (InfoDef) :: Info  ! Data associated with this grid
      REAL (KIND=qPrec), POINTER, DIMENSION (:,:,:,:)     :: q
      INTEGER :: i,j,k,mx,my,mz,rmbc,zrmbc,i0(2),j0(2), ii0, jj0, sample_res, ii, jj,level
      REAL(KIND=xprec) :: x,y,z,xl,yl,zl,dx,dy,dz,timeyr, xpos, xbounds(2), overlapxbounds(2), &
           overlapybounds(2), fact, pos(2)
      !
      level=Info%level
      q=>Info%q
      rmbc=levels(level)%gmbc(1)


      mx=Info%mX(1); my=Info%mX(2); mz=Info%mX(3)
      dx=levels(level)%dX; xl=Info%XBounds(1,1);yl=Info%xBounds(2,1)

      !xposition of start of marqui
      xpos=GxBounds(1,2)-textspeed*(levels(level)%tnow-tstart)
      !      write(*,*) "xpos= ", xpos
      !x bounds of launch region
      xbounds(1)=max(xpos, GxBounds(1,2))
      xbounds(2)=min(xpos+msize(1)*messagedx, GxBounds(1,2)+levels(ROOT_LEVEL)%gmbc(1)*levels(ROOT_LEVEL)%dX)

      !overlap physical bounds of launch region with extended grid
      overlapxbounds(1)=max(xbounds(1),Info%xBounds(1,1)-rmbc*levels(level)%dx)
      overlapxbounds(2)=min(xbounds(2),Info%xBounds(1,1)+(Info%mX(1)+rmbc)*levels(level)%dx)
      overlapybounds(1)=max(ybounds(1),Info%xBounds(2,1)-rmbc*levels(level)%dx)
      overlapybounds(2)=min(ybounds(2),Info%xBounds(2,1)+(Info%mX(2)+rmbc)*levels(level)%dx)
      sample_res=max(nint(levels(level)%dx/messagedx),8)
      fact=1d0/real(sample_res)**2
      !      write(*,'(A,4E13.4)') "overlapbounds=", overlapxbounds, overlapybounds
      !      write(*,*) "sample_res=", sample_res
      IF (overlapxbounds(2) >= overlapxbounds(1) .AND. overlapybounds(2) >= overlapybounds(1)) THEN

         i0(1)=max(1-rmbc, ceiling((overlapxbounds(1)-Info%xBounds(1,1))/levels(level)%dx)) !max(product(coarsenratio(0:Info%level-1))-Info%mGlobal(1,1)+1, 1-rmbc)
         i0(2)=min(Info%mX(1)+rmbc, floor((overlapxbounds(2)-Info%xBounds(1,1))/levels(level)%dx))
         j0(1)=max(1-rmbc, ceiling((overlapybounds(1)-Info%xBounds(2,1))/levels(level)%dx)) !max(product(coarsenratio(0:Info%level-1))-Info%mGlobal(1,1)+1, 1-rmbc)
         j0(2)=min(Info%mX(2)+rmbc, floor((overlapybounds(2)-Info%xBounds(2,1))/levels(level)%dx))
         !         write(*,'(A,4I)') "i0,j0= ", i0,j0
         IF (i0(1) <= i0(2) .AND. j0(1) <= j0(2)) THEN !zero out overlap region with background values
            q(i0(1):i0(2),j0(1):j0(2),1,:) = 0
            q(i0(1):i0(2),j0(1):j0(2),1,1) = rhoBackground
            q(i0(1):i0(2),j0(1):j0(2),1,2) = -Backtextspeed*rhoBackGround
            !            q(:,:,:,1,2)=-Backtextspeed*rhoBackGround
            !            write(*,*) -Backtextspeed*rhoBackGround
            !            IF (iE .ne. 0) q(i0(1):i0(2),j0(1):j0(2),:,1,iE) = gamma7*pOut!EnergyBackground
            !            return
            DO i=i0(1),i0(2)
               DO j=j0(1),j0(2)
                  DO ii=1,sample_res
                     DO jj=1,sample_res
                        pos=Info%xBounds(1:2,1)+((real((/i,j/))-1d0) + &
                             ((/REAL(ii,8),REAL(jj,8)/)-half)/REAL(sample_res,8))*levels(level)%dX 
                        ii0=nint((pos(1)-xpos)/messagedx)
                        jj0=nint((pos(2)-ybounds(1))/messagedx)
                        IF (ii0 >= 1 .AND. ii0 <= msize(1) .AND. jj0 >= 1 .AND. jj0 <= msize(2)) THEN
                           IF (message(ii0,jj0)==1) THEN
                              q(i,j,1,1)=q(i,j,1,1)+drho*fact
                              dp=-(rhoMessage*textspeed*(1d0+dv*cos(2d0*Pi*levels(level)%tnow/period))) + &
                                   (rhoBackground*backtextspeed)
                              q(i,j,1,3)=q(i,j,1,3)+rhoMessage*textspeed*ydv*sin(2d0*Pi*levels(level)%tnow/yperiod)*fact
                              q(i,j,1,2)=q(i,j,1,2)+dp*fact
                              !                              IF (iE .ne. 0) q(i,j,1,1,iE)=q(i,j,1,1,iE)+de*fact                     
                           END IF
                        END IF
                     END DO
                  END DO
                  IF (iE .ne. 0) q(i,j,1,iE)= gamma7*pOut+half*SUM(q(i,j,1,2:3)**2)/q(i,j,1,1)
               END DO
            END DO
         END IF
      END IF
   END SUBROUTINE ProblemBeforeStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemAfterStep(Info)
      !! @brief Performs any post-step corrections that are required.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      !! @brief Sets error flags according to problem-specific conditions..
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep


END MODULE Problem

