!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module MolecularCloudFormation is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MolecularCloudFormation
!! @brief Contains files necessary for the Molecular Cloud Formation problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MolecularCloudFormation Molecular Cloud Formation Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> Molecular Cloud Formation Module
!! @ingroup MolecularCloudFormation
MODULE Problem
  USE DataDeclarations
  USE ParticleDeclarations
  USE ProcessingDeclarations  
  USE CollidingFlows
  USE CoolingSrc
  USE Winds  
  USE Ambients
  USE Histograms
  USE Fields
  USE Totals
  USE PDFs
  USE Projections
  USE Clumps
  USE Shapes
  USE Refinements
  IMPLICIT NONE
  SAVE

  REAL(KIND=qPREC), DIMENSION(0:MaxDepth) :: cells_per_cooling_length=0
  REAL(KIND=qPREC), DIMENSION(0:MaxDepth)  :: InterfaceWidth=0
  TYPE(CollidingFlowDef), POINTER :: CollidingFlow
  TYPE(ShapeDef), POINTER :: DerefineShape=>Null()
  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(CoolingDef),POINTER :: coolingobj
  REAL(KIND=qPREC) :: IIScaleCool, InterfaceTime, IIScaleHeat, derefine_radius=1e20
  INTEGER :: derefine_dist=0
  TYPE(AmbientDef), POINTER :: Ambient
  LOGICAL :: DeRefineOutSide=.false.
  LOGICAL :: lClumps=.false.
  LOGICAL :: lPlaceClumps=.true.
  REAL(KIND=qPREC) :: ClumpJeansFact = .5d0 !Make them 'X=.5' jeans lengths in diameter.
  REAL(KIND=qPREC) :: MeanDensitywClumps=3d0
  REAL(KIND=qPREC) :: ClumpChi=10
  REAL(KIND=qPREC) :: separation_param=.3d0
  REAL(KIND=qPREC) :: TShutOff=1d30
  REAL(KIND=qPREC) :: RampTime=1d0

CONTAINS

  !> Initializes module variables
  SUBROUTINE ProblemModuleInit()      
    INTEGER :: nRegions, nWaves
    REAL(KIND=qPREC) :: density, temperature, smooth_distance, velocity, position(3), size_param(3), psi, theta, phi, wavevector(2), amplitude, phase, vel(3), a, alpha, dk, kmag, beta, mach, interface_dist, clumprho, clumptemp, clumpff, clumpradius, clumpnumbervolume, region(3,2), min_separation, ram_dens, ram_press, ram_temp, phi2
    REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: positions
    INTEGER :: i, j, type, subsample=1, smooth_function, edge, kx, ky, interface_func
    LOGICAL :: lCooling
    TYPE(WindDef), POINTER ::  Wind
    TYPE(HistogramDef), POINTER :: HISTOGRAM
    TYPE(PDFDef), POINTER :: PDF
    TYPE(ProjectionDef), POINTER :: Projection
    INTEGER :: nWinds, nbins=100, clumprefinelevel
    INTEGER, DIMENSION(10) :: myseed  
    INTEGER :: kmax,nclumps,nParticleCells=0
    TYPE(ClumpDef), POINTER :: Clump
    TYPE(ShapeDef), POINTER :: LeftClumpRegion, RightClumpRegion, TotalShape
    LOGICAL :: lPersist=.true.
    REAL(KIND=qpREC) :: TotalWidth, FlowDiameter(2), ShearAngle, MaxDiameter
    TYPE(RefinementDef), POINTER :: Refinement
    INTEGER :: MustRefineDepth=MAXDEPTH
    INTEGER :: CanRefineDepth=MAXDEPTH
    REAL(KIND=qPREC) :: MustRefineWidth=0d0, CanRefineWidth=1d30
    NAMELIST /ProblemData/ density, velocity, mach, temperature, smooth_distance, smooth_function, subsample, lCooling, &
         myseed, A, alpha, kmax, beta, &
         interface_func, interface_dist, nbins, lClumps, ClumpJeansFact, MeanDensitywClumps, &
         ClumpChi, separation_param, nParticleCells,CellsPerJeansLength, clumprefinelevel, IICoolPar, TShutOff, RampTime, lPlaceClumps, &
         TotalWidth, FlowDiameter, ShearAngle, MustRefineWidth, MustRefineDepth, CanRefineWidth, CanRefineDepth

    OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
    READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
    CLOSE(PROBLEM_DATA_HANDLE)

    write(*,*) 'Shear angle=', ShearAngle
    ! Rescale various physical parameters to computational units
    FlowDiameter=FlowDiameter*pc/lScale
    density=density/nScale
    temperature=temperature/TempScale
    velocity=velocity*1e5/VelScale
    TotalWidth=TotalWidth*pc/lScale
    RampTime=RampTime*1e6*yr/TimeScale
    TShutOff=TShutOff*1e6*yr/TimeScale
    InterfaceTime=InterfaceTime*1e6*yr/TimeScale

    IF (nParticleCells > 0) DefaultParticleRadius=nParticleCells  
 
    IF (lCooling) THEN
       IF (.NOT. lRestart) THEN
          CALL CreateCoolingObject(coolingobj)
       ELSE
          coolingobj => firstcoolingobj
       END IF
       coolingobj%iCooling=IICOOL
       coolingobj%floortemp=1d0
       coolingobj%mintemp=1d0
    END IF
    
    IF (temperature == 0d0) THEN
       CALL InitIICool(coolingobj)
       temperature=GetIICoolEqTemp(density*nScale)/TempScale
       IF (MPI_ID == 0) write(*,*) 'Background Density = ', density*nScale, ' particles/cc'
       IF (MPI_ID == 0) write(*,*) 'Equilibrium temp = ', temperature*TempScale, ' K'
       IF (MPI_ID == 0) write(*,*) 'sound speed = ', sqrt(gamma*temperature), sqrt(gamma*temperature)*velscale/1e5, 'km/s'
    END IF
    IF (mach /= 0d0) velocity=mach*sqrt(gamma*temperature)

    IF (MPI_ID == 0) write(*,*) 'Background Jeans Length =', JeansLength(density, temperature)*lScale/pc, ' pc'
    IF (MPI_ID == 0) write(*,*) 'Background free fall time =', sqrt(3d0*pi/32d0/ScaleGrav/density)*TimeScale/yr/1e6, 'myr'
    IF (MPI_ID == 0) write(*,*) 'flow velocity = ', velocity*velscale/1e5,  'km/s'

    CALL CreateAmbient(Ambient)
    Ambient%density=density
    Ambient%pressure=density*temperature
    IF (beta /= 0d0) THEN
       Ambient%B=sqrt(2d0*temperature*density/beta)*(/1d0,0d0,0d0/)
       IF (MPI_ID == 0) write(*,'(A,3E25.15)') 'Ambient%B=',Ambient%B
    END IF

    DO i=1,nDim
       DO edge=1,2
          IF (Gmthbc(i,edge) == 1) THEN 
             CALL CreateWind(Wind)
             Wind%dir=i
             Wind%edge=edge
             Wind%type=OUTFLOW_ONLY
             Wind%density=Ambient%density
             Wind%temperature=Ambient%Pressure/Ambient%Density
             Wind%B=Ambient%B
          END IF
       END DO
    END DO



    CALL CreateCollidingFlow(CollidingFlow)

    MaxDiameter=maxval(FlowDiameter)
    CollidingFlow%SubSample = subsample
    CollidingFlow%density = density
    CollidingFlow%velocity = velocity
    CollidingFlow%Temperature = Temperature
    CollidingFlow%PersistInBoundaries = lPersist
    CollidingFlow%smooth_function = smooth_function
    CollidingFlow%smooth_distance = smooth_distance*MaxDiameter
    CollidingFlow%interface_func = interface_func
    CollidingFlow%interface_dist = interface_dist*MaxDiameter
    CollidingFlow%tShutOff = TShutOff
    CollidingFlow%RampTime = RampTime
    CALL AddTracer(CollidingFlow%iTracer(1),'FlowTracer1')
    CALL AddTracer(CollidingFlow%iTracer(2),'FlowTracer2')
  
    CALL SetShapeType(CollidingFlow%Shape, ELLIPTICAL_PRISM, half*(/FlowDiameter(:),2d0*(GxBounds(1,2)-GxBounds(1,1))/))
    CALL SetShapeOrientation(CollidingFlow%Shape, 0d0, Pi/2d0, 0d0)
    CollidingFlow%Shape%Position=half*SUM(GxBounds(:,:), 2)
    CALL SetShapeBounds(CollidingFlow%Shape)

    IF (derefine_radius > 0d0) THEN
       CALL CreateShape(DerefineShape)
       CALL SetShapeType(DerefineShape, ELLIPTICAL_PRISM, (/derefine_radius*half*FlowDiameter(:), 2d0*(GxBounds(1,2)-GxBounds(1,1))/))
       CALL SetShapeOrientation(DerefineShape, 0d0, Pi/2d0, 0d0)
       DerefineShape%position=half*SUM(GxBounds(:,:),2)
       CALL SetShapeBounds(DerefineShape)
    END IF
    IF (MPI_ID == 0) THEN
       write(*,*) 'mass flux = ', 2d0*velocity*velScale * (SUM(FlowDiameter(1:nDim-1)**2))*Pi/4d0*density*rScale*lScale**2/(mSolar/(1e6*yr)), 'm_sun/Myr'
       ram_press=velocity**2*TempScale*density*nScale
       write(*,*) 'ram pressure = ', ram_press, 'particles K/cc'
       ram_dens=GetIICoolEqDensity(ram_press)
       write(*,*) 'ram density = ', ram_dens,  'particles per cc'
       ram_temp=ram_press/ram_dens
       write(*,*) 'ram temp = ', ram_temp,  'K'
       write(*,*) 'ram density jeans length =', JeansLength(ram_dens/nScale, ram_temp/TempScale)*lScale/pc, ' pc'
    END IF
    ! Then set up interface
    CollidingFlow%InterfaceObj%Position=CollidingFlow%Shape%Position 
    
    IF (nDim == 2) THEN
       CALL SetInterfaceOrientation(CollidingFlow%InterfaceObj, Pi/2d0,ShearAngle/180d0*Pi)
    ELSE
       CALL SetInterfaceOrientation(CollidingFlow%InterfaceObj, Pi/2d0-ShearAngle/180d0*Pi,0d0)
    END IF
    CALL RANDOM_SEED(size=j)
    CALL RANDOM_SEED(PUT=myseed(1:j))
    nWaves=0
    DO kx=1, kMax
       DO ky=1, kMax
          kmag=sqrt(real(kx**2+ky**2))
          if (kmag < kMax) THEN
             nwaves=nwaves+1
          END if
       END DO
    END DO
    dk=2d0*Pi/MaxDiameter
    CALL InitInterfaceWaves(CollidingFlow%InterfaceObj, nWaves)
    DO kx=1, kMax
       DO ky=1, kMax
          kmag=sqrt(real(kx**2+ky**2))
          if (kmag < kMax) THEN
             CALL RANDOM_NUMBER(phase)
             phase=phase*2d0*Pi
             wavevector=(/kx*dk,ky*dk/)
             amplitude=A*kmag**alpha*MaxDiameter
             CALL AddInterfaceWave(CollidingFlow%InterfaceObj, wavevector, phase, amplitude)
             !             write(*,*) wavevector, amplitude, phase
          END if
       END DO
    END DO

    CALL ClearAllRefinements()


    !First create refinement object to refine around interface
    CALL CreateRefinement(Refinement)    
    CALL CreateShape(Refinement%Shape)
    CALL SetShapeType(Refinement%Shape, ELLIPTICAL_PRISM, half*(/FlowDiameter(1)/cos(ShearAngle/180d0*Pi), FlowDiameter(2), MustRefineWidth/))
    IF (nDim == 2) THEN
       CALL SetShapeOrientation(Refinement%Shape, 0d0, Pi/2d0, ShearAngle/180d0*Pi)
    ELSE
       CALL SetShapeOrientation(Refinement%Shape, 0d0, Pi/2d0-ShearAngle/180d0*Pi, 0d0)
    END IF
    Refinement%Shape%Position=CollidingFlow%Shape%Position
    CALL SetShapeBounds(Refinement%Shape)
!    Refinement%field=Mass_Field
!    Refinement%tolerance=1
!    Refinement%scale=LOGSCALE
    Refinement%BufferCells=1
    Refinement%MaxLevel=MustRefineDepth

    CALL CreateRefinement(Refinement)
    CALL CreateShape(Refinement%Shape)
    CALL SetShapeType(Refinement%Shape, ELLIPTICAL_PRISM, half*(/FlowDiameter(1)/cos(ShearAngle/180d0*Pi), FlowDiameter(2), CanRefineWidth/))
    IF (nDim == 2) THEN
       CALL SetShapeOrientation(Refinement%Shape, 0d0, Pi/2d0, ShearAngle/180d0*Pi)
    ELSE
       CALL SetShapeOrientation(Refinement%Shape, 0d0, Pi/2d0-ShearAngle/180d0*Pi, 0d0)
    END IF
    Refinement%Shape%Position=CollidingFlow%Shape%Position
    CALL SetShapeBounds(Refinement%Shape)
    Refinement%field=Mass_Field
    Refinement%tolerance=1
    Refinement%scale=LOGSCALE
    Refinement%BufferCells=2
    Refinement%MaxLevel=CanRefineDepth


    CALL AddRefinementThreshold(JeansLength_Field, LESSTHAN, (/(CellsPerJeansLength*levels(i)%dx,i=0,MaxLevel)/))


!    CALL AddDiagnosticVar(ErrFlag_Field)


! Uncomment this to turn off processing stuff 
   RETURN

    CALL CreateShape(TotalShape)
    CALL SetShapeType(TotalShape, ELLIPTICAL_PRISM, half*(/FlowDiameter(:),TotalWidth/))
    CALL SetShapeOrientation(TotalShape, 0d0, Pi/2-ShearAngle/180d0*Pi,0d0)
    TotalShape%position(:) = CollidingFlow%Shape%position
    CALL SetShapeBounds(TotalShape)

    CALL AddAllTotals(GASCOMP, TotalShape)
    CALL AddAllTotals(PARTICLECOMP,TotalShape)
    CALL AddAllTotals(BOTHCOMP,TotalShape)

    CALL CreatePDF(PDF)
    PDF%Field(1)%iD=Mass_Field
    PDF%Field(1)%name='density'
    PDF%Field(1)%component=GASCOMP
    PDF%Field(2)%iD=VMag_Field
    PDF%Field(2)%name='velocity'
    PDF%Field(2)%component=GASCOMP
    PDF%minvalue=(/.01,.001/)
    PDF%maxvalue=(/1e7,1e4/)
    PDF%nbins=(/400,400/)
    PDF%Scale=(/LOGSCALE,LOGSCALE/)
    PDF%WeightField=BINBYVOLUME
    PDF%Shape=>TotalShape


    CALL CreateHistogram(Histogram)
    Histogram%Field%iD=1
    Histogram%Field%name='density'
    Histogram%Field%component=GASCOMP
    Histogram%minvalue=.1d0
    Histogram%maxvalue=1d8
    Histogram%nbins=nbins
    Histogram%scale=LOGSCALE
    Histogram%shape=>TotalShape


    CALL CreateHistogram(Histogram)
    Histogram%Field%iD=MixingRatio12_Field
    Histogram%Field%name='Mixing_Ratio'
    Histogram%Field%component=GASCOMP
    Histogram%minvalue=0d0
    Histogram%maxvalue=1d0
    Histogram%nbins=nbins
    Histogram%scale=LINEARSCALE
    Histogram%WeightField=BINBYVOLUME
    Histogram%shape=>TotalShape

    CALL CreatePDF(PDF)
    PDF%Field(:)%iD=(/Mass_Field, P_Field/)
    PDF%Field(1)%name='density'
    PDF%Field(2)%name='pressure'
    PDF%Field(:)%component=GASCOMP
    PDF%minvalue=(/.01,100.0/)
    PDF%maxvalue=(/1e7,1e6/)
    PDF%nbins=(/400,400/)
    PDF%Scale=(/LOGSCALE,LOGSCALE/)
    PDF%WeightField=BINBYMASS
    PDF%Shape=>TotalShape


    CALL CreateProjection(projection)
    Projection%Field%iD=Mass_Field
    Projection%Field%component=BOTHCOMP
    Projection%dim=1
!    Projection%Shape=>TotalShape

    CALL CreateProjection(projection)
    Projection%Field%iD=Mass_Field
    Projection%Field%component=BOTHCOMP
    Projection%dim=2

    CALL CreateProjection(projection)
    Projection%Field%iD=Mass_Field
    Projection%Field%component=BOTHCOMP
    Projection%dim=3

  END SUBROUTINE ProblemModuleInit


  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Could be used to update grids pre-step
  !! @param Info Info Object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
 END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info

  END SUBROUTINE ProblemAfterStep


  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag


  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n     
  END SUBROUTINE ProblemBeforeGlobalStep


  SUBROUTINE SetupTotals()

  END SUBROUTINE SetupTotals

END MODULE Problem

