!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module MultiClumps is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiClumps
!! @brief Contains files necessary for the MultiClumps problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiClumps Multiple Clumps Module
!! @brief Module for setting up multiple clumps
!! @ingroup Modules

!> MultiClump Module
!! @ingroup MultiClumps
MODULE Problem
  USE DataDeclarations
  USE ParticleDeclarations
  USE Clumps
  USE CoolingSrc
  USE Winds  
  USE Ambients
  USE Projections
  USE Fields
  USE Cameras
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(CoolingDef),POINTER :: coolingobj
  REAL(KIND=qPREC) :: wind_thickness, wind_vel, wind_refinement_time
  TYPE(WindDef), POINTER :: Wind
  TYPE(ProjectionDef), POINTER :: Projection
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      INTEGER :: nClumps, ClumpTracer, AltClumpTracer,i, dir, edge, ClumpToTrace
      TYPE(ClumpDef), POINTER :: Clump
      LOGICAL :: lCooling
      REAL(KIND=qPREC) :: density, temperature, velocity, smooth_distance, position(3), radius, B(3)
      TYPE(AmbientDef), POINTER :: Ambient
      REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /ProblemData/ nClumps, lCooling, ClumpToTrace
      NAMELIST /WindData/ density, temperature, velocity, B,  wind_thickness, wind_refinement_time, dir, edge
      NAMELIST /ClumpData/ density, temperature, radius, smooth_distance, position
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      dir=1
      edge=1
      CALL AddTracer(ClumpTracer, 'ClumpTracer')
      CALL AddTracer(AltClumpTracer, 'AltClumpTracer')
      CALL CreateAmbient(Ambient)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)

      READ(PROBLEM_DATA_HANDLE,NML=WindData)
      CALL CreateWind(Wind)
      Wind%density=density
      Wind%temperature=temperature
      Wind%velocity=velocity
      Wind%B=B
      wind_vel=velocity
      CALL AddTracer(Wind%iTracer, 'Wind_Tracer')
      Wind%dir=dir
      Wind%edge=edge
      DO i=1, nClumps
         READ(PROBLEM_DATA_HANDLE,NML=ClumpData)
         CALL CreateClump(Clump)
         Clump%density=density
         Clump%temperature=temperature
         Clump%position=position
         IF (i == ClumpToTrace) THEN
            Clump%iTracer=AltClumpTracer
         ELSE
            Clump%iTracer=ClumpTracer
         END IF
         Clump%thickness=smooth_distance
         Clump%radius=radius
         CALL UpdateClump(Clump)
      END DO

      IF (lCooling) THEN
         IF (.NOT. lRestart) THEN
            ! see sources/cooling.f90::CreateCoolingObject for
            ! default values of a cooling source term
            CALL CreateCoolingObject(coolingobj)
         ELSE
            coolingobj => firstcoolingobj
         END IF
         coolingobj%iCooling=DMCOOL
         coolingobj%floortemp=100d0
         coolingobj%mintemp=1
      END IF

      CALL CreateProjection(Projection)
      Projection%Field%id=Mass_Field
      Projection%Field%component=GASCOMP
      Projection%Field%name='density'
      Projection%dim=3d0
   END SUBROUTINE ProblemModuleInit

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info

   END SUBROUTINE ProblemGridInit

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: ip(3,2)
      REAL(KIND=qPREC) :: wind_pos, dx
      
      IF (levels(Info%level)%tnow <= wind_refinement_time) THEN
         dx=levels(Info%level)%dx
         wind_pos=wind_vel/2d0*levels(Info%level)%tnow
         ip(:,1)=1
         ip(:,2)=Info%mX
         ip(wind%dir,1)=ceiling((wind_pos-wind_thickness-Info%xBounds(wind%dir,1))/dx)
         ip(wind%dir,2)=ceiling((wind_pos+wind_thickness-Info%xBounds(wind%dir,1))/dx)
         ip(wind%dir,1)=max(ip(wind%dir,1), 1)
         ip(wind%dir,2)=min(ip(wind%dir,2),Info%mX(wind%dir))
         IF (ip(wind%dir,2) >= ip(wind%dir,1)) Info%ErrFlag(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2))=1
      END IF
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

