!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module MultiJets is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiJets
!! @brief Contains files necessary for the MultiJets Calculation

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiJets Module
!! @brief Module for simulating 2D turbulence
!! @ingroup Modules

!> Multiple Jets module
!! @ingroup MultiJets
MODULE Problem
  USE DataDeclarations
  USE Ambients
  USE Fields
  USE Refinements
  USE Clumps
  USE ProcessingDeclarations
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
     TYPE(AmbientDef), POINTER :: Ambient
     REAL(KIND=qPREC) :: density !Background density
     REAL(KIND=qPREC) :: Mach !mach number of induced turbulence
     REAL(KIND=qPREC) :: length !length scale of injection
     REAL(KIND=qPREC) :: eta !percent in mass increase per turbulence time scale
     REAL(KIND=qPREC) :: Temp !temperature of background 
     REAL(KIND=qPREC) :: decay_frac !Fraction of duration to use for decay
     REAL(KIND=qPREC) :: nOverlap !number of active overlapping outflows per interaction scale (sets duration)
     REAL(KIND=qPREC) :: numcellsinradius !sets size of outflow sphere
     REAL(KIND=qPREC) :: numcellsinshell !sets thickness of accelerating region  
     TYPE(OutflowDef), POINTER :: Outflow
     REAL(KIND=qPREC) :: outflow_density, outflow_temp, outflow_duration, outflow_radius, outflow_decay, outflow_velocity
     REAL(KIND=qPREC) :: velocity, momentum, volrate, time, LaunchPeriod, massloss, rand
     INTEGER :: i,j,nOutflows
     NAMELIST /ProblemData/ density, Mach, length, eta, Temp, numcellsinradius, numcellsinshell, nOverlap, decay_frac
     NAMELIST /OutflowData/ velocity, momentum, volrate, nOutflows, time, LaunchPeriod, massloss, outflow_temp, outflow_velocity, outflow_density, outflow_radius, outflow_duration, outflow_decay

     OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
     READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
     CALL CreateAmbient(Ambient)
     Temp=Temp/TempScale
     Ambient%density=density    
     IF (iE == EOS_ISOTHERMAL) Temp=Iso_Speed2
     Ambient%pressure=density*Temp
     
     ! Use density, mach, length
     ! to get P, S, and V (from eta)


     IF (.NOT. lRestart) THEN
        velocity=sqrt(gamma*Ambient%pressure/Ambient%density)*Mach
        
        momentum = density*length**(nDim)*velocity
        volrate = length**(-nDim-1)*velocity
        nOutflows=volrate*product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))*(final_time-start_time)
        time=length*velocity
        LaunchPeriod=1d0/(product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))*volrate)
        massloss=eta*Ambient%density/(volrate*time)
        outflow_temp=Temp
        outflow_velocity=momentum/massloss
        outflow_radius=levels(MaxLevel)%dx*(numcellsinradius)

        outflow_duration=nOverlap*time
        outflow_decay=outflow_duration*decay_frac
        IF (nDim == 2) THEN
           outflow_density=massloss/(outflow_duration*2d0*Pi*outflow_radius*outflow_velocity)
        ELSEIF (nDim == 3) THEN
           outflow_density=massloss/(outflow_duration*4d0*Pi*outflow_radius**2*outflow_velocity)
        END IF

        IF (MPI_ID == 0) THEN
           write(*,NML=ProblemData)
           write(*,NML=OutflowData)
        END IF
        DO i=1,nOutflows
           NULLIFY(Outflow)
           CALL CreateOutflowObject(Outflow)
           DO j=1,nDim
              CALL random_number(rand)
              Outflow%position(j)=GxBounds(j,1)+rand*(GxBounds(j,2)-GxBounds(j,1))
           END DO
           write(*,*) 'outflow%position=', outflow%position
           Outflow%open_angle=half*Pi
           Outflow%radius=levels(MaxLevel)%dx*(numcellsinradius-numcellsinshell)
           Outflow%thickness=levels(MaxLevel)%dx*numcellsinshell
           Outflow%begin=LaunchPeriod*(1d0*i-1d0)
           Outflow%velocity=outflow_velocity
           Outflow%duration=outflow_duration
           Outflow%decay=outflow_decay
           Outflow%density=outflow_density
           Outflow%temperature=outflow_temp
           CALL UpdateOutflow(Outflow)
        END DO

     ELSE
        ! Need to write IO routines for restarts...

     END IF
   END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: j
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: i
!    Info%q(1,1,1,iE)=0
  END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
    INTEGER, DIMENSION(3,2) :: mO
!    mO(:,1)=max(1, RefineBox(:,1)-Info%mGlobal(:,1)+1)
!    mO(:,2)=min(Info%mX, RefineBox(:,2)-Info%mGlobal(:,1)+1)
!    Info%ErrFlag(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=1
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

