!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module RadShock is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir RadShock
!! @brief Contains files necessary for the RadShock Problem

!> @file problem.f90
!! @brief Main file for module Prolem

!> @defgroup RadShock 1D Radiative Shock Module
!! @brief Module for calculating a stable 1D radiative shock
!! @ingroup Modules

!> 1D Radiative Shock Module
!! @ingroup RadShock
MODULE Problem

  USE DataDeclarations
  USE PhysicsDeclarations
  USE GlobalDeclarations
  USE SourceDeclarations
  USE CoolingSrc
  USE NEQCoolingSrc
  USE Emissions
  USE Projections
  USE Fields
  USE Cameras
  USE ProcessingDeclarations
  IMPLICIT NONE
  SAVE
  
  PUBLIC ProblemModuleInit, ProblemGridInit, &
     ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  
  TYPE(CoolingDef),POINTER :: coolingobj  
  INTEGER :: iCooling
  REAL(KIND=qPREC) :: n_amb, v_amb, T_amb, By_amb, Xh_amb, Hefrac, alpha, beta
  REAL(KIND=qPREC) :: rho_amb, p_amb, rho_ps, v_ps, p_ps, By_ps, mu

  NAMELIST /ProblemData/ n_amb, v_amb, T_amb, By_amb, Xh_amb, Hefrac, alpha, beta, iCooling

CONTAINS

  SUBROUTINE ProblemModuleInit()
    INTEGER :: j
    REAL(KIND=qPREC) :: cs, M, a, b, c, d
    COMPLEX(8) :: MyRoots(3)
    REAL(KIND=qPREC) :: ValidRoots(3)
    TYPE(ProjectionDef), POINTER :: Projection

    OPEN(UNIT=Problem_Data_Handle, FILE='problem.data', STATUS="OLD")
    READ(Problem_Data_Handle, NML=ProblemData)
    CLOSE(Problem_Data_Handle)

    IF (.NOT. lRestart) THEN
       CALL CreateCoolingObject(coolingobj)
    ELSE
       coolingobj => firstcoolingobj
    END IF

    coolingobj%iCooling = iCooling; coolingobj%floortemp = 1d3; coolingobj%mintemp = 1d-2;

    coolingobj%alpha = alpha; coolingobj%beta = beta

    lHe = .TRUE.; !lHeII = .TRUE.

    ! solve for initial postshock conditions ("ps" values)
    ! v_ps can be found using ambient mach number M and the shock jump equation in hydro case
    ! for MHD, v_ps is found via a more complicated cubic equation
    ! then rho_ps and p_ps can be found from mass flux and momentum flux conservation
    rho_amb = hMass*n_amb; p_amb = n_amb*Boltzmann*T_amb

    IF(lMHD) THEN
       ValidRoots = 2d0*v_amb
       a = rho_amb*v_amb*(half-gamma15)
       b = gamma15*(rho_amb*v_amb**2d0 + p_amb + By_amb**2d0/8d0/Pi)
       c = -(half*rho_amb*v_amb**3d0 + gamma15*p_amb*v_amb + By_amb**2d0*v_amb/4d0/Pi)
       d = By_amb**2d0*v_amb**2d0/4d0/Pi*(1d0-gamma15/2d0)
       MyRoots = CubicRoots(a,b,c,d)
       DO j=1, 3
          IF(AIMAG(MyRoots(j)) == 0d0 .AND. REAL(MyRoots(j)) > 0d0) THEN
             ValidRoots(j) = REAL(MyRoots(j))
          END IF
       END DO
       v_ps = MINVAL(ValidRoots)
       rho_ps = rho_amb*v_amb/v_ps
       By_ps = By_amb*v_amb/v_ps
       p_ps = rho_amb*v_amb**2d0 + p_amb + By_amb**2d0/(8d0*Pi) - rho_ps*v_ps**2d0 - By_ps**2d0/(8d0*Pi)
    ELSE
       cs = SQRT(gamma*p_amb/rho_amb)
       M = v_amb/cs
       v_ps = v_amb*(gamma1*M**2d0 + 2d0)/((gamma+1d0)*M**2d0)
       rho_ps = rho_amb*v_amb/v_ps
       p_ps = rho_amb*v_amb**2d0 + p_amb  - rho_ps*v_ps**2d0
    END IF

    CALL CreateProjection(Projection)
    Projection%Field%id = SII_6716_Field
    Projection%Field%component = GASCOMP
    Projection%Field%name = 'SII_6716'
    Projection%dim = 3d0

!    CALL AddDiagnosticVar(MPI_ID_FIELD, 'MPI_ID')
!    CALL AddDiagnosticVar(ChildMask_Field, 'ChildMask')
!    CALL AddDiagnosticVar(ErrFlag_Field, 'ErrFlags')
   
!    CALL AddDiagnosticVar(SII_6716_FIELD, 'SII_6716')
!     CALL StoreEmiss(iSII_6731)
!     CALL StoreEmiss(iOI)         ! stores 6300 line
!     CALL StoreEmiss(iNII)        ! stores 6583 line
!     CALL StoreEmiss(iHalpha)

  END SUBROUTINE ProblemModuleInit

  SUBROUTINE ProblemGridInit(Info)

    TYPE(InfoDef) :: Info
    REAL(KIND=qPREC), POINTER, DIMENSION(:,:,:,:) :: q
    INTEGER :: rmbc, mx, i
    REAL(KIND=qPREC) :: x, Lx, xlower, shock_pos, dx, xold
    REAL(KIND=qPREC) :: p_cool, v_cool, rho_cool, By_cool, Eflux_cool, HIIflux_cool, HeIIflux_cool, Heflux_cool

    q => Info%q
    q = 0d0

    rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)
    mx=Info%mX(1); dx=levels(Info%level)%dX*lscale
    xlower=Info%xbounds(1,1)*lscale; Lx=Gxbounds(1,2)-Gxbounds(1,1)
    shock_pos = 1d-1*Lx*lscale
    xold = shock_pos
    Eflux_cool = half*rho_ps*v_ps**3d0 + gamma15*p_ps*v_ps
    IF(lMHD) Eflux_cool = Eflux_cool + By_ps**2d0*v_ps/4d0/Pi
    HIIflux_cool = Xh_amb*rho_ps*v_ps
    HeIIflux_cool = 0d0
    Heflux_cool = Hefrac*(muHe/muH)*rho_ps*v_ps

    DO i=1-rmbc, mx+rmbc
       x=(xlower + (REAL(i) - half) * dx)

       IF(x < shock_pos) THEN
          ! ambient preshock region ("amb" values)
          q(i,:,:,1) = n_amb / nScale
          q(i,:,:,2) = q(i,:,:,1) * v_amb / VelScale
          q(i,:,:,iE) = gamma7*p_amb/pScale + half*q(i,:,:,1)*(v_amb/VelScale)**2d0
          IF(lMHD) THEN
             q(i,:,:,iBy) = By_amb/BScale
             q(i,:,:,iE) = q(i,:,:,iE) + half*q(i,:,:,iBy)**2d0
          END IF
          
          IF(iCooling == NEQCool .OR. iCooling == ZCool) THEN
             q(i,:,:,iHe) = Hefrac*q(i,:,:,1)
!             q(i,:,:,iHeII) = 0d0
             q(i,:,:,iHII) = Xh_amb*(q(i,:,:,1)-q(i,:,:,iHe))
             q(i,:,:,iH) = q(i,:,:,1)-q(i,:,:,iHe)-q(i,:,:,iHII)
          END IF

       ELSE
          ! postshock region ("cool" values)
          ! 4th order Runge-Kutta method is used to find energy flux Eflux_cool
          CALL RK4(x-xold, 1d-1*dx, Eflux_cool, HIIflux_cool, HeIIflux_cool, Heflux_cool)
          xold = x

          CALL EfluxVars(Eflux_cool, v_cool, rho_cool, By_cool, p_cool)

          q(i,:,:,1) = rho_cool / rScale
          q(i,:,:,2) = q(i,:,:,1) * v_cool / VelScale
          q(i,:,:,iE) = gamma7*p_cool/pScale + half*q(i,:,:,1)*(v_cool/VelScale)**2d0
          IF(lMHD) THEN
             q(i,:,:,iBy) = By_cool/BScale
             q(i,:,:,iE) = q(i,:,:,iE) + half*q(i,:,:,iBy)**2d0
          END IF
         
          IF(iCooling == NEQCool .OR. iCooling == ZCool) THEN
             q(i,:,:,iHII) = HIIflux_cool/v_cool/rScale/muH
!             q(i,:,:,iHeII) = HeIIflux_cool/v_cool/rScale/muHe
             q(i,:,:,iHe) = Heflux_cool/v_cool/rScale/muHe
             q(i,:,:,iH) = q(i,:,:,1) - q(i,:,:,iHe) - q(i,:,:,iHII) !- q(i,:,:,iHeII)
          END IF
          
       END IF
    END DO
  END SUBROUTINE ProblemGridInit

    ! This subroutine returns a numerical value for p_cool. It follows the 4th order
    ! Runge-Kutta integration method to solve for the pressure profile in the energy flux ODE.
    ! RK4(upper bound, pressure at lower bound, desired step size)
    SUBROUTINE RK4(dx, h, Eflux, HIIflux, HeIIflux, Heflux)
       INTEGER :: i, steps
       REAL(KIND=qPREC) :: k1, k2, k3, k4, dx, dh, h, Eflux, ddxEflux
       REAL(KIND=qPREC) :: l1, l2, l3, l4, HIIflux, ddxHIIflux
       REAL(KIND=qPREC) :: j1, j2, j3, j4, HeIIflux, ddxHeIIflux, Heflux

       ! In case dx/h is not an integer, CEILING function is used, and dh is
       ! used to ensure that all of dx is divided into evenly distributed steps.
       steps = CEILING(dx/h)
       dh = dx/REAL(steps)

       DO i=1, steps
          CALL Derivs(Eflux, ddxEflux, HIIflux, ddxHIIflux, HeIIflux, ddxHeIIflux, Heflux)
          k1 = dh*ddxEflux ; l1 = dh*ddxHIIflux; j1 = dh*ddxHeIIflux
          CALL Derivs(Eflux + half*k1, ddxEflux, HIIflux + half*l1, ddxHIIflux, HeIIflux + half*j1, ddxHeIIflux, Heflux - half*j1)
          k2 = dh*ddxEflux ; l2 = dh*ddxHIIflux; j2 = dh*ddxHeIIflux
          CALL Derivs(Eflux + half*k2, ddxEflux, HIIflux + half*l2, ddxHIIflux, HeIIflux + half*j2, ddxHeIIflux, Heflux - half*j2)
          k3 = dh*ddxEflux ; l3 = dh*ddxHIIflux; j3 = dh*ddxHeIIflux
          CALL Derivs(Eflux + k3, ddxEflux, HIIflux + l3, ddxHIIflux, HeIIflux + j3, ddxHeIIflux, Heflux - j3)
          k4 = dh*ddxEflux ; l4 = dh*ddxHIIflux; j4 = dh*ddxHeIIflux
       
          Eflux = Eflux + k1/6d0 + k2/3d0 + k3/3d0 + k4/6d0
          HIIflux = HIIflux + l1/6d0 + l2/3d0 + l3/3d0 + l4/6d0
          HeIIflux = HeIIflux + j1/6d0 + j2/3d0 + j3/3d0 + j4/6d0
          Heflux = Heflux - HeIIflux
       END DO
       
    END SUBROUTINE RK4

    ! This function provides values for the right hand side of the equation in the Runge-Kutta
    ! method in order to find the change in energy flux.
    SUBROUTINE Derivs(Eflux, ddxEflux, HIIflux, ddxHIIflux, HeIIflux, ddxHeIIflux, Heflux)
       REAL(KIND=qPREC) :: ddxEflux, ddxHIIflux, ddxHeIIflux
       REAL(KIND=qPREC) :: p, v, rho, By, T, T_floor, lambda, Eflux, HIIflux, HeIIflux, Heflux, ne, Xh, nH
       REAL(KIND=qPREC),DIMENSION(0:nSpeciesHi) :: nvec
       REAL(KIND=qPREC), DIMENSION(NrHydroVars) :: qin, f

       CALL EfluxVars(Eflux, v, rho, By, p)

       T = p/Boltzmann/(rho/hMass)     
       T_floor = 1d3

       SELECT CASE(iCooling)
          CASE(NoCool)
             lambda = 0d0
          CASE(AnalyticCool)
             lambda = (rho/hMass)**2d0*alpha*T**beta
             IF(T<=T_floor) lambda = 0d0 
         CASE(DMCool)
             lambda = (rho/hMass)**2d0*DMCoolingRate(T)
             IF(T<=T_floor) lambda = 0d0 
          CASE(NEQCool)
             f = 0d0
             qin = 0d0
             qin(1) = rho/rScale
             qin(iHII) = HIIflux/v/rScale/muH
!             qin(iHeII) = HeIIflux/v/rScale/muHe
             qin(iHe) = Heflux/v/rScale/muHe
             qin(iH) = qin(1)-qin(iHII)-qin(iHe) !-qin(iHeII)
             CALL GetNEQvars(qin, mu, nvec)
             T = mu*T
             CALL Cool_Derivatives(qin,f,T,nvec)
             lambda = f(iE)*pScale/timescale
             ddxHIIflux = f(iHII)*rScale/timescale
!             ddxHeIIflux = f(iHeII)*rScale/timescale
             ddxHeIIflux = 0d0
             IF(T<=T_floor) lambda = 0d0 
          CASE(ZCool)
             f = 0d0
             qin = 0d0
             qin(1) = rho/rScale
             qin(iHII) = HIIflux/v/rScale/muH
!             qin(iHeII) = HeIIflux/v/rScale/muHe
             qin(iHe) = Heflux/v/rScale/muHe
             qin(iH) = qin(1) - qin(iHII) - qin(iHe) !- qin(iHeII)
             CALL GetNEQvars(qin, mu, nvec)
             T = mu*T
             CALL Cool_Derivatives(qin,f,T,nvec,ZCool)
             CALL GetZvars(nvec,ne,Xh,nH)
             IF(ZCoolingRate(ne,T,Xh) == 0d0) THEN
                lambda = f(iE)*pScale/timescale
             ELSE
                lambda = (f(iE) + (1d0-Zweight(T))*metal_loss)*pScale/timescale + Zweight(T) * nH**2d0 * ZCoolingRate(ne,T,Xh)
             END IF
             ddxHIIflux = f(iHII)*rScale*muH/timescale
!             ddxHeIIflux = f(iHeII)*rScale*muHe/timescale
             ddxHeIIflux = 0d0
             IF(T<=T_floor) lambda = 0d0
       END SELECT

       ddxEflux = -lambda

    END SUBROUTINE Derivs

    ! Given the energy flux, returns hydro variables
    SUBROUTINE EfluxVars(Eflux, v, rho, By, p)
       INTEGER :: j
       REAL(KIND=qPREC) :: Eflux, v, rho, By, p, a, b, c, d
       COMPLEX(8) :: MyRoots(3)
       REAL(KIND=qPREC) :: ValidRoots(3)

       IF(lMHD) THEN
          ValidRoots = 2d0*v_amb
          a = rho_ps*v_ps*(half-gamma15)
          b = gamma15*(rho_ps*v_ps**2d0 + p_ps + By_ps**2d0/8d0/Pi)
          c = -Eflux
          d = By_ps**2d0*v_ps**2d0/4d0/Pi*(1d0-gamma15/2d0)
          MyRoots = CubicRoots(a,b,c,d)
          DO j=1, 3
             IF(AIMAG(MyRoots(j)) == 0d0 .AND. REAL(MyRoots(j)) > 0d0) THEN
                ValidRoots(j) = REAL(MyRoots(j))
             END IF
          END DO
          v = MINVAL(ValidRoots)
          rho = rho_ps*v_ps/v
          By = By_ps*v_ps/v
          p = rho_ps*v_ps**2d0 + p_ps + By_ps**2d0/(8d0*Pi) - rho*v**2d0 - By**2d0/(8d0*Pi)
       ELSE
          a = rho_ps*v_ps*(half-gamma15)
          b = gamma15*(rho_ps*v_ps**2d0 + p_ps)
          c = -Eflux
          v = (-b + SQRT(b**2d0 - 4d0*a*c))/(2d0*a)
          rho = rho_ps*v_ps/v
          By = 0d0
          p = rho_ps*v_ps**2d0 + p_ps - rho*v**2d0
       END IF
    END SUBROUTINE EfluxVars

    ! Gives the roots to the cubic equation ax^3 + bx^2 + cx + d = 0
    FUNCTION CubicRoots(a, b, c, d)
       COMPLEX(8) :: CubicRoots(3)
       REAL(KIND=qPREC) :: a, b, c, d, p, q, DD, y1, y2, y3
       REAL(KIND=qPREC) :: phi, temp1, temp2, y2r, y2i, u, v

       ! Step 1: Calculate p and q --------------------------------------------
       p  = c/a - b*b/a/a/3d0
       q  = (2d0*b*b*b/a/a/a - 9d0*b*c/a/a + 27d0*d/a) / 27d0

       ! Step 2: Calculate DD (discriminant) ----------------------------------
       DD = p*p*p/27d0 + q*q/4d0

       ! Step 3: Branch to different algorithms based on DD ------------------

       IF(DD .LT. 0d0) THEN
          !       Step 3b:
          !       3 real unequal roots -- use the trigonometric formulation
          phi = ACOS(-q/2d0/SQRT(ABS(p*p*p)/27d0))
          temp1=2d0*SQRT(ABS(p)/3d0)
          y1 =  temp1*COS(phi/3d0)
          y2 = -temp1*COS((phi+pi)/3d0)
          y3 = -temp1*COS((phi-pi)/3d0)
          temp1 = b/a/3d0
          y2 = y2-temp1
          y3 = y3-temp1
       ELSE
          !       Step 3a:
          !       1 real root & 2 conjugate complex roots OR 3 real roots (some are equal)
          temp1 = -q/2d0 + SQRT(DD)
          temp2 = -q/2d0 - SQRT(DD)
          u = ABS(temp1)**(1d0/3d0)
          v = ABS(temp2)**(1d0/3d0)
          IF(temp1 .LT. 0d0) u = -u
          IF(temp2 .LT. 0d0) v = -v
          y1  = u + v
          y2r = -(u+v)/2d0
          y2i =  (u-v)*SQRT(3d0)/2d0
          temp1 = b/a/3d0
          y2r = y2r-temp1
       END IF

        ! Step 4: Final transformation -----------------------------------------

        y1 = y1-temp1

        ! Assign answers -------------------------------------------------------
        IF(DD .LT. 0d0) THEN
           CubicRoots(1) = CMPLX( y1,  0d0)
           CubicRoots(2) = CMPLX( y2,  0d0)
           CubicRoots(3) = CMPLX( y3,  0d0)
        ELSE IF(DD .EQ. 0d0) THEN
           CubicRoots(1) = CMPLX( y1,  0d0)
           CubicRoots(2) = CMPLX(y2r,  0d0)
           CubicRoots(3) = CMPLX(y2r,  0d0)
        ELSE
           CubicRoots(1) = CMPLX( y1,  0d0)
           CubicRoots(2) = CMPLX(y2r, y2i)
           CubicRoots(3) = CMPLX(y2r,-y2i)
        END IF

    END FUNCTION CubicRoots

  ! Place any pre-processing operations here
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemBeforeStep

  ! Place any post-processing operations here
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  ! Can be used to set additional refinement
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem
