!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module SingleClump is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiClumps
!! @brief Contains files necessary for the Shape Tester problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiClumps Shape Tester Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> MultiClump Module
!! @ingroup MultiClumps
MODULE Problem
  USE DataDeclarations
  USE ParticleDeclarations
  USE Clumps
  USE CoolingSrc
  USE Winds  
  USE Ambients
!  USE VectorPerturbation
!  USE Histograms
!  USE Totals
!  USE Fields
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(CoolingDef),POINTER :: coolingobj
  REAL(KIND=qPREC) :: wind_thickness, wind_vel, wind_refinement_time
  TYPE(WindDef), POINTER :: Wind
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      INTEGER :: ClumpTracer, AltClumpTracer,i,j,dir,edge
      Logical :: ClumpToTrace
      TYPE(ClumpDef), POINTER :: Clump
      !TYPE(HistogramDef), POINTER :: Histogram
      !TYPE(TotalDef), POINTER :: KE,TE,BE,ENSTROPHY
      !LOGICAL :: lCooling!, lMagClump, lMHDPerturbation
      LOGICAL :: lCooling
      REAL(KIND=qPREC) :: density, temperature, velocity, smooth_distance, position(3), radius, B(3)
      REAL(KIND=qPREC) :: realden, realradius, realbeta, realtemp, realspeed, realcs, realpressure
      REAL(KIND=qPREC) :: theta=0
      REAL(KIND=qPREC) :: phi=0
      REAL(KIND=qPREC) :: B_tor=0
      REAL(KIND=qPREC) :: B_pol=0
      REAL(KIND=qPREC) :: B_phi=0
      REAL(KIND=qPREC) :: B_theta=0
      TYPE(AmbientDef), POINTER :: Ambient
      REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      !INTEGER :: kmin, kmax
      !REAL(KIND=qPREC) :: beta, field_amp, kscale
      !INTEGER :: nwaves=0, nMHDwaves=0
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /ProblemData/ lCooling, ClumpToTrace!, lMagClump, lMHDPerturbation
      NAMELIST /WindData/ density, temperature, velocity, B,  wind_thickness, wind_refinement_time, dir, edge
      NAMELIST /ClumpData/ density, temperature, radius, smooth_distance, theta, phi, B_tor, B_pol, B_theta, B_phi, position
      !NAMELIST /PerturbData/ nwaves, nMHDwaves
      !NAMELIST /MHDPerturbationData/ kmin, kmax, beta, field_amp, kscale

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      dir=1
      edge=1
      CALL AddTracer(ClumpTracer, 'ClumpTracer')
      CALL AddTracer(AltClumpTracer, 'AltClumpTracer')
      CALL CreateAmbient(Ambient)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      !Ambient%v(:)=(/vxOut, vyOut, vzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)


      READ(PROBLEM_DATA_HANDLE,NML=ClumpData)
      CALL CreateClump(Clump)
      Clump%density=density
      Clump%temperature=temperature
      Clump%position=position
      IF (ClumpToTrace) THEN
         Clump%iTracer=AltClumpTracer
      ELSE
         Clump%iTracer=ClumpTracer
      END IF
      Clump%thickness=smooth_distance
      Clump%radius=radius
      !Clump%Magnetized=lMagClump
      Clump%B_toroidal=B_tor
      Clump%B_poloidal=B_pol
      Clump%theta=theta*pi/180d0
      Clump%phi=phi*pi/180d0
      Clump%B_theta=B_theta*pi/180d0
      Clump%B_phi=B_phi*pi/180d0
      CALL UpdateClump(Clump)

      !IF (lMHDPerturbation) THEN
      !   READ(PROBLEM_DATA_HANDLE, NML=MHDPerturbationData)
      !   CALL CreateVectorPerturbationObj(Clump%MagneticPerturbation)
      !   CALL CreateSolenoidalSpectra(Clump%MagneticPerturbation, kmin, kmax, beta, field_amp, kscale)
      !END IF

      IF (lCooling) THEN
         IF (.NOT. lRestart) THEN
            ! see sources/cooling.f90::CreateCoolingObject for
            ! default values of a cooling source term
            CALL CreateCoolingObject(coolingobj)
         ELSE
            coolingobj => firstcoolingobj
         END IF
         coolingobj%iCooling=DMCOOL
         coolingobj%floortemp=100d0
         coolingobj%mintemp=1
      END IF

      realtemp=(TempScale*ambient%pressure/ambient%density)
      realpressure=pScale*ambient%pressure
      realbeta=2.0*ambient%pressure/SUM((ambient%B(:))**2)
      realden=(rScale*ambient%density)
      realradius=(lScale)*10.0
      realcs=sqrt(gamma*realpressure/realden)

      READ(PROBLEM_DATA_HANDLE,NML=WindData)
      CALL CreateWind(Wind)
      Wind%density=density
      Wind%temperature=temperature
      Wind%velocity=realcs*velocity/VelScale
      Wind%B=B
      wind_vel=realcs*velocity/VelScale
      CALL AddTracer(Wind%iTracer, 'Wind_Tracer')
      Wind%dir=dir
      Wind%edge=edge

      realspeed=(wind%velocity*VelScale)
      IF(mpi_id==0)THEN
        PRINT*, "realdensity = ", realden*1d3, "mg/cc"
        PRINT*, "realtemp = ", Boltzmann*realtemp/1.602d-12, "ev"
        PRINT*, "realradius = ", realradius, "mm"
        PRINT*, "realbeta = ", realbeta
        PRINT*, "realspeed = ", realspeed/1d5, "km/s"
        PRINT*, "timescale = ", timescale
        !PRINT*, "Reynolds# = ", wind%velocity*clump%radius/resistivity
      END IF
      !STOP

      !CALL CreateTotal(KE)
      !KE%Field%id=PKE_Field
      !KE%Field%component=GASCOMP
      !KE%Field%name='PKE'

      !CALL CreateTotal(TE)
      !TE%Field%id=PiE_Field
      !TE%Field%component=GASCOMP
      !TE%Field%name='PTE'

      !CALL CreateTotal(BE)
      !BE%Field%id=BE_Field
      !BE%Field%component=GASCOMP
      !BE%Field%name='BE'

!      CALL CreateHistogram(Histogram)
!      Histogram%Field%id=MixingRatio12_Field
!      Histogram%Field%component=GASCOMP
!      Histogram%minvalue=0d0
!      Histogram%maxvalue=1d0
!      Histogram%nbins=100
!      Histogram%scale=LINEARSCALE
!      Histogram%WeightField=BINBYMASS

       !CALL CreateTotal(ENSTROPHY)
       !ENSTROPHY%Field%iD=Enstrophy_Field
       !ENSTROPHY%Field%Component=GASCOMP
       !ENSTROPHY%Field%name='ENSTROPHY'
       !Processing_mbc=max(Processing_mbc,1)


   END SUBROUTINE ProblemModuleInit

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemGridInit

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: ip(3,2)
      REAL(KIND=qPREC) :: wind_pos, dx
      
      IF (levels(Info%level)%tnow <= wind_refinement_time) THEN
         dx=levels(Info%level)%dx
         wind_pos=wind_vel/2d0*levels(Info%level)%tnow
         ip(:,1)=1
         ip(:,2)=Info%mX
         ip(wind%dir,1)=ceiling((wind_pos-wind_thickness-Info%xBounds(wind%dir,1))/dx)
         ip(wind%dir,2)=ceiling((wind_pos+wind_thickness-Info%xBounds(wind%dir,1))/dx)
         ip(wind%dir,1)=max(ip(wind%dir,1), 1)
         ip(wind%dir,2)=min(ip(wind%dir,2),Info%mX(wind%dir))
         IF (ip(wind%dir,2) >= ip(wind%dir,1)) Info%ErrFlag(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2))=1
      END IF
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep


END MODULE Problem

