!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module SodShockTube is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiClumps
!! @brief Contains files necessary for the Shape Tester problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiClumps Shape Tester Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> MultiClump Module
!! @ingroup MultiClumps
MODULE Problem
  USE DataDeclarations
  USE SplitRegions
  USE Shapes
  USE EOS
  USE Ambients
  USE RiemannSolvers
  USE Totals
  USE Fields
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  REAL(KIND=qPREC), DIMENSION(MaxVars) :: qabove, qbelow
  TYPE(SplitRegionDef), POINTER :: SplitRegion
  REAL(KIND=qPREC) :: phi, theta, position(3)
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      TYPE(AmbientDef), POINTER :: Ambient
      REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      TYPE(TotalDef), POINTER :: Total
      NAMELIST /ProblemData/ position, phi, theta, qabove, qbelow      
      CALL CreateAmbient(Ambient)
      CALL CreateTotal(Total)
      Total%Field%Component=GASCOMP
      Total%Field%id=ivx
      Total%Field%name=FieldName(ivx)
      CALL CreateTotal(Total)
      Total%Field%Component=GASCOMP
      Total%Field%id=1
      Total%Field%name=FieldName(1)
      CALL CreateTotal(Total)
      Total%Field%Component=GASCOMP
      Total%Field%id=iE
      Total%Field%name=FieldName(iE)

      position=(/1d-6,1d-6,0d0/)
      phi=Pi
      theta=half*Pi
      qabove=0
      qbelow=0
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CALL CreateSplitRegion(SplitRegion)     
      SplitRegion%Shape%type=Rectangular_Prism
      SplitRegion%Shape%size_param=(/.5,.5,.5/)
      CALL SetShapeOrientation(Splitregion%Shape,0d0,0d0,0d0)
      CALL SetShapeBounds(splitregion%Shape)
!      write(*,*) splitregion%shape%xbounds
      SplitRegion%InterfaceObj%position=position
      CALL SetInterfaceOrientation(SplitRegion%InterfaceObj, theta, phi)
      SplitRegion%qabove=qabove
      SplitRegion%qbelow=qbelow
      CALL prim_to_cons(SplitRegion%qabove)
      CALL prim_to_cons(SplitRegion%qbelow)
      IF (lMHD .AND. GmX(1)==GmX(2)) THEN
         SplitRegion%PersistInBoundaries=.true.
         SplitRegion%subsample=20
      ELSE
         SplitRegion%PersistInBoundaries=.false.
      END IF

!      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
!      Ambient%density=qabove(1)!rhoOut
!      Ambient%pressure=qabove(2)!pOut
!      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
!      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)
!      CLOSE(PROBLEM_DATA_HANDLE)

   END SUBROUTINE ProblemModuleInit

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j
      IF (MaintainAuxArrays) THEN
         IF (Info%mx(1) > Info%mX(2)) Info%aux(1:Info%mX(1),1:Info%mx(2)+1,1,2) = SPREAD(Info%q(1:Info%mX(1),1,1,iBy),2,Info%mX(2)+1)
         IF (Info%mX(2) > Info%mX(1)) Info%aux(1:Info%mX(1)+1,1:Info%mx(2),1,1) = SPREAD(Info%q(1,1:Info%mX(2),1,iBx),1,Info%mX(1)+1)
         IF (lMHD .AND. Info%mX(1) == Info%mX(2)) THEN !Angled aux fields -  need to calculate potential
            DO j=1, Info%mX(2)
               Info%aux(1:j,Info%mx(2)+1-j,1,1)=Info%q(1,1,1,iBx)
               Info%aux(j+1:Info%mX(1)+1,Info%mx(2)+1-j,1,1)=Info%q(Info%mX(1),Info%mX(2),1,iBx)
            END DO
            DO i=1, Info%mX(1)
               Info%aux(info%mx(1)+1-i,1:i,1,2)=Info%q(1,1,1,iBy)
               Info%aux(info%mx(1)+1-i,i+1:Info%mX(2)+1,1,2)=Info%q(Info%mX(1),Info%mX(2),1,iBy)
            END DO
            Info%q(1:Info%mX(1),1:Info%mX(2),1,iBx)=half*(Info%aux(1:Info%mX(1),1:Info%mX(2),1,1)+Info%aux(2:Info%mX(1)+1,1:Info%mX(2),1,1))
            Info%q(1:Info%mX(1),1:Info%mX(2),1,iBy)=half*(Info%aux(1:Info%mX(1),1:Info%mX(2),1,2)+Info%aux(1:Info%mX(1),2:Info%mX(2)+1,1,2))
         END IF
      END IF
   END SUBROUTINE ProblemGridInit

   !> Applies Boundary conditions
   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: rmbc,i,j
      REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: tempaux
      rmbc=levels(Info%level)%gmbc(1)
      IF (lMHD .AND. nDim == 2 .AND. Info%mX(1) == Info%mX(2)) THEN !Angled aux fields -  need to calculate potential
         ALLOCATE(tempaux(1:Info%mX(1)+1,1:Info%mX(2)+1,2))
         tempaux=Info%aux(1:Info%mX(1)+1,1:Info%mX(2)+1,1,1:2)

         DO j=-rmbc, Info%mX(2)+rmbc
            Info%aux(1-rmbc:j,Info%mx(2)+1-j,1,1)=Info%q(1,1,1,iBx)
            Info%aux(j+1:Info%mX(1)+1+rmbc,Info%mx(2)+1-j,1,1)=Info%q(Info%mX(1),Info%mX(2),1,iBx)
         END DO
         DO i=1-rmbc, Info%mX(1)+rmbc
            Info%aux(info%mx(1)+1-i,1-rmbc:i,1,2)=Info%q(1,1,1,iBy)
            Info%aux(info%mx(1)+1-i,i+1:Info%mX(2)+1+rmbc,1,2)=Info%q(Info%mX(1),Info%mX(2),1,iBy)
         END DO
         Info%aux(1:Info%mX(1)+1,1:Info%mX(2),1,1)=tempaux(1:Info%mX(1)+1,1:Info%mX(2),1)
         Info%aux(1:Info%mX(1),1:Info%mX(2)+1,1,2)=tempaux(1:Info%mX(1),1:Info%mX(2)+1,2)

         Info%q(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,iBx)=half*(Info%aux(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,1)+Info%aux(2-rmbc:Info%mX(1)+1+rmbc,1-rmbc:Info%mX(2)+rmbc,1,1))
         Info%q(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,iBy)=half*(Info%aux(1-rmbc:Info%mX(1)+rmbc,1-rmbc:Info%mX(2)+rmbc,1,2)+Info%aux(1-rmbc:Info%mX(1)+rmbc,2-rmbc:Info%mX(2)+1+rmbc,1,2))
         DEALLOCATE(tempaux)
      END IF

   END SUBROUTINE ProblemBeforeStep

   !> Could be used to update grids pre-output
   !! @param Info Info Object
   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:) :: qExact
      REAL(KIND=qPREC) :: um, s, max_speed
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: wmiddle
      IF (levels(info%level)%tnow+levels(info%level)%dt == final_time) THEN
         OPEN(UNIT=11, FILE='out/data.curve', status='unknown')
         write(11,*) '# rho'         
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,1)
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,1)
         END DO
         write(11,*) 
         write(11,*) 
         write(11,*) '# vx'
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivx)/Info%q(i,1,1,1)
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivx)/Info%q(i,i,1,1)
         END DO
         write(11,*) 
         write(11,*) 
         write(11,*) '# P'
         DO i=1, Info%mX(1)
            IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, (Press(Info%q(i,1,1,:)))
            IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), (Press(Info%q(i,i,1,:)))
         END DO
         write(11,*) 
         write(11,*) 

         IF (lMHD) THEN
            write(11,*) '# vy'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivy)/Info%q(i,1,1,1)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivy)/Info%q(i,i,1,1)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# vz'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivz)/Info%q(i,1,1,1)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,ivz)/Info%q(i,i,1,1)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# Bx'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBx)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBx)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# By'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBy)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBy)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# Bz'
            DO i=1, Info%mX(1)
               IF (Info%mX(2) /= Info%mX(1)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,iBz)
               IF (Info%mX(1)==Info%mX(2)) write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx*sqrt(2d0), Info%q(i,i,1,iBz)
            END DO
            write(11,*) 
            write(11,*) 
         ELSE
            ALLOCATE(qExact(Info%mX(1), NrHydroVars))
            ALLOCATE(wmiddle(1:NrHydroVars))
            DO i=1, Info%mX(1)
               S=(Info%xBounds(1,1)+(REAL(i)-half)*Levels(Info%level)%dx-position(1))/final_time
               IF (Info%mX(1)==Info%mX(2)) S=S*sqrt(2d0)
               CALL vacuum_solve(qabove((/1,3,2/)), qbelow((/1,3,2/)), wmiddle, um, s, max_speed)
               qExact(i,:)=wmiddle
            END DO
            write(11,*) '# rho_Exact'
            DO i=1, Info%mX(1)
               write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, qExact(i,1)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# vx_Exact'
            DO i=1, Info%mX(1)
               write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, qExact(i,3)
            END DO
            write(11,*) 
            write(11,*) 
            write(11,*) '# P_Exact'
            DO i=1, Info%mX(1)
               write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, qExact(i,2)
            END DO
            CLOSE(11)
            IF (Info%mX(2) /= Info%mX(1)) write(*,*) 'dx, L2 Norm=', levels(Info%level)%dx, sum(abs(qExact(:,1)-Info%q(1:Info%mX(1),1,1,1)))/Info%mX(1)
            IF (Info%mX(1)==Info%mX(2)) write(*,*) 'dx, L2 Norm=', levels(Info%level)%dx, sum(abs(qExact(:,1)-(/(Info%q(i,i,1,1), i=1,Info%mX(1))/)))/Info%mX(1)
         END IF
      END IF

   END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

