!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module StreamDisk is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir StreamDisk
!! @brief Contains files necessary for the StreamDisk Calculation

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup StreamDisk StreamDisk Interaction Module
!! @brief Module for studying the impact of a stream on a disk
!! @ingroup Modules

!> StreamDisk Interaction Module
!! @ingroup StreamDisk
MODULE Problem
  USE DataDeclarations
  USE PhysicsDeclarations
  USE ProcessingDeclarations
  USE Ambients
  USE Disks
  USE PointGravitySrc
  USE Profiles
  USE CoolingSrc
  USE Refinements
  USE Projections
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep

  PRIVATE
  REAL(KIND=qPREC) :: ambient_density, ambient_temperature, & !ambient params
       disk_softlength, disk_density, disk_temperature, disk_radius, disk_smoothing, disk_height, & !disk params
       stream_density, stream_temperature, stream_velocity, stream_radius, stream_halfwidth, stream_halfheight, stream_xpos, & !stream params
       central_mass, & !star params
       hotspot_width, hotspot_pos(3) !hot spot params
  INTEGER :: disk_maxlevel !maximum level to refine disk
  LOGICAL :: lCooling,lGlobalSim, lMatchPressure, lRefineHotSpot !Logical switches
  INTEGER :: stream_tracer=0 !module integer handle for stream tracer
  TYPE(AmbientDef), POINTER :: Ambient !module pointer for ambient

CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
      ! Declare Objects initialized here
      TYPE(DiskDef), POINTER :: Disk
      TYPE(PointGravityDef), POINTER :: PointGravity
      TYPE(CoolingDef),POINTER :: coolingobj  
      TYPE(RefinementDef), POINTER :: DiskRefinement, HotSpotRefinement, DiskAntiRefinement, StreamInjectionRefinement, StreamRefinement
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: i, j, k  ! Miscellaneous counters
      INTEGER :: nPoints=400 !number of points in reconstructing hydrostatic equilibrium solution
      REAL(KIND=qPREC) :: r, r_max, p_inf, phi, rho

      NAMELIST /ProblemData/ ambient_density, ambient_temperature, &
           disk_softlength, disk_density, disk_temperature, disk_radius, disk_height, disk_smoothing, disk_maxlevel, &
           stream_density, stream_temperature, stream_velocity, stream_halfwidth, stream_halfheight, stream_xpos, &
           central_mass, &
           lRefineHotSpot, hotspot_width, hotspot_pos, &
           lCooling, lGlobalSim, lMatchPressure

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CLOSE(PROBLEM_DATA_HANDLE)


      !Point Gravity Stuff
      IF (.NOT. lRestart) THEN
         CALL CreatePointGravityObject(PointGravity)
      ELSE
         PointGravity=>FirstPointGravityObj
      END IF
      PointGravity%soft_length=disk_softlength*disk_radius/lScale
      PointGravity%soft_function=SPLINESOFT
      PointGravity%mass=central_mass*mSolar/mScale

      GravityDim = 3 !We want point gravity objects to have 1/r^2 force - not 1/r 

      ! Check that softening length is resolved to avoid lots of numerical dissipation and heating at disk center
      IF (PointGravity%soft_length/levels(0)%dx < 4d0) THEN
         IF (MPI_ID == 0) WRITE(*,*) 'PointGravity Softening Radius is only ', PointGravity%soft_length/levels(0)%dx, ' root cells'
         STOP
      END IF


      !Ambient Stuff
      CALL CreateAmbient(Ambient)
      Ambient%density=ambient_density/rScale
      Ambient%pressure=Ambient%density*ambient_temperature/TempScale
      Ambient%PersistInBoundaries=(gmthbc == 1) !Any open boundary should maintain ambient values

      !Disk Stuff
      CALL CreateDisk(Disk)
      Disk%density=disk_density/rScale
      Disk%pressure=Disk%density*disk_temperature/TempScale
      Disk%radius=disk_radius/lScale
      Disk%soft_length=PointGravity%soft_length
      Disk%soft_function=PointGravity%soft_function
      Disk%central_mass=PointGravity%mass
      Disk%thickness=disk_smoothing
      Disk%height=disk_height
      Disk%PersistInBoundaries(1,1)=.NOT. lGlobalSim
      CALL AddTracer(Disk%iTracer, 'Disk Tracer')
      CALL Updatedisk(Disk)


      !Now we need to calculate maximum distance from disk to any given cell
      !First get maximum distance to each corner from (0d0,0d0,0d0)
      r_max=0d0
      DO i=1,2
         DO j=1,merge(2,1,nDim >= 2)
            DO k=1,merge(2,1,nDim >= 3)
               r_max=max(r_max, sqrt(GxBounds(1,i)**2+GxBounds(2,j)**2+GxBounds(3,k)**2))
            END DO
         END DO
      END DO
      !Add in extra ghost zones
      r_max=r_max+sqrt(REAL(nDim))*levels(0)%dx*6 !make sure we have enough distance to include ghost zones...
      IF (lMatchPressure) THEN
         !For isothermal atmosphere, rho = rho_amb * exp(-phi/T_amb)
         !and pressure goes as P_amb = rho_amb * T_amb * exp(-phi/T_amb)
         !So we need to solve for rho_amb so that P_amb = P_disk at r=r_disk
         ! rho_amb = P_disk/T_amb/exp(-phi/T_amb)
         phi=GravityPotential(PointGravity%mass, (/disk_radius/lScale,0d0,0d0/), PointGravity%soft_length,PointGravity%soft_function)
         ambient%density=(Disk%pressure/(ambient_temperature/TempScale))/exp(-phi/(ambient_temperature/TempScale))
         ! More Ambient stuff to get pressure matching at disk edge
         CALL CreateProfile(Ambient%profile, nPoints, (/Mass_Field, P_Field/), RADIAL)
         DO i=1,nPoints
            r=REAL(i-1)/REAL(nPoints)*r_max
            !Here we are not just using softening to get ambient density near star, but also fixing the ambient density within the sphere surrounding the disk to be constant to avoid Nan's in the ambient solution
            phi=GravityPotential(PointGravity%mass, (/max(r, disk_radius/lScale),0d0,0d0/), PointGravity%soft_length,PointGravity%soft_function)
            rho=ambient%density*exp(-phi/(ambient_temperature/TempScale))
            Ambient%profile%data(i,1:2) = (/r, rho/)         
         END DO

         CALL Profile_PointGravityHSE(Ambient%profile, PointGravity, ambient%pressure)
      END IF

      CALL UpdateAmbient(Ambient)


      ! Stream stuff
      stream_density=stream_density/rScale
      stream_temperature=stream_temperature/TempScale
      stream_velocity=stream_velocity*1e5/velscale
      stream_xpos=stream_xpos/lScale
      stream_halfwidth=stream_halfwidth/lScale
      stream_halfheight=stream_halfheight/lScale
      CALL AddTracer(stream_tracer, 'Stream Tracer')

      ! Cooling Stuff
      IF (lCooling) THEN
         IF (.NOT. lRestart) THEN
            CALL CreateCoolingObject(coolingobj)
            coolingobj%iCooling = DMCool 
            coolingobj%floortemp = MinTemp
            coolingobj%mintemp = MinTemp
         END IF
      END IF

      ! Any diagnostics in chombo file?
      CALL AddDiagnosticVar(CoolingStrength_Field, 'Cooling Strength')

      ! Refinement Stuff     
      CALL ClearAllRefinements()

      ! Refine Disk
      CALL CreateRefinement(DiskRefinement)
      CALL CreateShape(DiskRefinement%Shape, CYLINDER, 1.2*(/Disk%radius, Disk%radius, Disk%height/))
      DiskRefinement%MaxLevel=disk_maxlevel

      ! Derefine Disk center
      CALL CreateRefinement(DiskAntiRefinement)
      DiskAntiRefinement%Tolerance=DEREFINE_INSIDE
      DiskAntiRefinement%MaxLevel=disk_maxlevel
      CALL CreateShape(DiskAntiRefinement%Shape, CYLINDER, .5*(/Disk%radius, Disk%radius, Disk%height/))

      ! Hotspot refinement
      IF (lRefineHotSpot) THEN
         CALL CreateRefinement(HotSpotRefinement)
         CALL CreateShape(HotSpotRefinement%Shape, RECTANGULAR_PRISM, hotspot_width*(/half,half,half/))
         HotSpotRefinement%Shape%position=hotspot_pos
      END IF

      ! Stream refinement of injection region
      CALL CreateRefinement(StreamInjectionRefinement)
      CALL CreateShape(StreamInjectionRefinement%Shape, RECTANGULAR_PRISM, 1.8*stream_radius*(/stream_halfwidth, stream_halfwidth, stream_halfheight/)) !make it 20% larger than stream
      StreamInjectionRefinement%Shape%position=(/stream_xpos, GxBounds(2,1), 0d0/)

      ! Stream refinement based on gradients in stream_density
      CALL CreateRefinement(StreamRefinement)
      StreamRefinement%Field=stream_tracer
      StreamRefinement%scale=LINEARSCALE
      StreamRefinement%tolerance=1d0/stream_density

      ! Processing Stuff
      CALL CreateProjection(Projection)
      Projection%Field%id=Mass_Field
      Projection%Field%component=GASCOMP
      Projection%Field%name='ColumnDensity'
      Projection%dim=3d0

      CALL CreateProjection(Projection)
      Projection%Field%id=Mass_Field
      Projection%Field%component=GASCOMP
      Projection%Field%name='ColumnDensity'
      Projection%dim=1d0

   END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k, mbc(3)
     REAL(KIND=xPREC) :: pos(3), dx(3), rho, r, x, z

     !determine number of ghost zones for each dimension
     mbc=levels(Info%level)%gmbc(levels(Info%level)%step)*merge((/1,1,1/),(/0,0,0/),nDim>=(/1,2,3/))
     dx=levels(Info%level)%dx*merge((/1,1,1/),(/0,0,0/),nDim>=(/1,2,3/))

     ! Initialize stream from y=0 boundary
     DO j=1-mbc(2), Info%mX(2)+mbc(2)
        pos(2)=Info%xBounds(2,1)+(REAL(j, KIND=qPREC)-half)*dx(2)
        IF (pos(2) < GxBounds(2,1)) THEN
           DO i=1-mbc(1), Info%mX(1)+mbc(1)
              pos(1)=Info%xBounds(1,1)+(REAL(i, KIND=qPREC)-half)*dx(1)
              x = (pos(1)-stream_xpos)/stream_halfwidth
              DO k=1-mbc(3), Info%mX(3)+mbc(3)
                 pos(3)=Info%xBounds(3,1)+(REAL(k, KIND=qPREC)-half)*dx(3)
                 z = pos(3)/stream_halfheight
                 r=sqrt(x**2+z**2)
                 rho=stream_density*exp(-r**2)
                 IF (rho >= ambient%density) THEN
                    IF (stream_tracer /= 0) Info%q(i,j,k,stream_tracer)=rho
                    Info%q(i,j,k,irho) = rho
                    Info%q(i,j,k,ivx)=0d0
                    Info%q(i,j,k,ivy)=rho*stream_velocity
                    IF (ivz /= 0) Info%q(i,j,k,ivz)=0d0
                    IF (iE /= 0) Info%q(i,j,k,iE)=gamma7*rho*stream_temperature+half*rho*stream_velocity**2        
                 END IF
              END DO
           END DO
        END IF
     END DO
  END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

