!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module UniformCollapse is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir UniformCollapse
!! @brief Contains files necessary for the UniformCollapse Calculation

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup UniformCollapse Uniform Collapse Module
!! @brief Module for calculating collapse of a uniform cloud
!! @ingroup Modules

!> Uniform Collapse Module 
!! @ingroup UniformCollapse
MODULE Problem
  USE Clumps
  USE Winds
  USE DataDeclarations
  USE Ambients
  IMPLICIT NONE
  SAVE
  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(ClumpDef), POINTER :: myclump
  TYPE(WindDef), POINTER :: Wind
  INTEGER :: nWinds
  TYPE(AmbientDef), POINTER :: Ambient
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
      INTEGER :: i,edge
      REAL(KIND=qPREC) :: radius=1    !clump radius
      REAL(KIND=qPREC) :: thickness=.1 !thickness of clump smoothing region
      REAL(KIND=qPREC) :: density=10   !Clump peak density
      REAL(KIND=qPREC) :: temp=.1      !Clump temperature
      REAL(KIND=qPREC), DIMENSION(3) :: velocity= (/0,0,0/)  !Clump velocity (in direction of clump axis)
      REAL(KIND=qPREC) :: theta=0     !Angle between X-axis and clump axis (towards y-axis)
      REAL(KIND=qPREC) :: phi=0       !Angle around X-axis to clump axis
      !     REAL(KIND=qPREC) :: mu=0        !Defines ratio of maximum magnetic pressure to ambient pressure
      !     REAL(KIND=qPREC) :: eta=0       !Parameter that determines ratio of maximum poloidal pressure to maximum toroidal pressure
      !     REAL(KIND=qPREC) :: B_theta=0   !Angle from clump axis to define clump orientation
      !     REAL(KIND=qPREC) :: B_phi=0     !Rotation around clump axis to define clump orientation(velocity)
      REAL(KIND=qPREC) :: B_tor=0     !Maximum Bfield for toroidal configuration
      REAL(KIND=qPREC) :: B_pol=0     !Maximum Bfield for poloidal configuration
      REAL(KIND=qPREC) :: Omega=0     !Solid body angular rotation
      REAL(KIND=qPREC) :: m2A         !Azimuthal density perturbation
      REAL(KIND=qPREC), DIMENSION(3) :: xloc=(/0,0,0/)  !Clump location
      INTEGER :: iTracer                                !Clump Tracer


      REAL(KIND=qPREC) :: alpha        !< True love alpha parameter (ratio of thermal to gravitational energy). \f$ \alpha = \frac{5}{2} \left ( \frac{3}{4 \pi \rho_o M^2} \right )^(1/3) \frac{c_s^2}{G} \f$
      REAL(KIND=qPREC) :: beta_rot   !< True love beta rotational parameter (ratio of rotational to gravitational energy). \f$ \beta_{\Omega} = \frac{1}{4 \pi} \frac{\Omega^2}{G \rho_o} \f$
      REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut

      NAMELIST /ProblemData/ density,velocity, xloc, radius, thickness, temp,theta, phi, B_tor, B_pol, beta_rot, alpha, m2A
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      CALL CreateAmbient(Ambient)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)
      CLOSE(PROBLEM_DATA_HANDLE)
      CALL CreateClump(myclump)
      ! Modify density or temperature based on alpha parameter
      IF (alpha == 0 .OR. .NOT. lSelfGravity) THEN
!         myclump%density=density
         IF (MPI_ID == 0 .AND. lSelfGravity) write(*,*) 'alpha = ', 5d0/2d0*gamma*Temp/ScaleGrav/density/(4d0/3d0*pi*radius**2)
      ELSE
         IF (iEOS == EOS_ISOTHERMAL) THEN
            density=5d0/2d0*Iso_Speed2/ScaleGrav/alpha/(4d0/3d0*pi*radius**2)
            IF (MPI_ID == 0) write(*,*) "adjusting clump density to", density
         ELSE
            temp=2d0/5d0*density*ScaleGrav*alpha*(4d0/3d0*pi*radius**2)/gamma
            IF (MPI_ID == 0) write(*,*) "adjusting clump temperature to", temp
         END IF
      END IF

      myclump%temperature=temp
      myclump%density=density
      ! Calculate solid body rotational velocity by beta_rot
      myclump%omega = sqrt(4d0*pi*ScaleGrav*density*beta_rot)
      myclump%velocity=velocity
      myclump%theta=theta
      myclump%phi=phi
      myclump%B_toroidal=B_tor
      myclump%B_poloidal=B_pol
      myclump%position=xloc
      myclump%thickness=thickness
      myclump%radius=radius
      myclump%m2A=m2A
      CALL UpdateClump(myclump)

      nWinds=0
      DO i=1,nDim
         DO edge=1,2
            IF (Gmthbc(i,edge) == 1) THEN 
               nWinds=nWinds+1
               CALL CreateWind(Wind)
               Wind%dir=i
               Wind%edge=edge
            END IF
         END DO
      END DO
   END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: i
    INTEGER, DIMENSION(0:2) :: Steps = (/1,4,11/)
    LOGICAL, DIMENSION(0:2) :: RestartTriggered=(/.false.,.false.,.false./)
    IF (Info%level > 2) RETURN
    i = levels(Info%level)%CurrentLevelStep
    IF (steps(Info%level) == i .AND. MPI_ID == 0 .AND. .NOT. RestartTriggered(Info%level)) THEN
       write(*,*) 'Processor ', MPI_ID, ' purposely requesting restart on level ', Info%level, 'step ', i, 'to test code'
       lRequestRestart=.true.
       RestartTriggered(Info%level)=.true.
    END IF

!    DO i=1,nWinds
!       CALL BeforeStepWind(Info,Wind(i))
!    END DO
  END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

