!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    module_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir modules
!! @brief Contains modules for handling initial and boundary conditions

!> @file module_control.f90
!! @brief Main file for module ModuleControl

!> @defgroup Modules Modules
!! @brief Initial and boundary condition modules

!> @defgroup ModuleControl Module Control
!! @brief Manages various initial and boundary condition modules
!! @ingroup Modules

!> Manages various initial and boundary condition modules
!! @ingroup ModuleControl
MODULE ModuleControl

   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE Clumps
   USE Disks
   USE UniformRegions
   USE SplitRegions
   USE CollidingFlows
   USE Winds
   USE Ambients
   USE Outflows
   USE Problem
   USE ParticleDeclarations
   USE Refinements
   USE ObjectControl
   IMPLICIT NONE

   PUBLIC ModuleObjectsInit, ModuleProblemInit, GridInit, BeforeStep, SetErrFlag, ApplyPhysicalBC, AfterFixup, BeforeGlobalStep

!   NAMELIST /ModulesData/ rhoOut,pOut,vxOut,vyOut,vzOut,BxOut,ByOut,BzOut
CONTAINS

   SUBROUTINE BeforeGlobalStep(n)
      USE PointGravitySrc
      INTEGER :: n
      IF (n == MaxLevel) THEN
         CALL CheckParticlePositions()
      END IF
      CALL ClumpBeforeGlobalStep(n)
      CALL ProblemBeforeGlobalStep(n)
   END SUBROUTINE BeforeGlobalStep
   
   !> Reads in active modules and initializes modules
   SUBROUTINE ModuleObjectsInit
      !CALL InitAmbients
      !CALL UpdateClumps
      !CALL UpdateDisks
      !CALL InitWinds
      !CALL InitOutflows
      !CALL InitCollidingFlows
      !CALL InitUniformRegions
      !CALL InitSplitRegions
      !CALL InitRefinements
      
      ! Read in modules' data files
   END SUBROUTINE ModuleObjectsInit


   !> Reads in active modules and initializes modules
   SUBROUTINE ModuleProblemInit()
      INTEGER :: i
      !Need to create default refinements based on refinevariablefactor
      IF (RefineVariableFactor(1) /= 0d0) CALL AddRefinementCriterion(Mass_Field, RefineVariableFactor(1))
      IF (iE /= 0) THEN
         IF (RefineVariableFactor(iE) /=0d0) CALL AddRefinementCriterion(P_Field, RefineVariableFactor(iE))
      END IF
      DO i=m_low, m_high
         IF (RefineVariableFactor(i) /= 0d0) CALL AddRefinementCriterion(i, RefineVariableFactor(i), RhoSoundSpeed_Field)
      END DO
      IF (lMHD) THEN
         DO i=iBx, iBz
            IF (RefineVariableFactor(i) /= 0d0) CALL AddRefinementCriterion(i, RefineVariableFactor(i), SqrtPress_Field)
         END DO
      END IF
      IF (lSelfGravity) THEN
         CALL AddRefinementCriterion(GasPotential_Field, RefineVariableFactor(i), SoundSpeed2_Field, LINEARSCALE)
         CALL AddRefinementThreshold(JeansLength_Field, LESSTHAN, (/(4d0*levels(i)%dx,i=0,MaxLevel)/))
      END IF
      CALL ProblemModuleInit()
!      CALL InitAmbient
   END SUBROUTINE ModuleProblemInit


   !> Manages initial conditions for an info object
   !! @param Info Info object
   SUBROUTINE GridInit(Info)
      TYPE (InfoDef) :: Info            ! Data associated with this grid
      INTEGER :: mB(3,2), rmbc

      mb(ndim+1:3,:)=1
      mb(1:nDim,1)=1
      mb(1:nDim,2)=Info%mX(1:nDim)
      IF (Info%level == 0) THEN
         IF (iPhiDot /= 0) Info%q(:,:,:,iPhiDot)=0
         IF (iPhiGas /= 0) Info%q(:,:,:,iPhiGas) = 0
      END IF
      !CALL AmbientGridInit(Info) 
      !CALL CollidingFlowGridInit(Info)
      !CALL ClumpGridInit(Info)
      !CALL DiskGridInit(Info)
      !CALL WindGridInit(Info)
      !CALL OutflowGridInit(Info)
      !CALL UniformRegionGridInit(Info)
      !CALL SplitRegionGridInit(Info)

!      CALL ObjectsGridInit(Info, AMBIENTOBJ)
!      CALL ObjectsGridInit(Info, COLLIDINGFLOWOBJ)
!      CALL ObjectsGridInit(Info, CLUMPOBJ)
!      CALL ObjectsGridInit(Info, DISKOBJ)
!      CALL ObjectsGridInit(Info, WINDOBJ)
!      CALL ObjectsGridInit(Info, OUTFLOWOBJ)
!      CALL ObjectsGridInit(Info, UNIFORMREGIONOBJ)
!      CALL ObjectsGridInit(Info, SPLITREGIONOBJ)

      CALL ObjectsGridInit(Info)

      CALL ProblemGridInit(Info)
      CALL Protectq(Info, mb, 'grid init', .TRUE.)
      CALL CheckDivergence(Info, .TRUE.)
   END SUBROUTINE GridInit

   !> Manages boundary conditions (and possibly source conditions) for active modules
   !! @param Info Info object
   !! @details Source terms are better handled within the source module
   SUBROUTINE BeforeStep(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i, mB(3,2), rmbc
      IF (irhoOld /= 0) Info%q(:,:,:,irhoOld)=Info%q(:,:,:,irho)
      DO i=1,nDim
         IF (ivOld(i) /= 0) Info%q(:,:,:,ivOld(i))=Info%q(:,:,:,imom(i))/Info%q(:,:,:,irho)
      END DO
      mb(ndim+1:3,:)=1
      rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)
      mb(1:nDim,1)=1-rmbc
      mb(1:nDim,2)=Info%mX(1:nDim)+rmbc


      ! If the floor temperature is set, then make sure the energy density 
      ! throughout the grid does not drop below the minimum established by
      !  MinTemp * rho + 1/2 (rho*|v|^2 + |B|^2).

      !   IF(MinTemp > zero .AND. iE /= 0) THEN
      !      IF(lMHD) THEN
      !        Info%q(:,:,:,iE) = MAX(Info%q(:,:,:,iE), &
      !              MinTemp/EOSConstants*Info%q(:,:,:,1)/(gamma1)+ &
      !              half*SUM(Info%q(:,:,:,m_low:m_high)**2,DIM=4)/Info%q(:,:,:,1)+&
      !              half*SUM(Info%q(:,:,:,iBx:iBz)**2,DIM=4))
      !      ELSE
      !        Info%q(:,:,:,iE) = MAX( Info%q(:,:,:,iE), &
      !             MinTemp/EOSConstants*Info%q(:,:,:,1)/(gamma1)+ &
      !              half*SUM(Info%q(:,:,:,m_low:m_high)**2,DIM=4)/Info%q(:,:,:,1))
      !      END IF

      !    END IF

      ! set EMF mask on level 0.
      !    IF(lMHD .AND. MaintainAuxArrays .AND. Info%level==0) THEN
      !       Info%childemf = UNDEFINED
      !    END IF      

      ! *** NO GENERIC BEFORESTEP ***
      !CALL Protectq(Info, mb, 'before step')
      !IF (lWinds) CALL WindsBeforeStep(Info)
      !IF (lOutflows) CALL OutflowBeforeStep(Info)
      !CALL UniformRegionBeforeStep(Info)
      !CALL SplitRegionBeforeStep(Info)
      !CALL CollidingFlowBeforeStep(Info)
      !CALL ClumpBeforeStep(Info)
      ! *** NO GENERIC BEFORESTEP ***

      CALL Protectq(Info, mb, 'before step')
 
!      CALL ObjectsBeforeStep(Info, AMBIENTOBJ)
!      CALL ObjectsBeforeStep(Info, WINDOBJ)
!      CALL ObjectsBeforeStep(Info, DISKOBJ)
!      CALL ObjectsBeforeStep(Info, OUTFLOWOBJ)
!      CALL ObjectsBeforeStep(Info, UNIFORMREGIONOBJ)
!      CALL ObjectsBeforeStep(Info, SPLITREGIONOBJ)
!      CALL ObjectsBeforeStep(Info, COLLIDINGFLOWOBJ)
!      CALL ObjectsBeforeStep(Info, CLUMPOBJ)

      CALL ObjectsBeforeStep(Info)

      CALL ProblemBeforeStep(Info)
      IF (lCheckDivergence) CALL CheckDivergence(Info)
      
   END SUBROUTINE BeforeStep


   !> Performs any module updates that should happen after a step
   !! @param Info Info object
   SUBROUTINE AfterStep(Info)
  
      TYPE(InfoDef) :: Info
      INTEGER :: i, mB(3,2), rmbc
      CALL ProblemAfterStep(Info)
      mb(ndim+1:3,:)=1
      rmbc=levels(Info%level)%ambc(levels(Info%level)%step)
      mb(1:nDim,1)=1-rmbc
      mb(1:nDim,2)=Info%mX(1:nDim)+rmbc
      CALl Protectq(Info, mb, 'after step')
   END SUBROUTINE AfterStep

   !> Performs any grid updates that should happen after a grid is updated from finer levels
   !! @param Info Info object
   SUBROUTINE AfterFixup(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE AfterFixup


   !> Sets err flags based on refinement criteria
   !! @param Info Info object
   SUBROUTINE SetErrFlag(Info)
      USE ParticleInfoOps
      USE ParticleDeclarations
      TYPE(InfoDef) :: Info
      INTEGER :: rmbc, iErr, n, level
      INTEGER, DIMENSION(3) :: mxL, mxH
      INTEGER :: i,j,k
      REAL(KIND=qPrec) :: x, y, z
      REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: qbuf, cs
      level=Info%level
      rmbc = levels(level)%gmbc(levels(level)%step)
      Info%ErrFlag=0
      CALL SinkSetErrFlag(Info)
      CALL ObjectsSetErrFlag(Info)
      CALL ProblemSetErrFlag(Info)
   END SUBROUTINE SetErrFlag

   !> Manages physical boundary conditions for an info object
   !! @param Info Info object
   SUBROUTINE ApplyPhysicalBC(Info)
      ! Interface declarations  
      TYPE (InfoDef) :: Info  
      ! Internal declarations    
      INTEGER rmbc, level,dim,edge,start,ncells
      INTEGER, DIMENSION(3,2):: lGmGlobal, mB
      level=Info%level
      rmbc=levels(level)%gmbc(levels(level)%step)
      lGmGlobal(:,1)=GmGlobal(:,1)
      lGmGlobal(:,2)=GmGlobal(:,2)*PRODUCT(levels(0:level-1)%CoarsenRatio)  
      DO dim=1,nDim
         IF (lHydroPeriodic(dim)) CYCLE
         DO edge=1,2
            IF (edge == 1) THEN
               start=lGmGlobal(dim,1)-Info%mGlobal(dim,1)  !first cell on left boundary
               nCells=start-(1-rmbc)+1
            ELSE
               start=(lGmGlobal(dim,2)+1)-(Info%mGlobal(dim,1)-1) !first ghost cell on right boundary
               nCells=Info%mx(dim)+rmbc-start+1
            END IF
            IF (nCells > 0) CALL SetGhost(dim,edge,start,nCells,Info,Gmthbc(dim,edge),rmbc) 
         END DO
      END DO

   CONTAINS

      SUBROUTINE SetGhost(dim, edge, start, nCells, Info, mthbc,rmbc)
         INTEGER :: dim, start, nCells, mthbc,edge,dir, level, rmbc,i,j,m
         TYPE(InfoDef) :: Info
         INTEGER, DIMENSION(3,2) :: ip,iq,ir,is
         LOGICAL :: lFirst
         INTEGER, DIMENSION(2) :: AuxParFields
         REAL(KIND=qPREC) :: aux_psign, aux_nsign
         INTEGER, DIMENSION(4) :: ReflectVars
         INTEGER :: nReflect
         IF (nCells == 0) RETURN
!         write(*,*) dim, edge, start, nCells, mthbc
         level=Info%level
         dir=(-1)**edge !(edge*2-3)  !direction of edge (1=>-1, 2=>1)
         ip(nDim+1:3,:)=1
         iq(nDim+1:3,:)=1
         AuxParFields(1:nDim-1)=modulo(dim+(/(i,i=0,nDim-2)/),nDim)+1

         ! Stretch bounds by nCells
         DO i=1,nDim
            IF (i==dim) CYCLE
            ip(i,1)=1-rmbc
            ip(i,2)=Info%mX(i)+rmbc
            iq(i,:)=ip(i,:)
         END DO
         SELECT CASE (mthbc)
         CASE(EXTRAPOLATED_BOUND)
            iq(dim,edge)=start+dir*(nCells-1)
            iq(dim,3-edge)=start
            ip(dim,:)=start-dir

            Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),1:NrHydroVars)=&
                 SPREAD(SUM(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1:NrHydroVars),dim),dim,nCells)
            !         write(*,*) "spreading ", ip, "along ",dim,"into", iq
            IF (MaintainAuxArrays) THEN
               lFirst=.true.
               DO j=1,nDim !Copy parallel auxfields and update b-norm
                  IF (j == dim) CYCLE
                  iq(dim,edge)=start+dir*(nCells-1)
                  iq(dim,3-edge)=start
                  ip(dim,:)=start-dir
                  ir=ip
                  is=iq
                  ir(dim,:)=ip(dim,:)
                  is(dim,:)=iq(dim,:)
                  ir(j,2)=ir(j,2)+1 !stretch aux bounds
                  is(j,2)=is(j,2)+1
                  Info%aux(is(1,1):is(1,2),is(2,1):is(2,2),is(3,1):is(3,2),j)=&
                       SPREAD(SUM(Info%aux(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2),j),dim),dim,nCells)
                  !               write(*,*) "spreading aux field", j, "at ", ir, "along dim", dim, "into", is
                  ir(j,2)=ir(j,2)-1
                  is(j,1)=is(j,1)+1            
                  DO i=1,nCells
                     iq(dim,:)=start+dir*(i-1)+(edge-1)!start+2-i
                     ip(dim,:)=start+dir*(i-2)+(edge-1)!start+1-i
                     ir(dim,:)=start+dir*(i-1)
                     is(dim,:)=start+dir*(i-1)
                     IF (lFirst) THEN
                        Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),dim)=&
                             Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),dim)
                        !                     write(*,*) "copying aux field", dim, "from" , ip, "into", iq
                     END IF
                     !                  write(*,*) "and adding...", dim, j, iq, ir, is
                     Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),dim)= &
                          Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),dim)+ &
                          dir*(Info%aux(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2),j)- &
                          Info%aux(is(1,1):is(1,2),is(2,1):is(2,2),is(3,1):is(3,2),j))                                   
                  END DO
                  is(j,:)=is(j,:)-1
                  lFirst=.false.
               END DO
               iq(dim,edge)=start+dir*(nCells-1)
               iq(dim,3-edge)=start
               ip(dim,:)=iq(dim,:)+1 !tart-dir!iq(dim,:)+1         
!               write(*,*) "updating fields", auxfields(dim), "at ", iq, "with average of ", iq, ip
               Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),AuxFields(dim))=half*(&
                    Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),dim)+&
                    Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),dim))
            END IF
         CASE(REFLECT_WALL, REFLECT_BPARALLEL, REFLECT_CYLINDRICAL)
            aux_psign=1d0
            aux_nsign=1d0
            SELECT CASE (mthbc)
            CASE (REFLECT_WALL)
               IF (lMHD) THEN
                  ReflectVars(1:2)=(/imom(dim), iB(dim)/); nReflect=2; aux_nsign=-1d0
               ELSE
                  ReflectVars(1)=imom(dim); nReflect=1
               END IF
            CASE(REFLECT_BPARALLEL)
               IF (lMHD) THEN
                  ReflectVars(1:3)=(/imom(dim), iB(modulo(dim+(/0,1/),3)+1)/); nReflect=3; aux_psign=-1d0
               ELSE
                  ReflectVars(1)=imom(dim); nReflect=1
               END IF
            CASE(REFLECT_CYLINDRICAL)
               IF (lMHD) THEN
                  ReflectVars(1:4)=(/imom(dim), imom(3), iB(dim), iB(3)/); nReflect=4;  aux_nsign=-1d0
               ELSE
                  IF (iCylindrical.eq.WithAngMom) THEN
                     ReflectVars(1:2)=(/imom(dim), imom(3)/); nReflect=2
                  ELSE
                     ReflectVars(1:1)=(/imom(dim)/); nReflect=1
                  END IF
               END IF
            END SELECT
            DO j=1,nCells
               iq(dim,:)=start+dir*(j-1)
               ip(dim,:)=start-dir*(j)

               Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),1:NrHydroVars)=&
                    Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1:NrHydroVars)
               Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),ReflectVars(1:nReflect)) = &
                    -Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),ReflectVars(1:nReflect))

               IF (MaintainAuxArrays) THEN
                  DO i=1,nDim-1
                     m=AuxParFields(i)
                     ip(m,2)=ip(m,2)+1
                     iq(m,2)=iq(m,2)+1
                     Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) = &
                          aux_psign*Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),m)
                  
                     ip(m,2)=ip(m,2)-1
                     iq(m,2)=iq(m,2)-1

                  END DO
                  iq(dim,:)=iq(dim,:)+(edge-1)
                  ip(dim,:)=ip(dim,:)+(2-edge)
                  Info%aux(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),dim) = &
                       aux_nsign*Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),dim)
                  
               END IF
            END DO
         CASE DEFAULT
            PRINT*, 'boundary condition type not implemented yet'
            STOP
         END SELECT
      END SUBROUTINE SetGhost
   END SUBROUTINE ApplyPhysicalBC


   !> Checks for divergence of aux fields
   !! @param Info Info object
   SUBROUTINE CheckDivergence(Info, lStopDivergence)
     USE ProcessingDeclarations
     TYPE(InfoDef) :: Info
     INTEGER :: i
     REAL :: ri,rl,rh
     REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: divergence
     LOGICAL, OPTIONAL :: lStopDivergence

     IF (MaintainAuxArrays) THEN
        ALLOCATE(divergence(Info%mX(1),Info%mX(2),Info%mX(3)))        

        IF (nDim == 2) THEN
           IF (iCylindrical==NoCyl) THEN
              divergence(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3))=&
                   (Info%aux(2:Info%mX(1)+1,1:Info%mX(2),1:Info%mX(3),1)+&
                   Info%aux(1:Info%mX(1),2:Info%mX(2)+1,1:Info%mX(3),2)-&
                   Info%aux(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),1) - &
                   Info%aux(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),2))/levels(Info%level)%dx
           ELSE
              DO i=1,Info%mX(1)
                 ri=1.d0/(Info%xBounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx)
                 rl=(Info%xBounds(1,1)+(i-1)*levels(Info%level)%dx)
                 rh=(Info%xBounds(1,1)+(i)*levels(Info%level)%dx)
                 divergence(i,1:Info%mX(2),1:Info%mX(3))=&
                      (rl*ri*Info%aux(i+1,1:Info%mX(2),1:Info%mX(3),1)+&
                      Info%aux(i,2:Info%mX(2)+1,1:Info%mX(3),2)-&
                      rh*ri*Info%aux(i,1:Info%mX(2),1:Info%mX(3),1) - &
                      Info%aux(i,1:Info%mX(2),1:Info%mX(3),2))/levels(Info%level)%dx
              END DO
           END IF
        ELSE
           divergence(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3))=&
                (Info%aux(2:Info%mX(1)+1,1:Info%mX(2),1:Info%mX(3),1)+&
                Info%aux(1:Info%mX(1),2:Info%mX(2)+1,1:Info%mX(3),2)+&
                Info%aux(1:Info%mX(1),1:Info%mX(2),2:Info%mX(3)+1,3)-&
                Info%aux(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),1) - &
                Info%aux(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),2) - &
                Info%aux(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),3))/levels(Info%level)%dx
        END IF

        IF (ANY(ABS(divergence(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3))) > &
             1d-13/levels(Info%level)%dx*max(1d0, sqrt(SUM(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),AuxFields(1:nDim))**2, 4))))) THEN
           !         CALL WriteDataFrame(100)
           write(*,*) "Divergence Warning", Info%level, MAXVAL(ABS(divergence(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3))))
             
           IF (PRESENT(lStopDivergence) .AND. lStopDivergence) THEN
              WRITE(*,*) "*** Simulation stopped due to divergence in the initialization! ***"
              STOP
           END IF 
        END IF

        DEALLOCATE(divergence)
        NULLIFY(divergence)
     END IF
   END SUBROUTINE CheckDivergence
   
   !> Sets err flags based on Jeans Criteria
   !! @param rho array of density
   !! @param cs array of sound speed
   !! @param errorflags array of error flags
   SUBROUTINE JeansCriterionTest(rho, cs, errorflags, dx)

     REAL(KIND=qPrec), DIMENSION(:,:,:) :: cs, rho
     INTEGER, DIMENSION(:,:,:) :: errorflags
     REAL(KIND=qPrec) :: jl_coeff, dx
     REAL(KIND=qPrec) :: jeans_length

     jeans_length = dx * CellsPerJeansLength
     jl_coeff = PI / ScaleGrav

     WHERE(SQRT(jl_coeff / rho) * cs < jeans_length)  ErrorFlags = 1

!     rho > jl_coeff/(jeans_length)**2*(cs**2)


!      JeansFact=sqrt(pi*gamma/ScaleGrav)/(JEAN_CELLS*levels(MaxLevel)%dx)
     !rhoJeans=JeansFact*sqrt(press(q(i,j,k,:)))  Sink Particle test
   END SUBROUTINE JeansCriterionTest

 END MODULE ModuleControl

