!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    ambients.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @Filegotoli ambients.f90
!! @brief Main file for module Ambients

!> @defgroup Ambients Ambients Object
!! @brief Module that handles the placement of ambients
!! @ingroup ModuleObjects

!> Module that handles the placement of ambients
!! @ingroup Ambients
MODULE Ambients
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE Perturbation
   USE VectorPerturbation
   USE ObjectDeclarations
   USE Profiles
   IMPLICIT NONE
   !> Ambient data type
   TYPE AmbientDef
      REAL(KIND=qPREC) :: density=1d0
      REAL(KIND=qPREC) :: velocity(3)=0d0
      REAL(KIND=qPREC) :: pressure=1d0
      REAL(KIND=qPREC) :: B(3) = 0d0
      REAL(KIND=qPREC), DIMENSION(3,2) :: xBounds
      TYPE(PerturbationDef), POINTER :: DensityPerturbation => null()
      TYPE(VectorPerturbationDef), POINTER :: VelocityPerturbation => null()
      TYPE(VectorPerturbationDef), POINTER :: MagneticPerturbation => null()
      TYPE(ProfileDef), POINTER :: profile => null()
      LOGICAL :: PersistInBoundaries(3,2) = .false.
      INTEGER :: iTracer=0
      INTEGER :: ObjId
      TYPE(PointGravityDef), POINTER :: PointGravity => null()
   END TYPE AmbientDef

   !new declaration
   TYPE pAmbientDef
      TYPE(AmbientDef), POINTER :: ptr
   END TYPE pAmbientDef
   TYPE(pAmbientDef) :: pAmbient
   !new declaration

CONTAINS


   !> Creates an Ambient object
   !! @param Ambient Ambient object
   SUBROUTINE CreateAmbient(Ambient, density, pressure)
      TYPE(AmbientDef), POINTER :: Ambient
      REAL(KIND=qPREC), OPTIONAL :: density, pressure
      INTEGER :: i
      ALLOCATE(Ambient)
      IF (Present(density)) Ambient%density=density
      IF (Present(pressure)) Ambient%pressure=pressure
      Ambient%xBounds(:,1)=GxBounds(:,1)-(/(merge(1,0,ndim>=i),i=1,3)/)*levels(0)%dx*levels(0)%gmbc(1)
      Ambient%xBounds(:,2)=GxBounds(:,2)+(/(merge(1,0,ndim>=i),i=1,3)/)*levels(0)%dx*levels(0)%gmbc(1)
      CALL AddAmbientToList(Ambient)
   END SUBROUTINE CreateAmbient


   !> Updates an Ambient object
   !! @param Ambient Ambient object
   SUBROUTINE UpdateAmbient(Ambient)
     TYPE(AmbientDef), POINTER :: Ambient
     INTEGER :: i,j
     !Do a sanity check on magnetic fields and boundary conditions
     DO i=1,nDim
        IF (ANY(Gmthbc(i,:) == REFLECT_WALL) .AND. Ambient%B(i) /= 0) THEN
           print*, 'cannot have boundary conditions of type ', REFLECT_WALL, ' along dimension ', i, ' with B(',i,') /= 0'
           STOP
        END IF
        
        IF (ANY(Gmthbc(i,:) == REFLECT_BPARALLEL) .AND. ANY(Ambient%B(modulo(i+(/(j,j=0,nDim-2)/),nDim)+1) /= 0)) THEN
           print*, 'cannot have boundary conditions of type ', REFLECT_BPARALLEL,' along dimension ', i, &
                ' with B(',modulo(i,3)+1,') or B(',modulo(i+1,3)+1,') /= 0'
           STOP
        END IF
     END DO
   END SUBROUTINE UpdateAmbient


   SUBROUTINE AddAmbientToList(Ambient)
     TYPE(AmbientDef), POINTER :: Ambient
     TYPE(ObjectDef), POINTER :: Object
     Ambient%ObjId = ObjectListAdd(Object,AMBIENTOBJ)
     pAmbient%ptr => Ambient
     len = size(transfer(pAmbient, dummy_char))
     ALLOCATE(Object%storage(len))
     Object%storage = transfer(pAmbient, Object%storage)
   END SUBROUTiNE AddAmbientToList

   SUBROUTINE DestroyAmbient(Ambient)
      TYPE(AmbientDef), POINTER :: Ambient
      TYPE(ObjectDef), POINTER :: Object
      INTEGER :: id
      id=Ambient%ObjId
      Object => ObjectListFind(id)
      IF (ASSOCIATED(Object) .AND. Object%type == AMBIENTOBJ) THEN
         pAmbient = transfer(Object%storage,pAmbient)
         DEALLOCATE(pAmbient%ptr)
         NULLIFY(pAmbient%ptr)
         CALL ObjectListRemove(id)
      ENDIF
      NULLIFY(Ambient)
   END SUBROUTINE DestroyAmbient


   SUBROUTINE AmbientGridInit(Info, Ambient)
     TYPE(InfoDef) :: Info
     TYPE(AmbientDef), POINTER :: Ambient
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     CALL CalcPhysicalOverlaps(Info, Ambient%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
     IF (nOverlaps > 0) THEN
        CALL PlaceAmbient(Info, Ambient, nOverlaps, mSs, offsets)
        DEALLOCATE(mSs, offsets)
     END IF
   END SUBROUTINE AmbientGridInit


   SUBROUTINE AmbientBeforeStep(Info, Ambient)
     TYPE(InfoDef) :: Info
     TYPE(AmbientDef), POINTER :: Ambient
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     INTEGER :: i,j
     DO i=1,nDim
        DO j=1,2
           IF (Ambient%PersistInBoundaries(i,j)) THEN
              CALL CalcPhysicalOverlaps(Info, Ambient%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
              IF (nOverlaps > 0) THEN
                 CALL PlaceAmbient(Info, Ambient, nOverlaps, mSs, offsets)
                 DEALLOCATE(mSs, offsets)
              END IF
           END IF
        END DO
     END DO
   END SUBROUTINE AmbientBeforeStep

   SUBROUTINE AmbientSetErrFlag(Info, Ambient)
      TYPE(InfoDef) :: Info
      Type(AmbientDef), POINTER :: Ambient
   END SUBROUTINE AmbientSetErrFlag
  

   !> Places ambients in info object
   !! @param Info Info object
   SUBROUTINE PlaceAmbient(Info, Ambient, nOverlaps, mSs, offsets)
     TYPE(InfoDef) :: Info
     TYPE(AmbientDef), POINTER :: Ambient
     INTEGER :: i, j, k, l, n
     REAL(KIND=qPREC) :: pos(3), rho, vel(3), x, y, z, press, dx, r, phi
     INTEGER :: location, nOverlaps
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     INTEGER, DIMENSION(3,2) :: mS
     REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
     REAL(KIND=qPREC), DIMENSION(3) :: offset
     REAL(KIND=qPREC), DIMENSION(3,2) :: universe

     dx=levels(Info%level)%dx
     IF (nOverlaps > 0) THEN
        DO n=1, nOverlaps
           mS=mSs(n,:,:)
           offset=offsets(n,:)
           IF(MaintainAuxArrays) THEN
              DO i=1,nAux
                 mS(i,2)=mS(i,2)+1
                 Info%aux(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),i)=Ambient%B(i)
                 mS(i,2)=mS(i,2)-1
              END DO
           END IF

           DO i=mS(1,1), mS(1,2)
              x=Info%xBounds(1,1)+(real(i)-half)*dx + offset(1)
              DO j=mS(2,1), mS(2,2)
                 y=Info%xBounds(2,1)+(real(j)-half)*dx + offset(2)
                 DO k=mS(3,1), mS(3,2)
                    z=Info%xBounds(3,1)+(real(k)-half)*merge(dx, 0d0, nDim==3) + offset(3)
                    pos=(/x,y,z/)
                    rho=Ambient%Density
                    press=Ambient%pressure
                    vel=Ambient%velocity
                    IF (Associated(Ambient%DensityPerturbation)) rho = Ambient%Density+PerturbationValue(Ambient%DensityPerturbation,pos)
                    IF (ASSOCIATED(Ambient%VelocityPerturbation)) vel = Ambient%velocity+VectorPerturbationValue(Ambient%VelocityPerturbation,pos)
                    IF (ASSOCIATED(Ambient%PointGravity)) THEN
                       r = sqrt(sum(pos(1:nDim)**2))
                       phi=GravityPotential(Ambient%PointGravity%mass, pos, Ambient%PointGravity%soft_length, Ambient%PointGravity%soft_function)
                       rho=min(exp(-phi/(ambient%pressure/ambient%density)+log(ambient%density)), 1d300)
                       press=rho*ambient%pressure/ambient%density

                       !                        write(*,*) phi, ambient%pressure/ambient%density, phi/(ambient%pressure/ambient%density), exp(phi/(ambient%pressure/ambient%density))
                       !                        STOP

                    ELSEIF (ASSOCIATED(Ambient%Profile)) THEN
                       r = sqrt(sum(pos(1:nDim)**2))
                       DO l=1, size(Ambient%Profile%fields)
                          IF (Ambient%Profile%fields(l) == Mass_Field) rho=GetProfileValue(r,Mass_Field,Ambient%Profile)
                          IF (Ambient%Profile%fields(l) == P_Field) press=GetProfileValue(r,P_Field,Ambient%Profile)
                       END DO

                    END IF
                    Info%q(i,j,k,1:NrHydroVars)=0
                    Info%q(i,j,k,1)=rho
                    IF (ivx /= 0) Info%q(i,j,k,ivx)=vel(1)
                    IF (ivy /= 0) Info%q(i,j,k,ivy)=vel(2)
                    IF (ivz /= 0) Info%q(i,j,k,ivz)=vel(3)
                    IF (iE /= 0) Info%q(i,j,k,iE)=press
                    IF (iBx /= 0) Info%q(i,j,k,iBx)=Ambient%B(1)
                    IF (iBy /= 0) Info%q(i,j,k,iBy)=Ambient%B(2)
                    IF (iBz /= 0) Info%q(i,j,k,iBz)=Ambient%B(3)     
                    IF (Ambient%iTracer > 0) Info%q(i,j,k,Ambient%iTracer) = Ambient%density
                 END DO
              END DO
           END DO
           CALL Prim_to_cons(Info, mS)
        END DO
     END IF
   END SUBROUTINE PlaceAmbient

 END MODULE Ambients

