!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    clumps.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file clumps.f90
!! @brief Main file for module Clumps

!> @defgroup Clumps Clumps Object
!! @brief Module that handles the placement of clumps
!! @ingroup ModuleObjects

!> Module that handles the placement of clumps
!! @ingroup Clumps
MODULE Clumps
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   USE Shapes
   USE Perturbation
   USE VectorPerturbation
   USE DataInfoOps
   USE EOS
   USE BE_MODULE
   USE ObjectDeclarations
   IMPLICIT NONE
   !> Clump Data Type
   TYPE ClumpDef
      REAL(KIND=qPREC) :: density=10   !Clump peak density
      REAL(KIND=qPREC) :: pressure=0   !Clump Pressure
      REAL(KIND=qPREC) :: temperature=.1      !Clump temperature
      REAL(KIND=qPREC), DIMENSION(3) :: velocity= (/0,0,0/)  !Clump velocity 
      REAL(KIND=qPREC), DIMENSION(3) :: position = (/0,0,0/)  !Clump velocity 
      REAL(KIND=qPREC) :: radius=1
      REAL(KIND=qPREC) :: phi=0
      REAL(KIND=qPREC) :: theta=0
      REAL(KIND=qPREC) :: thickness=.1 !thickness of clump smoothing region
      !      REAL(KIND=qPREC) :: B_tor=0     !Maximum Bfield for toroidal configuration
      !      REAL(KIND=qPREC) :: B_pol=0     !Maximum Bfield for poloidal configuration
      REAL(KIND=qPREC) :: B_phi=0
      REAL(KIND=qPREC) :: B_theta=0
      REAL(KIND=qPREC) :: B_toroidal=0     !Maximum Bfield for toroidal configuration
      REAL(KIND=qPREC) :: B_poloidal=0     !Maximum Bfield for poloidal configuration
      REAL(KIND=qPREC) :: omega=0     !Angular velocity for solid body rotation
      INTEGER :: RefineLevel=MaxDepth
      INTEGER :: SubSample=1
      INTEGER :: iTracer=0                                !Clump Tracer
      INTEGER :: density_profile=0
      REAL(KIND=qPREC) :: m2A=0 !Amplitude of an m=2 density perturbation [  rho = rho_0*(1+m2A*cos(2*phi))   ]
      LOGICAL :: PersistInBoundaries(3,2) = .false.
      INTEGER :: id
      TYPE(ShapeDef), POINTER :: Shape => Null()
      TYPE(PerturbationDef), POINTER :: DensityPerturbation => Null()
      TYPE(VectorPerturbationDef), POINTER :: VelocityPerturbation => Null()
      TYPE(PerturbationDef), DIMENSION(:), POINTER :: MagneticPerturbation => Null()
      INTEGER :: ObjId
   END TYPE ClumpDef

   !new declaration
   TYPE pClumpDef
      TYPE(ClumpDef), POINTER :: ptr
   END TYPE pClumpDef
   TYPE(pClumpDef) :: pClump   
   !

   INTEGER, PARAMETER :: UNIFORM = 0, BE_PROFILE = 1
   SAVE
CONTAINS


   !> Creates a clump object
   !! @param Clump Clump object
   SUBROUTINE CreateClump(Clump, density, pressure, position)
      TYPE(ClumpDef), POINTER :: Clump
      REAL(KIND=qPREC), OPTIONAL :: density, pressure, position(3)
      ALLOCATE(Clump) 
      ALLOCATE(clump%Shape)
      IF (Present(density)) Clump%density=density
      IF (Present(pressure)) Clump%pressure=pressure
      IF (Present(position)) Clump%position=position
      CALL AddClumpToList(Clump)
      CALL UpdateClump(Clump)
   END SUBROUTINE CreateClump

   !> Updates a clump object
   !! @param Clump Clump object
   SUBROUTINE UpdateClump(Clump)
      TYPE(ClumpDef), POINTER :: Clump
      CALL SetShapeType(Clump%Shape, SPHERE, (/Clump%radius,0d0,0d0/))
      CALL SetShapeOrientation(Clump%Shape, 0d0, Clump%theta, Clump%phi)
      Clump%Shape%Position=Clump%position
      Clump%Shape%velocity=Clump%velocity
      Clump%Shape%t0=levels(0)%tnow
      CALL SetShapeBounds(Clump%Shape)         
   END SUBROUTINE UpdateClump

   SUBROUTINE AddClumpToList(Clump)
      TYPE(ClumpDef), POINTER :: Clump
      TYPE(ObjectDef), POINTER :: Object
      Clump%ObjId = ObjectListAdd(Object,CLUMPOBJ)
      pClump%ptr => Clump
      len = size(transfer(pClump, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pClump, Object%storage)
   END SUBROUTINE AddClumpToList

   SUBROUTINE DestroyClump(Clump)
      TYPE(ClumpDef), POINTER :: Clump
      TYPE(ObjectDef),POINTER :: Object
      INTEGER :: id
      id=Clump%ObjId
      Object => ObjectListFind(id)
      IF (ASSOCIATED(Object) .AND. Object%type == CLUMPOBJ) THEN
         pClump = transfer(Object%storage,pClump)
         DEALLOCATE(pClump%ptr)
         NULLIFY(pClump%ptr)
         CALL ObjectListRemove(id)
      ENDIF
      NULLIFY(Clump)
   END SUBROUTINE DestroyClump

   SUBROUTINE ClumpGridInit(Info, Clump)
      TYPE(InfoDef) :: Info
      TYPE(ClumpDef), POINTER :: Clump
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      INTEGER :: nOverlaps
      CALL CalcPhysicalOverlaps(Info, Clump%Shape%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
      IF (nOverlaps > 0) THEN
         CALL PlaceClump(Info, Clump, nOverlaps, mSs, offsets)
         DEALLOCATE(mSs, offsets)
      END IF
   END SUBROUTINE ClumpGridInit

   SUBROUTINE ClumpBeforeStep(Info, Clump)
      TYPE(InfoDef) :: Info
      TYPE(ClumpDef), POINTER :: Clump
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      INTEGER :: nOverlaps
      INTEGER :: i,j
      DO i=1,nDim
         DO j=1,2
            IF (Clump%PersistInBoundaries(i,j)) THEN
               CALL CalcPhysicalOverlaps(Info, Clump%Shape%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
               IF (nOverlaps > 0) THEN
                  CALL PlaceClump(Info, Clump, nOverlaps, mSs, offsets)
                  DEALLOCATE(mSs, offsets)
               END IF
            END IF
         END DO
      END DO
   END SUBROUTINE ClumpBeforeStep

   SUBROUTINE ClumpSetErrFlag(Info, Clump)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC) :: sample_res
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSIOn(3,2) :: tempbounds, tempbounds2
      Type(ClumpDef), POINTER :: Clump
      INTEGER :: nOverlaps, n
      LOGICAL :: lrefine
      IF (Info%level < Clump%RefineLevel) THEN
         IF (levels(Info%level)%dx > .125*clump%radius) THEN !May need to trigger refinement manually            
            tempbounds=GetShapeBounds(Clump%Shape, levels(Info%level)%tnow)
            tempbounds2=GetShapeBounds(Clump%Shape, levels(Info%level)%tnow+levels(Info%level)%dt)
            tempbounds(:,1)=min(tempbounds(:,1), tempbounds2(:,1))
            tempbounds(:,2)=max(tempbounds(:,2), tempbounds2(:,2))
            tempbounds(1:nDim,1)=tempbounds(1:nDim,1)-.5d0*levels(Info%level)%dx
            tempbounds(1:nDim,2)=tempbounds(1:nDim,2)+.5d0*levels(Info%level)%dx
            IF (levels(Info%level)%tnow == start_time) THEN !initial setup
               lRefine=.true.
            ELSEIF((ALL(tempbounds(1:nDim,1) > GxBounds(1:nDim,1)) .AND. ALL(tempbounds(1:nDim,2) < GxBounds(1:nDim,2)))) THEN !completely inside the grid
               lRefine=.false.
            ELSEIF (ANY(tempbounds(1:nDim,2) < GxBounds(1:nDim,1)) .OR. ANY(tempbounds(1:nDim,1) > GxBounds(1:nDim,2))) THEN !completely outside the grid
               lRefine=.false.
            ELSE
               lRefine=.true.
            END IF
            IF (lRefine) THEN
               CALL CalcPhysicalOverlaps(Info, tempbounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic, 0)
               IF (nOverlaps > 0) THEN
                  DO n=1,nOverlaps
                     mS=mSs(n,:,:)
                     Info%ErrFlag(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2))=1
                  END DO
               END IF
            END IF
         END IF
      END IF

   END SUBROUTINE ClumpSetErrFlag

   !includes decoding for now
   SUBROUTINE ClumpBeforeGlobalStep(n)
      INTEGER :: n
      !Check if clump is entirely inside of domain and if so - stop placing it in boundary zones in before step
      TYPE(ObjectDef), POINTER :: Object
      REAL(KIND=qPREC), DIMENSIOn(3,2) :: tempbounds
      IF (n == MaxLevel) THEN
         Object => ListHead
         DO WHILE(ASSOCIATED(Object))
            IF (Object%type == CLUMPOBJ) THEN
               pClump = transfer(Object%storage,pClump)
               tempbounds=GetShapeBounds(pClump%ptr%Shape, levels(MaxLevel)%tnow)
               IF((ALL(tempbounds(1:nDim,1) > GxBounds(1:nDim,1)) .AND. ALL(tempbounds(1:nDim,2) < GxBounds(1:nDim,2)))) THEN !completely inside the grid
                  pClump%ptr%PersistInBoundaries=.false.
               END IF
            END IF
            Object => Object%next
         END DO
      END IF
   END SUBROUTINE ClumpBeforeGlobalStep

   !> Place a clump in an info object
   !! @param Info Info object
   !! @param Clump Clump object
   SUBROUTINE PlaceClump(Info,Clump, nOverlaps, mSs, offsets)
      TYPE(InfoDef) :: Info
      Type(ClumpDef) :: Clump
      INTEGER :: i,j,k,n,m,ii,jj,kk, location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos, rpos
      REAL(KIND=qPREC) :: sample_fact(3), q_fact, dx,dz
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(3,2) :: tempbounds
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source, emf_source, qacc
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: emf
      !      RETURN
      dx=levels(Info%level)%dX
      dz=merge(dx,0d0,nDim==3)
      xpos=0          
      IF (nOverlaps > 0) THEN
         ALLOCATE(q_Source(NrHydroVars), qacc(NrHydroVars))
         sample_res=1
         sample_fact=0d0
         IF (Clump%SubSample > 0) THEN
            sample_res(1:nDim)=Clump%SubSample !min(Clump%SubSample,2**(MaxLevel-Info%level))
         ELSE
            sample_res(1:nDim)=1!max(Clump%SubSample,2**(MaxLevel-Info%level))
         END IF
         sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
         q_fact=product(sample_fact(1:nDim))
         DO n=1,nOverlaps
            mS=mSs(n,:,:)
            offset=offsets(n,:)
            !Set up emf first - aux fields first
            CALL ConvertTotalToInternalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
            IF (MaintainAuxArrays) THEN
               IF (nDim == 2) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,1,3:3), emf_source(3:3))
                  emf=0
                  DO i=mS(1,1),mS(1,2)+1
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)+1
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        IF (IsInShape(Clump%Shape, xpos, rpos, levels(Info%level)%tnow)) THEN                         
                           emf_source(3)=emf_clump_2D(Clump, rpos) !RotateVectorFromShape(Clump%Shape, emf_2D(Clump, rpos)) !Assume shape is not rotated in 2D
                           emf(i,j,1,3)=emf(i,j,1,3)+emf_source(3)
                        END IF
                     END DO
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,1,1:2),emf(:,:,1,3),dx)
                  DEALLOCATE(emf, emf_source)
               ELSE IF (nDim == 3) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,3), emf_source(3))
                  emf=0
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        DO k=mS(3,1),mS(3,2)
                           xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dz
                           DO m=1,3
                              pos=xpos
                              DO ii=1,sample_res(1)
                                 pos(m)=xpos(m)+(REAL(ii, 8)-half)*dx*sample_fact(m)
                                 IF (IsInShape(Clump%Shape, pos, rpos, levels(Info%level)%tnow)) THEN
                                    emf_source=RotateVectorFromShape(Clump%Shape, emf_Clump_3D(Clump, rpos))
                                    emf(i,j,k,m)=emf(i,j,k,m)+emf_source(m)
                                 END IF
                              END DO
                              emf(i,j,k,m)=emf(i,j,k,m)*sample_fact(m)
                           END DO
                        END DO
                     END DO
                     !                     CALL OutputDoubleArray(emf(i,:,:,3))
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,1:3),emf,dx)
                  DEALLOCATE(emf, emf_source)                 
               END IF
               CALL UpdateAux(Info, mS)          
            END IF
            ! Now set up cell centered quantities (density and momentum)
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+REAL(k-1,8)*dz
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+REAL(j-1,8)*dx
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+REAL(i-1,8)*dx
                     qacc=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dz*sample_fact(3)
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact(2)
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact(1)
                              IF (IsInShape(Clump%Shape, pos, rpos, levels(Info%level)%tnow)) THEN

                                 CALL q_ClumpSource(Clump,rpos,Info%q(i,j,k,1:NrHydroVars), q_source)
                                 qacc=qacc+q_source

                              END IF
                           END DO
                        END DO
                     END DO
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+qacc*q_fact                 
                  END DO
               END DO
            END DO
            CALL ConvertInternalToTotalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
         END DO
         DEALLOCATE(q_Source, qacc)
      END IF

   END SUBROUTINE PlaceClump

   SUBROUTINE q_ClumpSource(Clump, pos, q, s)
      TYPE(ClumpDef) :: Clump
      REAL(KIND=qPREC), DIMENSION(:) :: pos, q, s
      REAL(KIND=qPREC) :: r, f, rotation_speed(3), perturb, vel(3)
      r=sqrt(sum(pos(1:nDim)**2))
      f=smooth_tanh(r/Clump%Shape%size_param(1),Clump%thickness)
      perturb=Clump%Density
      s=0
      IF (Clump%density_profile==BE_profile) THEN
         perturb=BE_rho(REAL(r*Sqrt((4*pi*scalegrav*clump%density)/(gamma*clump%temperature))))*Clump%Density
      END IF
      IF (Clump%m2A > 0) perturb=perturb+Clump%Density*(1d0+Clump%m2A*(pos(1)**2-pos(2)**2)/r**2)  !cos(2a)=cos(a)^2-sin(a)^2
      IF (ASSOCIATED(Clump%DensityPerturbation)) perturb=perturb*exp(PerturbationValue(Clump%DensityPerturbation, pos))

      s(1)=(perturb-q(1))*f !fade into clump        
      vel=0
      IF (ASSOCIATED(Clump%VelocityPerturbation)) THEN
         vel=vel+VectorPerturbationValue(Clump%VelocityPerturbation,pos)
      END IF
      vel=vel*f
      IF (Clump%Omega /= 0) THEN 
         vel=vel+RotateVectorFromShape(Clump%Shape, Clump%Omega*(/pos(2),-pos(1),0d0/))
      END IF

      vel(1:nDim)=vel(1:nDim)+Clump%velocity(1:nDim)! + rotation_speed(1:nDim)

      s(ivx:ivx+nDim-1)=(perturb*vel(1:nDim)-q(ivx:ivx+nDim-1))*f!momentum associated with clump mass + rotational energy
      IF (iE .ne. 0) THEN
         IF (CLump%Pressure > 0) THEN 
            s(iE)=(gamma7*Clump%pressure-q(iE))*f !Internal energy associated with clump material
         ELSE
            s(iE)=(gamma7*perturb*Clump%temperature-q(iE))*f !Internal energy associated with clump material
         END IF
      END IF
      IF (Clump%iTracer .ne. 0) s(Clump%iTracer) = perturb*f-q(Clump%iTracer)

   END SUBROUTINE q_ClumpSource


   FUNCTION emf_clump_3D(Clump,pos)
      TYPE(ClumpDef) :: Clump
      REAL(KIND=qPrec) :: x_loc, s, r, w(3), emf_clump_3D(3),  tor_source, pol_source,r0,s2,rz,rz2,pos(3), f
      REAL(KIND=qPREC), PARAMETER :: r_m =.9 !Magnetic field cutoff in clump radii
      INTEGER :: i
      s2=pos(1)**2+pos(2)**2
      r=sqrt(s2+pos(3)**2)
      s=sqrt(s2)
      rz2=Clump%Shape%size_param(1)**2-pos(3)**2
      rz=sqrt(rz2)
      IF (s <= r_m*rz) THEN
         tor_source=Clump%B_toroidal*(r_m*rz2 - s2)/(2d0*r_m*Clump%Shape%size_param(1))
      ELSE
         tor_source=Clump%B_toroidal*(rz-s)**2/(2d0*(1d0-r_m)*Clump%Shape%size_param(1))
      END IF
      r0 = r/Clump%Shape%size_param(1)
      pol_source=Clump%B_poloidal/(2d0*Clump%Shape%size_param(1)**2)*(Clump%Shape%size_param(1)-r)**2  !planar version
      emf_clump_3D=tor_source*(/0d0,0d0,1d0/)+pol_source*(/pos(2),-pos(1),0d0/)
      IF (ASSOCIATED(Clump%MagneticPerturbation)) THEN         
         IF (r0 < r_m) THEN
            f=exp(2d0*r0/r_m)*(1d0-r0/r_m)**2
         ELSE
            f=0d0
         END IF
         DO i=1,3
            emf_clump_3D(i)=emf_clump_3D(i)+f*PerturbationValue(Clump%MagneticPerturbation(i), pos)
         END DO
      END IF
   END FUNCTION emf_clump_3D


   FUNCTION emf_clump_2D(Clump,pos)
      REAL(KIND=qPrec), DIMENSION(:) :: pos
      TYPE(ClumpDef) :: Clump
      REAL(KIND=qPREC) :: emf_clump_2D, r
      r=sqrt(SUM(pos(1:2)**2))
      emf_clump_2D=Clump%B_toroidal*(Clump%Shape%size_param(1)-r)
      IF (ASSOCIATED(Clump%MagneticPerturbation)) THEN
         emf_clump_2D=emf_clump_2D+PerturbationValue(Clump%MagneticPerturbation(1), pos)
      END IF
   END FUNCTION emf_clump_2D


END MODULE Clumps


