!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    collidingflows.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file collidingflows.f90
!! @brief Main file for module CollidingFlows

!> @defgroup CollidingFlows CollidingFlows Object
!! @brief Module that handles the placement of CollidingFlows
!! @ingroup ModuleObjects

!> Module that handles the placement of CollidingFlows
!! @ingroup CollidingFlows
MODULE CollidingFlows
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE DataInfoOps
   USE Shapes
   USE Interfaces
   USE CommonFunctions
   USE EOS
   USE ObjectDeclarations
   IMPLICIT NONE
   !> CollidingFlow Data Type
   TYPE CollidingFlowDef
      REAL(KIND=qPREC) :: density
      REAL(KIND=qPREC) :: velocity
      REAL(KIND=qPREC) :: temperature
      REAL(KIND=qPREC), DIMENSION(3) :: B=0
      TYPE(ShapeDef), POINTER :: shape => null()
      TYPE(InterfaceDef), POINTER :: InterfaceObj => null()
      REAL(KIND=qPREC) :: smooth_distance=0
      INTEGER :: smooth_function=0
      REAL(KIND=qPREC) :: interface_dist=0
      INTEGER :: interface_func=0
      INTEGER :: iTracer(2) = 0
      INTEGER :: SubSample=1
      LOGICAL :: PersistInBoundaries(3,2) = .false.
      LOGICAL :: PersistInternal = .false.
      REAL(KIND=qPREC) :: tShutOff = 1e30
      REAL(KIND=qPREC) :: RampTime = 1
      INTEGER :: ObjId
   END TYPE CollidingFlowDef

   !new declaration
   TYPE pCollidingFlowDef 
      TYPE(CollidingFlowDef), POINTER :: ptr
   END TYPE pCollidingFlowDef
   TYPE(pCollidingFlowDef)  :: pCollidingFlow
   !

   SAVE
CONTAINS

   !
   !> Initializes a CollidingFlow object
   !! @param CollidingFlow CollidingFlow object
   SUBROUTINE CreateCollidingFlow(CollidingFlow, density, temperature, velocity)
      TYPE(CollidingFlowDef), POINTER :: CollidingFlow
      REAL(KIND=qPREC), OPTIONAL :: density, temperature, velocity
      ALLOCATE(CollidingFlow)
      ALLOCATE(CollidingFlow%InterfaceObj)
      ALLOCATE(CollidingFlow%Shape)
      IF (Present(density)) CollidingFlow%density=density
      IF (Present(temperature)) CollidingFlow%temperature=temperature
      IF (Present(velocity)) CollidingFlow%velocity=velocity
      CALL addcollidingFlowToList(CollidingFlow) 
   END SUBROUTINE CreateCollidingFlow

  SUBROUTINE UpdateCollidingFlow(CollidingFlow)
    TYPE(CollidingFlowDef), POINTER :: CollidingFlow
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdateCollidingFlow


   SUBROUTINE AddCollidingFlowToList(CollidingFlow)
      TYPE(CollidingFlowDef), POINTER :: CollidingFlow
      TYPE(ObjectDef), POINTER :: Object
      CollidingFlow%ObjId = ObjectListAdd(Object,COLLIDINGFLOWOBJ)
      pCollidingFlow%ptr => CollidingFlow
      len = size(transfer(pCollidingFlow, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pCollidingFlow, Object%storage)
   END SUBROUTINE AddCollidingFlowToList


   SUBROUTINE DestroyCollidingFlowObject(CollidingFlow)
      TYPE(CollidingFlowDef),POINTER :: CollidingFlow
      CALL ObjectListRemove(CollidingFlow%ObjId)
      DEALLOCATE(CollidingFlow%Shape)
      CALL DestroyInterface(CollidingFlow%InterfaceObj)
      DEALLOCATE(CollidingFlow)
      NULLIFY(CollidingFlow)
   END SUBROUTINE DestroyCollidingFlowObject



   SUBROUTINE CollidingFlowGridInit(Info, CollidingFlow)
     TYPE(InfoDef) :: Info
     TYPE(CollidingFlowDef), POINTER :: CollidingFlow
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     CALL CalcPhysicalOverlaps(Info, CollidingFlow%Shape%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
     IF (nOverlaps > 0) THEN
        CALL PlaceCollidingFlow(Info, CollidingFlow, nOverlaps, mSs, offsets)
        DEALLOCATE(mSs, offsets)
     END IF
   END SUBROUTINE CollidingFlowGridInit


   SUBROUTINE CollidingFlowBeforeStep(Info, CollidingFlow)
     TYPE(InfoDef) :: Info
     TYPE(CollidingFlowDef), POINTER :: CollidingFlow
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     INTEGER :: i,j
     DO i=1,nDim
        DO j=1,2
           IF (CollidingFlow%PersistInBoundaries(i,j)) THEN
              CALL CalcPhysicalOverlaps(Info, CollidingFlow%Shape%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
              IF (nOverlaps > 0) THEN
                 CALL PlaceCollidingFlow(Info, CollidingFlow, nOverlaps, mSs, offsets)
                 DEALLOCATE(mSs, offsets)
              END IF
           END IF
        END DO
     END DO
   END SUBROUTINE CollidingFlowBeforeStep

   SUBROUTINE CollidingFlowSetErrFlag(Info, CollidingFlow)
      TYPE(InfoDef) :: Info
      Type(CollidingFlowDef), POINTER :: CollidingFlow
   END SUBROUTINE CollidingFlowSetErrFlag


   SUBROUTINE PlaceCollidingFlow(Info, CollidingFlow, nOverlaps, mSs, offsets)
      TYPE(InfoDef) :: Info
      Type(CollidingFlowDef) :: CollidingFlow
      INTEGER :: i,j,k,n,m,ii,jj,kk, location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos,coords,vel
      REAL(KIND=qPREC) :: sample_fact(3), q_fact, dx, r, fact, dz, dist
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source

      dx=levels(Info%level)%dX
      dz=merge(dx, 0d0, nDim == 3)

      xpos=0          
      IF (nOverlaps > 0) THEN
         sample_res=1
         sample_fact=0d0
         IF (CollidingFlow%SubSample > 0) THEN
            sample_res(1:nDim)=CollidingFlow%SubSample !max(CollidingFlow%SubSample,2**(MaxLevel-Info%level))
         END IF
         sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
         q_fact=product(sample_fact(1:nDim))
         ALLOCATE(q_Source(NrHydroVars))
         DO n=1,nOverlaps
            mS=mSs(n,:,:)
            offset=offsets(n,:)
            ! Now set up cell centered quantities (density and momentum)
            CALL ConvertTotalToInternalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dz
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     q_Source=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dz*sample_fact(3)
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact(2)
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact(1)
                              IF (IsInShape(CollidingFlow%Shape, pos, coords)) THEN
                                 q_source(1)=q_source(1)+CollidingFlow%density-Info%q(i,j,k,1)
                                 r=sqrt(sum(coords(1:2)**2/CollidingFlow%Shape%size_param(1:2)**2))
                                 fact=SmoothFunction(r, CollidingFlow%smooth_function,CollidingFlow%smooth_distance/sqrt(sum(CollidingFlow%Shape%Size_param(1:2)**2)))
                                 vel=fact*CollidingFlow%velocity*RotateVectorFromShape(CollidingFlow%Shape, (/0d0,0d0,1d0/))
                                 IF (levels(Info%level)%tnow > CollidingFlow%tShutOff) THEN
                                    IF (levels(Info%level)%tnow > CollidingFlow%tShutOff+CollidingFlow%RampTime) THEN
                                       vel=0d0
                                    ELSE
                                       vel=vel*(1d0-(levels(Info%level)%tnow - CollidingFlow%tShutOff)/CollidingFlow%RampTime)
                                    END IF
                                 END IF
                                 IF (iE .ne. 0) q_source(iE)=q_source(iE)+gamma7*CollidingFlow%density*CollidingFlow%Temperature - Info%q(i,j,k,iE)
                                 IF (IsAboveInterface(CollidingFlow%InterfaceObj, pos, dist)) THEN
                                    q_source(ivx:ivx+nDim-1)=q_source(ivx:ivx+nDim-1)-CollidingFlow%density*vel(1:nDim)*SmoothFunction(1d0-dist/CollidingFlow%interface_dist, CollidingFlow%interface_func, 1d0) -Info%q(i,j,k,ivx:ivx+nDim-1)
                                    IF (CollidingFlow%iTracer(1) /= 0) q_source(CollidingFlow%iTracer(1)) = q_source(CollidingFlow%iTracer(1)) + CollidingFlow%density- Info%q(i,j,k,CollidingFlow%iTracer(1))
                                    IF (CollidingFlow%iTracer(2) /= 0) q_source(CollidingFlow%iTracer(2)) = q_source(CollidingFlow%iTracer(2)) + 0d0 - Info%q(i,j,k,CollidingFlow%iTracer(2))
                                 ELSE
                                    q_source(ivx:ivx+nDim-1)=q_source(ivx:ivx+nDim-1)+CollidingFlow%density*vel(1:nDim)*SmoothFunction(1d0-dist/CollidingFlow%interface_dist, CollidingFlow%interface_func, 1d0)-Info%q(i,j,k,ivx:ivx+nDim-1)
                                    IF (CollidingFlow%iTracer(2) /= 0) q_source(CollidingFlow%iTracer(2)) = q_source(CollidingFlow%iTracer(2)) + CollidingFlow%density- Info%q(i,j,k,CollidingFlow%iTracer(2))
                                    IF (CollidingFlow%iTracer(1) /= 0) q_source(CollidingFlow%iTracer(1)) = q_source(CollidingFlow%iTracer(1)) + 0d0 - Info%q(i,j,k,CollidingFlow%iTracer(1))
                                 END IF
                                 IF (lMHD) THEN
                                    q_source(iBx:iBz)=CollidingFlow%B(:)
                                 END IF
                              END IF
                           END DO
                        END DO
                     END DO
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+q_source*q_fact                 
                  END DO
               END DO
            END DO
            CALL ConvertInternalToTotalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
         END DO
         DEALLOCATE(q_Source)
      END IF

   END SUBROUTINE PlaceCollidingFlow




END MODULE CollidingFlows

