!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    disks.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file disks.f90
!! @brief Main file for module Disks

!> @defgroup Disks Disks Object
!! @brief Module that handles the placement of disks
!! @ingroup ModuleObjects

!> Module that handles the placement of disks
!! @ingroup Disks
MODULE Disks
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   USE Shapes
   USE Perturbation
   USE DataInfoOps
   USE EOS
   USE BE_MODULE
   USE ObjectDeclarations
   IMPLICIT NONE

   INTEGER, PARAMETER :: UNIFORM = 0, HYDROSTATIC = 1

   !> Disk Data Type
   TYPE DiskDef
      REAL(KIND=qPREC), DIMENSION(3) :: velocity= (/0,0,0/)  !Disk velocity 
      REAL(KIND=qPREC), DIMENSION(3) :: position = (/0,0,0/)  !Disk velocity 
      REAL(KIND=qPREC) :: density=10   !Disk peak density
      REAL(KIND=qPREC) :: pressure=1   !External Disk Pressure
      REAL(KIND=qPREC) :: radius=1
      REAL(KIND=qPREC) :: height=.25
      REAL(KIND=qPREC) :: thickness=.1 !thickness of disk smoothing region
      REAL(KIND=qPREC) :: phi=0
      REAL(KIND=qPREC) :: theta=0
      REAL(KIND=qPREC) :: central_mass  !Central Mass
!      REAL(KIND=qPREC) :: temp=.1      !Disk temperature
      REAL(KIND=qPREC) :: t0=0         !Last time velocity and position were updated.
      REAL(KIND=qPREC) :: background_density=1   !Last time velocity and position were updated.
!      REAL(KIND=qPREC) :: alpha=.5d0
!      REAL(KIND=qPREC) :: B_tor=0     !Maximum Bfield for toroidal configuration
!      REAL(KIND=qPREC) :: B_pol=0     !Maximum Bfield for poloidal configuration
!      REAL(KIND=qPREC) :: Omega=0     !Angular velocity for solid body rotation
      INTEGER :: soft_function         !Gravitational softening function
      REAL(KIND=qPREC) :: soft_length  !Gravitational softening radius
      INTEGER :: SubSample=1           !Subsample factor
      INTEGER :: iTracer=0             !Disk Tracer
      INTEGER :: type = UNIFORM
      LOGICAL :: PersistInternal=.false.
      LOGICAL :: PersistInBoundaries(3,2)=.false.  !keep disk in each boundary
!     REAL(KIND=qPREC) :: m2A=0 !Amplitude of an m=2 density perturbation [  rho = rho_0*(1+m2A*cos(2*phi))   ]
      INTEGER :: id
      TYPE(ShapeDef), POINTER :: Shape
!    TYPE(PerturbationDef), POINTER :: DensityPerturbation => Null()
!    TYPE(PerturbationDef), DIMENSION(:), POINTER :: MagneticPerturbation
      INTEGER :: ObjId
   END TYPE DiskDef

  !new declaration
  TYPE pDiskDef
    TYPE(DiskDef), POINTER :: ptr
  END TYPE pDiskDef
  TYPE(pDiskDef) :: pDisk
  !

   SAVE
CONTAINS


   !> Creates a disk object
   !! @param Disk Disk object
   SUBROUTINE CreateDisk(Disk, density, pressure, central_mass)
      TYPE(DiskDef), POINTER :: Disk
      REAL(KIND=qPREC), OPTIONAL :: density, pressure, central_mass
      IF (.NOT. ASSOCIATED(Disk)) ALLOCATE(Disk)  
      ALLOCATE(disk%Shape)
      IF(Present(density)) Disk%density = density
      IF(Present(pressure)) Disk%pressure = pressure
      IF(Present(central_mass)) Disk%central_mass = central_mass 
      CALL UpdateDisk(Disk)
      CALL AddDiskToList(Disk)
   END SUBROUTINE CreateDisk

   !> Updates a disk object
   !! @param Disk Disk object
   SUBROUTINE UpdateDisk(Disk)
      TYPE(DiskDef), POINTER :: Disk
      IF (Disk%Type == UNIFORM) THEN
         CALL SetShapeType(Disk%Shape, CYLINDER, (/Disk%radius,Disk%radius, Disk%height/))
      ELSE
         CALL SetShapeType(Disk%Shape, CYLINDER, (/Disk%radius*1e2,Disk%radius*1e2, Disk%height*1e2/))
      END IF
      CALL SetShapeOrientation(Disk%Shape, 0d0, Disk%theta, Disk%phi)
      Disk%Shape%Position=Disk%position
      CALL SetShapeBounds(Disk%Shape)         
      
   END SUBROUTINE UpdateDisk

   SUBROUTINE AddDiskToList(Disk)
      TYPE(DiskDef), POINTER :: Disk
      TYPE(ObjectDef), POINTER :: Object
      Disk%ObjId = ObjectListAdd(Object,DISKOBJ)
      pDisk%ptr => Disk
      len = size(transfer(pDisk, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pDisk, Object%storage)
   END SUBROUTINE AddDiskToList

   SUBROUTINE DestroyDisk(Disk,id)
      TYPE(DiskDef),POINTER :: Disk
      TYPE(ObjectDef),POINTER :: Object
      INTEGER,OPTIONAL :: id

      IF(PRESENT(id)) THEN
        Object => ObjectListFind(id)
        IF (ASSOCIATED(Object) .AND. Object%type == DISKOBJ) THEN
          write(*,*) "found obj"
          pDisk = transfer(Object%storage,pDisk)
          IF(ASSOCIATED(pDisk%ptr%Shape)) THEN
            DEALLOCATE(pDisk%ptr%Shape)
          ENDIF
          DEALLOCATE(pDisk%ptr)
          NULLIFY(pDisk%ptr)
          CALL ObjectListRemove(id)
        ENDIF
      ELSE
        CALL ObjectListRemove(Disk%ObjId)
        IF(ASSOCIATED(Disk%Shape)) THEN
            DEALLOCATE(Disk%Shape)
          ENDIF
        DEALLOCATE(Disk)
        NULLIFY(Disk)
      ENDIF
   END SUBROUTINE DestroyDisk

   SUBROUTINE DiskGridInit(Info, Disk)
      TYPE(InfoDef) :: Info
      TYPE(DiskDef), POINTER :: Disk
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      INTEGER :: nOverlaps
      CALL CalcPhysicalOverlaps(Info, Disk%Shape%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
      IF (nOverlaps > 0) THEN
         CALL PlaceDisk(Info, Disk, nOverlaps, mSs, offsets)
         DEALLOCATE(mSs, offsets)
      END IF
    END SUBROUTINE DiskGridInit

   
   SUBROUTINE DiskBeforeStep(Info, Disk)
      TYPE(InfoDef) :: Info
      TYPE(DiskDef), POINTER :: Disk
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      INTEGER :: nOverlaps
      INTEGER :: i,j
      IF (Disk%PersistInternal) THEN
         CALL DiskGridInit(Info, Disk)
      ELSE 
         DO i=1,nDim
            DO j=1,2
               IF (Disk%PersistInBoundaries(i,j)) THEN
                  CALL CalcPhysicalOverlaps(Info, Disk%Shape%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
                  IF (nOverlaps > 0) THEN
                     CALL PlaceDisk(Info, Disk, nOverlaps, mSs, offsets)
                     DEALLOCATE(mSs, offsets)
                  END IF
               END IF
            END DO
         END DO
      END IF
   END SUBROUTINE DiskBeforeStep

   SUBROUTINE DiskSetErrFlag(Info, Disk)
      TYPE(InfoDef) :: Info
      Type(DiskDef), POINTER :: Disk
   END SUBROUTINE DiskSetErrFlag

   SUBROUTINE DiskBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE DiskBeforeGlobalStep

   !> Place a disk in an info object
   !! @param Info Info object
   !! @param Disk Disk object
   SUBROUTINE PlaceDisk(Info,Disk,nOverlaps,mSs,offsets)
      TYPE(InfoDef) :: Info
      Type(DiskDef) :: Disk
      INTEGER :: i,j,k,n,m,ii,jj,kk, location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos, rpos
      REAL(KIND=qPREC) :: sample_fact(3), q_fact, dx(3)
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(3,2) :: tempbounds
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source, emf_source, qacc
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: emf
!      RETURN
      dx=levels(Info%level)%dX*(/(merge(1,0,nDim >= i), i=1,3)/)
      xpos=0          
      ALLOCATE(q_Source(NrHydroVars), qacc(NrHydroVars))
      sample_res=1
      sample_fact=0d0
      IF (Disk%SubSample > 0) sample_res(1:nDim)=max(Disk%SubSample,2**(MaxLevel-Info%level))
      sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
      q_fact=product(sample_fact(1:nDim))
      DO n=1,nOverlaps
         mS=mSs(n,:,:)
         offset=offsets(n,:)
         !Set up emf first - aux fields first
         CALL ConvertTotalToInternalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))            ! Now set up cell centered quantities (density and momentum)
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dx(3)
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx(2)
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx(1)
                     qacc=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dx(3)*sample_fact(3)
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx(2)*sample_fact(2)
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx(1)*sample_fact(1)
                              IF (IsInShape(Disk%Shape, pos, rpos)) THEN
                                 CALL q_DiskSource(Disk,rpos,Info%q(i,j,k,1:NrHydroVars), q_source)
                                 qacc=qacc+q_source
!                              ELSE
!                                 write(*,*) pos
                              END IF
                           END DO
                        END DO
                     END DO
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+qacc*q_fact                 
                  END DO
               END DO
            END DO
            CALL ConvertInternalToTotalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
         END DO
         DEALLOCATE(q_Source, qacc)
   END SUBROUTINE PlaceDisk

   SUBROUTINE q_DiskSource(Disk, pos, q, s)
      TYPE(DiskDef) :: Disk
      REAL(KIND=qPREC), DIMENSION(:) :: pos, q, s
      REAL(KIND=qPREC) :: r, f, rotation_speed(3), perturb, r_s, omega, f_rad, f_grav(3), c_s, rho, r_disk
      !     r=SQRT(SUM((pos(1:nDim)-Disk%position(1:nDim))**2))
      r=sqrt(sum(pos**2))
      f=smooth_tanh(r/Disk%Shape%size_param(1),Disk%thickness)
      perturb=1d0
      s=0
      r_s=sqrt(sum(pos(1:2)**2))

      IF (Disk%Type == UNIFORM) THEN
         rho=Disk%density*perturb
         s(1)=(rho-q(1))*f !fade into disk        
         f_grav=GravityForce(Disk%central_mass, TransformCoordsFromShape(Disk%Shape,pos), Disk%soft_length, Disk%soft_function)

         !gravitational force of rotated position relative to particl
         f_grav=RotateVectorToShape(Disk%Shape, f_grav)
         IF (r > TINY(r)) THEN
            f_rad=sum(-f_grav(1:2)*pos(1:2))/r_s
            omega=sqrt(f_rad/r_s)
         ELSE
            omega=0
         END IF
      ELSEIF (Disk%Type == HYDROSTATIC) THEN
         IF (iEOS == EOS_ISOTHERMAL) THEN
            c_s=Iso_Speed
         ELSE
            c_s=sqrt(gamma*Disk%Pressure/Disk%density)
         END IF
         r_disk=Disk%radius
         IF (Disk%soft_function == PLUMMERSOFT) THEN
            r_s=sqrt(r_s**2+disk%soft_length**2)
            r=sqrt(r**2+disk%soft_length**2)
            r_disk=sqrt(r_disk**2+disk%soft_length**2)
         END IF

         IF (r_s > r_disk) THEN
            rho=Disk%density*perturb*exp(Disk%central_mass*ScaleGrav/c_s**2*(1d0/r-1d0/r_disk))
            omega=0
            IF (rho < Disk%Background_density) THEN
               rho=q(1)
               s(1)=0
            ELSE
               s(1)=(rho-q(1))*f
            END IF
         ELSE
            rho=Disk%density*perturb*exp(Disk%central_mass*ScaleGrav/c_s**2*(1d0/r-1d0/r_s))
            IF (rho < Disk%background_density) THEN
               rho=q(1)
               s(1)=0d0!(rho-q(1))*f
               omega=0d0!sqrt((Disk%alpha)*Disk%central_mass*ScaleGrav/r_s**2)
            ELSE
               s(1)=(rho-q(1))*f
               omega=sqrt(Disk%central_mass*ScaleGrav/r_s**3)
            END IF

        END IF
      END IF
      rotation_speed(1:3)=RotateVectorFromShape(Disk%Shape, Omega*(/-pos(2),pos(1),0d0/)) 
      s(ivx:ivx+nDim-1)=(rho*(Disk%velocity(1:nDim) + rotation_speed(1:nDim))-q(ivx:ivx+nDim-1))*f!momentum associated with disk mass + rotational energy
      IF (iCylindrical == WITHANGMOM) s(iAngMom)=(rho*(rotation_speed(3))-q(iAngMom))*f
      IF (iE .ne. 0) s(iE)=(gamma7*Disk%pressure-q(iE))*f !Internal energy associated with disk material
      IF (Disk%iTracer .ne. 0) s(Disk%iTracer) = Disk%density*perturb*f
   END SUBROUTINE q_DiskSource


END MODULE Disks


