!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    interfaces.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file interfaces.f90
!! @brief Main file for module Interfaces

!> @defgroup Interfaces Interfaces Object
!! @brief Module that handles the placement of interfaces
!! @ingroup ModuleObjects

!> Module that handles the placement of interfaces
!! @ingroup Interfaces
MODULE Interfaces
   USE GlobalDeclarations
   IMPLICIT NONE
   !> Interface data type
   TYPE InterfaceDef      
     REAL(KIND=qPREC) :: position(3)=(/0,0,0/)
     REAL(KIND=qPREC) :: RotationMatrix(3,3)=RESHAPE((/1d0,0d0,0d0,0d0,1d0,0d0,0d0,0d0,1d0/),(/3,3/))    !Rotates lab frame into object frame
     REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:) :: WaveVector
     REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: amplitude
     REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: phase
     REAL(KIND=qPREC) :: HalfWidth=1e30
     INTEGER :: nWaves=0
     INTEGER :: iWaves=0
     INTEGER :: ObjId
   END TYPE InterfaceDef

CONTAINS

  SUBROUTINE CreateInterface(InterfaceObj,theta,phi)
    TYPE(InterfaceDef), POINTER :: InterfaceObj
    REAL(KIND=qPREC), OPTIONAL :: theta,phi
    ALLOCATE(InterfaceObj)
    IF(Present(theta) .AND. Present(phi)) THEN
        CALL SetInterfaceOrientation(InterfaceObj, theta,phi)
    ENDIF
  END SUBROUTINE  CreateInterface

  SUBROUTINE UpdateInterface(InterfaceObj)
    TYPE(InterfaceDef), POINTER :: InterfaceObj
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdateInterface

   SUBROUTINE SetInterfaceOrientation(InterfaceObj, theta, phi)
      TYPE(InterfaceDef), POINTER :: InterfaceObj
      REAL(KIND=qPREC) :: c2,c3,s2,s3,theta,phi
      c2=cos(theta)
      c3=cos(phi)
      s2=sin(theta)
      s3=sin(phi)      
      InterfaceObj%RotationMatrix=RESHAPE((/ &
           c2*c3, -s3, +c3*s2,&
           c2*s3, c3, s2*s3, &
           -s2, 0d0, c2/),(/3,3/),(/0d0/),(/2,1/))
   END SUBROUTINE SetInterfaceOrientation

   SUBROUTINE InitInterfaceWaves(InterfaceObj, nWaves)
      TYPE(InterfaceDef), POINTER :: InterfaceObj
      INTEGER :: nWaves
      IF (nWaves == 0) RETURN
      ALLOCATE(InterfaceObj%WaveVector(nWaves,2), InterfaceObj%amplitude(nWaves), InterfaceObj%Phase(nWaves))
      InterfaceObj%iWaves=0
      InterfaceObj%nWaves=nWaves
   END SUBROUTINE InitInterfaceWaves

   SUBROUTINE AddInterfaceWave(InterfaceObj, wavevector, phase, amplitude)      
      TYPE(InterfaceDef), POINTER :: InterfaceObj
      REAL(KIND=qPREC), DIMENSION(2) :: wavevector
      REAL(KIND=qPREC) :: phase, amplitude
      INTEGER :: iwaves
      iwaves=InterfaceObj%iwaves+1
      InterfaceObj%iwaves=iwaves
      InterfaceObj%WaveVector(iwaves,:)=wavevector
      InterfaceObj%phase(iwaves)=phase
      InterfaceObj%amplitude(iwaves)=amplitude
   END SUBROUTINE AddInterfaceWave
      
   SUBROUTINE DestroyInterface(InterfaceObj)
      TYPE(InterfaceDef), POINTER :: InterfaceObj
      IF (ALLOCATED(InterfaceObj%WaveVector)) DEALLOCATE(InterfaceObj%WaveVector)
      IF (ALLOCATED(InterfaceObj%Amplitude)) DEALLOCATE(InterfaceObj%Amplitude)
      IF (ALLOCATED(InterfaceObj%Phase)) DEALLOCATE(InterfaceObj%Phase)
      DEALLOCATE(InterfaceObj)
      NULLIFY(InterfaceObj)
   END SUBROUTINE DestroyInterface

   FUNCTION IsAboveInterface(InterfaceObj, pos, dist)
     TYPE(InterfaceDef), POINTER :: InterfaceObj
     REAL(KIND=qPREC), DIMENSION(3) :: pos, coords
     REAL(KIND=qPREC) :: z
     REAL(KIND=qPREC), OPTIONAL :: dist
     LOGICAL :: IsAboveInterface
     INTEGER :: l,i
     DO l=1,3
        coords(l)=SUM(InterfaceObj%RotationMatrix(:,l)*(pos(:)-InterfaceObj%position(:))) !Inverse rotation
     END DO
     IF (coords(3) > InterfaceObj%HalfWidth) THEN
        IsAboveInterface=.true.
     ELSEIF (coords(3) < -InterfaceObj%HalfWidth) THEN
        IsAboveInterface=.false.
     ELSE
        z=0         
        DO i=1,InterfaceObj%nWaves
           z=z+InterfaceObj%amplitude(i)*cos(SUM(InterfaceObj%WaveVector(i,:)*coords(1:2))+InterfaceObj%Phase(i))
        END DO
        IsAboveInterface=coords(3)>z
     END IF
     IF (Present(dist)) dist=abs(coords(3)-z)
  END FUNCTION IsAboveInterface


  FUNCTION DistanceToInterface(InterfaceObj, pos)
     TYPE(InterfaceDef), POINTER :: InterfaceObj
     REAL(KIND=qPREC), DIMENSION(3) :: pos, coords
     REAL(KIND=qPREC) :: z
     REAL(KIND=qPREC) :: DistanceToInterface
     INTEGER :: l,i
     DO l=1,3
        coords(l)=SUM(InterfaceObj%RotationMatrix(:,l)*(pos(:)-InterfaceObj%position(:))) !Inverse rotation
     END DO
     z=0         
     DO i=1,InterfaceObj%nWaves
        z=z+InterfaceObj%amplitude(i)*cos(SUM(InterfaceObj%WaveVector(i,:)*coords(1:2))+InterfaceObj%Phase(i))
     END DO
     DistanceToInterface=abs(coords(3)-z)
     
   END FUNCTION DistanceToInterface

END MODULE Interfaces

