!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    outflows.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file outflows.f90
!! @brief Main file for module Outflows

!> @defgroup Outflows Outflows Object
!! @brief Module that handles the placement of outflows
!! @ingroup ModuleObjects

!> Module that handles the placement of outflows
!! @ingroup Outflows
MODULE Outflows
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   USE DataInfoOps
   USE EOS
   USE ObjectDeclarations
   IMPLICIT NONE
   private :: emf_source_3D

   !> Outflow Data Type
   REAL(KIND=qPREC), PARAMETER :: default_buffer=2d0

   TYPE OutflowDef
      LOGICAL :: active=.false.
      LOGICAL :: initialize=.false.
      REAL(KIND=qPREC) :: density=0             !Density of Jet
      REAL(KIND=qPREC) :: temperature=0         !Temperature of Jet
      REAL(KIND=qPREC) :: velocity=0            !Velocity of Jet
      REAL(KIND=qPREC) :: theta=0               !Angle between jet and x-axis 
      REAL(KIND=qPREC) :: phi=0                 !3D: Angle around x-axis from xy plane // 2D: angle around z-axis from x-axis
      REAL(KIND=qPREC) :: B=0                   !Strength of peak magnetic field in computational units
      REAL(KIND=qPREC) :: begin=0               !When does jet turn on
      REAL(KIND=qPREC) :: duration=0            !How long does jet run for
      REAL(KIND=qPREC) :: decay=0               !How quickly does the jet decay
      REAL(KIND=qPREC) :: radius=0              !Radius of jet launching region
      REAL(KIND=qPREC) :: thickness=0           !Thickness of source region
      REAL(KIND=qPREC) :: open_angle=0          !Jet opening angle (half angle) in radians
      REAL(KIND=qPREC) :: oscAmplitude=0        !Amplitude of velocity Oscillations
      REAL(KIND=qPREC) :: oscPeriod=0           !Period of velocity oscillations
      REAL(KIND=qPREC) :: oscPhase=0            !Initial phase of velocity oscillations
      REAL(KIND=qPREC) :: PrecessAmplitude=0    !Precession angle in radians - measured from x' axis
      REAL(KIND=qPREC) :: PrecessPeriod=0       !Period of precession
      REAL(KIND=qPREC) :: PrecessPhase=0        !Initial phase of precession - measured from x'y' plane in rotated jet frame.
      REAL(KIND=qPREC) :: mu=0                  !Inverse beta for jet magnetic field
      REAL(KIND=qPREC) :: finish=0      
      REAL(KIND=qPREC) :: buffer=default_buffer !Distance from center of jet to source region
      REAL(KIND=qPREC) :: offset=0
      REAL(KIND=qPREC) :: r_inner=0
      REAL(KIND=qPREC) :: r_outer=0
      REAL(KIND=qPREC) :: radius_needed
      REAL(KIND=qPREC), DIMENSION(3) :: source_vel=(/0,0,0/)  !Outflow velocity (in direction of outflow axis)
      REAL(KIND=qPREC), DIMENSION(3) :: position=(/0,0,0/)  !Outflow location
      REAL(KIND=qPREC) :: t0=0
      REAL(KIND=qPREC), DIMENSION(3,2) :: xBounds !physical extent of outflow box
      REAL(KIND=qPREC) :: mass=0                    !Current mass of outflow object
      REAL(KIND=qPREC) :: accretionrate=0                !Current accretion rate
      REAL(KIND=qPREC) :: massloss=0                !mass lossed to interior cells of grid
      INTEGER :: id
      INTEGER :: ObjId


   END TYPE OutflowDef

   !new declaration
   TYPE pOutflowDef
      TYPE(OutflowDef), POINTER :: ptr
   END TYPE pOutflowDef
  TYPE(pOutflowDef) :: pOutflow
  !

   INTEGER ::  nOutflowObjects = 0
   SAVE
CONTAINS


   !> Initializes Outflow Module
  ! SUBROUTINE InitOutflows()
  !    nOutflowObjects=0
  !    lOutflows=.false.
  ! END SUBROUTINE InitOutflows


     SUBROUTINE CreateOutflowObject(Outflow, density, temperature, velocity)
    !INTEGER :: dummy
    TYPE(OutflowDef),POINTER :: Outflow
    REAL(KIND=qPREC), OPTIONAL :: density, temperature, velocity
    
    ALLOCATE(Outflow)
    nOutflowObjects=nOutflowObjects+1
    Outflow%id=nOutflowObjects

    IF (Present(density)) OutFlow%density=density
    IF (Present(temperature)) Outflow%temperature=temperature
    IF (Present(velocity)) OutFlow%velocity=velocity
    
    CALL UpdateOutflow(Outflow)
    CALL AddOutflowObjToList(Outflow)
  END SUBROUTINE CreateOutflowObject


   !> Initializes a outflow object
   !! @param Outflow Outflow object
   SUBROUTINE UpdateOutflow(Outflow)
      TYPE(OutflowDef), POINTER :: Outflow
      CALL SetOutflowGeometry(Outflow)
      CALL SetOutflowBounds(Outflow)     
      Outflow%finish=outflow%begin+outflow%duration+outflow%decay
   END SUBROUTINE UpdateOutflow

   SUBROUTINE AddOutflowObjToList(Outflow)
      TYPE(OutflowDef), POINTER :: Outflow
      TYPE(ObjectDef), POINTER :: Object
      Outflow%ObjId = ObjectListAdd(Object,OutflowOBJ)
      pOutflow%ptr => Outflow
      len = size(transfer(pOutflow, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pOutflow,Object%storage)
   END SUBROUTINE AddOutflowObjToList

  SUBROUTINE DestroyOutflowObject(Outflow,id)
      TYPE(OutflowDef),POINTER :: Outflow
      TYPE(ObjectDef),POINTER :: Object
      INTEGER,OPTIONAL :: id

      IF(PRESENT(id)) THEN
        Object => ObjectListFind(id)
        IF (ASSOCIATED(Object) .AND. Object%type == OUTFLOWOBJ) THEN
          pOutflow = transfer(Object%storage,pOutFlow)
          DEALLOCATE(pOutflow%ptr)
          NULLIFY(pOutflow%ptr)
          CALL ObjectListRemove(id)
        ENDIF
      ELSE
        CALL ObjectListRemove(Outflow%ObjId)
        DEALLOCATE(Outflow)
        NULLIFY(Outflow)
      ENDIF
  END SUBROUTINE DestroyOutflowObject

  SUBROUTINE OutflowGridInit(Info, Outflow)
      TYPE(InfoDef) :: Info
      TYPE(OutflowDef), POINTER :: Outflow
      CALL OutflowBeforeStep(Info, Outflow)
  END SUBROUTINE OutflowGridInit

   SUBROUTINE OutflowBeforeStep(Info, Outflow)
      TYPE(InfoDef) :: Info
      TYPE(OutflowDef), POINTER :: Outflow
       IF (Outflow%begin < levels(Info%level)%tnow .AND. Outflow%finish > levels(Info%level)%tnow) THEN
          CALL SourceOutflow(Info, Outflow, levels(Info%level)%tnow, levels(Info%level)%dt, IEVERYWHERE)
       END IF
   END SUBROUTINE OutflowBeforeStep

  SUBROUTINE OutflowSetErrFlag(Info, Outflow)
      TYPE(InfoDef) :: Info
      TYPE(OutflowDef), POINTER :: Outflow
      INTEGER, DIMENSION(:,:,:), POINTER :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      INTEGER :: nOverlaps, nGhost
      INTEGER, DIMENSION(3,2) :: mS
      INTEGER :: i
      nghost=0
         CALL CalcPhysicalOverlaps(Info, Outflow%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic, nGhost)
         DO i=1,nOverlaps
            mS=mSs(i,:,:)
            Info%ErrFlag(ms(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):ms(3,2))=1
         END DO
         IF (nOverlaps > 0) THEN
            DEALLOCATE(mSs)
            NULLIFY(mSs)
         END IF
   END SUBROUTINE OutflowSetErrFlag

   SUBROUTINE OutflowBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE OutflowBeforeGlobalStep



   !> Place a outflow in an info object
   !! @param Info Info object
   !! @param Outflow Outflow object
   SUBROUTINE PlaceOutflow(Info,Outflow,location)
      TYPE(InfoDef) :: Info
      Type(OutflowDef) :: Outflow
      INTEGER :: i,j,k,n,m,ii,jj,kk,location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos
      REAL(KIND=qPREC) :: sample_fact, q_fact, dx
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source, emf_source
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: emf
      dx=levels(Info%level)%dX
      xpos=0          
      CALL CalcPhysicalOverlaps(Info, Outflow%xBounds, mSs, nOverlaps, offsets, location,lHydroPeriodic)
      IF (nOverlaps > 0) THEN
         sample_res=1
         sample_res(1:nDim)=min(2**(MaxLevel-Info%level),8)
         sample_fact=1d0/REAL(sample_res(1),8)
         q_fact=sample_fact**nDim
         ALLOCATE(q_Source(NrHydroVars))
         DO n=1,nOverlaps
            mS=mSs(n,:,:)
            offset=offsets(n,:)
            !Set up emf first - aux fields first
            CALL ConvertTotalToInternalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
            IF (MaintainAuxArrays .AND. Outflow%B > 0d0) THEN
               IF (nDim == 2) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,1,1))
                  emf=0
                  DO i=mS(1,1),mS(1,2)+1
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)+1
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        emf(i,j,1,1)=emf_Launch_2D(Outflow,xpos)                   
                     END DO
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,1,1:2),emf(:,:,1,1),dx)
                  DEALLOCATE(emf)
               ELSE IF (nDim == 3) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,3), emf_source(3))
                  emf=0
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        DO k=mS(3,1),mS(3,2)
                           xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dx
                           DO m=1,3
                              pos=xpos
                              DO ii=1,sample_res(1)
                                 pos(m)=xpos(m)+(REAL(ii, 8)-half)*dx*sample_fact
                                 emf_source=emf_launch_3D(Outflow,pos)
                                 emf(i,j,k,m)=emf(i,j,k,m)+emf_source(m)
                              END DO
                              emf(i,j,k,m)=emf(i,j,k,m)*sample_fact
                           END DO
                        END DO
                     END DO
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,1:3),emf,dx)
                  DEALLOCATE(emf, emf_source)                 
               END IF
               CALL UpdateAux(Info, mS)          
            END IF
            ! Now set up cell centered quantities (density and momentum)
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+REAL(k-1,8)*dx
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+REAL(j-1,8)*dx
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+REAL(i-1,8)*dx
                     q_Source=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dx*sample_fact
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact
                              CALL OutflowLaunch(Outflow,pos,Info%q(i,j,k,1:NrHydroVars),q_source)
                           END DO
                        END DO
                     END DO
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+q_source*q_fact                 
                     IF (i >= 1 .AND. i <= Info%mX(1) .AND. j >= 1 .AND. j <= Info%mX(2) .AND. k >= 1 .AND. k <= Info%mX(3)) Outflow%massloss=Outflow%massloss+q_source(1)*q_fact
                  END DO
               END DO
            END DO
            CALL ConvertInternalToTotalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
         END DO
         DEALLOCATE(q_Source)
      if (associated(mSs)) then
      	DEALLOCATE(mSs)
      	nullify(mSs)
      endif
      if (associated(offsets)) then
      	DEALLOCATE(offsets)
      	nullify(offsets)
      endif
 END IF
   END SUBROUTINE PlaceOutflow


   !> Place a outflow in an info object
   !! @param Info Info object
   !! @param Outflow Outflow object
   SUBROUTINE SourceOutflow(Info,Outflow,t,dt,location)
      TYPE(InfoDef) :: Info
      Type(OutflowDef) :: Outflow
      INTEGER :: i,j,k,n,m,ii,jj,kk,location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos
      REAL(KIND=qPREC) :: sample_fact, q_fact, dx, t, dt
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source, emf_source
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: emf
      dx=levels(Info%level)%dX
      xpos=0          
      CALL CalcPhysicalOverlaps(Info, Outflow%xBounds, mSs, nOverlaps, offsets,location,lHydroPeriodic)
      IF (nOverlaps > 0) THEN
         sample_res=1
         sample_res(1:nDim)=min(2**(MaxLevel-Info%level),8)
         sample_fact=1d0/REAL(sample_res(1),8)
         q_fact=sample_fact**nDim
         ALLOCATE(q_Source(NrHydroVars))
         DO n=1,nOverlaps
            mS=mSs(n,:,:)
            offset=offsets(n,:)
            !Set up emf first - aux fields first
            CALL ConvertTotalToInternalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
            IF (MaintainAuxArrays .AND. Outflow%B > 0d0) THEN
               IF (nDim == 2) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,1,1))
                  emf=0
                  DO i=mS(1,1),mS(1,2)+1
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)+1
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        emf(i,j,1,1)=emf_Source_2D(Outflow,xpos,t,dt)                   
                     END DO
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,1,1:2),emf(:,:,1,1),dx)
                  DEALLOCATE(emf)
               ELSE IF (nDim == 3) THEN
                  ALLOCATE(emf(mS(1,1):mS(1,2)+1, mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,3), emf_source(3))
                  emf=0
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     DO j=mS(2,1),mS(2,2)
                        xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                        DO k=mS(3,1),mS(3,2)
                           xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dx
                           DO m=1,3
                              pos=xpos
                              DO ii=1,sample_res(1)
                                 pos(m)=xpos(m)+(REAL(ii, 8)-half)*dx*sample_fact
                                 emf_source=emf_source_3D(Outflow,pos,t,dt)
                                 emf(i,j,k,m)=emf(i,j,k,m)+emf_source(m)
                              END DO
                              emf(i,j,k,m)=emf(i,j,k,m)*sample_fact
                           END DO
                        END DO
                     END DO
                  END DO
                  CALL AddCurl(Info%aux(mS(1,1):mS(1,2)+1,mS(2,1):mS(2,2)+1,mS(3,1):mS(3,2)+1,1:3),emf,dx)
                  DEALLOCATE(emf, emf_source)                 
               END IF
               CALL UpdateAux(Info, mS)          
            END IF
            ! Now set up cell centered quantities (density and momentum)
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+REAL(k-1,8)*dx
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+REAL(j-1,8)*dx
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+REAL(i-1,8)*dx
                     q_Source=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dx*sample_fact
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact
                              CALL OutflowSource(Outflow,pos,Info%q(i,j,k,1:NrHydroVars),q_source,t,dt)
                           END DO
                        END DO
                     END DO
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+q_source*q_fact                 
                     IF (i >= 1 .AND. i <= Info%mX(1) .AND. j >= 1 .AND. j <= Info%mX(2) .AND. k >= 1 .AND. k <= Info%mX(3)) Outflow%massloss=Outflow%massloss+q_source(1)*q_fact
                  END DO
               END DO
            END DO
            CALL ConvertInternalToTotalEnergy(Info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2),mS(3,1):mS(3,2),:))           
         END DO
      if (associated(mSs)) then
      	DEALLOCATE(mSs)
      	nullify(mSs)
      endif
      if (associated(offsets)) then
      	DEALLOCATE(offsets)
      	nullify(offsets)
      endif
   END IF
   END SUBROUTINE SourceOutflow



   SUBROUTINE OutflowLaunch(Outflow, pos, q, dq)
      REAL(KIND=qPrec), DIMENSION(:) :: pos,q,dq
      REAL(KIND=qPrec) :: s, r, angle, w(3), vddr, pr, pT, mag_press,therm_press,rz,rz2,s2,x,t,dt,rho_Outflow,f,b_curr(3),precessPhase,oscPhase,velocity,rpos(3)
      TYPE(OutflowDef) :: Outflow
      pos=pos-Outflow%position !refere to the relative position of source
      dq=0
      precessphase=0
      oscphase=0

      IF (Outflow%precessperiod > 0) PrecessPhase=Outflow%PrecessPhase
      IF (Outflow%oscPeriod > 0) oscPhase=Outflow%oscPhase
      velocity=Outflow%velocity+sin(oscPhase)*Outflow%oscAmplitude
      !    write(*,*) t,dt,precessphase,oscphase,velocity
      SELECT CASE(nDim)
      CASE(2)

         rpos(1:2)=rotate_z2D(pos(1:2),-Outflow%phi-PrecessPhase) !rotate to coordinates in Outflow frame
         x=Outflow%offset+abs(rpos(1)) !distance to 'center' for Outflows
         s2=rpos(2)**2
         s=abs(rpos(2))

         IF (Outflow%open_angle == 0) THEN
            IF (s < Outflow%radius .AND. x < Outflow%r_outer) THEN
               IF (Outflow%B > 0) THEN
                  write(*,*) "Magnetized Outflows in 2D not supported"
                  STOP
               END IF
               dq(1)=Outflow%density-q(1)
               therm_press=Outflow%density*Outflow%temperature
               IF (iE .ne. 0) dq(iE)=therm_press-q(iE)
               f=fade(s/Outflow%radius, .5d0)
               vddr=f*velocity/Outflow%thickness
               IF (x > Outflow%buffer) THEN
                  dq(ivx:ivy)=Outflow%density*(vddr*(x-Outflow%buffer)*rotate_z2D((/sign(1d0,rpos(1)), 0d0/),Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         ELSEIF (abs(Outflow%open_angle-half*Pi) < .01) THEN
            !          angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  vddr=velocity/Outflow%thickness
                  dq(ivx:ivy)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_z2D(rpos(1:2)/r,Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         ELSE
            angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (angle < Outflow%open_angle .AND. r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  f=fade(angle/Outflow%open_angle, .1d0)
                  vddr=f*velocity/Outflow%thickness
                  dq(ivx:ivy)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_z2D((/sign(cos(angle),rpos(1)), sin(angle)/),Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         END IF

      CASE(3)

         rpos=rotate_z(rotate_x(pos,-Outflow%phi),-Outflow%theta) !rotate coordinates again so that vector along Outflow-axis is along x-axis
         rpos=rotate_z(rotate_x(rpos,-PrecessPhase),-Outflow%PrecessAmplitude)
         x=Outflow%offset+abs(rpos(1)) !distance to 'center' for Outflows
         s2=rpos(2)**2+rpos(3)**2
         s=sqrt(s2)

         IF (Outflow%open_angle == 0) THEN
            IF (s < Outflow%radius .AND. x < Outflow%r_outer) THEN
               IF (Outflow%B > 0) THEN
                  IF (.NOT. ALL(Outflow%source_vel==0)) THEN
                     write(*,*) "moving magnetized Outflows not supported"
                     STOP
                  END IF
                  !First calculate existing b field in terms of local rotated cylindrical coordinates
                  !                b_curr=rotate_z(rotate_x(q(iBx:iBz),-Outflow%phi),-Outflow%theta)
                  !                b_curr(1)=0d0 !We don't want to change B along the axis
                  IF (s <= .8*Outflow%radius) THEN !magnetic radius inside Outflow beam
                     mag_press=half*(Outflow%B*s/(.8*Outflow%radius))**2
                     !                   dq(iBx:iBz)=rotate_x(rotate_z(Outflow%B/(.8*Outflow%radius)*(/0d0,-pos(3),pos(2)/)-b_curr,Outflow%theta),Outflow%phi)
                  ELSE
                     mag_press=half*(Outflow%B*(Outflow%radius-s)/(.2*Outflow%radius))**2
                     !                   dq(iBx:iBz)=rotate_x(rotate_z(Outflow%B*(Outflow%radius-s)/(.2*Outflow%radius*s)*(/0d0,-pos(3),pos(2)/)-b_curr,Outflow%theta),Outflow%phi)
                  END IF

               ELSE
                  mag_press=0
               END IF
               pT=Outflow%density*Outflow%temperature
               therm_press=pT-mag_press
               rho_Outflow=therm_press/Outflow%temperature  
               dq(1)=rho_Outflow-q(1)
               IF (iE .ne. 0) dq(iE)=therm_press-q(iE)
               f=fade(s/Outflow%radius, .1d0)
               vddr=f*velocity/Outflow%thickness
               IF (x > Outflow%buffer) THEN
                  dq(ivx:ivz)=rho_Outflow*(vddr*(x-Outflow%buffer)*rotate_x(rotate_z(rotate_x(rotate_z((/sign(1d0,rpos(1)), 0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)+Outflow%source_vel)-q(ivx:ivz) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivz)=rho_Outflow*Outflow%source_vel-q(ivx:ivz)
               END IF
            END IF
         ELSE
            angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (angle < Outflow%open_angle .AND. r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  f=fade(angle/Outflow%open_angle, .1d0)
                  vddr=f*velocity/Outflow%thickness
                  dq(ivx:ivz)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_x(rotate_z(rotate_x(rotate_z((/sign(cos(angle),rpos(1)), sin(angle)*rpos(2)/s, sin(angle)*rpos(3)/s/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)+Outflow%source_vel)-q(ivx:ivz) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivz)=Outflow%density*Outflow%source_vel-q(ivx:ivz)
               END IF
            END IF
         END IF
      END SELECT
   END SUBROUTINE OutflowLaunch

   SUBROUTINE OutflowSource(Outflow, pos, q, dq, t, dt)
      REAL(KIND=qPrec), DIMENSION(:) :: pos,q, dq
      REAL(KIND=qPrec) :: s, r, angle, w(3), vddr, pr, pT, mag_press,therm_press,rz,rz2,s2,x,t,dt,rho_Outflow,f,b_curr(3),precessPhase,oscPhase,velocity,rpos(3)
      TYPE(OutflowDef) :: Outflow
      pos=pos-Outflow%position !refere to the relative position of source
      dq=0
      precessphase=0
      oscphase=0
      IF (Outflow%precessperiod > 0) PrecessPhase=Outflow%PrecessPhase+2*Pi*(t-Outflow%begin)/Outflow%precessPeriod
      IF (Outflow%oscPeriod > 0) oscPhase=Outflow%oscPhase+2d0*Pi*(t-Outflow%begin)/Outflow%oscPeriod
      velocity=Outflow%velocity+sin(oscPhase)*Outflow%oscAmplitude
      !    write(*,*) t,dt,precessphase,oscphase,velocity
      SELECT CASE(nDim)
      CASE(2)

         rpos(1:2)=rotate_z2D(pos(1:2),-Outflow%phi-PrecessPhase) !rotate to coordinates in Outflow frame
         x=Outflow%offset+abs(rpos(1)) !distance to 'center' for Outflows
         s2=rpos(2)**2
         s=abs(rpos(2))

         IF (Outflow%open_angle == 0) THEN
            IF (s < Outflow%radius .AND. x < Outflow%r_outer) THEN
               IF (Outflow%B > 0) THEN
                  write(*,*) "Magnetized Outflows in 2D not supported"
                  STOP
               END IF
               dq(1)=Outflow%density-q(1)
               therm_press=Outflow%density*Outflow%temperature
               IF (iE .ne. 0) dq(iE)=therm_press-q(iE)
               f=fade(s/Outflow%radius, .5d0)
               vddr=f*velocity/Outflow%thickness
               IF (x > Outflow%buffer) THEN
                  dq(ivx:ivy)=Outflow%density*(vddr*(x-Outflow%buffer)*rotate_z2D((/sign(1d0,rpos(1)), 0d0/),Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         ELSEIF (abs(Outflow%open_angle-half*Pi) < .01) THEN
            !          angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  vddr=velocity/Outflow%thickness
                  dq(ivx:ivy)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_z2D(rpos(1:2)/r,Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         ELSE
            angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (angle < Outflow%open_angle .AND. r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  f=fade(angle/Outflow%open_angle, .1d0)
                  vddr=f*velocity/Outflow%thickness
                  dq(ivx:ivy)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_z2D((/sign(cos(angle),rpos(1)), sin(angle)/),Outflow%phi+precessPhase)+Outflow%source_vel(1:2))-q(ivx:ivy) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivy)=Outflow%density*(Outflow%source_vel(1:2))-q(ivx:ivy)
               END IF
            END IF
         END IF

      CASE(3)

         rpos=rotate_z(rotate_x(pos,-Outflow%phi),-Outflow%theta) !rotate coordinates again so that vector along Outflow-axis is along x-axis
         rpos=rotate_z(rotate_x(rpos,-PrecessPhase),-Outflow%PrecessAmplitude)
         x=Outflow%offset+abs(rpos(1)) !distance to 'center' for Outflows
         s2=rpos(2)**2+rpos(3)**2
         s=sqrt(s2)

         IF (Outflow%open_angle == 0) THEN
            IF (s < Outflow%radius .AND. x < Outflow%r_outer) THEN
               IF (Outflow%B > 0) THEN
                  IF (.NOT. ALL(Outflow%source_vel==0)) THEN
                     write(*,*) "moving magnetized Outflows not supported"
                     STOP
                  END IF
                  !First calculate existing b field in terms of local rotated cylindrical coordinates
                  !                b_curr=rotate_z(rotate_x(q(iBx:iBz),-Outflow%phi),-Outflow%theta)
                  !                b_curr(1)=0d0 !We don't want to change B along the axis
                  IF (s <= .8*Outflow%radius) THEN !magnetic radius inside Outflow beam
                     mag_press=half*(Outflow%B*s/(.8*Outflow%radius))**2
                     !                   dq(iBx:iBz)=rotate_x(rotate_z(Outflow%B/(.8*Outflow%radius)*(/0d0,-pos(3),pos(2)/)-b_curr,Outflow%theta),Outflow%phi)
                  ELSE
                     mag_press=half*(Outflow%B*(Outflow%radius-s)/(.2*Outflow%radius))**2
                     !                   dq(iBx:iBz)=rotate_x(rotate_z(Outflow%B*(Outflow%radius-s)/(.2*Outflow%radius*s)*(/0d0,-pos(3),pos(2)/)-b_curr,Outflow%theta),Outflow%phi)
                  END IF

               ELSE
                  mag_press=0
               END IF
               pT=Outflow%density*Outflow%temperature
               therm_press=pT-mag_press
               rho_Outflow=therm_press/Outflow%temperature  
               dq(1)=rho_Outflow-q(1)
               IF (iE .ne. 0) dq(iE)=therm_press-q(iE)
               f=fade(s/Outflow%radius, .1d0)
               vddr=f*velocity/Outflow%thickness
               IF (x > Outflow%buffer) THEN
                  dq(ivx:ivz)=rho_Outflow*(vddr*(x-Outflow%buffer)*rotate_x(rotate_z(rotate_x(rotate_z((/sign(1d0,rpos(1)), 0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)+Outflow%source_vel)-q(ivx:ivz) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivz)=rho_Outflow*Outflow%source_vel-q(ivx:ivz)
               END IF
            END IF
         ELSE
            angle=atan(s/x)
            r=sqrt(s2+x**2)
            IF (angle < Outflow%open_angle .AND. r < Outflow%r_outer) THEN
               dq(1)=Outflow%density-q(1)
               IF (iE .ne. 0) dq(iE)=Outflow%density*Outflow%temperature-q(iE)
               IF (r > Outflow%r_inner) THEN
                  f=fade(angle/Outflow%open_angle, .1d0)
                  vddr=f*velocity/Outflow%thickness
                  dq(ivx:ivz)=Outflow%density*(vddr*(r-Outflow%r_inner)*rotate_x(rotate_z(rotate_x(rotate_z((/sign(cos(angle),rpos(1)), sin(angle)*rpos(2)/s, sin(angle)*rpos(3)/s/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)+Outflow%source_vel)-q(ivx:ivz) !rotate coordinates again so that vector along Outflow-axis is along x-axis
               ELSE
                  dq(ivx:ivz)=Outflow%density*Outflow%source_vel-q(ivx:ivz)
               END IF
            END IF
         END IF
      END SELECT
   END SUBROUTINE OutflowSource

   FUNCTION emf_Launch_2D(Outflow,pos)
      REAL(KIND=qPREC) :: pos(:),tor,s2,s,x,t,precessPhase,oscPhase
      TYPE(OutflowDef) :: Outflow
      REAL(KIND=qPREC) :: emf_Launch_2D
      emf_launch_2d=0
      print*, '2D emfs not implemented for outflows'
      STOP
   END FUNCTION emf_Launch_2D

   FUNCTION emf_Launch_3D(Outflow,pos)
      REAL(KIND=qPREC) :: pos(:),tor,s2,s,x,t,precessPhase,oscPhase
      TYPE(OutflowDef) :: Outflow
      REAL(KIND=qPREC) :: emf_Launch_3D(3)
      pos=pos-Outflow%position !refere to the relative position of source
      precessphase=0
      oscphase=0
      IF (Outflow%precessperiod > 0) PrecessPhase=Outflow%PrecessPhase
      IF (Outflow%oscPeriod > 0) oscPhase=Outflow%oscPhase
      pos=rotate_z(rotate_x(pos,-Outflow%phi),-Outflow%theta) !rotate coordinates again so that vector along Outflow-axis is along x-axis
      pos=rotate_z(rotate_x(pos,-precessPhase),-Outflow%PrecessAmplitude)
      !    velocity=Outflow%velocity+sin(oscPhase)*Outflow%oscAmplitude

      x=Outflow%offset+abs(pos(1)) !distance to 'center' for Outflows
      s2=pos(2)**2+pos(3)**2
      s=sqrt(s2)
      emf_launch_3D=0
      IF (x < Outflow%r_outer) THEN
         IF (s <= .8*Outflow%radius) THEN
            tor=Outflow%B*(.8*Outflow%radius**2-s2)/(2d0*.8*Outflow%radius)
            emf_launch_3D=tor*rotate_x(rotate_z(rotate_x(rotate_z((/1d0,0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),precessPhase)
         ELSEIF (s < Outflow%radius) THEN
            tor=Outflow%B*(Outflow%radius-s)**2/(2d0*(.2)*Outflow%radius)
            emf_launch_3D=tor*rotate_x(rotate_z(rotate_x(rotate_z((/1d0,0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)
         END IF
      END IF
   END FUNCTION emf_Launch_3D


   
  FUNCTION emf_source_3D(Outflow,pos,t,dt)
      REAL(KIND=qPREC) :: pos(:),tor,s2,s,x,t,dt,vddr,PrecessPhase,OscPhase,velocity
      TYPE(OutflowDef) :: Outflow
      REAL(KIND=qPREC), DIMENSION(3) :: emf_source_3D
      pos=pos-Outflow%position !refere to the relative position of source
      precessphase=0
      oscphase=0
      IF (Outflow%precessperiod > 0) PrecessPhase=Outflow%PrecessPhase+2*Pi*(t-Outflow%begin)/Outflow%precessPeriod
      IF (Outflow%oscPeriod > 0) oscPhase=Outflow%oscPhase+2d0*Pi*(t-Outflow%begin)/Outflow%oscPeriod
      pos=rotate_z(rotate_x(pos,-Outflow%phi),-Outflow%theta) !rotate coordinates again so that vector along Outflow-axis is along x-axis
      pos=rotate_z(rotate_x(pos,-precessPhase),-Outflow%PrecessAmplitude)
      velocity=Outflow%velocity+sin(oscPhase)*Outflow%oscAmplitude

      x=Outflow%offset+abs(pos(1)) !distance to 'center' for Outflows
      s2=pos(2)**2+pos(3)**2
      s=sqrt(s2)
      emf_source_3D=0
      IF (x < Outflow%r_outer .AND. x > Outflow%buffer) THEN
         vddr=velocity/Outflow%thickness
         IF (s <= .8*Outflow%radius) THEN
            tor=vddr*Outflow%B*(.8*Outflow%radius**2-s2)/(2d0*.8*Outflow%radius)
            emf_source_3D=tor*rotate_x(rotate_z(rotate_x(rotate_z((/1d0,0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)*dt
         ELSEIF (s < Outflow%radius) THEN
            tor=vddr*Outflow%B*(Outflow%radius-s)**2/(2d0*(.2)*Outflow%radius)
            emf_source_3D=tor*rotate_x(rotate_z(rotate_x(rotate_z((/1d0,0d0,0d0/),Outflow%theta),Outflow%phi),Outflow%PrecessAmplitude),PrecessPhase)*dt
         END IF
      END IF
   END FUNCTION emf_source_3D

  FUNCTION emf_source_2D(Outflow,pos,t,dt)
      REAL(KIND=qPREC) :: pos(:),tor,s2,s,x,t,dt,vddr,PrecessPhase,OscPhase,velocity
      TYPE(OutflowDef) :: Outflow
      REAL(KIND=qPREC) :: emf_source_2D
      pos=pos-Outflow%position !refere to the relative position of source
      emf_source_2d=0
      print*, 'emf source not supported for outflows in 2d'
      STOP
   END FUNCTION emf_source_2D


   !> Initializes Outflow geometric variables using radius and thickness
   !! @param Outflow Outflow object
   SUBROUTINE SetOutflowGeometry(Outflow)
      TYPE(OutflowDef) :: Outflow
      !      Outflow%temperature=Outflow%temperature/EosConstants
      IF (Outflow%open_angle == 0) THEN
         Outflow%offset=0
         Outflow%r_inner=Outflow%radius
         Outflow%r_outer=Outflow%buffer+Outflow%thickness
      ELSEIF (abs(Outflow%open_angle-half*Pi) < .01) THEN
         Outflow%offset=0
         Outflow%buffer=0
         Outflow%r_inner=Outflow%radius
         Outflow%r_outer=Outflow%radius+Outflow%thickness
      ELSE
         Outflow%offset=max(0d0,Outflow%radius/tan(Outflow%open_angle)-Outflow%buffer)
         Outflow%buffer=min(Outflow%buffer,Outflow%offset)
         Outflow%r_inner=sqrt((Outflow%offset+Outflow%buffer)**2+Outflow%radius**2)
         Outflow%r_outer=Outflow%r_inner+Outflow%thickness
      END IF
      IF (Outflow%open_angle==0) THEN
         Outflow%radius_needed=sqrt(Outflow%r_inner**2+Outflow%r_outer**2)
      ELSE
         Outflow%radius_needed=sqrt((Outflow%r_outer*sin(Outflow%open_angle))**2+(Outflow%r_outer*cos(Outflow%open_angle)-Outflow%offset)**2)
      END IF
   END SUBROUTINE SetOutflowGeometry

   SUBROUTINE SetOutflowBounds(Outflow)
      TYPE(OutflowDef) :: Outflow
      Outflow%xBounds=0
      Outflow%xBounds(1:nDim,1)=Outflow%position(1:nDim)-Outflow%radius_needed
      Outflow%xBounds(1:nDim,2)=Outflow%position(1:nDim)+Outflow%radius_needed
   END SUBROUTINE SetOutflowBounds

END MODULE Outflows


