!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    perturbations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file perturbation.f90
!! @brief Main file for module Perturbation

!> @defgroup Perturbation Perturbation Object
!! @brief Module that handles the placement of perturbation
!! @ingroup ModuleObjects

!> Module that handles the placement of perturbation
!! @ingroup Perturbation
MODULE Perturbation
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   USE EOS
   USE ObjectDeclarations
   IMPLICIT NONE
   !> Perturbation data type
   TYPE PerturbationDef
      INTEGER :: nWaves=0
      INTEGER :: iWaves=0
      INTEGER :: Type=0
      INTEGER :: geometry=0
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:) :: WaveVector
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: amplitude
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: phase
      INTEGER :: ObjId
   END TYPE PerturbationDef

   !new declaration
   TYPE pPerturbationDef
      TYPE(PerturbationDef), POINTER :: p
   END type pPerturbationDef
  TYPE(pPerturbationDef) :: pPerturbation
  !

   INTEGER, PARAMETER :: COSINESERIES=0, SINESERIES=1, EXPONENTIAL=2
   INTEGER, PARAMETER :: CARTESIAN=0, CYLINDRICAL=1, SPHERICAL=2
CONTAINS

   SUBROUTINE CreatePerturbation(Perturbation)
      TYPE(PerturbationDef), POINTER :: Perturbation
      ALLOCATE(Perturbation)
   END SUBROUTINE CreatePerturbation

  SUBROUTINE UpdatePerturbation(Perturbation)
    TYPE(PerturbationDef), POINTER :: Perturbation
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdatePerturbation


   SUBROUTINE CreatePerturbationSpectra(PerturbationObj, kmin, kmax, beta, amplitude,scale)
      TYPE(PerturbationDef), POINTER :: PerturbationObj
      INTEGER :: kmin, kmax,i,j,k,k2,ikrange(3),nwaves
      REAL(KIND=qPREC) :: beta, amplitude, phase, realpart, imagpart,dk
      REAL(KIND=qPREC), OPTIONAL :: scale
      IF (PRESENT(scale)) THEN
         dk=2d0*Pi/scale
      ELSE
         dk=2d0*Pi/maxval(GxBounds(:,2)-GxBounds(:,1))
      END IF
      PerturbationObj%TYPE=EXPONENTIAL
      ikrange=0
      ikrange(1:nDim)=kmax
      nwaves=0
      !Get degeneracy of each radial wavenumber
      DO i=-ikrange(1),ikrange(1)
         DO j=-ikrange(2),ikrange(2)
            DO k=-ikrange(3),ikrange(3)
               k2 = i**2+j**2+k**2
               IF (k2 <= kmax**2 .AND. k2 >= kmin**2) THEN
                  nwaves=nwaves+1
               END IF
            END DO
         END DO
      END DO
      CALL InitPerturbationWaves(PerturbationObj, nwaves)
      DO i=-ikrange(1),ikrange(1)
         DO j=-ikrange(2),ikrange(2)
            DO k=-ikrange(3),ikrange(3)
               k2 = i**2+j**2+k**2
               IF (k2 <= kmax**2 .AND. k2 >= kmin**2) THEN
                  realpart=amplitude*cos(getrand()*2d0*Pi)*sqrt(REAL(k2))**beta
                  imagpart=amplitude*sin(getrand()*2d0*Pi)*sqrt(REAL(k2))**beta                  
                  CALL AddPerturbationWave(PerturbationObj, REAL((/i,j,k/))*dk, realpart, imagpart)
               END IF
            END DO
         END DO
      END DO
    END SUBROUTINE CreatePerturbationSpectra

   SUBROUTINE InitPerturbationWaves(PerturbationObj, nWaves)
      TYPE(PerturbationDef) :: PerturbationObj
      INTEGER :: nWaves
      IF (nWaves == 0) RETURN
      ALLOCATE(PerturbationObj%WaveVector(nWaves,nDim), PerturbationObj%amplitude(nWaves), PerturbationObj%Phase(nWaves))
      PerturbationObj%iWaves=0
      PerturbationObj%nWaves=nWaves
   END SUBROUTINE InitPerturbationWaves

   SUBROUTINE AddPerturbationWave(PerturbationObj, wavevector, phase, amplitude)      
      TYPE(PerturbationDef) :: PerturbationObj
      REAL(KIND=qPREC), DIMENSION(:) :: wavevector
      REAL(KIND=qPREC) :: phase, amplitude
      INTEGER :: iwaves
      iwaves=PerturbationObj%iwaves+1
      PerturbationObj%iwaves=iwaves
      PerturbationObj%WaveVector(iwaves,1:nDim)=wavevector(1:nDim)
      PerturbationObj%phase(iwaves)=phase
      PerturbationObj%amplitude(iwaves)=amplitude
   END SUBROUTINE AddPerturbationWave
      
   SUBROUTINE DestroyPerturbation(PerturbationObj)
      TYPE(PerturbationDef), POINTER :: PerturbationObj
      IF (ALLOCATED(PerturbationObj%WaveVector)) DEALLOCATE(PerturbationObj%WaveVector)
      IF (ALLOCATED(PerturbationObj%Amplitude)) DEALLOCATE(PerturbationObj%Amplitude)
      IF (ALLOCATED(PerturbationObj%Phase)) DEALLOCATE(PerturbationObj%Phase)
      DEALLOCATE(PerturbationObj)
      NULLIFY(PerturbationObj)
   END SUBROUTINE DestroyPerturbation

   FUNCTION PerturbationValue(PerturbationObj, pos)
     TYPE(PerturbationDef) :: PerturbationObj
     REAL(KIND=qPREC), DIMENSION(3) :: pos, geopos
     REAL(KIND=qPREC) :: z
     REAL(KIND=qPREC) :: PerturbationValue
     INTEGER :: l,i    
     PerturbationValue=0d0         
     SELECT CASE (PerturbationObj%geometry)
     CASE(CARTESIAN)
        geopos=pos
     CASE(CYLINDRICAL)
        geopos=ConvertCylindrical(pos)
     CASE(Spherical)
        geopos=ConvertSpherical(pos)
     END SELECT

     SELECT CASE (PerturbationObj%Type)
        CASE(COSINESERIES)
           DO i=1,PerturbationObj%nWaves
              PerturbationValue=PerturbationValue+PerturbationObj%amplitude(i)*cos(SUM(PerturbationObj%WaveVector(i,:)*geopos(:))+PerturbationObj%Phase(i))
           END DO
        CASE(SINESERIES)
           DO i=1,PerturbationObj%nWaves
              PerturbationValue=PerturbationValue+PerturbationObj%amplitude(i)*cos(SUM(PerturbationObj%WaveVector(i,:)*geopos(:))+PerturbationObj%Phase(i))
           END DO
        CASE(EXPONENTIAL)
           DO i=1,PerturbationObj%nWaves
              PerturbationValue=PerturbationValue+REAL(cmplx(PerturbationObj%amplitude(i), PerturbationObj%Phase(i))*exp(cmplx(0d0,SUM(PerturbationObj%WaveVector(i,:)*geopos(:)))), 8)
           END DO
        END SELECT
     END FUNCTION PerturbationValue


     FUNCTION getrand()
        REAL(KIND=qPREC) :: getrand
        REAL :: temp
        CALL Random_number(temp)
        getrand=REAL(temp, KIND=qPREC)
     END FUNCTION getrand
END MODULE Perturbation

