!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    profiles.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> Profiles module

MODULE Profiles
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE Splines
   USE CommonFunctions
   USE PointGravitySrc
   USE Fields
   IMPLICIT NONE


   INTEGER, PARAMETER :: RADIAL = 0

   !> Profile data type
   TYPE ProfileDef
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: data => null()
      INTEGER, DIMENSION(:), ALLOCATABLE :: fields
      REAL(KIND=qPREC) :: dr = 0d0 ! length change for each index
      INTEGER :: mode = RADIAL ! profile type
   END TYPE ProfileDef

CONTAINS

   SUBROUTINE createProfile(Profile, nPoints, fields, mode)
      TYPE(ProfileDef), POINTER :: Profile
      INTEGER :: nPoints
      INTEGER, DIMENSION(:) :: fields
      INTEGER, OPTIONAL :: mode
      ALLOCATE(Profile)
      IF (present(mode)) Profile%mode=mode
      ALLOCATE(Profile%fields(size(fields)))
      Profile%fields=fields
      ALLOCATE(Profile%data(nPoints, size(fields)+1)) !one slot for position
   END SUBROUTINE   CreateProfile

   SUBROUTINE UpdateProfile(Profile)
    TYPE(ProfileDef), POINTER :: Profile
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdateProfile

   ! cubic spline for arbitrary value in array
   FUNCTION getProfileValue(r,field,Profile)
      TYPE(ProfileDef), POINTER :: Profile
      REAL(KIND=qPREC) :: getProfileValue, r
      TYPE(SplineDef), POINTER :: spl
      INTEGER :: index, i, field, n
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: data
      LOGICAL :: lWarned = .false.
      data => Profile%data
      n=sum(minloc(profile%fields, profile%fields == field))+1
      IF (n == 0) THEN
         IF (MPI_ID == 0) write(*,*) 'error unable to find field ', field, 'in profile%fields ', profile%fields
         STOP
      END IF
      IF ((r < data(1,1) .OR. r > data(size(data, 1), 1)) .AND. .NOT. lWarned) THEN
         PRINT*, '*** WARNING - profile is not defined for r = ', r, data(1,1), data(size(data,1),1)
         lWarned=.true.
      END IF
      DO index=2, size(data,1)-2
         IF (data(index,1) .gt. r) EXIT
      END DO
      CALL CreateSpline(spl, 4)

      spl%x(1) = data(index - 1,1)
      spl%x(2) = data(index ,1)
      spl%x(3) = data(index + 1,1)
      spl%x(4) = data(index + 2,1)
      spl%y(1) = data(index - 1, n)
      spl%y(2) = data(index , n)
      spl%y(3) = data(index + 1, n)
      spl%y(4) = data(index + 2, n)
      CALL SolveSpline(spl)
      getProfileValue = GetSplineValue(spl, r)
      CALL DestroySpline(spl)
   END FUNCTION getProfileValue

   ! calculated ideal pressure for hydrostatic equilibrium
   SUBROUTINE Profile_PointGravityHSE(Profile, PointGravityObj, p_inf)
      TYPE(ProfileDef), POINTER :: Profile
      TYPE(PointGravityDef), POINTER :: PointGravityObj
      REAL(KIND=qPREC) :: p_inf
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: data
      INTEGER ::  i, npoints,  i_rho, i_press

      data => Profile%data

      npoints=size(data, 1)
      i_rho=sum(minloc(profile%fields, profile%fields == Mass_Field))+1
      i_Press=sum(minloc(profile%fields, profile%fields == P_Field))+1

      data(npoints, i_Press) = p_inf

      ! start from last entry and end up to first entry
      DO i=npoints-1, 2, -1
         !   dP/dr = f
         !   [P(i+1)-P(i)] / [r(i+1)-r(i)] = g(r(i+1/2))*rho(i+1/2)
         !   P(i) = P(i+1) - (r(i+1)-r(i))*g(r(i+1/2))*rho(i+1/2) where f(i+1/2) = half*(f(i)+f(i+1))
         data(i, i_Press)=data(i+1, i_Press)-sum(GravityForce( PointGravityObj%mass, (/half*(data(i,1)+data(i+1,1)),0d0,0d0/), PointGravityObj%soft_length, PointGravityObj%soft_function))*half*(data(i,i_rho)+data(i+1,i_rho)) * (data(i+1,1)-data(i,1))
      END DO

      

      
   END SUBROUTINE Profile_PointGravityHSE


END MODULE Profiles
