!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    refinements.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @Filegotoli refinements.f90
!! @brief Main file for module Refinements

!> @defgroup Refinements Refinements Object
!! @brief Module that handles the placement of refinements
!! @ingroup ModuleObjects

!> Module that handles the placement of refinements
!! @ingroup Refinements
MODULE Refinements
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE Fields
   USE ObjectDeclarations
   IMPLICIT NONE
   !> Refinement data type

   LOGICAL, PARAMETER :: GREATERTHAN=.false., LESSTHAN=.true.
   TYPE RefinementDef
      INTEGER :: field = 0 !Field id.  See fields module for a complete list
      INTEGER :: ScaleField = 0 !Field id.  See fields module for a complete list
      REAL(KIND=qPREC) :: tolerance=1d0 !Relative tolerance (multipled by qtolerance for level
      REAL(KIND=qPREC), DIMENSION(0:MaxDepth) :: Threshold = UNDEFINED
      INTEGER :: scale = LOGSCALE !Scale to use for gradients
      INTEGER :: BufferCells = 2 !Number of cells to nest regions
      LOGICAL :: limit = GREATERTHAN
      INTEGER :: MaxLevel = MAXDEPTH
      TYPE(ShapeDef), POINTER :: Shape=>NULL()
      INTEGER :: RefinementId
      INTEGER :: ObjId
   END TYPE RefinementDef

   !new declaration
   TYPE pRefinementDef
      TYPE(RefinementDef), POINTER :: ptr
   END TYPE pRefinementDef
   TYPE(pRefinementDef) :: pRefinement

   INTEGER :: nRefinements=0
   INTEGER, PARAMETER :: DEREFINE_OUTSIDE=-1d0, DEREFINE_INSIDE=-2d0
   !


CONTAINS

   SUBROUTINE AddRefinementCriterion(field, tolerance, scaleField, Scale)
      INTEGER :: Field
      REAL(KIND=qPREC), OPTIONAL :: Tolerance
      INTEGER, OPTIONAL :: ScaleField
      INTEGER, OPTIONAL :: scale
      TYPE(RefinementDef), POINTER :: Refinement
      CALL CreateRefinement(Refinement)
      Refinement%field=field
      IF (Present(tolerance)) Refinement%tolerance=tolerance
      IF (Present(ScaleField)) Refinement%scalefield=ScaleField
      IF (Present(Scale)) Refinement%scale=scale
   END SUBROUTINE AddRefinementCriterion

   SUBROUTINE AddRefinementThreshold(field, limit, threshold, scaleField)
      INTEGER :: Field
      REAL(KIND=qPREC), DIMENSION(:) :: threshold
      INTEGER, OPTIONAL :: ScaleField
      TYPE(RefinementDef), POINTER :: Refinement
      LOGICAL :: limit
      CALL CreateRefinement(Refinement)
      Refinement%field=field     
      Refinement%threshold(0:size(threshold)-1)=threshold
      Refinement%limit=limit
      IF (Present(ScaleField)) Refinement%scalefield=ScaleField
   END SUBROUTINE AddRefinementThreshold

   SUBROUTINE CreateRefinement(Refinement)
      TYPE(RefinementDef), POINTER :: Refinement
      ALLOCATE(Refinement)
      CALL AddRefinementToList(Refinement)
      nRefinements=nRefinements+1
      Refinement%RefinementID=nRefinements
   END SUBROUTINE CreateRefinement

   SUBROUTINE UpdateRefinement(Refinement)
    TYPE(RefinementDef), POINTER :: Refinement
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdateRefinement

   SUBROUTINE AddRefinementToList(Refinement)
      TYPE(RefinementDef), POINTER :: Refinement
      TYPE(ObjectDef), POINTER :: Object
      Refinement%ObjId = ObjectListAdd(Object,REFINEMENTOBJ)
      pRefinement%ptr => Refinement
      len = size(transfer(pRefinement, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pRefinement, Object%storage)
   END SUBROUTINE AddRefinementToList


   SUBROUTINE DestroyRefinement(Refinement)
      TYPE(RefinementDef),POINTER :: Refinement
      CALL ObjectListRemove(Refinement%ObjId)
      IF (ASSOCIATED(Refinement%Shape)) DEALLOCATE(Refinement%Shape)
      DEALLOCATE(Refinement)
      NULLIFY(Refinement)
   END SUBROUTINE DestroyRefinement


   !SUBROUTINE ClearAllRefinements()
   !   TYPE(RefinementDef), POINTER :: temp
   !   DO WHILE (ASSOCIATED(FirstRefinement))
   !      temp=>FirstRefinement
   !      FirstRefinement=>FirstRefinement%next
   !      CALL DestroyRefinement(temp)
   !   END DO
   !END SUBROUTINE ClearAllRefinements

   SUBROUTINE ClearAllRefinements()
      TYPE(ObjectDef), POINTER :: Object
      Object => ListHead
      DO WHILE (ASSOCIATED(Object))
         IF(Object%type == REFINEMENTOBJ) THEN
            pRefinement = transfer(Object%storage,pRefinement)
            CALL DestroyRefinement(pRefinement%ptr)
         ENDIF
         Object => Object%next
      END DO
      nRefinements=0
   END SUBROUTINE ClearAllRefinements

 
    SUBROUTINE RefinementSetErrFlag(Info, Refinement)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mS, mT
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC) :: dx, dy, dz, MinScale, w
      REAL(KIND=qPREC), DIMENSION(3) :: offset, pos, rpos
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSIOn(3,2) :: tempbounds
      Type(RefinementDef), POINTER :: Refinement
      INTEGER :: nOverlaps, n, i, j, k
      LOGICAL :: lrefine
      LOGICAL, DIMENSION(:,:,:), POINTER :: flags, inshape
      REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: buffer


      IF (Info%level >= Refinement%MaxLevel) RETURN
      dx=levels(Info%level)%dx
      dy=merge(dx,0d0,nDim>=2)
      dz=merge(dx,0d0,nDim>=3)

         minScale=merge(1d0,0d0,Refinement%ScaleField /= 0) 
         IF (ASSOCIATED(Refinement%Shape)) THEN
            Refinement%Shape%size_param=Refinement%Shape%size_param+(Refinement%BufferCells)*levels(Info%level)%dx
            CALL SetShapeBounds(Refinement%Shape)
         END IF
         IF (ASSOCIATED(Refinement%Shape) .AND. Refinement%Tolerance >= 0) THEN
            CALL CalcPhysicalOverlaps(Info, GetShapeBounds(Refinement%Shape, levels(Info%level)%tnow), mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic, 0)
         ELSE
            nOverlaps=1
            ALLOCATE(mSs(1,3,2))
            ALLOCATE(offsets(1,3))
            offsets=0d0
            mSs(1,:,1)=1
            mSs(1,:,2)=Info%mX(:)
         END IF
         IF (nOverlaps > 0) THEN
            DO n=1,nOverlaps
               mS=mSs(n,:,:)
               offset=offsets(n,:)
!               write(*,*) n, offset

               mT=mS
               mS(1:nDim,1)=mS(1:nDim,1)-1
               mS(1:nDim,2)=mS(1:nDim,2)+1

               ALLOCATE(buffer(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2)))
               ALLOCATE(inshape(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2)))
               ALLOCATE(flags(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2)))
               flags=.false.
               DO i=mS(1,1), mS(1,2)
                  pos(1)=Info%xBounds(1,1)+(REAL(i)-half)*dx-offset(1)
                  DO j=mS(2,1), mS(2,2)
                     pos(2)=Info%xBounds(2,1)+(REAL(j)-half)*dy-offset(2)
                     DO k=mS(3,1), mS(3,2)
                        pos(3)=Info%xBounds(3,1)+(REAL(k)-half)*dz-offset(3)
                        IF (ASSOCIATED(Refinement%Shape)) THEN
                           inshape(i,j,k)=IsInShape(Refinement%Shape, pos, rpos, levels(Info%level)%tnow)
!                           IF (ALL((/i,j,k/)==(/16,16,16/))) THEN
!                              write(*,*) pos, rpos, levels(Info%level)%tnow
!                              STOP
!                           END IF

                        END IF
!                        write(*,*) i,j,k
                        IF (Refinement%Field /= 0) THEN
                           IF (Refinement%ScaleField /= 0) THEN
                              buffer(i,j,k)=GetField(Info, i, j, k, Refinement%Field, pos)/GetField(Info, i, j, k, Refinement%ScaleField, pos)
                           ELSE
                              buffer(i,j,k)=GetField(Info, i, j, k, Refinement%Field, pos)
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
               IF (ANY(Refinement%Threshold(0:MaxLevel) /= UNDEFINED)) THEN !ignore gradients and just look at thresholds
                  IF (Refinement%Threshold(Info%level) /= UNDEFINED) THEN !have a defined threshold for this level
                     IF (Refinement%limit .EQV. GREATERTHAN) THEN
                        IF (ASSOCIATED(Refinement%Shape)) THEN
                           WHERE (inshape(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) .AND. &
                                buffer(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) >= Refinement%Threshold(Info%level))
                              Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))+2**Refinement%RefinementID
                           END WHERE
                        ELSE
                           WHERE (buffer(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) >= Refinement%threshold(Info%level))
                              Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))+2**Refinement%RefinementID
                           END WHERE
                        END IF
                     ELSE !Refinement%Limit = Upper
                        IF (ASSOCIATED(Refinement%Shape)) THEN
                           WHERE (inshape(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) .AND. &
                                buffer(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) <= Refinement%threshold(Info%level))
                              Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))+2**Refinement%RefinementID
                           END WHERE
                        ELSE
                           WHERE (buffer(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) <= Refinement%threshold(Info%level))
                              Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))+2**Refinement%RefinementID
                           END WHERE
                        END IF
                     END IF
                  END IF
               ELSE
                  IF (Refinement%Tolerance == DEREFINE_OUTSIDE) THEN
                     WHERE (.NOT. inshape(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)))
                        Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=0
                     END WHERE
                     !                  write(*,*) inshape
                  ELSEIF (Refinement%Tolerance == DEREFINE_INSIDE) THEN
                     WHERE (inshape(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)))
                        Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=0
                     END WHERE
                  ELSE
                     IF (Refinement%Field /= 0) THEN
                        CALL SetFlags(mT, flags, buffer, Refinement%Scale, Refinement%Tolerance/levels(Info%level)%qtolerance, MinScale)
                     ELSE
                        flags=.true.
                     END IF
                     IF (ASSOCIATED(Refinement%Shape)) THEN
                        WHERE((flags(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)) .AND. &
                             inshape(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))))
                           Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))+2**Refinement%RefinementID
                        END WHERE
                          
!                     write(*,*) inshape
                     ELSE
                        WHERE(flags(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2)))
                           Info%ErrFlag(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=1
                        END WHERE
                     END IF
                  END IF
               END IF
               DEALLOCATE(buffer, inshape, flags)
            END DO
            DEALLOCATE(mSs, offsets)
         END IF
         IF (ASSOCIATED(Refinement%Shape)) THEN
            Refinement%Shape%size_param=Refinement%Shape%size_param-(Refinement%BufferCells)*levels(Info%level)%dx
            CALL SetShapeBounds(Refinement%Shape)
         END IF
   END SUBROUTINE RefinementSetErrFlag

   SUBROUTINE SetFlags(mT,flags, buffer, scale, tolerance, MinScale)
      INTEGER, DIMENSION(3,2) :: mA, mB, mT
      LOGICAL, DIMENSION(:,:,:), POINTER :: flags
      REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: buffer
      INTEGER :: scale, i
      REAL(KIND=qPREC) :: tolerance, MinScale
      mA=mT
      mB=mT
      DO i=1,nDim
         mA(i,:)=mT(i,:)-1
         mB(i,:)=mT(i,:)+1
         IF (scale==LOGSCALE) THEN
            WHERE(ABS(buffer(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2)) - &
                 buffer(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2))) * tolerance  > &                
                 
                 half*MAX((ABS(buffer(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2))) + &
                 ABS(buffer(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2)))), MinScale))

               Flags(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=.true.
!               Flags(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2))=.true.
            END WHERE
         ELSE
            WHERE(ABS(buffer(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2)) - &
                 buffer(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2))) * tolerance > &                
                 1d0)
               Flags(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2))=.true.
!               Flags(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2))=.true.
            END WHERE
         END IF
         
         mA(i,:)=mT(i,:)
         mB(i,:)=mT(i,:)

      END DO

   END SUBROUTINE SetFlags


END MODULE Refinements

