!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    shapes.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file shapes.f90
!! @brief Main file for module Shapes

!> @defgroup Shapes Shapes Object
!! @brief Module that handles the placement of shapes
!! @ingroup ModuleObjects

!> Module that handles the placement of shapes
!! @ingroup Shapes
MODULE Shapes
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE ObjectDeclarations
  IMPLICIT NONE
  INTEGER, PARAMETER :: SPHERE=1, ELLIPSOID=2, CYLINDER=3, ELLIPTICAL_PRISM=4, RECTANGULAR_PRISM=5
  !> Shape data type
  TYPE ShapeDef
     INTEGER :: Type = SPHERE
     REAL(KIND=qPREC) :: position(3)=(/0d0,0d0,0d0/)
     !      REAL(KIND=qPREC) :: psi=0   !Initial rotation about z axis
     !      REAL(KIND=qPREC) :: theta=0 !Angle rotated around y (from z towards x)
     !      REAL(KIND=qPREC) :: phi=0   !Angle rotated around z (from x towards y)
     INTEGER :: xy(3) = (/1,2,3/) !integer indices corresponding to frame reference x and y directions
     REAL(KIND=qPREC) :: size_param(3) = (/1d0,1d0,1d0/)
     REAL(KIND=qPREC) :: RotationMatrix(3,3)=RESHAPE((/1d0,0d0,0d0,0d0,1d0,0d0,0d0,0d0,1d0/),(/3,3/))    !Rotates lab frame into object frame
     REAL(KIND=qPREC) :: xBounds(3,2)=RESHAPE((/-1d0,-1d0,-1d0,1d0,1d0,1d0/),(/3,2/))
     REAL(KIND=qPREC) :: velocity(3)=0
     REAL(KIND=qPREC) :: t0 = 0
     LOGICAL :: offcenter = .false.
     REAL(KIND=qPREC) :: bounds_param(3,2) = RESHAPE((/-1d0,-1d0,-1d0,1d0,1d0,1d0/),(/3,2/))

  END TYPE ShapeDef

CONTAINS

  SUBROUTINE CreateShape(Shape, type, dimensions)
    TYPE(ShapeDef), POINTER :: Shape
    INTEGER, OPTIONAL :: type
    REAL(KIND=qPREC), OPTIONAL :: dimensions(3)
    ALLOCATE(Shape)
    IF(Present(type) .AND. Present(dimensions)) THEN
      CALL SetShapeType(shape, type, dimensions)
    ENDIF
    CALL SetShapeBounds(Shape)
  END SUBROUTINE CreateShape

  SUBROUTINE UpdateShape(Shape)
    TYPE(ShapeDef), POINTER :: Shape
    CALL SetShapeBounds(Shape)
  END SUBROUTINE UpdateShape

  SUBROUTINE SetShapeOrientation(Shape, psi, theta, phi)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC) :: c1,c2,c3,s1,s2,s3
    REAL(KIND=qPREC) :: theta,phi,psi
    REAL(KIND=qPREC) :: theta_,phi_,psi_


    IF (nDim == 2) THEN
       !theta must be 0 Pi/2 or Pi
       !psi should be 0 Pi/2 Pi, 3Pi/2, 2Pi, etc...
       !phi should be 0 Pi/2 Pi, etc...
!       psi_=Pi/2d0*nint(2d0*psi/Pi)
!       theta_=Pi/2d0*nint(2d0*theta/Pi)
!       phi_=Pi/2d0*nint(2d0*phi/Pi)
!       write(*,*) 'adjusted psi, theta, and phi to', psi, theta, phi

       psi_=psi
       theta_=theta
       phi_=phi

   ELSE
       psi_=psi
       theta_=theta
       phi_=phi
    END IF

    c1=cos(psi_)
    c2=cos(theta_)
    c3=cos(phi_)
    s1=sin(psi_)
    s2=sin(theta_)
    s3=sin(phi_)      


    IF(abs(real(nint(c1),8)-c1)<1d-8) c1=real(nint(c1),8)
    IF(abs(real(nint(c2),8)-c2)<1d-8) c2=real(nint(c2),8)
    IF(abs(real(nint(c3),8)-c3)<1d-8) c3=real(nint(c3),8)
    IF(abs(real(nint(s1),8)-s1)<1d-8) s1=real(nint(s1),8)
    IF(abs(real(nint(s2),8)-s2)<1d-8) s2=real(nint(s2),8)
    IF(abs(real(nint(s3),8)-s3)<1d-8) s3=real(nint(s3),8)
    !      Shape%RotationMatrix=RESHAPE((/ &
    !           c1*c2*c3-s1*s3, -c2*s3*c1-c3*s1, c1*s2,&
    !           c1*s3+c3*c2*s1, c1*c3-c2*s1*s3, s2*s1, &
    !           -c3*s2, s3*s2, c2/),(/3,3/),(/0d0/),(/2,1/))


    Shape%RotationMatrix=RESHAPE((/ &
         c1*c2*c3-s1*s3, -c3*c2*s1-c1*s3, +c3*s2,&
         c1*c2*s3+c3*s1, c1*c3-c2*s1*s3, s2*s3, &
         -c1*s2, s1*s2, c2/),(/3,3/),(/0d0/),(/2,1/))


!    IF (nDim == 2) THEN
       shape%xy(1)=maxval(maxloc(abs(RotateVectorToShape(shape,(/1d0,0d0,0d0/)))))
       shape%xy(2)=maxval(maxloc(abs(RotateVectorToShape(shape,(/0d0,1d0,0d0/)))))
       shape%xy(3)=maxval(maxloc(abs(RotateVectorToShape(shape,(/0d0,0d0,1d0/)))))
!       IF (MPI_ID == 0) write(*,*) 'shape%xy=', shape%xy
!    END IF

       CALL SetShapeBounds(Shape)
  END SUBROUTINE SetShapeOrientation


  SUBROUTINE SetShapeType(Shape, type, dimensions)
    TYPE(ShapeDef), POINTER :: Shape
    INTEGER :: type
    REAL(KIND=qPREC) :: dimensions(3)
    Shape%type=type
    SELECT CASE (type)
    CASE (SPHERE)         
       Shape%size_param=dimensions(1)
    CASE (CYLINDER)
       Shape%size_param=(/dimensions(1),dimensions(1),dimensions(3)/)
    CASE (ELLIPSOID, ELLIPTICAL_PRISM, RECTANGULAR_PRISM)
       SHape%size_param=dimensions
    CASE DEFAULT
       PRINT*, 'unrecognized shape type in shapes.f90'
       STOP
    END SELECT
  END SUBROUTINE SetShapeType

  FUNCTION TransformCoordsFromShape(Shape, pos, tnow)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: TransformCoordsFromShape, pos
    REAL(KIND=qPREC), OPTIONAL :: tnow
    INTEGER :: l
    IF (PRESENT(tnow)) THEN
       DO l=1,3
          TransformCoordsFromShape(l)=Shape%position(l)+Shape%velocity(l)*(tnow-Shape%t0)+SUM(Shape%RotationMatrix(l,:)*(pos(:))) !Inverse rotation
       END DO

    ELSE
       DO l=1,3
          TransformCoordsFromShape(l)=Shape%position(l)+SUM(Shape%RotationMatrix(l,:)*(pos(:))) !Inverse rotation
       END DO
    END IF
  END FUNCTION TransformCoordsFromShape

  FUNCTION TransformCoordsToShape(Shape, pos, tnow)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: TransformCoordsToShape, pos
    REAL(KIND=qPREC), OPTIONAL :: tnow
    INTEGER :: l
    IF (Present(tnow)) THEN
       DO l=1,3
          TransformCoordsToShape(l)=SUM(Shape%RotationMatrix(:,l)*(pos(:)-(Shape%position(:)+Shape%velocity*(tnow-Shape%t0)))) !Inverse rotation
       END DO
    ELSE
       DO l=1,3
          TransformCoordsToShape(l)=SUM(Shape%RotationMatrix(:,l)*(pos(:)-Shape%position(:))) !Inverse rotation
       END DO

    END IF
  END FUNCTION TransformCoordsToShape

  FUNCTION RotateVectorFromShape(Shape, pos)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: RotateVectorFromShape, pos
    INTEGER :: l
    DO l=1,3
       RotateVectorFromShape(l)=SUM(Shape%RotationMatrix(l,:)*pos(:)) !Inverse rotation
    END DO
  END FUNCTION RotateVectorFromShape

  FUNCTION RotateVectorToShape(Shape, pos)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: RotateVectorToShape, pos
    INTEGER :: l
    DO l=1,3
       RotateVectorToShape(l)=SUM(Shape%RotationMatrix(:,l)*pos(:)) !Inverse rotation
    END DO
  END FUNCTION RotateVectorToShape


  SUBROUTINE SetShapeBounds(Shape)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: pos  !Position of each corner in object frame
    REAL(KIND=qPREC), DIMENSION(3) :: rpos !Rotated position in simulation frame
    INTEGER :: i,j,k
    Shape%xBounds=SPREAD(Shape%position,2,2)
    SELECT CASE (Shape%Type)
    CASE(SPHERE) !Orientation does not matter
       Shape%xBounds(:,1)=Shape%position-Shape%size_param(1)
       Shape%xBounds(:,2)=Shape%position+Shape%size_param(1)
    CASE(ELLIPSOID, CYLINDER, ELLIPTICAL_PRISM, RECTANGULAR_PRISM) !Bound shape by rectangle and find max extents of each corner.
       IF(.NOT. Shape%offcenter) THEN
          DO i=1,2
             pos(1)=(-1d0)**i*Shape%size_param(1)
             DO j=1,2
                pos(2)=(-1d0)**j*Shape%size_param(2)
                DO k=1,2
                   pos(3)=(-1d0)**k*Shape%size_param(3)                 
                   rpos=TransformCoordsFromShape(Shape, pos)
                   Shape%xBounds(:,1)=min(Shape%xBounds(:,1),rpos)
                   Shape%xBounds(:,2)=max(Shape%xBounds(:,2),rpos)
                END DO
             END DO
          END DO
       ELSE
          DO i=1,2
             pos(1) = Shape%bounds_param(1,i)
             DO j=1,2
                pos(2) = Shape%bounds_param(2,j)
                DO k=1,2
                   pos(3) = Shape%bounds_param(3,k)
                   rpos = TransformCoordsFromShape(Shape,pos)
                   Shape%xBounds(:,1)=min(Shape%xBounds(:,1),rpos)
                   Shape%xBounds(:,2)=max(Shape%xBounds(:,2),rpos)
                END DO
             END DO
          END DO   
       END IF         
    END SELECT
  END SUBROUTINE SetShapeBounds

  FUNCTION GetShapeBounds(Shape, tnow)
    TYPE(ShapeDef) :: Shape
    REAL(KIND=qPREC), DIMENSION(3,2) :: GetShapeBounds
    REAL(KIND=qPREC), OPTIONAL :: tnow
    IF (Present(tnow)) THEN
       GetShapeBounds=Shape%xBounds+SPREAD(Shape%velocity*(tnow-Shape%t0),2,2)
    ELSE
       GetShapeBounds=Shape%xBounds
    END IF
  END FUNCTION GetShapeBounds

  FUNCTION IsInShape(Shape, pos, coords, tnow)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), DIMENSION(3) :: pos
    REAL(KIND=qPREC), DIMENSION(3) :: coords
    REAL(KIND=qPREC), DIMENSION(3) :: rpos
    REAL(KIND=qPREC), OPTIONAL :: tnow
    LOGICAL :: IsInShape
    IsInShape=.false.
    IF (PRESENT(tnow)) THEN
       coords=TransformCoordsToShape(Shape, pos, tnow)
    ELSE
       coords=TransformCoordsToShape(Shape, pos)
    END IF
    SELECT CASE (SHAPE%TYPE)
    CASE(SPHERE)        
       IsInShape=(sum(coords(:)**2) <= Shape%size_param(1)**2)
    CASE (CYLINDER)
       IsInShape=(sum(coords(1:2)**2) <= Shape%size_param(1)**2) .AND. (abs(coords(3)) <= Shape%size_param(3))
    CASE (RECTANGULAR_PRISM)
       IsInShape=ALL(abs(coords(:)) <= Shape%size_param(:))
    CASE (ELLIPSOID)
       IsInShape=(SUM(coords(:)**2/Shape%size_param(:)**2) <= 1d0)
    CASE (ELLIPTICAL_PRISM)
       IsInShape=(SUM(coords(1:2)**2/Shape%size_param(1:2)**2) <= 1d0) .AND. (abs(coords(3)) <= Shape%size_param(3))
    END SELECT

  END FUNCTION IsInShape

  SUBROUTINE PrintShape(shape, tnow)
    TYPE(ShapeDef), POINTER :: Shape
    REAL(KIND=qPREC), OPTIONAL :: tnow
    REAL(KIND=qPREC) :: pos(3)
    IF (PRESENT(tnow)) THEN
       pos=Shape%position+Shape%velocity*(tnow-shape%t0)
    ELSE
       pos=Shape%position
    END IF
    write(*,*) '================= Shape ===================='
    SELECT CASE (shape%Type)
    CASE(SPHERE)
       write(*,'(A,E24.15,A,3E24.15)') 'Sphere of radius ', shape%size_param(1), 'centered at ', pos(1:nDim)
    CASE(CYLINDER)
       write(*,'(A,E24.15,A,E24.15,A,3E24.15)') 'Cylinder of radius', shape%size_param(1), ' and height ', 2d0*shape%size_param(3), ' centered at ', pos(1:nDim), ' with axial vector ', RotateVectorFromShape(shape,(/0d0,0d0,01d0/))
    CASE(RECTANGULAR_PRISM)
       write(*,'(A,E24.15,A,E24.15,A,E24.15,A,3E24.15)') 'Rectangular prism of width', 2d0*shape%size_param(1), ' height ', 2d0*shape%size_param(3), ' and depth ', 2d0*shape%size_param(2), ' centered at ', pos(1:nDim)
       write(*,'(A)') 'Vectors normal to the x, y, and z faces are: '
       write(*,'(3E24.15)') RotateVectorFromShape(shape,(/1d0,0d0,0d0/)), RotateVectorFromShape(shape,(/0d0,1d0,0d0/)), RotateVectorFromShape(shape,(/0d0,0d0,1d0/))
    CASE(ELLIPSOID)
       write(*,'(A,E24.15,A,E24.15,A,E24.15,A,3E24.15)') 'Ellipsoid of width', 2d0*shape%size_param(1), ' height ', 2d0*shape%size_param(3), ' and depth ', 2d0*shape%size_param(2), ' centered at ', pos(1:nDim)
       write(*,'(A)') 'Vectors corresponding to the x, y, and z axis are: '
       write(*,'(3E24.15)') RotateVectorFromShape(shape,(/1d0,0d0,0d0/)), RotateVectorFromShape(shape,(/0d0,1d0,0d0/)), RotateVectorFromShape(shape,(/0d0,0d0,1d0/))
    CASE(ELLIPTICAL_PRISM)
       write(*,'(A,2E24.15,A,E24.15,A,3E24.15)') 'Elliptical prism with radii',shape%size_param(1:2), ' and height ', 2d0*shape%size_param(3), ' centered at ', pos(1:nDim)
       write(*,'(A)') 'Vectors corresponding to the x, y, and z axis are: '
       write(*,'(3E24.15)') RotateVectorFromShape(shape,(/1d0,0d0,0d0/)), RotateVectorFromShape(shape,(/0d0,1d0,0d0/)), RotateVectorFromShape(shape,(/0d0,0d0,1d0/))

    END SELECT
    write(*,*) '============================================'
  END SUBROUTINE PrintShape

  SUBROUTINE GetRandomPointsInShape(shape, min_separation, boundary_separation, positions, tnow)
    TYPE(ShapeDef), POINTER :: shape
    INTEGER :: npoints, recheck_count, j,k,l
    REAL(KIND=qPREC) :: min_separation,sep,minsep,boundary_separation
    REAL :: a
    REAL(KIND=qPREC), DIMENSION(:,:) :: positions
    REAL(KIND=qPREC), OPTIONAL :: tnow !current time (needed if shape is moving)
    INTEGER, PARAMETER :: MaxIter=1000, MaxRetries=10
    LOGICAL :: lSuccess, lReCheck
    INTEGER :: ii,jj,kk,ioffset(3,2), iErr
    REAL(KIND=qPREC), DIMENSION(3) :: poffset, minpos,temp,temp2, periodic_pos
    ioffset=0
    WHERE (lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=1 
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)

!    write(*,*) min_separation, boundary_separation, shape%size_param
    npoints=size(positions,1)
    positions=0d0 !spread(shape%position, 1, npoints)
    DO k=1,npoints
       DO j=1,MaxIter
          lSuccess=.true.
          CALL Random_number(a)
          positions(k,shape%xy(1))=(a*2d0-1d0)*(shape%size_param(shape%xy(1))-boundary_separation)
          CALL Random_number(a)
          positions(k,shape%xy(2))=(a*2d0-1d0)*(shape%size_param(shape%xy(2))-boundary_separation)
          IF (nDim == 3) THEN
             CALL Random_number(a)
             positions(k,shape%xy(3))=(a*2d0-1d0)*(shape%size_param(shape%xy(3))-boundary_separation)
          END IF

          IF (MPI_ID == 0) writE(*,*) 'placing particle ', k,  'of ', npoints ![Bpositions(k,:)
          recheck_count=0 !how many times have we tried shifting the point away from the boundary and other points
          lReCheck=.true. !Do we need to recheck the distance from other points after moving this one?
          DO WHILE (lReCheck .AND. recheck_count < MaxReTries)
             lSuccess=.true. !Assume success in finding a point with no conflicts
             lReCheck=.false. !Assume that we will not shift the point
             DO l=1,k-1       ! Loop over all current existing points
                minsep=1d30
                minpos=positions(l,:)
                DO ii=ioffset(1,1),ioffset(1,2)
                   DO jj=ioffset(2,1),ioffset(2,2)
                      DO kk=ioffset(3,1),ioffset(3,2)
                         pOffSet=(/ii,jj,kk/)*(GxBounds(:,2)-GxBounds(:,1))
                         periodic_pos=positions(l,:)+RotateVectorToShape(shape,pOffSet)
                         sep=sqrt(sum((positions(k,:)-periodic_pos)**2)) !distance from point to existing point
                         IF (sep < minsep) THEN
!                            IF (MPI_ID == 0) write(*,'(A1,2I,7E24.15)') 'a',k,l,sep,positions(k,:),periodic_pos
                            minpos=periodic_pos
                            minsep=sep
                         END IF
                      END DO
                   END DO
                END DO
                temp=positions(k,:)
!                IF (MPI_ID == 0) write(*,*) 'temp=', temp
                IF (minsep < min_separation) THEN
                   !try shifting it away from that particle so it is 1.1 min_separations away
                   ! make sure it is still inside the shape bounds
!                   IF (MPI_ID == 0) write(*,*) 'shifting from particle', l, positions(l,:), minpos, minsep
                   positions(k,shape%xy(1:nDim))=positions(k,shape%xy(1:nDim))+min_separation/minsep*1.1d0*(positions(k,shape%xy(1:nDim))-minpos(shape%xy(1:nDim)))
                END IF
                temp2=positions(k,:)
!                IF (MPI_ID == 0) write(*,*) 'temp2=', temp2
                IF (ANY(temp /= positions(k,:)) .OR. l == 1) THEN
                   SELECT CASE(Shape%Type)
                   CASE(SPHERE)                              
                      positions(k,shape%xy(1:nDim))=positions(k,shape%xy(1:nDim))*min(1d0,(shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,shape%xy(1:nDim))**2)))
                   CASE (CYLINDER)
                      IF (nDim == 2) THEN
                         IF (shape%xy(3) == 3) THEN !cylinder projects to a circle in 2D
                            positions(k,shape%xy(1:2))=positions(k,shape%xy(1:2))*min(1d0,(shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,shape%xy(1:2))**2)))
                         ELSE
                            positions(k,shape%xy(1:2))=positions(k,shape%xy(1:2))*min(1d0, (shape%size_param(shape%xy(1:2))-boundary_separation)/abs(positions(k,shape%xy(1:2))))
                         END IF
                      ELSE
                         positions(k,shape%xy(1:2))=positions(k,shape%xy(1:2))*min(1d0, (shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,shape%xy(1:2))**2)))
                         positions(k,shape%xy(3))=positions(k,shape%xy(3))*min(1d0, (shape%size_param(3)-boundary_separation)/abs(positions(k,shape%xy(3))))
                      END IF
                   CASE (RECTANGULAR_PRISM)
                      positions(k,shape%xy(1:nDim))=positions(k,shape%xy(1:nDim))*min(1d0,(shape%size_param(shape%xy(1:nDim))-boundary_separation)/abs(positions(k,shape%xy(1:nDim))))
                   CASE (ELLIPSOID, ELLIPTICAL_PRISM)
                      IF (Shape%Type == ELLIPSOID .AND. ALL(Shape%size_param(shape%xy(1:nDim))==maxval(shape%size_param(shape%xy(1:nDim))))) THEN
                         positions(k,shape%xy(1:nDim))=positions(k,shape%xy(1:nDim))*min(1d0,(shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,shape%xy(1:nDim)**2))))
                      ELSEIF (Shape%Type == ELLIPTICAL_PRISM .AND. (nDim == 2 .OR.  ALL(Shape%size_param(1:2)==maxval(shape%size_param(1:2))))) THEN
                         IF (nDim == 2) THEN
                            IF (shape%xy(3) == 3) THEN
                               positions(k,1:2)=positions(k,1:2)*min(1d0,(shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,1:2)**2)))
                            ELSE
                               positions(k,shape%xy(1:2))=positions(k,shape%xy(1:2))*min(1d0,(shape%size_param(shape%xy(1:2))-boundary_separation)/abs(positions(k,shape%xy(1:2))))
                            END IF
                         ELSE
                            positions(k,1:2)=positions(k,1:2)*min(1d0,(shape%size_param(1)-boundary_separation)/sqrt(sum(positions(k,1:2)**2)))
                            positions(k,3)=positions(k,3)*min(1d0,(shape%size_param(3)-boundary_separation)/abs(positions(k,3)))
                         END IF
                      ELSE
                         IF (MPI_ID == 0) THEN
                            write(*,*) 'getrandompointsinshape not implemented for ellipsoids or elliptical prisms yet that are not effectively spheres/circles/ellipses/cylinders.  '
                            write(*,*) 'Prescription is available online at http://www.geometrictools.com/Documentation/DistancePointEllipseEllipsoid.pdf  '
                            write(*,*) 'STOPPING'
                         END IF
                         CALL MPI_BARRIER(MPI_COMM_WORLD, iErr)
                         STOP
                      END IF
                   END SELECT
                END IF
                IF (ANY(temp2 /= positions(k,:))) THEN
!                   IF (MPI_ID == 0) write(*,*) 'moved away from boundary'
                END IF
                IF (ANY(temp /= positions(k,:))) THEN
!                   IF (MPI_ID == 0) write(*,*) 'rechecking', temp, temp2, positions(k,:)
                   lRecheck=.true.
                   recheck_count=recheck_count+1
                   lSuccess=.false.
                   EXIT
                END IF
             END DO
             !Do we need to recheck? 
          END DO
          ! Have we succeeded in this particle
          IF (lSuccess) EXIT
       END DO
       IF (.NOT. lSuccess) THEN
          IF (MPI_ID == 0) THEN
             write(*,'(A,I6,A,I6,A,E24.15, A)') 'Only was able to place ', k-1 , ' of ', npoints, ' points with a min separation of ', min_separation, ' inside of : '
             CALL PrintShape(shape)
             write(*,'(A,E24.15,A)') 'min_separation is ', min_separation/(shapevolume(shape)/npoints)**(1d0/REAL(nDim)), ' of the grid packing value.  Try reducing the number of points or the min_separation'
          END IF
          CALL MPI_BARRIER(MPI_COMM_WORLD, iErr)
          STOP
       END IF
    END DO
    DO k=1,npoints
       IF (PRESENT(tnow)) THEN
          positions(k,:)=TransformCoordsFromShape(shape, positions(k,:), tnow)
       ELSE
          positions(k,:)=TransformCoordsFromShape(shape, positions(k,:))
       END IF
    END DO
!    write(*,*) positions(:,:)
!    STOP

  END SUBROUTINE GetRandomPointsInShape


  FUNCTION ShapeVolume(shape)
    REAL(KIND=qPREC) :: ShapeVolume
    TYPE(ShapeDef), POINTER :: shape
    SELECT CASE(Shape%TYPE)
    CASE(SPHERE)        
       IF (nDim == 2) THEN
          ShapeVolume=pi*shape%size_param(1)**2
       ELSE
          ShapeVolume=4d0/3d0*pi*shape%size_param(1)**3
       END IF
    CASE (CYLINDER)
       IF (nDim == 2) THEN
          IF (shape%xy(3) == 3) THEN !
             ShapeVolume=pi*shape%size_param(1)**2
          ELSE
             ShapeVolume=4d0*product(shape%size_param(shape%xy(1:2)))
          END IF
       ELSE
          ShapeVolume=2d0*pi*shape%size_param(1)**2*shape%size_param(3)
       END IF
    CASE (RECTANGULAR_PRISM)
       ShapeVolume=2d0**nDim*product(shape%size_param(shape%xy(1:nDim)))
    CASE (ELLIPSOID)
       IF (nDim == 2) THEN
          ShapeVolume=pi*product(shape%size_param(shape%xy(1:2)))
       ELSE
          ShapeVolume=4d0/3d0*pi*product(shape%size_param(:))
       END IF
    CASE (ELLIPTICAL_PRISM)
       IF (nDim == 2) THEN
          IF (shape%xy(3) == 3) THEN !
             ShapeVolume=pi*product(shape%size_param(shape%xy(1:2)))
          ELSE
             ShapeVolume=4d0*product(shape%size_param(shape%xy(1:2)))
          END IF          
       ELSE
          ShapeVolume=2d0*pi*product(shape%size_param(1:3))
       END IF
    END SELECT       
  END FUNCTION ShapeVolume

END MODULE Shapes

