!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    splitregions.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file splitregions.f90
!! @brief Main file for module Splitregions

!> @defgroup Splitregions Splitregions Object
!! @brief Module that handles the placement of splitregions
!! @ingroup ModuleObjects

!> Module that handles the placement of splitregions
!! @ingroup Splitregions
MODULE Splitregions
  USE GlobalDeclarations
  USE DataDeclarations
  USE PhysicsDeclarations
  USE DataInfoOps
  USE Shapes
  USE Interfaces
  USE ObjectDeclarations
  IMPLICIT NONE
  !> SplitRegion Data Type
  TYPE SplitRegionDef
     TYPE(ShapeDef), POINTER :: Shape => null()
     TYPE(InterfaceDef), POINTER :: InterfaceObj => null()
     REAL(KIND=qPREC), DIMENSION(MaxVars) :: qabove, qbelow
     INTEGER :: SubSample=1
     LOGICAL :: PersistInBoundaries(3,2) = .false.
     LOGICAL :: PersistInternal = .false.
     INTEGER :: ObjId
  END TYPE SplitRegionDef

  !new declaration
  TYPE pSplitRegionDef
    TYPE(SplitRegionDef), POINTER :: ptr
  END TYPE pSplitRegionDef
    TYPE(pSplitRegionDef) :: pSplitRegion
  !
  SAVE
CONTAINS


   !> Initializes a splitregion object
   !! @param SplitRegion SplitRegion object
     SUBROUTINE CreateSplitRegion(SplitRegion, Shape, InterfaceObj)
      TYPE(SplitRegionDef), POINTER :: SplitRegion
      TYPE(ShapeDef), POINTER, OPTIONAL :: Shape
      TYPE(InterfaceDef), POINTER, OPTIONAL :: InterfaceObj
      ALLOCATE(SplitRegion)

      IF (Present(Shape)) THEN
        SplitRegion%Shape = Shape
      ELSE
        ALLOCATE(SplitRegion%Shape)
      ENDIF

      IF (Present(InterfaceObj)) THEN
        SplitRegion%InterfaceObj = InterfaceObj
      ELSE
        ALLOCATE(SplitRegion%InterfaceObj)
      ENDIF

!      ALLOCATE(SplitRegion%qabove(NrHydroVars))
!      ALLOCATE(SplitRegion%qbelow(NrHydroVars))
      CALL AddSplitRegionToList(SplitRegion) 
  END SUBROUTINE CreateSplitRegion

   SUBROUTINE UpdateUniformRegion(SplitRegion)
    TYPE(SplitRegionDef), POINTER :: SplitRegion
    !update attributes that needs to be updated, if any
   END SUBROUTINE UpdateUniformRegion

  
  SUBROUTINE AddSplitRegionToList(SplitRegion)
     TYPE(SplitRegionDef), POINTER :: SplitRegion
     TYPE(ObjectDef), POINTER :: Object
     SplitRegion%ObjId = ObjectListAdd(Object,SplitRegionOBJ)
     pSplitRegion%ptr => SplitRegion
     len = size(transfer(pSplitRegion, dummy_char))
     ALLOCATE(Object%storage(len))
     Object%storage = transfer(pSplitRegion,Object%storage)
  END SUBROUTINE AddSplitRegionToList


  SUBROUTINE DestroySplitRegionObject(SplitRegionObj)
    TYPE(SplitRegionDef),POINTER :: SplitRegionObj
    CALL ObjectListRemove(SplitRegionObj%ObjId)
!    DEALLOCATE(SplitRegionObj%qabove)
!    DEALLOCATE(SplitRegionObj%qbelow)
    DEALLOCATE(SplitRegionObj%Shape)
    CALL DestroyInterface(SplitRegionObj%InterfaceObj)
    DEALLOCATE(SplitRegionObj)
    NULLIFY(SplitRegionObj)
  END SUBROUTINE DestroySplitRegionObject


   SUBROUTINE SplitRegionGridInit(Info, SplitRegion)
     TYPE(InfoDef) :: Info
     TYPE(SplitRegionDef), POINTER :: SplitRegion
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     CALL CalcPhysicalOverlaps(Info, SplitRegion%Shape%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
     IF (nOverlaps > 0) THEN
        CALL PlaceSplitRegion(Info, SplitRegion, nOverlaps, mSs, offsets)
        DEALLOCATE(mSs, offsets)
     END IF
   END SUBROUTINE SplitRegionGridInit


   SUBROUTINE SplitRegionBeforeStep(Info, SplitRegion)
     TYPE(InfoDef) :: Info
     TYPE(SplitRegionDef), POINTER :: SplitRegion
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     INTEGER :: i,j
     DO i=1,nDim
        DO j=1,2
           IF (SplitRegion%PersistInBoundaries(i,j)) THEN
              CALL CalcPhysicalOverlaps(Info, SplitRegion%Shape%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
              IF (nOverlaps > 0) THEN
                 CALL PlaceSplitRegion(Info, SplitRegion, nOverlaps, mSs, offsets)
                 DEALLOCATE(mSs, offsets)
              END IF
           END IF
        END DO
     END DO
   END SUBROUTINE SplitRegionBeforeStep

   SUBROUTINE SplitRegionSetErrFlag(Info, SplitRegion)
      TYPE(InfoDef) :: Info
      Type(SplitRegionDef), POINTER :: SplitRegion
   END SUBROUTINE SplitRegionSetErrFlag

   SUBROUTINE SplitRegionBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE SplitRegionBeforeGlobalStep 
  


   SUBROUTINE PlaceSplitRegion(Info, SplitRegion, nOverlaps, mSs, offsets)
      TYPE(InfoDef) :: Info
      Type(SplitRegionDef) :: SplitRegion
      INTEGER :: i,j,k,n,m,ii,jj,kk, location
      INTEGER, DIMENSION(3,2) :: mS    
      INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos,coords
      REAL(KIND=qPREC) :: sample_fact(3), q_fact, dx,dy,dz
      INTEGER :: sample_res(3), nOverlaps
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source

      dx=levels(Info%level)%dX
      dy=0
      dz=0
      IF (nDim >= 2) dy=dx
      IF (nDim >= 3) dz=dx
      xpos=0          
      IF (nOverlaps > 0) THEN
         sample_res=1
         sample_fact=0d0
         sample_res(1:nDim)=SplitRegion%SubSample!min(2**(MaxLevel-Info%level),4)
         sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
         q_fact=product(sample_fact(1:nDim))
         ALLOCATE(q_Source(NrHydroVars))
         DO n=1,nOverlaps
            mS=mSs(n,:,:)
            offset=offsets(n,:)
            !           write(*,*) MPI_ID, mS, q_fact
            ! Now set up cell centered quantities (density and momentum)
            DO k=mS(3,1),mS(3,2)
               xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dz
               DO j=mS(2,1),mS(2,2)
                  xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dy
                  DO i=mS(1,1),mS(1,2)
                     xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                     q_Source=0
                     DO kk=1,sample_res(3)
                        pos(3)=xpos(3)+(REAL(kk, 8)-half)*dz*sample_fact(3)
                        DO jj=1,sample_res(2)
                           pos(2)=xpos(2)+(REAL(jj, 8)-half)*dy*sample_fact(2)
                           DO ii=1,sample_res(1)
                              pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact(1)
                              IF (IsInShape(SplitRegion%Shape, pos, coords)) THEN
                                 IF (IsAboveInterface(SplitRegion%InterfaceObj, pos)) THEN
                                    q_source=q_source+(SplitRegion%qabove-Info%q(i,j,k,1:NrHydroVars))
                                 ELSE
                                    q_source=q_source+(SplitRegion%qbelow-Info%q(i,j,k,1:NrHydroVars))
                                 END IF
                              END IF
                           END DO
                        END DO
                     END DO
                     !                    write(*,*) MPI_ID, Info%q(i,j,k,1)
                     Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+q_source*q_fact                 
                     !                    write(*,*) MPI_ID, Info%q(i,j,k,1)
                  END DO
               END DO
            END DO
         END DO
         DEALLOCATE(q_Source)
      END IF

   END SUBROUTINE PlaceSplitRegion

END MODULE SplitRegions

