!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    uniformregions.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir objects
!! @brief contains object modules

!> @defgroup ModuleObjects Module Objects
!! @brief Objects that can be manipulated by modules to set initial conditions and boundary conditions
!! @ingroup Modules

!> @file uniformregions.f90
!! @brief Main file for module Uniformregions

!> @defgroup Uniformregions Uniformregions Object
!! @brief Module that handles the placement of uniformregions
!! @ingroup ModuleObjects

!> Module that handles the placement of uniformregions
!! @ingroup Uniformregions
MODULE Uniformregions
  USE GlobalDeclarations
  USE DataDeclarations
  USE PhysicsDeclarations
  USE DataInfoOps
  USE Shapes
  USE ObjectDeclarations
  IMPLICIT NONE
  !> UniformRegion Data Type
  TYPE UniformRegionDef
     TYPE(ShapeDef), POINTER :: Shape => null()
     REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q
     INTEGER :: SubSample=1
     LOGICAL :: PersistInBoundaries(3,2) = .false.
     LOGICAL :: PersistInternal = .false.
     INTEGER :: ObjId
  END TYPE UniformRegionDef

  !new declaration
  TYPE pUniformRegionDef
    TYPE(UniformRegionDef), POINTER :: ptr
  END TYPE pUniformRegionDef
    TYPE(pUniformRegionDef) :: pUniformRegion
  !
 
  !TYPE(UniformRegionDef), POINTER :: FirstUniformRegion, LastUniformRegion
  SAVE
CONTAINS


   !> Initializes a uniformregion object
   !! @param UniformRegion UniformRegion object
   SUBROUTINE CreateUniformRegion(UniformRegion, Shape)
      TYPE(UniformRegionDef), POINTER :: UniformRegion
      TYPE(ShapeDef), POINTER, OPTIONAL :: Shape
      ALLOCATE(UniformRegion)
      IF (Present(Shape)) THEN
        UniformRegion%Shape => Shape
      ELSE
        ALLOCATE(UniformRegion%Shape)
      ENDIF
      ALLOCATE(UniformRegion%q(NrHydroVars))
      CALL AddUniformRegionToList(UniformRegion)
   END SUBROUTINE CreateUniformRegion

   SUBROUTINE UpdateUniformRegion(UniformRegion)
    TYPE(UniformRegionDef), POINTER :: UniformRegion
    !update attributes that needs to be updated,
    !if any
   END SUBROUTINE UpdateUniformRegion

   SUBROUTINE AddUniformRegionToList(UniformRegion)
      TYPE(UniformRegionDef), POINTER :: UniformRegion
      TYPE(ObjectDef), POINTER :: Object
      UniformRegion%ObjId = ObjectListAdd(Object,UniformRegionOBJ)
      pUniformRegion%ptr => UniformRegion
      len = size(transfer(pUniformRegion, dummy_char))
      ALLOCATE(Object%storage(len))
      Object%storage = transfer(pUniformRegion,Object%storage)
   END SUBROUTINE AddUniformRegionToList


  SUBROUTINE DestroyUniformRegionObject(UniformRegionObj)
    TYPE(UniformRegionDef),POINTER :: UniformRegionObj
    CALL ObjectListRemove(UniformRegionObj%ObjId)
    DEALLOCATE(UniformRegionObj%q)
    DEALLOCATE(UniformRegionObj%Shape)
    DEALLOCATE(UniformRegionObj)
    NULLIFY(UniformRegionObj)
  END SUBROUTINE DestroyUniformRegionObject



   SUBROUTINE UniformRegionGridInit(Info, UniformRegion)
     TYPE(InfoDef) :: Info
     TYPE(UniformRegionDef), POINTER :: UniformRegion
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     CALL CalcPhysicalOverlaps(Info, UniformRegion%Shape%xBounds, mSs, nOverlaps, offsets, IEVERYWHERE, lHydroPeriodic)
     IF (nOverlaps > 0) THEN
        CALL PlaceUniformRegion(Info, UniformRegion, nOverlaps, mSs, offsets)
        DEALLOCATE(mSs, offsets)
     END IF
   END SUBROUTINE UniformRegionGridInit


   SUBROUTINE UniformRegionBeforeStep(Info, UniformRegion)
     TYPE(InfoDef) :: Info
     TYPE(UniformRegionDef), POINTER :: UniformRegion
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     INTEGER :: nOverlaps
     INTEGER :: i,j
     DO i=1,nDim
        DO j=1,2
           IF (UniformRegion%PersistInBoundaries(i,j)) THEN
              CALL CalcPhysicalOverlaps(Info, UniformRegion%Shape%xBounds, mSs, nOverlaps, offsets, IBOUNDARY(i,j), lHydroPeriodic)
              IF (nOverlaps > 0) THEN
                 CALL PlaceUniformRegion(Info, UniformRegion, nOverlaps, mSs, offsets)
                 DEALLOCATE(mSs, offsets)
              END IF
           END IF
        END DO
     END DO
   END SUBROUTINE UniformRegionBeforeStep

   SUBROUTINE UniformRegionSetErrFlag(Info, UniformRegion)
      TYPE(InfoDef) :: Info
      Type(UniformRegionDef), POINTER :: UniformRegion
   END SUBROUTINE UniformRegionSetErrFlag

   SUBROUTINE UniformRegionBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE UniformRegionBeforeGlobalStep 

  SUBROUTINE PlaceUniformRegion(Info, UniformRegion, nOverlaps, mSs, offsets)
     TYPE(InfoDef) :: Info
     Type(UniformRegionDef) :: UniformRegion
     INTEGER :: i,j,k,n,m,ii,jj,kk, location
     INTEGER, DIMENSION(3,2) :: mS    
     INTEGER, POINTER, DIMENSION(:,:,:) :: mSs
     REAL(KIND=qPREC), DIMENSION(3) :: offset
     REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: offsets
     REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos,coords
     REAL(KIND=qPREC) :: sample_fact(3), q_fact, dx, xBounds(3,2)
     INTEGER :: sample_res(3), nOverlaps
     REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: q_source

     dx=levels(Info%level)%dX
     xpos=0          
     IF (nOverlaps > 0) THEN
        sample_res=1
        sample_fact=0d0
        sample_res(1:nDim)=UniformRegion%SubSample!min(2**(MaxLevel-Info%level),4)
        sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
        q_fact=product(sample_fact(1:nDim))
        ALLOCATE(q_Source(NrHydroVars))
        DO n=1,nOverlaps
           mS=mSs(n,:,:)
           offset=offsets(n,:)
           ! Now set up cell centered quantities (density and momentum)
           DO k=mS(3,1),mS(3,2)
              xpos(3)=Info%xBounds(3,1)+offset(3)+(k-1)*dx
              DO j=mS(2,1),mS(2,2)
                 xpos(2)=Info%xBounds(2,1)+offset(2)+(j-1)*dx
                 DO i=mS(1,1),mS(1,2)
                    xpos(1)=Info%xBounds(1,1)+offset(1)+(i-1)*dx
                    q_Source=0
                    DO kk=1,sample_res(3)
                       pos(3)=xpos(3)+(REAL(kk, 8)-half)*dx*sample_fact(3)
                       DO jj=1,sample_res(2)
                          pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact(2)
                          DO ii=1,sample_res(1)
                             pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact(1)
                             IF (IsInShape(UniformRegion%Shape, pos, coords)) THEN
                                q_source=q_source+(UniformRegion%q-Info%q(i,j,k,1:NrHydroVars))
                             END IF
                          END DO
                       END DO
                    END DO
                    Info%q(i,j,k,1:NrHydroVars)=Info%q(i,j,k,1:NrHydroVars)+q_source*q_fact                 
                 END DO
              END DO
           END DO
        END DO
        DEALLOCATE(q_Source)
     END IF


  END SUBROUTINE PlaceUniformRegion

END MODULE UniformRegions


