!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    vectorperturbations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file vectorperturbation.f90
!! @brief Main file for module Vectorperturbation

!> @defgroup Vectorperturbation Vectorperturbation Object
!! @brief Module that handles the placement of vectorperturbation
!! @ingroup ModuleObjects

!> Module that handles the placement of vectorperturbation
!! @ingroup Vectorperturbation
MODULE Vectorperturbation
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   USE EOS
   USE Perturbation
   USE ObjectDeclarations
  IMPLICIT NONE
   !> Vectorperturbation data type
   TYPE VectorperturbationDef
      INTEGER :: nwaves = 0
      INTEGER :: iwaves = 0
      INTEGER :: Type = 0
      INTEGER :: geometry = 0
      TYPE(pPerturbationDef), DIMENSION(3) :: comp
      INTEGER :: ObjId
   END TYPE VectorperturbationDef

  ! new declaration
  TYPE pVectorperturbationDef
    TYPE(VectorperturbationDef), POINTER :: ptr
  END TYPE pVectorperturbationDef
    TYPE(pVectorperturbationDef) :: pVectorperturbation
  !

!   INTEGER, PARAMETER :: COSINESERIES=0, SINESERIES=1, EXPONENTIAL=2
!   INTEGER, PARAMETER :: CARTESIAN=0, CYLINDRICAL=1, SPHERICAL=2
CONTAINS

   SUBROUTINE CreateVectorPerturbation(VectorPerturbationObj)
      TYPE(VectorPerturbationDef), POINTER :: VectorPerturbationObj
      INTEGER :: j
      ALLOCATE(VectorPerturbationObj)      
      DO j=1,3
         CALL CreatePerturbation(VectorPerturbationObj%comp(j)%p)
      END DO
   END SUBROUTINE CreateVectorPerturbation

  SUBROUTINE UpdateVectorPerturbation(VectorPerturbation)
    TYPE(VectorPerturbationDef), POINTER :: VectorPerturbation
    !update attributes that needs to be updated, if any
  END SUBROUTINE UpdateVectorPerturbation


   SUBROUTINE CreateSolenoidalSpectra(VectorPerturbationObj, kmin, kmax, beta, amplitude, scale)
      TYPE(VectorPerturbationDef), POINTER :: VectorPerturbationObj
      INTEGER :: kmin, kmax,i,j,k,k2,ikrange(3),nwaves
      REAL(KIND=qPREC) :: beta, amplitude, phase, realpart, imagpart, dk
      REAL(KIND=qPREC) :: A(3), E(3), K_real(3)
      INTEGER ::  kx, ky, kz,Kvec(3)
      REAL(KIND=qPREC), OPTIONAL :: scale
      IF (PRESENT(scale)) THEN
         dk=2d0*Pi/scale
      ELSE
         dk=2d0*Pi/maxval(GxBounds(:,2)-GxBounds(:,1))
      END IF

      VectorPerturbationObj%TYPE=EXPONENTIAL
      nwaves=0
      ikrange=0
      ikrange(1:nDim)=kmax
      nwaves=0
      DO kx = -ikrange(1), ikrange(1)
         DO ky = -ikrange(2),ikrange(2)
            DO kz = -ikrange(3),ikrange(3)
               Kvec=(/kx,ky,kz/)
               k2 = DOT_PRODUCT(Kvec(1:nDim),Kvec(1:nDim))
               IF (k2 >= kmin**2 .AND. k2 <= kmax**2) THEN
                  nwaves=nwaves+1
               END IF
            END DO
         END DO
      END DO
      CALL InitVectorPerturbationWaves(VectorPerturbationObj, nwaves)

      DO kx = -ikrange(1), ikrange(1)
         DO ky = -ikrange(2),ikrange(2)
            DO kz = -ikrange(3),ikrange(3)
               Kvec=(/kx,ky,kz/)
               k2 = DOT_PRODUCT(Kvec(1:nDim),Kvec(1:nDim))
               IF (k2 >= kmin**2 .AND. k2 <= kmax**2) THEN
                  K_real=REAL(Kvec, 8)
                  IF (nDim == 2) THEN
                     E=(/0d0,0d0,1d0/)
                  ELSE
                     CALL random_sphere(E)  !get emf
                     E=E-DOT_PRODUCT(K_real,E)*K_real/DOT_PRODUCT(K_real,K_real) !subtract off extra dilational component
                  END IF

                  E=E/magnitude(E) !Normalize E
                  A=cross_product(K_real, E)*amplitude*magnitude(K_real)**(beta-1d0) !
                  phase=2d0*getrand()*ACOS(-1d0)
                  CALL AddVectorPerturbationWave(VectorPerturbationObj, REAL((/kx,ky,kz/), KIND=qPREC)*dk, A*cos(phase), A*sin(phase))
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE CreateSolenoidalSpectra

   SUBROUTINE InitVectorperturbationWaves(VectorperturbationObj, nWaves)
      TYPE(VectorperturbationDef) :: VectorperturbationObj
      INTEGER :: nWaves,i
      IF (nWaves == 0) RETURN
      DO i=1,3
         CALL InitPerturbationWaves(VectorPerturbationObj%comp(i)%p, nwaves)
         VectorPerturbationObj%comp(i)%p%Type=VectorPerturbationObj%Type
         VectorPerturbationObj%comp(i)%p%Geometry=VectorPerturbationObj%geometry         
      END DO
      VectorperturbationObj%iWaves=0
      VectorperturbationObj%nWaves=nWaves
   END SUBROUTINE InitVectorperturbationWaves

   SUBROUTINE AddVectorperturbationWave(VectorperturbationObj, wavevector, phase, amplitude)      
      TYPE(VectorperturbationDef) :: VectorperturbationObj
      REAL(KIND=qPREC), DIMENSION(:) :: wavevector
      REAL(KIND=qPREC) :: phase(3), amplitude(3)
      INTEGER :: iwaves,i
      iwaves=VectorperturbationObj%iwaves+1
      VectorPerturbationObj%iwaves=iwaves
      DO i=1,3
         VectorperturbationObj%comp(i)%p%iwaves=iwaves      
         VectorperturbationObj%comp(i)%p%WaveVector(iwaves,:)=wavevector(:)
         VectorperturbationObj%comp(i)%p%phase(iwaves)=phase(i)
         VectorperturbationObj%comp(i)%p%amplitude(iwaves)=amplitude(i)
      END DO
   END SUBROUTINE AddVectorperturbationWave
      
   SUBROUTINE DestroyVectorperturbation(VectorperturbationObj)
      TYPE(VectorperturbationDef), POINTER :: VectorperturbationObj
      INTEGER :: i
      DO i=1,3
         CALL DestroyPerturbation(VectorPerturbationObj%comp(i)%p)
      END DO
      NULLIFY(VectorperturbationObj)
   END SUBROUTINE DestroyVectorperturbation

   FUNCTION VectorperturbationValue(VectorperturbationObj, pos)
      TYPE(VectorperturbationDef) :: VectorperturbationObj
      REAL(KIND=qPREC), DIMENSION(3) :: pos, geopos
      REAL(KIND=qPREC) :: z
      REAL(KIND=qPREC) :: VectorperturbationValue(3)
      INTEGER :: l,i,j
      VectorperturbationValue=0d0         
      SELECT CASE (VectorperturbationObj%geometry)
      CASE(CARTESIAN)
         geopos=pos
      CASE(CYLINDRICAL)
         geopos=ConvertCylindrical(pos)
      CASE(Spherical)
         geopos=ConvertSpherical(pos)
      END SELECT

      SELECT CASE (VectorperturbationObj%Type)
      CASE(COSINESERIES)
         DO j=1,3
            DO i=1,VectorperturbationObj%nWaves
               VectorperturbationValue(j)=VectorperturbationValue(j)+VectorperturbationObj%comp(j)%p%Amplitude(i)*cos(SUM(VectorperturbationObj%comp(j)%p%WaveVector(i,:)*geopos(:))+VectorperturbationObj%comp(j)%p%Phase(i))
            END DO
         END DO
      CASE(SINESERIES)
         DO j=1,3
            DO i=1,VectorperturbationObj%nWaves
               VectorperturbationValue(j)=VectorperturbationValue(j)+VectorperturbationObj%comp(j)%p%Amplitude(i)*cos(SUM(VectorperturbationObj%comp(j)%p%WaveVector(i,:)*geopos(:))+VectorperturbationObj%comp(j)%p%Phase(i))
            END DO
         END DO
      CASE(EXPONENTIAL)
         DO j=1,3
            DO i=1,VectorperturbationObj%nWaves
               VectorperturbationValue(j)=VectorperturbationValue(j)+REAL(cmplx(VectorperturbationObj%comp(j)%p%Amplitude(i), VectorperturbationObj%comp(j)%p%Phase(i))*exp(cmplx(0d0,SUM(VectorperturbationObj%comp(j)%p%WaveVector(i,:)*geopos(:)))), 8)
            END DO
         END DO
      END SELECT
   END FUNCTION VectorperturbationValue

  function unique(k,mymx)
    INTEGER :: k(:),mymx(:),i
    logical :: unique
    unique=.false.
    DO i=1,nDim
       IF (k(i) > mymx(i)/2 .OR. k(i) <= -mymx(i)/2) RETURN
    END DO
    IF (k(1) < 0) RETURN
    IF (k(1) == 0 .OR. k(1) == mymx(1)/2) THEN
       IF (k(2) < 0) RETURN
       IF (nDim > 2 .AND. (k(2) == 0 .OR. k(2) == mymx(2)/2)) THEN
          IF (k(3) < 0) RETURN
       END IF
    END IF
    unique=.true.
    return
  end function unique

  FUNCTION cross_product(A,B)
    REAL(8), DIMENSION(3) :: cross_product, A, B
    cross_product = (/A(2)*B(3)-A(3)*B(2), A(3)*B(1)-A(1)*B(3),A(1)*B(2)-A(2)*B(1)/)
  END FUNCTION cross_product

  FUNCTION magnitude(A)
    REAL(8), DIMENSION(:) :: A
    REAL(8) :: magnitude
    magnitude = sqrt(DOT_PRODUCT(A,A))
  END FUNCTION magnitude


  
END MODULE Vectorperturbation

