!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    winds.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @Filegotoli winds.f90
!! @brief Main file for module Winds

!> @defgroup Winds Winds Object
!! @brief Module that handles the placement of winds
!! @ingroup ModuleObjects

!> Module that handles the placement of winds
!! @ingroup Winds
MODULE Winds
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE ObjectDeclarations
   IMPLICIT NONE
   !> Wind data type

   INTEGER, PARAMETER :: NOWAVE = 0, SQUAREWAVE = 1, SINEWAVE = 2

   TYPE WindDef
      REAL(KIND=qPREC) :: velocity = 0d0
      REAL(KIND=qPREC) :: density=1d0
      REAL(KIND=qPREC) :: temperature=1d0
      REAL(KIND=qPREC) :: B(3) = 0d0 !magnetic field
      REAL(KIND=qPREC) :: period = 0d0
      REAL(KIND=qPREC) :: amplitude = 0d0
      INTEGER :: dir = 1
      INTEGER :: edge = 1      
      INTEGER :: waveform = NOWAVE
      INTEGER :: Type = 0
      INTEGER :: iTracer = 0 !Tracer slot index (1 - NrTracerVars)
      INTEGER :: ObjId
!      TYPE(ShapeDef), POINTER :: shape
   END TYPE WindDef

  ! new declaration
  TYPE pWindDef
     TYPE(WindDef), POINTER :: ptr
  END TYPE pWindDef
  TYPE(pWindDef) :: pWind
  !


  INTEGER, PARAMETER :: USER_DEFINED = 0, OUTFLOW_ONLY = 1
CONTAINS

   !> Initializes a wind object
   !! @param Wind Wind object
   SUBROUTINE CreateWind(Wind, density, temperature, velocity)
      TYPE(WindDef), POINTER :: Wind
      REAL(KIND=qPREC), OPTIONAL :: density, temperature, velocity
      ALLOCATE(Wind)
      IF (Present(density)) Wind%density=density
      IF (Present(velocity)) Wind%velocity=velocity
      IF (Present(temperature)) Wind%temperature=temperature
      CALL AddWindToList(Wind)
    END SUBROUTINE CreateWind
    

    SUBROUTINE UpdateWind(Wind)
      TYPE(WindDef), POINTER :: Wind
    END SUBROUTINE UpdateWind

    SUBROUTINE AddWindToList(Wind)
       TYPE(WindDef), POINTER :: Wind
       TYPE(ObjectDef), POINTER :: Object
       Wind%ObjId = ObjectListAdd(Object,WindOBJ)
       pWind%ptr => Wind
       len = size(transfer(pWind, dummy_char))
       ALLOCATE(Object%storage(len))
       Object%storage = transfer(pWind,Object%storage)
    END SUBROUTINE AddWindToList

  
    SUBROUTINE WindGridInit(Info, Wind)
       TYPE(InfoDef) :: Info
       TYPE(WindDef) :: Wind
       CALL WindBeforeStep(Info, Wind)
    END SUBROUTINE WindGridInit


  !> Places a wind boundary condition on an Info object
  !! @param Info Info object
  !! @param Wind Wind object
   SUBROUTINE WindBeforeStep(Info, Wind)
      TYPE(InfoDef) :: Info
      TYPE(WindDef) :: Wind
      INTEGER, DIMENSION(3,2) :: ip,iq
      INTEGER :: rmbc, m,j
      REAL(KIND=qPREC) :: windvals(5)
      INTEGER :: nwindvars
      INTEGER :: windvars(5)
      IF (GhostOverlap(Info, wind%dir, wind%edge,ip)) THEN
         SELECT CASE (Wind%Type)
            CASE (USER_DEFINED)
               Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1:NrHydroVars)=0d0
               Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1)=wind%density
               Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),m_low:m_high)=0
 
               SELECT CASE (wind%waveform)
                  CASE(NOWAVE)
                     Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir))=wind%velocity

                  CASE(SQUAREWAVE)
                     Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir))=wind%velocity*(1d0 + wind%amplitude*SIGN(1d0,SIN(2d0*pi*levels(info%level)%tnow/wind%period)))

                  CASE(SINEWAVE)
                     Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir))=wind%velocity*(1d0 + wind%amplitude*SIN(2d0*pi*levels(info%level)%tnow/wind%period))
               END SELECT
            
               IF (iE .ne. 0) Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),iE)=wind%density*wind%temperature 
               IF (lMHD) THEN
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBx)=wind%B(1)
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBy)=wind%B(2)
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBz)=wind%B(3)
                  IF(MaintainAuxArrays) THEN
                     DO j=1,nDim
                        IF (j == wind%dir) cycle
                        ip(j,2)=ip(j,2)+1
                        Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), j) = wind%B(j)
                        ip(j,2)=ip(j,2)-1
                     END DO
                     IF (wind%edge == 2) ip(wind%dir,:)=ip(wind%dir,:)+1
                     Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), wind%dir) = wind%B(wind%dir)
                     IF (wind%edge == 2) ip(wind%dir,:)=ip(wind%dir,:)-1
                     iq=ip
                     iq(wind%dir,:)=iq(wind%dir,:)+1
                     Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBx-1+wind%dir) = half*( &
                          Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), wind%dir) + &
                          Info%aux(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2), wind%dir))
                  END IF
               END IF
               CALL Prim_To_Cons(Info,ip)
            CASE (OUTFLOW_ONLY)
               CALL ConvertTotalToInternalEnergy(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:))
               IF (wind%edge == 1) THEN
                  Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir))=&
                       min(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir)),0d0)
               ELSE
                  Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir))=&
                       max(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir)),0d0)
               END IF
               IF (iE .ne. 0) THEN
                  IF (ivz /= 0) THEN
                     windvars(1:5)=(/1,ivx,ivy,ivz,iE/)
                     nwindvars=5
                  ELSEIF (ivy /= 0) THEN
                     windvars(1:4)=(/1,ivx,ivy,iE/)
                     nwindvars=4
                  ELSE
                     windvars(1:3)=(/1,ivx,iE/)
                     nwindvars=3
                  END IF
               ELSE
                  IF (ivz /= 0) THEN
                     windvars(1:4)=(/1,ivx,ivy,ivz/)
                     nwindvars=4
                  ELSEIF (ivy /= 0) THEN
                     windvars(1:3)=(/1,ivx,ivy/)
                     nwindvars=3
                  ELSE
                     windvars(1:2)=(/1,ivx/)
                     nwindvars=2
                  END IF
               END IF
               windvals(1)=wind%density
               windvals(m_low:m_high)=0
               IF (iE .ne. 0) windvals(iE)=gamma7*wind%density*wind%temperature

               FORALL (m=1:nwindvars)
                  WHERE(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),imom(wind%dir)) == 0)
                     Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),windvars(m)) = windvals(windvars(m))
                  END WHERE
               END FORALL
               IF (lMHD) THEN
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBx)=wind%B(1)
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBy)=wind%B(2)
                  Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBz)=wind%B(3)
                  IF(MaintainAuxArrays) THEN
                     DO j=1,nDim
                        IF (j == wind%dir) cycle
                        ip(j,2)=ip(j,2)+1
                        Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), j) = wind%B(j)
                        ip(j,2)=ip(j,2)-1
                     END DO
                     IF (wind%edge == 2) ip(wind%dir,:)=ip(wind%dir,:)+1
                     Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), wind%dir) = wind%B(wind%dir)
                     IF (wind%edge == 2) ip(wind%dir,:)=ip(wind%dir,:)-1
                     iq=ip
                     iq(wind%dir,:)=iq(wind%dir,:)+1
                     Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), iBx-1+wind%dir) = half*( &
                          Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), wind%dir) + &
                          Info%aux(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2), wind%dir))
                  END IF
               END IF


               CALL ConvertInternalToTotalEnergy(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:))
            END SELECT
            IF (wind%iTracer .ne. 0) Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), wind%iTracer)=wind%density
      END IF
   END SUBROUTINE WindBeforeStep
   

   SUBROUTINE WindSetErrFlag(Info, Wind)
      TYPE(InfoDef) :: Info
      Type(WindDef), POINTER :: Wind
   END SUBROUTINE WindSetErrFlag

   SUBROUTINE WindBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE WindBeforeGlobalStep 


END MODULE Winds

