!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    particle_advance.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file particle_advance.f90
!! @brief Main file for module ParticleAdvance

!> @defgroup ParticleAdvance Particle Advance
!! @brief Module containing routines for advancing particles
!! @ingroup Particles

!> Module containing particle operations for advancing particles
!! @ingroup ParticleAdvance
MODULE ParticleAdvance
   USE GlobalDeclarations
   USE ParticleDeclarations
   IMPLICIT NONE

   PUBLIC AdvanceParticles
   PRIVATE

CONTAINS
   !> Advances positions of particles
   SUBROUTINE AdvanceParticles
      !Subroutine that advances the particles positions using the particles positions, velocities, and gas_accel
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: vel, pos, gas_accel, sink_accel
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: gmasses
      LOGICAL, DIMENSION(:), ALLOCATABLE :: lFixed
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle
      INTEGER :: i,j, iters
      REAL(KIND=qPREC) :: dtmax, dt_remaining, dt_next, temp
      REAL(KIND=qPREC), DIMENSION(3) :: accumgas = 0
      INTEGER, PARAMETER :: maxiters=1e4
      ParticleMaxSpeed=0
      IF (NrSinkParticles == 0) RETURN

!      temp=SinkParticles%self%q(1)*SinkParticles%self%q(ivx)
      ALLOCATE(gmasses(NrSinkParticles), vel(NrSinkParticles,nDim), pos(NrSinkParticles,nDim), gas_accel(NrSinkParticles,nDerivatives), sink_accel(NrSinkParticles, nDim), lFixed(NrSinkParticles))
      particleList=>SinkParticles
      i=0
      DO WHILE (ASSOCIATED(particlelist))
         i=i+1
         particle=>particlelist%self
         gmasses(i)=particle%Q(1)*ScaleGrav
         vel(i,1:nDim)=particle%Q(imom(1:nDim))
         pos(i,1:nDim)=particle%xloc(1:nDim)
         lFixed(i)=particle%lFixed
         particlelist=>particlelist%next
    
         
      END DO
      dt_remaining=levels(MaxLevel)%dt
!      IF (MPI_ID == 0) write(*,*) 'preadvance pos=', pos
!      IF (MPI_ID == 0) write(*,*) 'preadvance vel=', vel

      !         sink_accel=gas_accel
      !      CALL GetGasAcceleration(particlelist, pos, sink_accel)!Start with gas acceleration
      CALL GetGasAcceleration(SinkParticles, pos, sink_accel)
!      write(*,*) 'gas_acceleration=', sink_accel
!      write(*,*) 'preadvance accel=', sink_accel
      CALL CalcSinkAcc(gmasses, pos, sink_accel, dtmax) !Add in particle-particle acceleration
      iters=0
      DO WHILE (dt_remaining > 0d0)
         iters=iters+1
         dt_next=min(dtmax,dt_remaining)         
         FORALL(i=1:NrSinkParticles, .NOT. lFixed(i))
            pos(i,1:nDim)=pos(i,1:nDim)+vel(i,1:nDim)*dt_next+half*(sink_accel(i,1:nDim))*dt_next**2
            vel(i,1:nDim)=vel(i,1:nDim)+half*sink_accel(i,1:nDim)*dt_next
         END FORALL
         !         sink_accel=gas_accel
!         particleList=>SinkParticles - commented out by Erica 1/16/2012 
!         sink_accel=0
         CALL GetGasAcceleration(SinkParticles, pos, sink_accel)!Recalculate gas force at new position
         CALL CalcSinkAcc(gmasses, pos, sink_accel, dtmax)
         FORALL(i=1:NrSinkParticles, .NOT. lfixed(i))
            vel(i,1:nDim)=vel(i,1:nDim)+half*sink_accel(i,1:nDim)*dt_next
         END FORALL
         dt_remaining=dt_remaining-dt_next
         IF (iters > MaxIters) THEN
            lRequestRestart=.true.
            IF (MPI_ID == 0) THEN
               write(*,*) 'too many iterations required '
               write(*,*) 'pos=', pos
            END IF
            EXIT
         END IF
      END DO
      particleList=>SinkParticles
      i=0
      DO WHILE (ASSOCIATED(particlelist))
         i=i+1
         particle=>particlelist%self
         !         gmasses(i)=particle%Q(1)*ScaleGrav
         particle%Q(imom(1:nDim))=vel(i,1:nDim)
         ParticleMaxSpeed=max(ParticleMaxSpeed, sqrt(sum(vel(i,1:nDim)**2)))
         particle%xloc(1:nDim)=pos(i,1:nDim)
         DO j=1,nDim
!            write(*,*) particle%xloc(j), floor((particle%xloc(j)-GxBounds(j,1))/(GxBounds(j,2)-GxBounds(j,1)))
            IF (lHydroPeriodic(j)) particle%xloc(j)=particle%xloc(j)-(GxBounds(j,2)-GxBounds(j,1))*floor((particle%xloc(j)-GxBounds(j,1))/(GxBounds(j,2)-GxBounds(j,1)))
            IF (lHydroPeriodic(j) .AND. (particle%xloc(j) < GxBounds(j,1) .OR. particle%xloc(j) > GxBounds(j,2))) THEN
               write(*,*) 'whoops in particleadvance', j,particle%xloc(j), GxBounds(j,:)
!               STOP
            END IF
         END DO

         !         gas_accel(i,1:nDim)=particle%gas_accel(1:nDim)
         particlelist=>particlelist%next

!         IF (MPI_ID == 0) THEN
!            write(*,*) 'pos= ', particle%xloc
!            write(*,*) 'vel= ', particle%vel
!            write(*,*) 'fgas= ', particle%Q(1)*particle%gas_accel(1:nDim)*levels(MaxLevel)%dt
!            accumgas=accumgas+particle%Q(1)*particle%gas_accel(1:nDim)*levels(MaxLevel)%dt
!            write(*,*) 'accumgas=', accumgas
!         END IF
      END DO
!      write(*,*) 'momenta', SinkParticles%self%q(1)*SinkParticles%self%q(ivx), temp, temp+sink_accel(1,1)*levels(MaxLevel)%dt*SinkParticles%self%q(1)
!      accumgas(1)=accumgas(1)+sink_accel(1,1)*levels(MaxLevel)%dt*SinkParticles%self%q(1)
!      write(*,*) 'accumulated gas back force', accumgas(1)
      DEALLOCATE(gmasses, vel, pos, gas_accel, sink_accel, lFixed)      

   END SUBROUTINE AdvanceParticles

   !> @name Auxilliary module routines
   !! @{

   !> Calculates the Gas acceleration on a particle using the gas force at the particle's initial position and the gradients of the gas force
   !! @param ParticleList List of particles
   !! @param pos Array of current positions
   !! @param sink_accel Output array of gas accelerations
   SUBROUTINE GetGasAcceleration(ParticleList, pos, sink_accel)
      TYPE(ParticleListDef), POINTER :: ParticleList, TempParticleList
      TYPE(ParticleDef), POINTER :: Particle
      REAL(KIND=qPREC), DIMENSION(:,:) :: pos, sink_accel
      INTEGER :: i, j
      i=0
      TempParticleList=>ParticleList
      DO WHILE (ASSOCIATED(TempParticleList))
         i=i+1
         Particle=>TempParticleList%self
         IF (ASSOCIATED(Particle%pointgravityobj) .and. Particle%q(1)>0d0&
			.AND. .NOT.Particle%lFixed) & !jonathan added 8mar '12
			THEN
            sink_accel(i,1:ndim)=(/(SUM(Particle%pointgravityobj%dmom(0:maxlevel,j)/(levels(0:maxlevel)%dt)), j=1, ndim)/)/particle%q(1)
         ELSE 
            Sink_accel(i,1:ndim) = 0d0
         END IF


         !         sink_accel(i,:)=Particle%gas_accel(nDim+1:2*nDim) !First derivatives at particles initial position (phi_x, phi_y, phi_z)
         !         IF (nDim == 2) THEN
         !            sink_accel=sink_accel+(pos(i,nDim:1:-1)-Particle%xloc(nDim:1:-1))*Particle%gas_accel(2*nDim+1) !Cross Derivatives (phi_xy)
         !         ELSE
         !            sink_accel=sink_accel+(/ &
         !                 (pos(i,(/2,3/))-Particle%xloc((/2,3/)))*Particle%gas_accel((/7,9/)), & !Cross Derivatives (phi_xy phi_xz)
         !                 (pos(i,(/1,3/))-Particle%xloc((/1,3/)))*Particle%gas_accel((/7,8/)), & !Cross Derivatives (phi_yx phi_yz)
         !                 (pos(i,(/1,2/))-Particle%xloc((/1,2/)))*Particle%gas_accel((/9,8/))/)  !Cross Derivatives (phi_zx phi_zx)                 
         !         END IF
         TempParticleList=>TempParticleList%next
      END DO
   END SUBROUTINE GetGasAcceleration


   !> Calculates the sink particles acceleration
   !! @param gmasses computational 'GM' for each particle
   !! @param pos Array of positions
   !! @param sink_accel Acceleration array to update
   !! @param dtmax Adjusted based on CFL_LEAPFROG condition
   SUBROUTINE CalcSinkAcc(gmasses, pos, sink_accel,dtmax)
      USE CommonFunctions
      REAL(KIND=qPREC), DIMENSION(:,:) :: pos, sink_accel
      REAL(KIND=qPREC), DIMENSION(:)  :: gmasses
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: minradii
      REAL(KIND=qpREC) :: dtmax, radius, gacc(3),poffset(3)
      INTEGER :: i, j, nParticles, ii, jj, kk, ioffset(3,2)
      nParticles=size(gmasses)
      ALLOCATE(minradii(nParticles))
      minradii=huge(minradii)
      dtmax=huge(dtmax)
      ioffset=0
      WHERE(lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=1
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=1,nParticles
         DO j=i+1,nParticles
            DO ii=ioffset(1,1),ioffset(1,2)
               DO jj=ioffset(2,1),ioffset(2,2)
                  DO kk=ioffset(3,1),ioffset(3,2)
                     pOffSet=(/ii,jj,kk/)*(GxBounds(:,2)-GxBounds(:,1))
                     radius=sqrt(SUM((pos(i,:)+pOffSet-pos(j,:))**2))
                     minradii(i)=min(radius,minradii(i))
                     minradii(j)=min(radius,minradii(j))
                     gacc(:)=SplineSoftening(pos(j,:)-pos(i,:)-pOffset,r_soft)
                     sink_accel(i,:)=sink_accel(i,:)+gacc(1:nDim)*gmasses(j)
                     sink_accel(j,:)=sink_accel(j,:)-gacc(1:nDim)*gmasses(i)
                  END DO
               END DO
            END DO
         END DO
         dtmax=min(dtmax,CFL_LEAPFROG*sqrt(sink_dx/sqrt(sum(sink_accel(i,:)**2))))
      END DO
      DEALLOCATE(minradii)

   END SUBROUTINE CalcSinkAcc

   !> @}

END MODULE ParticleAdvance
