!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    particle_comms.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file particle_comms.f90
!! @brief Main file for module ParticleComms

!> @defgroup ParticleComms Particle Communication
!! @brief Module for performing particle related communication
!! @ingroup Particles

!> Module for performing particle related communication
!! @ingroup ParticleComms
MODULE ParticleComms
   USE ParticleDeclarations
   USE GlobalDeclarations
   USE PhysicsDeclarations
   IMPLICIT NONE
   
   PUBLIC SynchronizeMoments, SynchronizeAccretions, CollectNewParticles, SynchronizeGasForces

CONTAINS

   !> Synchronize moment contributions from all processors
   SUBROUTINE SynchronizeMoments()
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: SinkData
      INTEGER :: i, iErr
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(ParticleDef), POINTER :: Particle

      IF (NrSinkParticles == 0) RETURN

      ALLOCATE(SinkData((nMoments+2),NrSinkParticles))
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         SinkData(1:nMoments,i)=particle%moments
         SinkData(nMoments+1,i)=particle%accretionrate
         SinkData(nmoments+2,i)=particle%bondi_kernel
         particlelist=>particlelist%next
      END DO
      CALL MPI_ALLREDUCE(MPI_IN_PLACE,SinkData,i*(nMoments+2),MPI_DOUBLE_PRECISION, MPI_SUM, levels(MaxLevel)%MPI_COMM, iErr)
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         particle%moments=SinkData(1:nMoments,i)
         particle%AccretionRate=SinkData(nMoments+1,i)
         Particle%Bondi_kernel=SinkData(nMoments+2,i)
         particlelist=>particlelist%next
      END DO
      DEALLOCATE(SinkData)
   END SUBROUTINE SynchronizeMoments

   !> Synchronizes accretions across processors
   SUBROUTINE SynchronizeAccretions
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: SinkData
      INTEGER :: i, iErr, NrSinkFields
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle

      IF (NrSinkParticles==0) RETURN
      NrSinkFields=NrHydroVars+nDim+nAngularMomentum

      ALLOCATE(SinkData(NrSinkFields,NrSinkParticles))
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         SinkData(:,i)=(/particle%dQ(1:NrHydroVars), particle%drmass(1:nDim),particle%dJ(1:nAngularMomentum)/)
         particlelist=>particlelist%next
      END DO
      CALL MPI_ALLREDUCE(MPI_IN_PLACE,SinkData,i*NrSinkFields,MPI_DOUBLE_PRECISION, MPI_SUM, levels(MaxLevel)%MPI_COMM, iErr)
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         !Get collected contributions
         particle%dQ(1:NrHydroVars)=SinkData(1:NrHydroVars,i)
         Particle%drmass(1:nDim)=SinkData(NrHydroVars+1:NrHydroVars+nDim,i)
         Particle%dJ(1:nAngularMomentum)=SinkData(NrHydroVars+nDim+1:NrHydroVars+nDim+nAngularMomentum,i)
         particlelist=>particlelist%next
      END DO

      DEALLOCATE(SinkData)

   END SUBROUTINE SynchronizeAccretions


   !> Synchronizes accretions across processors
   SUBROUTINE SynchronizeGasForces(n)
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: SinkData
      INTEGER :: i, iErr, n
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle
      REAL(KIND=qPREC) :: newmass

      IF (NrSinkParticles==0) RETURN

      ALLOCATE(SinkData(nDim,NrSinkParticles))
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         IF (ASSOCIATED(particle%pointgravityobj)) &
              SinkData(:,i)=particle%pointgravityobj%dMom(n,1:nDim)
         particlelist=>particlelist%next
      END DO
      CALL MPI_ALLREDUCE(MPI_IN_PLACE,SinkData,Size(SinkData),MPI_DOUBLE_PRECISION, MPI_SUM, levels(MaxLevel)%MPI_COMM, iErr)
      particlelist=>SinkParticles
      i=0
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
         i=i+1
         !Get collected contributions
         IF (ASSOCIATED(particle%pointgravityobj)) &
              Particle%pointgravityobj%dMom(n,1:nDim)=SinkData(:,i)
         particlelist=>particlelist%next
      END DO
      DEALLOCATE(SinkData)

   END SUBROUTINE SynchronizeGasForces


   !> Collect new particles across processors
   SUBROUTINE CollectNewParticles()
      !Global communication routine to gather new particles
      INTEGER :: nparticles !Number of local new particles
      INTEGER :: nGlobalParticles !Number of global new particles
      INTEGER, DIMENSION(:), ALLOCATABLE :: NParticlesByProcessor !Array containing the number of new particles on each processor
      INTEGER, DIMENSION(:), ALLOCATABLE :: OffsetsByProcessor !Array containing the offsets for storing the new particles from each processor
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: Particles !arrays containing pertinent particle information for local particles
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: GlobalParticles !arrays containing pertinent particle information for global particles
      INTEGER :: iErr
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle, NewParticle
      INTEGER :: i
      ! Get local contribution to new particles
      nParticles=ParticleCount(NewSinkParticles)
      ALLOCATE(Particles(nParticleFields,nParticles))
      particlelist=>NewSinkParticles
      i=0
      DO WHILE (ASSOCIATED(particlelist))
         i=i+1
         particle=>particlelist%self
         Particles(:,i)=(/particle%xloc(1:nDim), particle%moments(1:nMoments)/)
         particlelist=>particlelist%next
      END DO

      ! Create space and receive information on number of particles 
      ALLOCATE(NParticlesByProcessor(MPI_NP), OffsetsByProcessor(MPI_NP))
      CALL MPI_AllGather(nParticles, 1, MPI_INTEGER, NParticlesByProcessor, 1, MPI_INTEGER, levels(MaxLevel)%MPI_COMM, iErr)

!      write(*,*) "nParticlesByProcessor=", nParticlesByProcessor

      !Calculate Offsets
      OffsetsByProcessor(1)=0
      DO i=2,MPI_NP
         OffsetsByProcessor(i)=OffsetsByProcessor(i-1)+NParticlesByProcessor(i-1)
      END DO
      nGlobalParticles=OffsetsByProcessor(MPI_NP)+NParticlesByProcessor(MPI_NP)
      IF (nGlobalParticles > 0) THEN
         ! Allocate space for and receive (collect) all particle info
         ALLOCATE(GlobalParticles(nParticleFields,nGlobalParticles))
         CALL MPI_ALLGatherV(Particles,nParticles*nParticleFields,MPI_DOUBLE_PRECISION,GlobalParticles,NParticlesByProcessor*nParticleFields,OffsetsByProcessor*nParticleFields,MPI_DOUBLE_PRECISION,levels(MaxLevel)%MPI_COMM, iErr)
         !      write(*,*) "nGlobalParticles=",nGlobalParticles
         ! Replace local particle list with global particlelist
         CALL DestroyParticleList(NewSinkParticles)
         NULLIFY(NewSinkParticles, LastNewSinkParticle)
         DO i=1,nGlobalParticles         
            NULLIFY(NewParticle)
            CALL CreateParticle(NewParticle)
            NewParticle%xloc(1:nDim)=GlobalParticles(1:nDim,i)
            NewParticle%moments=GlobalParticles(nDim+1:,i)
            NewParticle%iAccrete=DefaultAccretionRoutine !FEDERRATH_ACCRETION
            CALL AddSinkParticle(NewParticle)
            CALL CreatePointGravityObject(NewParticle%PointGravityObj)
            NewParticle%PointGravityObj%x0(1:nDim)=GlobalParticles(1:nDim,i)
            NewParticle%PointGravityObj%soft_length=1d0*sink_dx !NewParticle%radius*sink_dx
            NewParticle%PointGravityObj%soft_function=SPLINESOFT
         END DO
         DEALLOCATE(GlobalParticles)
      END IF
   END SUBROUTINE CollectNewParticles


END MODULE ParticleComms
