!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    particle_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir particle
!! @brief Directory containing modules for managing embedded lagrangian particles

!> @defgroup Particles Particles
!! @brief Group of modules for managing embedded lagrangian particles

!> @file particle_control.f90
!! @brief Main file for module ParticleControl

!> @defgroup ParticleControl Particle Control
!! @brief Module for managing particles
!! @ingroup Particles


!> Module for managing particles
!! @ingroup ParticleControl
MODULE ParticleControl
   USE ParticleDeclarations
   USE ParticleComms
   USE ParticleLevelOps
   USE ParticleAdvance  
   USE Timing
   USE Bondi
   IMPLICIT NONE

   ! --------------------------------------------------------------------------------------------------------
   !This approach requires 3 collective calls before beginning the advance step.  
   !But minimizes the amount of data transferred - as well as additional computations.

   !1*            !Ghost r_acc cells

   !Sink Advance
   !Check for new particles and calculate moments
   !2             !Collect New Particles and moments 
   !Do accretion
   !3             !Collect new masses, momentum, and advance particles

   !Hydro Advance
   !Hydro Step

   !Sink PostAdvance
   !4             !Calculate new particle multipole moments and reduce along with call to elliptic

   !Elliptic
   !Calculate new gas potential


   ! ------------------------------------------------------

    PUBLIC Particle_WriteData, Particle_ReadData

CONTAINS


   SUBROUTINE SinkParticleBackup()
      TYPE(ParticleListDef), POINTER :: ParticleList, lastparticle
      TYPE(ParticleDef), POINTER :: BackupParticle
      !Need to duplicate particle list      

      ParticleList=>BackupParticles
      DO WHILE (ASSOCIATED(ParticleList))
         IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
            DEALLOCATE(ParticleList%self%PointGravityObj)
            NULLIFY(ParticleList%self%PointGravityObj)
         END IF                    

         IF (ASSOCIATED(ParticleList%self%OutflowObj)) THEN
            DEALLOCATE(ParticleList%self%OutflowObj)
            NULLIFY(ParticleList%self%OutflowObj)
         END IF                    
         ParticleList=>ParticleList%next         
      END DO      
      CALL DestroyParticleList(BackupParticles)
      NULLIFY(lastparticle)
      ParticleList=>SinkParticles
      DO WHILE (ASSOCIATED(ParticleList))
         NULLIFY(BackupParticle)
         ALLOCATE(BackupParticle)
         BackupParticle=ParticleList%self
         IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
            NULLIFY(BackupParticle%PointGravityObj)
            ALLOCATE(BackupParticle%PointGravityObj)
            BackupParticle%PointGravityObj=ParticleList%self%PointGravityObj
         END IF            
         IF (ASSOCIATED(ParticleList%self%OutflowObj)) THEN
            NULLIFY(BackupParticle%OutflowObj)
            ALLOCATE(BackupParticle%OutflowObj)
            BackupParticle%OutflowObj=ParticleList%self%OutflowObj
         END IF            
         CALL AddParticleToList(BackupParticle,BackupParticles,lastParticle)
         ParticleList=>ParticleList%next         
      END DO
   END SUBROUTINE SinkParticleBackup


   SUBROUTINE SinkParticleRestore()
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(ParticleDef), POINTER :: RestoreParticle
      !Need to remove old particles 
      ParticleList=>SinkParticles
      DO WHILE (ASSOCIATED(ParticleList))
         IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
            CALL DestroyPointGravityObject(ParticleList%self%PointGravityObj)
         END IF
         IF (ASSOCIATED(ParticleList%self%OutflowObj)) THEN
            CALL DestroyOutflowObject(ParticleList%self%OutflowObj)
         END IF
         ParticleList=>ParticleList%next         
      END DO

      CALL DestroyParticleList(SinkParticles)

      NrSinkParticles=0
      NULLIFY(LastSinkParticle)
      ParticleList=>BackupParticles
      DO WHILE (ASSOCIATED(ParticleList))
         NULLIFY(RestoreParticle)
         ALLOCATE(RestoreParticle)
         RestoreParticle=ParticleList%self
         IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
            NULLIFY(RestoreParticle%PointGravityObj)
            CALL CreatePointGravityObject(RestoreParticle%PointGravityObj)
            RestoreParticle%PointGravityObj=ParticleList%self%PointGravityObj
         END IF
         IF (ASSOCIATED(ParticleList%self%OutflowObj)) THEN
            NULLIFY(RestoreParticle%OutflowObj)
            CALL CreateOutflowObject(RestoreParticle%OutflowObj)
            RestoreParticle%OutflowObj=ParticleList%self%OutflowObj
            CALL AddOutflowObjToList(RestoreParticle%OutflowObj)
         END IF
         CALL AddParticleToList(RestoreParticle,SinkParticles,LastSinkParticle)
         ParticleList=>ParticleList%next
         NrSinkParticles=NrSinkParticles+1
      END DO
      
   END SUBROUTINE SinkParticleRestore

   !> Initializes variables for module
   SUBROUTINE SinkParticleInit
      INTEGER :: r2, i, j, k, ip(3,2)
      JeansFact=sqrt(pi*gamma/ScaleGrav)/(JEAN_CELLS*levels(MaxLevel)%dx)
      nParticleFields=nDim+nMoments
      sink_dv=levels(MaxLevel)%dx**nDim
      sink_dx=levels(MaxLevel)%dx
      ScaleGravdV=ScaleGrav*sink_dv
      r_acc=IR_ACC*levels(maxLevel)%dx
      r_soft=half*r_acc
      r_acc2=r_acc**2
      r_inner_acc=half*sink_dx*sqrt(REAL(nDim,8)) !This is the farthest that the host cell's center can be from the particle
      ip(nDim+1:3,:)=0
      ip(1:nDim,1)=0-IR_ACC
      ip(1:nDim,2)=0+IR_ACC      
      r2=IR_ACC**2
      NrSinkParticles=0
      NULLIFY(SinkParticles, LastSinkParticle, BackupParticles, LastNewSinkParticle, NewSinkParticles)
      IF (lSinkParticles) THEN
         IF (ANY(levels(MaxLevel)%mX(1:nDim) <= 2*(levels(MaxLevel)%gmbc(1)+IR_ACC) .AND. lHydroPeriodic(1:nDim))) THEN
            PRINT *,'WARNING: Highest level grid should be at least ', 2*(levels(MaxLevel)%gmbc(1)+IR_ACC)+1, 'cells across for periodic bcs to work with accretion by sink particles.  Unphysical results may ensue'
         END IF
      END IF
      ALLOCATE(lControlVolume(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2)))
      FORALL(i=ip(1,1):ip(1,2),j=ip(2,1):ip(2,2),k=ip(3,1):ip(3,2))
         lControlVolume(i,j,k)=SUM((/i,j,k/)**2) <= r2
      END FORALL
      IF (nDim == 2) THEN
         nAngularMomentum=1
      ELSE
         nAngularMomentum=3
      END IF
      IF (nDim == 2) THEN
         nDerivatives=5 !dx dy dxx dyy dxy
      ELSE
         nDerivatives=9 !dx dy dz dxx dyy dzz dxy dyz dzx
      END IF
      IF (lSinkParticles) THEN 
         particle_mbc=IR_ACC  !Minimum number of ghost zones required for determining new sink particles
      ELSEIF (lSelfGravity) THEN
         particle_mbc=0 !Need to get gas forces on particles
      ELSE
         particle_mbc=0
      END IF
      CALL InitializeBondi()
      
   END SUBROUTINE SinkParticleInit

   !> Update Particles with data from level n
   !! @param n level
   SUBROUTINE ParticlePreUpdate(n)
      INTEGER :: n
      IF (n < 0) RETURN
      CALL StartTimer(iParticleUpdate, n)
      IF (lSinkParticles) THEN
         CALL ClearParticleMomenta(n)
         IF (n == MaxLevel) THEN
            IF (NrSinkParticles > 0) THEN
               CALL CalcMoments()
               CALL SynchronizeMoments()
            END IF
            IF (lSelfGravity) THEN
               CALL CheckForNewParticles()
               CALL CollectNewParticles    
            END IF
!            END IF
            CALL DoAccretions()
            CALL SynchronizeAccretions()
            CALL FinalizeAccretions()
         END IF
      END IF
      CALL StopTimer(iParticleUpdate, n)
    END SUBROUTINE ParticlePreUpdate

    SUBROUTINE ClearParticleMomenta(n)
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle
      INTEGER :: n
      particlelist=>SinkParticles
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self
          IF (ASSOCIATED(Particle%PointGravityObj)) THEN
             Particle%PointGravityObj%dmom(n,:)=0d0
          END IF
         particlelist=>particlelist%next
      END DO
    END SUBROUTINE ClearParticleMomenta

    !> Update Particles with data from level n following a hydro step (but before an elliptic step)
    !! @param n level
    SUBROUTINE ParticlePostUpdate(n)
      INTEGER :: n      
      IF (n < 0) RETURN
!      RETURN
      IF (lSinkParticles) THEN
!         IF (lSelfGravity) THEN
         CALL SynchronizeGasForces(n)
        ! END IF
         IF (n == MaxLevel) THEN
!            write(*,*) 'advancing particles on level', n
            CALL AdvanceParticles
         END IF
      END IF
      IF (lParticles .AND. n == MaxLevel) THEN
         CALL UpdateParticleObjects()
      END IF
!         write(*,*) 'updating phi sinks on level', n
    END SUBROUTINE ParticlePostUpdate


    SUBROUTINE ParticlePostRestrict(n)
      INTEGER :: n
    END SUBROUTINE ParticlePostRestrict

    SUBROUTINE UpdateParticleObjects()
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle
      REAL(KIND=qPREC) :: newmass
      particlelist=>SinkParticles
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self     
         !Calculate new mass
         IF (ASSOCIATED(Particle%PointGravityObj)) THEN
            Particle%PointGravityObj%v0(1:nDim)=Particle%q(imom(1:nDim))
            Particle%PointGravityObj%x0=Particle%xloc
            Particle%PointGravityObj%t0=levels(MaxLevel)%tnow+levels(MaxLevel)%dt
            Particle%PointGravityObj%Mass=Particle%q(1)
         END IF
         IF (ASSOCIATED(Particle%OutflowObj)) THEN
            Particle%OutflowObj%source_vel=Particle%q(imom(1:nDim))
            Particle%OutflowObj%position=Particle%xloc
            Particle%OutflowObj%t0=levels(MaxLevel)%tnow+levels(MaxLevel)%dt
            Particle%OutflowObj%Mass=Particle%q(1)
            CALL SetOutflowBounds(Particle%OutflowObj)
         END IF
         particlelist=>particlelist%next
      END DO
    END SUBROUTINE UpdateParticleObjects

    SUBROUTINE FinalizeAccretions()
      TYPE(ParticleListDef), POINTER :: particlelist
      TYPE(ParticleDef), POINTER :: particle
      REAL(KIND=qPREC) :: newmass, newvel(3), temp
      REAL(KIND=qPREC), DIMENSION(3) :: accumdP=0
      particlelist=>SinkParticles
      DO WHILE(ASSOCIATED(particlelist))
         particle=>particlelist%self     
         !Calculate new mass
!         write(*,*) 'B', MPI_ID, particle%dq(1)
         temp=particle%q(1)*particle%q(ivx)
!         IF (MPI_ID == 0) write(*,'(A,10E24.15)') 'Particle%q= ',Particle%Q(1:NrHydroVars)
!         IF (MPI_ID == 0) write(*,'(A,10E24.15)') 'Particle%dq= ', Particle%dQ(1:NrHydroVars)

         IF (particle%dQ(1) > 0d0) THEN
            newmass=(Particle%Q(1)+Particle%dQ(1))
            IF (newmass <= 0) THEN 
               write(*,*) 'error in particle_control.f90'
               STOP
            END IF
            newvel(1:nDim)=(Particle%Q(imom(1:nDim))*Particle%Q(1)+Particle%dQ(imom(1:nDim)))/newmass
            IF (.NOT. Particle%lFixed) Particle%xloc=Particle%xloc+(Particle%drmass)/newmass
            Particle%Q(1:NrHydroVars)=Particle%Q(1:NrHydroVars)+Particle%dQ(1:NrHydroVars)
            Particle%Q(imom(1:nDim))=newvel
            ! Don't want fixed particles to give velocities to point gravity objects.
            IF (Particle%lFixed) Particle%q(imom(1:nDim))=0d0

!            mean_density=mean_density-Particle%dQ(1)/Product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))
         END IF
!         accumdP(1)=accumdP(1)+particle%dq(ivx)
!         write(*,*) 'accumulated dp=', accumdP(1)
!         write(*,*) 'particle p=', Particle%q(1)*particle%q(ivx)

         IF (ASSOCIATED(Particle%OutflowObj)) THEN
            Particle%OutflowObj%accretionrate=Particle%dQ(1)/levels(MaxLevel)%dt
         END IF

!         write(*,*) 'momenta', temp+particle%dq(ivx), particle%q(1)*particle%q(ivx)

         Particle%J=Particle%J+Particle%dJ


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!   Print statements added by Erica to diagnose psrticle kicks

!         IF (MPI_ID == 0) THEN

            ! write(*,*) 'dM=', Particle%dM
            ! write(*,*) 'dP=', Particle%dP
            ! write(*,*) 'avel=', Particle%vel
!            write(*,'(A,3E20.12,A,E20.12)') 'axloc=', Particle%xloc, 'time', Levels(MaxLevel)%tnow
!            write(*,'(A,5E20.12,A,E20.12)') 'dq:' , Particle%dQ(1:5), 'time' , Levels(MaxLevel)%tnow
            !   write(*,'(A,3E20.12,A,E20.12, A,10E20.12 )') 'Q:' , Particle%Q(1:5), 'time', Levels(MaxLevel)%tnow
            !  accumDP=accumdP+Particle%dP
            !  write(*,*) 'accumdP=', accumdP
            !  write(*,*) 'drmass=', Particle%drmass

!         END IF

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

         !zero out contributions
         Particle%dJ=0
         Particle%dQ=0
         Particle%drmass=0
         particlelist=>particlelist%next
      END DO
    END SUBROUTINE FinalizeAccretions

    !> Creates a group for particle data within the Chombo file and populates it with the contents of the SinkParticles list.
    !! @param chandle A chombo file handle.
    SUBROUTINE Particle_WriteData(chandle)

      USE ChomboDeclarations, ONLY: ChomboHandle, Chombo_OpenParticleGroup, Chombo_CloseParticleGroup
      USE HDF5Declarations, ONLY: Add_HDF5_Attribute_Int

      TYPE(ChomboHandle), POINTER :: chandle

      TYPE(ParticleListDef), POINTER :: particle_list
      TYPE(ParticleDef), POINTER :: particle
      INTEGER :: i_err


      ! Create a new particle group in the chombo file, and write the number of particles to it.
      i_err = Chombo_OpenParticleGroup(chandle)

      ! Store the number of components and the number of particles.	 
      CALL Add_HDF5_Attribute_Int("num_components", chandle%particle_group_id, IO_TRACKED_COMPONENTS)
      CALL Add_HDF5_Attribute_Int("num_particles", chandle%particle_group_id, NrSinkParticles)

      ! If there are any sink particles, then create datasets to store their data.  Otherwise,
      ! don't create the datasets (since there's no need to take up file space).
      !      IF (NrSinkParticles > 0) THEN

      ! Initialize the Chombo datasets that will store the sink particle data.
      CALL Particle_InitChomboDatasets(chandle)

      particle_list => SinkParticles

      ! Loop over the list of sink particles and write their data to the new component datasets.
      DO WHILE (ASSOCIATED(particle_list))
         particle => particle_list%self
         CALL Particle_WriteObjectToChomboFile(chandle, particle)
         particle_list => particle_list%next

      END DO

      !      END IF

      ! Close the particles group.
      CALL Chombo_CloseParticleGroup(chandle)

    END SUBROUTINE Particle_WriteData

    !> Read particle data in from a Chombo file.
    !! @param nframe The frame number to be opened.
    SUBROUTINE Particle_ReadData(nframe)

      USE ChomboDeclarations, ONLY: ChomboHandle, CHOMBO_HANDLE_READ, CreateChomboHandle, CloseChomboHandle, &
           Chombo_OpenParticleGroup, Chombo_CloseParticleGroup

      INTEGER :: nframe

      CHARACTER(LEN=23) :: s_filename
      TYPE(ChomboHandle), POINTER :: chandle
      TYPE(ParticleDef), POINTER :: particle
      INTEGER :: i_err
      INTEGER :: nr_particles


      ! Open a reading handle for the specified frame.
      WRITE(s_filename, '(A10,I5.5,A4)') 'out/chombo', nframe, '.hdf'
      CALL CreateChomboHandle(s_filename, chandle, CHOMBO_HANDLE_READ)

      ! Open the 'particles' group, save the number of elements in it.
      nr_particles = Chombo_OpenParticleGroup(chandle)

      chandle%particle_offset = 0
      IF (nr_particles > 0 .AND. MPI_ID == 0) write(*,*) "Found ", nr_particles, "particles"

      DO WHILE (chandle%particle_offset < nr_particles)

         ! Create a new particle object to be populated from the Chombo file.
         NULLIFY(particle)
         CALL CreateParticle(particle)

         ! Read in the particle data from the Chombo file.  This subroutine also advances the
         ! chandle%particle_offset variable.
         CALL Particle_ReadObjectFromChombo(chandle, particle)

         ! Adds the particle to the particle list.  I am only assuming at this point that I should
         ! be using AddSinkParticle and not AddNewSinkParticle; we will find out.
         CALL AddSinkParticle(particle)
      END DO

      ! Close the particles group.
      CALL Chombo_CloseParticleGroup(chandle)

      CALL CloseChomboHandle(chandle)

    END SUBROUTINE Particle_ReadData





    ! Other options for the sink particle algorithm



    !To do the accretion on a cell by cell basis we need to be able to calculate the binding energy
    !of each particle and it's surrounding gas...  This requires knowledge of the density in the accretion radius of each sink particle.  
    !If the accretion radius is 4 - this is a 8x8x8 array of floats for each particle.  

    !Alternatively we could use a multipole expansion and store just the first few terms
    !1             !Then we would calculate the expansion and reduceall the particle moments
    !Each processor could then do the accretion step independently
    !2             !Each processor would also create a list of potential new particles that would be shared...
    !*             !Then at the beginning of each hydro step during the overlap stage - the potential particles would collect grids as well.
    !These particles would then be checked and possibly created.
    !3             !Then a global reduce would collect all of the new particles
    !4             !Each grid would then receive updated data from the new particles first accretion...



    !  - after a ghosting for the hydro solver - new zero mass particles could be checked for and created - and could accrete
    !Then any grid that overlapped a new particle could receive new ghost data

    !As well as create new potential particles
    !Before creating new particles there would need to be a ghosting of all quantities in a 4 cell region
    !However if this is done before each hydro step - each particle creation did not modify any quantities - then every
    !This still requires ghosting of 4 cells around potential particles to check the other criteria...
    !Then there would be a collective gather of new particles and a reduceall of accreted masses etc...






    !Do accretion of gas (in ghost zones as well) (but only store contributions from local cells)

    !2             !Collect new particles position/mass/velocity and reduce accreted mass, momentum, energy...
    !3             !

    !1             !Alternatively we could ghost enough cells so that every cell has enough information to calculate the potential directly...
    !This requires a an additional ghosting call of density for 8 cells around each grid and everything else for 4 cells...
    !But allows for particles to be directly created/accreted.
    !Every cell could determine whether it should become a new sink particle and accrete - or whether it should be accreted by an existing sink particle...
    !2             !There would still need to be a collective reduceall to get the new particles mass, velocity, and postion...  
    !But this requires only two communication calls.


    !

    !Since we need to calculate the binding energy between the particles and 

    !We've just performed an accretion step so any particles inside the accretion radius of an existing particle cannot become new particles...

  END MODULE ParticleControl
