!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    particle_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file particle_declarations.f90
!! @brief Main file for module ParticleDeclarations

!> @defgroup ParticleDeclarations Particle Declarations
!! @brief Module containing declarations needed for Particles
!! @ingroup Particles


!> Module containing declarations needed for Particles
!! @ingroup ParticleDeclarations
MODULE ParticleDeclarations
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE PointGravitySrc
  USE Outflows
  IMPLICIT NONE
  SAVE
  PUBLIC
  INTEGER, PARAMETER :: nMoments=1
  INTEGER, PARAMETER :: MaxHydroVars = 20

  INTEGER, PARAMETER :: NOACCRETION=0, FEDERRATH_ACCRETION=1, KRUMHOLZ_ACCRETION=2
  !> Contains data for Particles
  TYPE, PUBLIC :: ParticleDef
     !     REAL(KIND=qPREC) :: mass !particle mass
     REAL(KIND=qPREC), DIMENSION(MaxHydroVars) :: Q  !Conserved quantities
     REAL(KIND=qPREC), DIMENSION(3) :: xloc !location of particle
     REAL(KIND=qPREC), DIMENSION(3) :: J
     REAL(KIND=qPREC), DIMENSION(MaxHydroVars) :: dQ !Accreted contributions to quantities
     REAL(KIND=qPREC), DIMENSION(3) :: dJ
     REAL(KIND=qPREC), DIMENSION(3) :: drmass     
     REAL(KIND=qPREC), DIMENSION(nMoments) :: moments
     REAL(KIND=qPREC), DIMENSION(3) :: gas_accel 
     REAL(KIND=qPREC) :: rho_inf !Used for Bondi Accretion
     INTEGER :: radius=4 
     INTEGER :: Buffer(0:MaxDepth)=0 !number of extra cells of buffer around sink region to set err flags
     LOGICAL :: lFixed=.false. !Particle is free to move
     INTEGER :: iAccrete=NOACCRETION !particle accretion routine
     REAL(KIND=qPREC) :: AccretionRate=0d0
     REAL(KIND=qPREC) :: Bondi_kernel=0d0
     TYPE(PointGravityDef), POINTER :: PointGravityObj ! Potential Source object
     TYPE(OutflowDef), POINTER :: OutflowObj      ! Potential Source object
  END TYPE ParticleDef

  !> List of ParticleDef's
  TYPE, PUBLIC :: ParticleListDef
     TYPE(ParticleDef), POINTER :: self=>Null()
     TYPE(ParticleListDef), POINTER :: next=>Null()
  END TYPE ParticleListDef


  PUBLIC :: CreateParticle
  PUBLIC :: IO_TRACKED_COMPONENTS, Particle_InitChomboDatasets, Particle_WriteObjectToChomboFile


  REAL(KIND=qPREC), PARAMETER :: JEAN_CELLS=4 !Number of cells needed to refine jeans length
  INTEGER, PARAMETER :: IR_ACC = 4 !Accretion radius in cells
  REAL(KIND=qpREC), PARAMETER :: CFL_LEAPFROG=.1  !CFL criteria for leap frog method
  REAL(KIND=qPREC) :: r_acc, r_acc2, r_inner_acc, r_soft!Accretion radius and inner accretion radius in physical units
  REAL(KIND=qPREC) :: JeansFact !Factor involving scalegrav, pi, gamma, and the fine grid dx
  REAL(KIND=qPREC) :: sink_dv, sink_dx !volume of fine grid cells, size of fine grid cells
  REAL(KIND=qPREC) :: ScaleGravdV !Product of ScaleGrav and cell volume dV
  INTEGER :: nParticleFields
  LOGICAL, DIMENSION(:,:,:), ALLOCATABLE :: lControlVolume
  TYPE(ParticleListDef), POINTER :: SinkParticles, LastSinkParticle, NewSinkParticles, LastNewSinkParticle, BackupParticles
  LOGICAL :: HaveSinkParticles=.false.
  INTEGER :: NrSinkParticles
  INTEGER :: nAngularMomentum
  INTEGER :: nDerivatives
  REAL(KIND=qPREC) :: ParticleMaxSpeed
  REAL(KIND=qPREC) :: DefaultParticleRadius = IR_ACC
  INTEGER, PARAMETER :: IO_TRACKED_COMPONENTS = 11

CONTAINS

  !> Creates a new particle
  !! @param particle Newly created particle
  SUBROUTINE CreateParticle(particle)
    TYPE(ParticleDef), POINTER :: particle           
    IF (Associated(particle)) THEN
       PRINT*, 'particle already associataed in CreateParticle'
       STOP
    END IF
    ALLOCATE(particle)
    IF (NrHydroVars > MaxHydroVars) THEN !
       PRINT*, 'err increase maxhydrovars in particle_declarations.'
       STOP
    END IF
!    write(*,*) 'made a particle'
    Particle%J=0
    Particle%dQ=0
    Particle%dJ=0
    Particle%drmass=0
    particle%radius=DefaultParticleRadius
    particle%Buffer=0
    particle%lfixed=.false.
    particle%Q=0
    particle%xloc=0
    particle%moments=0
    particle%gas_accel=0

    NULLIFY(particle%PointGravityObj)
    NULLIFY(particle%OutflowObj)
    particle%iaccrete=0
    particle%buffer(0:MaxLevel)=ceiling(real(max(particle%radius, IR_ACC)*sink_dx)/levels(0:MaxLevel)%dx)

    ! [BDS][20110114]:  On a restart, the point gravity object will be read in from the data files.
    ! Consequently, There is no need to create the particle here.  The point gravity object will be 
    ! associated in the ParticleDeclarations::Particle_ReadObjectFromChombo() subroutine.
    !    IF (.NOT. lRestart) THEN
    !      CALL CreatePointGravityObject(Particle%PointGravityObj)
    !      Particle%PointGravityObj%soft_length=Particle%radius*sink_dx
    !      Particle%PointGravityObj%soft_function=SPLINESOFT
    !  ELSE
    !      NULLIFY(Particle%PointGravityObj)
    !  END IF

  END SUBROUTINE CreateParticle

  !> Adds particle to list
  !! @param particle particle object
  !! @param particlelist particle list to add to
  !! @param lastparticle pointer to last particle in list
  SUBROUTINE AddParticleToList(particle, particlelist, lastparticle)
    TYPE(Particlelistdef), POINTER :: lastparticle
    TYPE(Particlelistdef), POINTER, OPTIONAL :: particlelist
    TYPE(ParticleDef), POINTER :: particle
    INTEGER :: iErr
    IF (.NOT. ASSOCIATED(LastParticle)) THEN     
       ALLOCATE(LastParticle, STAT=iErr)
       NULLIFY(LastParticle%next)
       NULLIFY(LastParticle%self)
       IF (iErr /= 0) THEN
          PRINT *, "AddParticle() error: unable to allocate LastParticle list object."
          STOP
       END IF
       IF (present(particlelist)) particlelist=>lastparticle
    ELSE
       IF (ASSOCIATED(LastParticle%next)) THEN
          PRINT *, "Error - last particle next allocated"
          STOP
       END IF
       ALLOCATE(LastParticle%next, STAT=iErr)
       IF (iErr /= 0) THEN
          PRINT *, "AddParticle() error: unable to allocate LastParticle%next list object."
          STOP
       END IF
       LastParticle=>LastParticle%next         
       NULLIFY(LastParticle%next)
       NULLIFY(LastParticle%self)

    END IF
    IF (ASSOCIATED(particle)) THEN
       LastParticle%self=>particle
    ELSE
       ALLOCATE(LastParticle%self, STAT=iErr)
       PRINT*, 'Expecting particle to be associated in AddParticleToList'

       IF (iErr /= 0) THEN
          PRINT *, "AddParticle() error: unable to allocate LastParticle%self object."
          STOP
       END IF
       particle=>LastParticle%self
    END IF
  END SUBROUTINE AddParticleToList

  !> Clears a particle list
  !! @param particlelist Particle list object
  RECURSIVE SUBROUTINE ClearParticleList(particlelist)
    TYPE(Particlelistdef), POINTER :: particlelist
    IF (.NOT. ASSOCIATED(particlelist))  RETURN
    IF (ASSOCIATED(particlelist%next)) CALL ClearParticleList(particlelist%next)
    DEALLOCATE(particlelist)
    NULLIFY(particlelist)
  END SUBROUTINE ClearParticleList


  !> Destroys a particle
  !! @param particle Particle object
  SUBROUTINE DestroyParticle(particle)
    TYPE(ParticleDef), POINTER :: particle
    IF (.NOT. ASSOCIATED(particle)) THEN
       PRINT*, 'asked to destroy unassociated particle in DestroyParticle'
    ELSE
       !       IF (ASSOCIATED(Particle%PointGravityObj)) CALL DestroyPointGravityObject(Particle%PointGravityObj)    
       !       IF (ASSOCIATED(Particle%OutflowObj)) CALL DestroyOutflowObject(Particle%OutflowObj)
       DEALLOCATE(particle)
       NULLIFY(particle)
    END IF
  END SUBROUTINE DestroyParticle


  !> Destroys a particle list
  !! @param particlelist Particle List
  RECURSIVE SUBROUTINE DestroyParticleList(particlelist)
    TYPE(Particlelistdef), POINTER :: particlelist
    IF (.NOT. ASSOCIATED(particlelist))  RETURN
    IF (ASSOCIATED(particlelist%next)) CALL DestroyParticleList(particlelist%next)
    IF (ASSOCIATED(particlelist%self)) CALL DestroyParticle(particlelist%self)
    DEALLOCATE(particlelist)
    NULLIFY(particlelist)
  END SUBROUTINE DestroyParticleList

  !> Counts the number of particles
  !! @param particlelist Particle List
  FUNCTION ParticleCount(particlelist)
    INTEGER :: ParticleCount
    TYPE(ParticleListDef), POINTER :: particlelist, temp
    ParticleCount=0

    temp=>particlelist
    DO WHILE (ASSOCIATED(temp))
       IF (ASSOCIATED(temp%self))  ParticleCount = ParticleCount + 1
       temp => temp%next
    END DO
  END FUNCTION ParticleCount

  !> Adds a New local sink particle
  !! @param Particle New local particle
  SUBROUTINE AddNewSinkParticle(Particle)
    TYPE(ParticleDef), POINTER :: Particle
    CALL AddParticleToList(Particle, NewSinkParticles, LastNewSinkParticle)
!    CALL PrintParticle(Particle)
  END SUBROUTINE AddNewSinkParticle

  !> Adds a new global particle
  !! @param Particle New global particle
  SUBROUTINE AddSinkParticle(Particle)
    TYPE(ParticleDef), POINTER :: Particle
    CALL AddParticleToList(Particle, SinkParticles, LastSinkParticle)
    NrSinkParticles=NrSinkParticles+1
    CALL PrintParticle(Particle)
    lParticles=.true.
    lSinkParticles=.true.
  END SUBROUTINE AddSinkParticle

  !> Outputs particle list in a pretty format
  !! @param ParticleList list of particles to print
  SUBROUTINE PrintParticleList(ParticleList)
    TYPE(ParticleListDef), POINTER :: ParticleList
    TYPE(ParticleListDef), POINTER :: TempParticleList
    TempParticleList=>ParticleList

    DO WHILE (ASSOCIATED(TempParticleList))
       CALL PrintParticle(TempParticleList%self)
       TempParticleList=>TempParticleList%next
    END DO

  END SUBROUTINE PrintParticleList

  !> Outputs a single particle in a pretty format
  !! @param Particle particle to print
  SUBROUTINE PrintParticle(Particle)
    TYPE(ParticleDef) :: Particle
    IF (MPI_ID == 0) THEN
       write(*,'(A,E18.6)') "Mass = ", Particle%Q(1)
       write(*,'(A,3E18.6)') "Position = ", Particle%xloc(1:nDim)
       write(*,'(A,3E18.6)') "Velocity = ", Particle%Q(imom(1:nDim))
!       write(*,'(A,3E18.6)') "Angular Momentum = ", Particle%J(1:nAngularMomentum)
!       write(*,'(A,20E18.6)') "Moments = ", Particle%moments
!       write(*,'(A,3E18.6)') "Gas acceleration = ", Particle%gas_accel(1:nDim)
!       write(*,'(A,20E18.6)') "Particle%Q", Particle%Q(1:NrHydroVars)
    END IF
  END SUBROUTINE PrintParticle


  !> Calculate Binding energy of cell to particle \f$E=(\mathbf{v}-\mathbf{v}_{Sink})^2+\phi_{Control Gas} + \phi_{Sink} \f$ where \f$ \phi_{Sink}= \frac{G\left(M_{Sink}+M_{Control}\right)}{gs(|\mathbf{x}-\mathbf{x}_{sink}|)} \f$
  !! @param Particle Particle object
  !! @param pos Position of cell center
  !! @param q cell fields
  !! @param r distance from cell to particle
  FUNCTION GetBindingEnergy(Particle, pos, q, r)
    REAL(KIND=qPREC) :: GetBindingEnergy
    TYPE(ParticleDef) :: Particle
    REAL(KIND=qPREC), DIMENSION(:) :: pos
    REAL(KIND=qPREC), DIMENSION(:) :: q
    REAL(KIND=qPREC) :: r
    !      r=sqrt(sum((pos(1:nDim)-Particle%xloc(1:nDim))**2))
    IF (nDim == 3 .OR. iCylindrical /= NOCYL) THEN
       GetBindingEnergy=SUM((q(m_low:m_high)/q(1)-Particle%Q(m_low:m_high))**2)-ScaleGrav*(Particle%Q(1)+Particle%moments(1))/r
    ELSE
       GetBindingEnergy=SUM((q(m_low:m_high)/q(1)-Particle%Q(m_low:m_high))**2)+2d0*ScaleGrav*(Particle%Q(1)+Particle%moments(1))*log(r/R2DEff)
    END IF
  END FUNCTION GetBindingEnergy


  !> Returns the number of sink particles attached to the SinkParticles list.
  INTEGER FUNCTION Particle_CountSinkParticles()

    TYPE(ParticleListDef), POINTER :: part_list
    INTEGER :: working_count


    part_list => SinkParticles
    working_count = 0

    DO WHILE (ASSOCIATED(part_list))
       working_count = working_count + 1
       part_list => part_list%next
    END DO

    Particle_CountSinkParticles = working_count

  END FUNCTION Particle_CountSinkParticles


  !> Initialize the particle datasets within a Chombo file.
  !! @param chandle An active Chombo handle.
  SUBROUTINE Particle_InitChomboDatasets(chandle)

    USE ChomboDeclarations, ONLY: ChomboHandle
    USE HDF5Declarations, ONLY: Initialize_HDF5_Dataset_Int, Initialize_HDF5_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: i
    CHARACTER(LEN=30) :: parttracername
    IF (.NOT. ASSOCIATED(chandle)) THEN
       PRINT *, "Particle_InitChomboDatasets error::chombo handle not associated."
       STOP
    END IF

    ! Position coordinates
    CALL Initialize_HDF5_Dataset_Double("position_x", chandle%particle_group_id, NrSinkParticles)
    CALL Initialize_HDF5_Dataset_Double("position_y", chandle%particle_group_id, NrSinkParticles)
    CALL Initialize_HDF5_Dataset_Double("position_z", chandle%particle_group_id, NrSinkParticles)

    ! Particle mass
    CALL Initialize_HDF5_Dataset_Double("P_mass", chandle%particle_group_id, NrSinkParticles)

    ! velocity
    CALL Initialize_HDF5_Dataset_Double("P_vx", chandle%particle_group_id, NrSinkParticles)
    CALL Initialize_HDF5_Dataset_Double("P_vy", chandle%particle_group_id, NrSinkParticles)
    IF (ivz /= 0)  CALL Initialize_HDF5_Dataset_Double("P_vz", chandle%particle_group_id, NrSinkParticles)

    IF (iE /= 0)  CALL Initialize_HDF5_Dataset_Double("P_E", chandle%particle_group_id, NrSinkParticles)
    ! Creates tags for tracer variables.
    DO i = 1, NrTracerVars !NrCons+1,NrHydroVars
       write(parttracername,'(A2,A)') 'P_', trim(TracerName(i))
       CALL Initialize_HDF5_Dataset_Double(trim(parttracername),chandle%particle_group_id, NrSinkParticles)
    END DO

    ! Angular momentum
    CALL Initialize_HDF5_Dataset_Double("P_Jx", chandle%particle_group_id, NrSinkParticles)
    CALL Initialize_HDF5_Dataset_Double("P_Jy", chandle%particle_group_id, NrSinkParticles)
    CALL Initialize_HDF5_Dataset_Double("P_Jz", chandle%particle_group_id, NrSinkParticles)


    ! Particle radius in cells
    CALL Initialize_HDF5_Dataset_Int("P_radius", chandle%particle_group_id, NrSinkParticles)

    ! A converted logical flag, indicating that the particle is free to move.
    CALL Initialize_HDF5_Dataset_Int("P_lFixed", chandle%particle_group_id, NrSinkParticles)

    ! A converted logical flag, indicating that the particle is free to move.
    CALL Initialize_HDF5_Dataset_Int("P_iAccrete", chandle%particle_group_id, NrSinkParticles)

    ! The cell buffer around each level.
    CALL Initialize_HDF5_Dataset_Int("Buffer", chandle%particle_group_id, NrSinkParticles * (MaxDepth + 1))

    ! A tag for the particle acceleration routine.
    CALL Initialize_HDF5_Dataset_Int("point_gravity_id", chandle%particle_group_id, NrSinkParticles)

  END SUBROUTINE Particle_InitChomboDatasets

  !> Writes the data from a single particle to a chombo file.  Also increments the chombo handle's particle offset.
  !! @param chandle An active Chombo handle.
  !! @param particle A particle object.
  SUBROUTINE Particle_WriteObjectToChomboFile(chandle, particle)

    USE ChomboDeclarations, ONLY: ChomboHandle
    USE PointGravitySrc, ONLY: PointGravityDef
    USE HDF5Declarations, ONLY: Write_Slab_To_Dataset_Int, Write_Slab_To_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle
    TYPE(ParticleDef), POINTER :: particle
    INTEGER :: i
    CHARACTER(LEN=30) :: parttracername

    IF (.NOT. ASSOCIATED(chandle)) THEN
       PRINT *, "Particle_WriteObjectToChomboFile error::Chombo handle not associated."
       STOP
    END IF

    IF (.NOT. ASSOCIATED(particle)) THEN
       PRINT *, "Particle_WriteObjectToChomboFile error::particle not associated."
       STOP
    END IF

    ! Write position components to datasets.  Be sure to index the xloc array using the i:i notation, as
    ! this passes a 1-element array to the subroutine instead of a scalar.
    CALL Write_Slab_To_Dataset_Double("position_x", chandle%particle_group_id, particle%xloc(1:1), chandle%particle_offset)		  
    CALL Write_Slab_To_Dataset_Double("position_y", chandle%particle_group_id, particle%xloc(2:2), chandle%particle_offset)		  
    CALL Write_Slab_To_Dataset_Double("position_z", chandle%particle_group_id, particle%xloc(3:3), chandle%particle_offset)		  

    ! Write mass to dataset.
    CALL Write_Slab_To_Dataset_Double("P_mass", chandle%particle_group_id, (/ particle%Q(1) /), chandle%particle_offset)		  

    ! Write velocity components to datasets.
    CALL Write_Slab_To_Dataset_Double("P_vx", chandle%particle_group_id, particle%Q(ivx:ivx), chandle%particle_offset)		  
    CALL Write_Slab_To_Dataset_Double("P_vy", chandle%particle_group_id, particle%Q(ivy:ivy), chandle%particle_offset)		  
    IF (ivz /= 0) THEN
       CALL Write_Slab_To_Dataset_Double("P_vz", chandle%particle_group_id, particle%Q(ivz:ivz), chandle%particle_offset)		  
    END IF

    IF (iE /= 0)  CALL Write_Slab_To_Dataset_Double("P_E", chandle%particle_group_id, particle%Q(iE:iE),chandle%particle_offset)
    ! Creates tags for tracer variables.
    DO i = 1, NrTracerVars !NrCons+1,NrHydroVars
       write(parttracername,'(A2,A)') 'P_', trim(TracerName(i))
       CALL Write_Slab_To_Dataset_Double(trim(partTracerName),chandle%particle_group_id, particle%Q(nTracerLo-1+i:nTracerLo-1+i),chandle%particle_offset)
    END DO

    ! Write angular momentum components to datasets.
    CALL Write_Slab_To_Dataset_Double("P_Jx", chandle%particle_group_id, particle%J(1:1), chandle%particle_offset)		  
    CALL Write_Slab_To_Dataset_Double("P_Jy", chandle%particle_group_id, particle%J(2:2), chandle%particle_offset)		  
    CALL Write_Slab_To_Dataset_Double("P_Jz", chandle%particle_group_id, particle%J(3:3), chandle%particle_offset)		  

    CALL Write_Slab_To_Dataset_Int("P_radius", chandle%particle_group_id, (/ particle%radius /), chandle%particle_offset)		  

    CALL Write_Slab_To_Dataset_Int("P_lFixed", &
         chandle%particle_group_id, &
         (/ BoolToInt(particle%lFixed) /), &
         chandle%particle_offset)

    CALL Write_Slab_To_Dataset_Int("P_iAccrete", &
         chandle%particle_group_id, &
         (/ (particle%iAccrete) /), &
         chandle%particle_offset)


    ! Write the whole buffer array to the appropriate dataset.
    CALL Write_Slab_To_Dataset_Int("Buffer", chandle%particle_group_id, particle%Buffer, chandle%particle_offset * (MaxDepth + 1))

    IF (ASSOCIATED(Particle%PointGravityObj)) THEN
       cALL Write_Slab_To_Dataset_Int("point_gravity_id", &
            chandle%particle_group_id, &
            (/ particle%PointGravityObj%id /), &
            chandle%particle_offset)
    ELSE
       cALL Write_Slab_To_Dataset_Int("point_gravity_id", &
            chandle%particle_group_id, &
            (/ -1 /), &
            chandle%particle_offset)
    END IF
    chandle%particle_offset = chandle%particle_offset + 1

  END SUBROUTINE Particle_WriteObjectToChomboFile

  !> Reads the data for one particle from an open Chombo file and increments the handle offset.
  !! @param chandle An open Chombo file handle.
  !! @param particle A non-null pointer to a ParticleDef object.  This object will be filled with data from the file.
  SUBROUTINE Particle_ReadObjectFromChombo(chandle, particle)

    USE ChomboDeclarations, ONLY: ChomboHandle
    USE HDF5Declarations, ONLY: Read_Slab_From_Dataset_Int, Read_Slab_From_Dataset_Double

    TYPE(ChomboHandle), POINTER :: chandle
    TYPE(ParticleDef), POINTER :: particle

    INTEGER, DIMENSION(1), TARGET :: int_buffer_array
    REAL(KIND=qPrec), DIMENSION(1), TARGET :: dbl_buffer_array
    INTEGER, DIMENSION(:), POINTER :: int_buffer
    REAL(KIND=qPrec), DIMENSION(:), POINTER :: dbl_buffer
    INTEGER :: i
    CHARACTER(LEN=30) :: parttracername

    ! Fail if a null pointer is passed in for a ChomboHandle.
    IF (.NOT. ASSOCIATED(chandle)) THEN
       PRINT "('Particle_ReadObjectFromChombo() error::particle number ', i4, ' object not created.')", &
            chandle%particle_offset
       STOP
    END IF

    ! Fail if a null pointer is passed in for a particle object.
    IF (.NOT. ASSOCIATED(particle)) THEN
       PRINT "('Particle_ReadObjectFromChombo() error::particle number ', i4, ' object not created.')", &
            chandle%particle_offset
       STOP
    END IF

    int_buffer => int_buffer_array
    dbl_buffer => dbl_buffer_array

    int_buffer = 0
    dbl_buffer = 0.d0

    ! Read in the position coordinates for this particle.  This takes three reads from the file, because
    ! the coordinates stored in three separate arrays.
    CALL Read_Slab_From_Dataset_Double("position_x", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%xloc(1) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("position_y", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%xloc(2) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("position_z", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%xloc(3) = dbl_buffer(1)

    ! Read in the particle mass.
    CALL Read_Slab_From_Dataset_Double("P_mass", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%Q(1) = dbl_buffer(1)

    ! Read in the velocity for this particle.  This takes three reads from the file, because
    ! the angular momentum components are stored in three separate arrays.
    CALL Read_Slab_From_Dataset_Double("P_vx", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%Q(ivx) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("P_vy", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%Q(ivy) = dbl_buffer(1)

    IF (ivz /= 0) THEN
       CALL Read_Slab_From_Dataset_Double("P_vz", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
       particle%Q(ivz) = dbl_buffer(1)
    END IF

    IF (iE /= 0)  THEN
       CALL Read_Slab_From_Dataset_Double("P_E", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
       particle%Q(iE) = dbl_buffer(1)
    END IF

    ! Creates tags for tracer variables.
    DO i = 1, NrTracerVars !NrCons+1,NrHydroVars
       write(parttracername,'(A2,A)') 'P_', trim(TracerName(i))
       CALL Read_Slab_From_Dataset_Double(trim(partTracerName), chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
       particle%Q(nTracerlo-1+i)=dbl_buffer(1)
    END DO


    ! Read in the angular momentum for this particle.  This takes three reads from the file, because
    ! the angular momentum components are stored in three separate arrays.
    CALL Read_Slab_From_Dataset_Double("P_Jx", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%J(1) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("P_Jy", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%J(2) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Double("P_Jz", chandle%particle_group_id, dbl_buffer, chandle%particle_offset)
    particle%J(3) = dbl_buffer(1)

    CALL Read_Slab_From_Dataset_Int("P_radius", chandle%particle_group_id, int_buffer, chandle%particle_offset)
    particle%radius = int_buffer(1)

    ! The ALL mask converts the integer value from the chombo file into a logical value.
    CALL Read_Slab_From_Dataset_Int("P_lFixed", chandle%particle_group_id, int_buffer, chandle%particle_offset)
    particle%lFixed = ALL(int_buffer /= 0)

    CALL Read_Slab_From_Dataset_Int("P_iAccrete", chandle%particle_group_id, int_buffer, chandle%particle_offset)
    particle%iAccrete = int_buffer(1)


    ! Most of the objects in particle are scalars, or their components are stored as scalars.  Particle%Buffer,
    ! on the other hand, is an array whose components are stored as a contiguous array.  Consequently, we
    ! can pass particle%Buffer directly to the Read_Slab function.  The buffer is of size MaxDepth + 1, hence the
    ! multiplier for the particle offset.
    CALL Read_Slab_From_Dataset_Int("Buffer", chandle%particle_group_id, particle%Buffer, &
         chandle%particle_offset * (MaxDepth + 1))

    ! Retrieve the ID of the point gravity object associated with this particle, then find that object
    ! and point the particle's PointGravityObj toward it.
    CALL Read_Slab_From_Dataset_Int("point_gravity_id", chandle%particle_group_id, int_buffer, chandle%particle_offset)


    IF (int_buffer(1) > -1) THEN
       CALL FindPointGravityObject(int_buffer(1), particle%PointGravityObj)
    END IF

    chandle%particle_offset = chandle%particle_offset + 1

  END SUBROUTINE Particle_ReadObjectFromChombo

  FUNCTION GetParticleBounds(Particle)
    TYPE(ParticleDef) :: Particle
    REAL(KIND=qPREC), DIMENSION(3,2) :: GetParticleBounds
    GetParticleBounds=GxBounds
    GetParticleBounds(1:nDim,1)=Particle%xloc(1:nDim)-Particle%radius*sink_dx
    GetParticleBounds(1:nDim,2)=Particle%xloc(1:nDim)+Particle%radius*sink_dx
  END FUNCTION GetParticleBounds


  FUNCTION GetParticlePos(Particle, t)
    TYPE(ParticleDef) :: Particle
    REAL(KIND=qPREC) :: t
    REAL(KIND=qPREC) :: GetParticlePos(3)
    GetParticlePos=Particle%PointGravityObj%x0+Particle%PointGravityObj%v0*(t-Particle%PointGravityObj%t0)

  END FUNCTION GetParticlePos

END MODULE ParticleDeclarations


