!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    particle_info_ops.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file particle_info_ops.f90
!! @brief Main file for module ParticleInfoOps

!> @defgroup ParticleInfoOps Particle Info Ops
!! @brief Module containing particle operations for info structures
!! @ingroup Particles

!> Module containing particle operations for info structures
!! @ingroup ParticleInfoOps
MODULE ParticleInfoOps
   USE ParticleDeclarations
   USE DataDeclarations
   USE GlobalDeclarations
   USE CommonFunctions
   USE PointGravitySrc
   USE Bondi
   IMPLICIT NONE
   PUBLIC CalcMoment, CheckForNewParticle, DoAccretion, GetGasForce, SinkSetErrFlag, BondiAccretionRate

   PRIVATE
CONTAINS

   !> @name Main Info Routines
   !! @{


   !> Set err flags around sink particles
   !! @param Info Info structure
   SUBROUTINE SinkSetErrFlag(Info)
      TYPE(InfoDef) :: Info
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(ParticleDef), POINTER :: Particle
      INTEGER, DIMENSION(:,:,:), POINTER :: mSs
      INTEGER :: nOverlaps, nGhost
      INTEGER, DIMENSION(3,2) :: mS
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
      REAL(KIND=qPREC), DIMENSION(3,2) :: xBounds
      INTEGER :: i

      ParticleList=>SinkParticles
      DO WHILE (ASSOCIATED(ParticleList))
         Particle=>ParticleList%self         
         xBounds=spread(particle%xloc,2,2)
         xBounds(1:nDim,1)=particle%xloc(1:nDim)-particle%buffer(Info%level)*levels(Info%level)%dx
         xBounds(1:nDim,2)=particle%xloc(1:nDim)+particle%buffer(Info%level)*levels(Info%level)%dx
         CALL CalcPhysicalOverlaps(Info, xBounds, mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic, 0)
         IF (nOverlaps > 0) THEN
            DO i=1,nOverlaps
               mS=mSs(i,:,:)
               Info%ErrFlag(ms(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):ms(3,2))=1
            END DO
            DEALLOCATE(mSs, offsets)
            NULLIFY(mSs, offsets)
         END IF
         ParticleList=>ParticleList%next
      END DO
   END SUBROUTINE SinkSetErrFlag





   !> Calculates a grids contribution to the moments of Particle
   !! @param Particle Particle
   !! @param Info Info structure
   !! @param lGhost Include contributions from ghost cells?
   SUBROUTINE CalcMoment(Particle, Info, lGhost)
      TYPE(ParticleDef) :: Particle
      TYPE(InfoDef) :: Info
      LOGICAL :: lPeriodic, lGhost
      INTEGER, DIMENSION(:,:,:), POINTER :: mSs
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
      INTEGER :: nOverlaps, nGhost
      INTEGER, DIMENSION(3,2) :: mS
      REAL(KIND=qPREC), DIMENSION(3) :: offset
      INTEGER :: i
      ! For new particles - include ghost cells and don't consider periodic overlaps
      ! For old particles - don't include ghost cells - except at physical boundaries
      IF (lGhost) THEN
         CALL CalcPhysicalOverlaps(Info, GetParticleBounds(Particle), mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic)
      ELSE
         CALL CalcPhysicalOverlaps(Info, GetParticleBounds(Particle), mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic, 0)
      END IF

      !      IF (nOverlaps > 0) write(*,*) "GridParticleOverlaps", nOverlaps, mSs
      IF (nOverlaps > 0) THEN         
         DO i=1,nOverlaps
            mS=mSs(i,:,:)
            offset=offsets(i,:)
            CALL CalcMomentContributions(Info, Particle, mS, offset)
         END DO
         DEALLOCATE(mSs)
         DEALLOCATE(offsets)
         NULLIFY(mSs)
         NULLIFY(offsets)
      END IF
      !      write(*,*) particle%moments(1)
   END SUBROUTINE CalcMoment


   !> Checks a grid for new particles
   !! @param Info Info structure
   SUBROUTINE CheckForNewParticle(Info)
      ! Federrath et al list the following checks:
      ! * Density Threshold (TrueLove Criteria -> JeansLength = sqrt(pi*cs^2/G/rho)=sqrt(pi*gamma*Press/G)/rho > 4 dx or rho < sqrt(pi*gamma*Press/G)/(4 dx) = sqrt(pi*gamma/G)/(4 dx) * sqrt(Press)
      ! * Refinement Check (not necessary as this routine is only called on the maxlevel)
      ! * Converging Flow in each direction
      ! * Minimum in the total Potential
      ! * Jeans Instability Check
      ! * Bound State within control volume
      ! * Proximity check - don't create sink particles that are close to already existing particles
      USE EOS
      INTEGER :: i,j,k,l,m,n,ll,mm,nn,ii,jj,kk
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC), POINTER, DIMENSION(:,:,:,:) :: q
      REAL(KIND=qPREC) :: rhojeans, dx, eth, egrav,mass,ekin,emag
      INTEGER, DIMENSION(3,2) :: ip,ioffset
      LOGICAL :: lsamex, lsamey, lConverging
      INTEGER, DIMENSION(3) :: ir, il, ipos
      REAL(KIND=qPREC), DIMENSION(3) :: pos,mom,poffset
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(ParticleDef), POINTER :: Particle, NewParticle      
      LOGICAL :: lFoundNearbyParticle
      q=>Info%q
      dx=levels(maxlevel)%dx
      ioffset=0
      WHERE (lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=2 !Do two copies...
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)

      !      write(*,*) "JeansFact=", JeansFact,JeansFact*sqrt(press(q(10,10,10,:)))
      DO i=1, Info%mX(1)
         DO j=1, Info%mX(2)
            DO k=1, Info%mX(3)               
               ! * Density Threshold                
               rhoJeans=JeansFact*sqrt(press(q(i,j,k,:)))
               IF (q(i,j,k,1) > rhoJeans) THEN !TrueLove threshold reached
                  ! Check for Proximity to existing particles
                  !                  write(*,'(A5,3I8,A,4E18.5)') "cell", Info%mGlobal(:,1)-1+(/i,j,k/), ' violates density threshhold', rhoJeans, q(i,j,k,1)
                  ipos=(/i,j,k/)
                  pos=Cellpos(Info,i,j,k)
                  particlelist=>SinkParticles
                  lFoundNearbyParticle=.false.
                  DO WHILE (ASSOCIATED(particlelist))
                     particle=>particlelist%self
                     IF (particle%iaccrete == FEDERRATH_ACCRETION .OR. particle%iaccrete == KRUMHOLZ_ACCRETION) THEN
                        !                     write(*,*) "proximity check", sqrt(SUM((pos(1:nDim)-particle%xloc(1:nDim))**2)), sqrt(r_acc2)
                        DO ii=ioffset(1,1),ioffset(1,2)
                           DO jj=ioffset(2,1),ioffset(2,2)
                              DO kk=ioffset(3,1),ioffset(3,2)
                                 pos=Cellpos(Info, i, j, k)+(/ii,jj,kk/)*(GxBounds(:,2)-GxBounds(:,1))                                 
                                 IF (SUM((pos(1:nDim)-particle%xloc(1:nDim))**2) <= r_acc2) THEN
                                    lFoundNearbyParticle=.true.
                                 END IF
                              END DO
                           END DO
                        END DO
                        IF (lFoundNearbyParticle) EXIT
                     END IF
                     particlelist=>particlelist%next
                  END DO

                  IF (.NOT. lFoundNearbyParticle) THEN !Did not find an existing particle...
                     !                     write(*,*) "did not find an existing particle"
                     ip=spread(ipos,2,2)
                     ! Check for converging flow
                     lConverging=.true.
                     ir=ipos
                     il=ipos
                     DO l=1,nDim
                        il(l)=ipos(l)-1 !Left indices
                        ir(l)=ipos(l)+1 !Right indices
                        !div(velocity) = v(ir)-v(il) < 0
                        lConverging=q(ir(1),ir(2),ir(3),imom(l))/q(ir(1),ir(2),ir(3),irho) < &
                             q(il(1),il(2),il(3),imom(l))/q(il(1),il(2),il(3),irho)                        
                        IF (.NOT. lConverging) EXIT
                        il(l)=ipos(l)
                        ir(l)=ipos(l)
                     END DO
                     IF (lConverging) THEN
                        !                                                write(*,*) "converging pass"
                        ! Check for Gravitational Minimum                        
                        ip(1:nDim,1)=ipos(1:nDim)-IR_ACC
                        ip(1:nDim,2)=ipos(1:nDim)+IR_ACC
                        IF (q(ipos(1),ipos(2),ipos(3),1) == MAXVAL(q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1),lControlVolume) .AND. COUNT(q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1)==q(ipos(1),ipos(2),ipos(3),1) .AND. lControlVolume)==1) THEN
                           !                           write(*,*) 'A', q(ipos(1),ipos(2),ipos(3),iPhi)
                           !                           write(*,*) 'B', MAXVAL(q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),iPhi),lControlVolume)
                           !                           write(*,*) 'C', lControlVolume
                           !                                                      write(*,*) "minimum potential pass at ", Info%mGlobal(:,1)-1+ipos! !NrSinkParticles
                           !                                                      write(*,'(9E13.2)') q(ipos(1)-1:ipos(1)+1,ipos(2)-1:ipos(2)+1,ipos(3)-1:ipos(3)+1,iphi)
                           ! Jeans Instability Check                           
                           eth=0
                           egrav=0
                           mass=0
                           mom=0
                           emag=0                           
                           !Sum up the total energies (divided by dV)
                           DO l=ip(1,1),ip(1,2)
                              DO m=ip(2,1),ip(2,2)
                                 DO n=ip(3,1),ip(3,2)
                                    IF (.NOT. lControlVolume(l-ipos(1),m-ipos(2),n-ipos(3))) CYCLE
                                    mass=mass+q(l,m,n,1)
                                    mom(1:nDim)=mom(1:nDim)+q(l,m,n,imom(1:nDim))
                                    eth=eth+1.5d0*Press(q(l,m,n,:))
                                    IF (lMHD) emag=emag+half*SUM(q(l,m,n,iBx:iBz)**2)
                                    DO ll=ip(1,1),ip(1,2)
                                       lsamex=ll==l
                                       DO mm=ip(2,1),ip(2,2)
                                          lsamey=lsamex .AND. mm==m
                                          DO nn=ip(3,1),ip(3,2)
                                             IF (lControlVolume(ll-ipos(1),mm-ipos(2),nn-ipos(3))) THEN
                                                IF (lsamey .AND. nn==n) THEN
                                                   IF (nDim == 3) THEN 
                                                      egrav=egrav-0.94115625*q(l,m,n,1)**2*ScaleGravdV/sink_dx ! Self-energy of a uniform cube... from arxiv.org/pdf/astro-ph/0002496
                                                   ELSE
                                                      egrav=egrav-2.94298464974921*q(l,m,n,1)**2*ScaleGravdV/sink_dx ! Self-energy of a uniform square..
                                                   END IF
                                                ELSE
                                                   IF (nDim == 3 .OR. iCylindrical /= NOCYL) THEN
                                                      egrav=egrav-half*q(l,m,n,1)*q(ll,mm,nn,1)*ScaleGravdV/sqrt(REAL((ll-l)**2+(mm-m)**2+(nn-n)**2))/sink_dx
                                                   ELSE
                                                      ! In 2D everything is bound - nothing can truly escape to infinity since potential goes like ln(r) so normalize potential to go to zero at r ~ box-size
                                                      egrav=egrav+half*q(l,m,n,1)*q(ll,mm,nn,1)*2d0*ScaleGravdV*log(sqrt(REAL((ll-l)**2+(mm-m)**2+(nn-n)**2))*sink_dx/sqrt(product(GxBounds(1:2,2)-GxBounds(1:2,1))))
                                                   END IF
                                                END IF
                                             END IF
                                          END DO
                                       END DO
                                    END DO
                                 END DO
                              END DO
                           END DO
                           IF (abs(egrav) > 2d0*eth+emag) THEN !Jeans unstable  - not sure if this should be 2d0*(eth+emag) instead of 2d0*eth+emag
                              !                              write(*,'(A,6E20.12)') "Jeans unstable", egrav, eth, emag,mom/mass

                              mom=mom/mass !mom is now center of motion
                              ekin=0
                              DO l=ip(1,1),ip(1,2)
                                 DO m=ip(2,1),ip(2,2)
                                    DO n=ip(3,1),ip(3,2)
                                       IF (.NOT. lControlVolume(l-ipos(1),m-ipos(2),n-ipos(3))) CYCLE
                                       Ekin=Ekin+half*q(l,m,n,1)*SUM((q(l,m,n,imom(1:nDim))/q(l,m,n,1)-mom(1:nDim))**2)
                                    END DO
                                 END DO
                              END DO
                              IF ((Egrav+Eth+Ekin+Emag < 0d0).and.(q(i,j,k,1)-0.25**2*pi*(soundspeed(q(i,j,k,:)))**2/(ScaleGrav*dx**2)>0d0)) THEN
                                 !                                 write(*,'(A,5E20.12)') "Bound System", Egrav, Eth, Emag, Ekin, Egrav+Eth+Ekin+Emag
                                 ! We made a new particle
                                 print*, "creating new particle..."
                                 NULLIFY(NewParticle)
                                 CALL CreateParticle(NewParticle)
                                 NewParticle%xloc(1:nDim)=pos(1:nDim)
                                 CALL CalcMoment(NewParticle, Info,.true.)
                                 CALL AddNewSinkParticle(NewParticle)
                              ELSE
                                 !                                 write(*,'(A,5E20.12)') "UnBound System", Egrav, Eth, Emag, Ekin, Egrav+Eth+Ekin+Emag
                              END IF
                           ELSE
                              !                              write(*,'(A,4E20.12)') "Jeans stable", egrav, eth, emag
                           END IF
                        END IF
                     ELSE
                        !                        write(*,*) "converging failed"
                     END IF
                  ELSE
                     !                     write(*,*) "near an existing particle"
                  END IF
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE CheckForNewParticle


   SUBROUTINE BondiAccretionRate(Particle, Info)
      TYPE(InfoDef) :: Info
      TYPE(ParticleDef) :: Particle
      INTEGER :: ipos(3), mb(3,2), i, j, k
      REAL(KIND=qPREC) :: v_inf, c_inf, r_bh, r_kernel, r, w, rho_inf, w_sum,dipos(3)
      IF (ParticleInGrid(Particle, Info, ipos)) THEN
         c_inf=soundspeed(Info%q(ipos(1),ipos(2),ipos(3),:))
		 !print*, "11111"
		 if (particle%q(1).eq.0.0) THEN
		 	!print*, "22222"
         	v_inf=0.0
 	        r_bh=ScaleGrav*info%q(ipos(1),ipos(2),ipos(3),1)*sink_dv/(c_inf**2)
		 ELSE
			v_inf=sqrt(sum((Info%q(ipos(1),ipos(2),ipos(3),imom(1:nDim))/Info%q(ipos(1),ipos(2),ipos(3),1)-Particle%q(imom(1:nDim))/Particle%q(1))**2))
 	        r_bh=ScaleGrav*particle%q(1)/(v_inf**2+c_inf**2)
		END IF
         r_kernel=min(max(r_bh, sink_dx/4d0), IR_ACC*sink_dx/2d0)
         mb=1
         mb(1:nDim,1)=ipos(1:nDim)-IR_ACC
         mB(1:nDim,2)=ipos(1:nDim)+IR_ACC
         w_sum=0d0
!         rho_inf=0d0
         dipos=(Particle%xloc-Info%xBounds(:,1))/sink_dx+half
         DO i=mb(1,1), mB(1,2)
            DO j=mB(2,1), mB(2,2)
               DO k=mb(3,1), mB(3,2)
                  r=sqrt(sum((real((/i,j,k/),8)-dipos)**2 ))*sink_dx
                  IF (r <= REAL(IR_ACC)*sink_dx) THEN
                     IF (iCylindrical /= NOCYL) THEN
                        w=exp(-(r/r_kernel)**2)*abs(Info%xBounds(1,1)+(real(i,8)-half)*levels(Info%level)%dx)
                     ELSE
                        w=exp(-(r/r_kernel)**2)
                     END IF
                     w_sum=w_sum+w
!                     rho_inf=rho_inf+w*Info%q(i,j,k,1)
                  END IF
               END DO
            END DO
         END DO
!         rho_inf=rho_inf/w_sum/BH_alpha(1.2*sink_dx/r_bh)
         Particle%AccretionRate=4d0*Pi*r_bh**2*sqrt(Bondi_lambda2*c_inf**2+v_inf**2)/BH_alpha(1.2*sink_dx/r_bh)/(w_sum*sink_dv) !normalized
         Particle%Bondi_kernel=r_kernel
!         write(*,*) 'ParticleAccretionRate=', Particle%AccretionRate, 1.2*sink_dx/r_bh
!         write(*,'(A,E24.15)') 'r_bh=', r_bh
!         write(*,'(A,E24.15)') 'r_kernel', r_kernel
!         write(*,'(A,E24.15)') 'c_inf', c_inf
!         write(*,'(A,2E24.15)') 'rho_inf', rho_inf, rho_inf*BH_alpha(1.2*sink_dx/r_bh)
!         write(*,'(A,E24.15)') 'v_inf', v_inf

!         write(*,'(A,E24.15)') 'Accretion Coeff', Particle%AccretionRate
!         write(*,'(A,E24.15)') 'Accretion Rate', 4d0*Pi*r_bh**2*rho_inf*sqrt(Bondi_lambda2*c_inf**2+v_inf**2)
!         write(*,*) 'rate/coeff= ', 4d0*Pi*r_bh**2*rho_inf*sqrt(Bondi_lambda2*c_inf**2+v_inf**2)/Particle%AccretionRate
      END IF
    END SUBROUTINE BondiAccretionRate


   !> Calculate accretion onto sink particles - including ghost regions - but don't accumulate contributions from ghost zones
   !! @param Info Info structure
   SUBROUTINE DoAccretion(Info) 
      TYPE(InfoDef) :: Info
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(ParticleDef), POINTER :: Particle
      ParticleList=>SinkParticles
      DO WHILE (ASSOCIATED(ParticleList))         
         Particle=>ParticleList%self
         SELECT CASE(Particle%iAccrete)
         CASE(FEDERRATH_ACCRETION)
            CALL SelfGravAccretion(Info, Particle)
         CASE(KRUMHOLZ_ACCRETION)
	     	CALL BondiAccretion(Info, Particle)
         END SELECT
         ParticleList=>ParticleList%next
      END DO
   END SUBROUTINE DoAccretion

   SUBROUTINE SelfGravAccretion(Info, Particle)
      USE EOS
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: q
      INTEGER, DIMENSION(3,2) :: ip
      INTEGER, DIMENSION(:,:,:), POINTER :: mSs
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
      INTEGER :: nParticles
      INTEGER :: nOverlaps
      INTEGER :: i,j,k,l,iOverlap,rmbc,ii,jj,kk,ioffset(3,2)
      INTEGER, DIMENSION(3) :: ipos
      REAL(KIND=qPREC), DIMENSION(3) :: pos, offset, poffset
      REAL(KIND=qPREC) :: rhoJeans, minCellBindingEnergy, minRadius, r, CellBindingEnergy,mass,rmass(3),p
      TYPE(ParticleDef), POINTER :: Particle, MostBoundParticle,ClosestParticle,TempParticle
      TYPE(ParticleListDef), POINTER :: ParticleList, TempParticleList
      LOGICAL :: lAccrete
      ioffset=0
      WHERE (lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=2 !Do two copies...
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)

      q=>Info%q
      !      ip(nDim+1:3,:)=1
      !      ip(1:nDim,1)=Info%mGlobal(1:nDim,1)-IR_ACC
      !      ip(1:nDim,2)=Info%mGlobal(1:nDim,2)+IR_ACC
      !      rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)

      !      CALL CreateOverlapList(ip, GridParticles, ips)
      !      nParticles=size(ips,1)

      !      DO iParticle=1, nParticles

         !         write(*,*) "checking on particle", loc(particle)
         CALL CalcPhysicalOverlaps(Info, GetParticleBounds(Particle), mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic)
         !         write(*,*) "found overlaps", nOverlaps
         !        write(*,*) Info%xbounds
         !         write(*,*) Particle%xloc        
         !         DO iOverlap=1,nOverlaps
         !            write(*,*) mSs(iOVerlap,:,:)
         !         END DO
         DO iOverlap=1,nOverlaps
            offset=offsets(iOverlap,:)
            ip=mSs(iOverlap,:,:) !overlap bounds for this particle
            !         CALL CalcOverlaps(Info%mGlobal, Particle%xloc, ip)

            !         ALLOCATE(potentialoverlaps(ParticleCount(ParticleOverlaps)))
            !                        write(*,*) "checking cells", ip

            DO k=ip(3,1),ip(3,2)
               DO j=ip(2,1),ip(2,2)
                  DO i=ip(1,1),ip(1,2)
                     IF (q(i,j,k,1) <= MinDensity) CYCLE
                     p=Press(q(i,j,k,:))
                     IF (p < q(i,j,k,1)*MinTemp) CYCLE
                     rhoJeans=(JeansFact)**2*p/q(i,j,k,1)
                     !                        rhoJeans=JeansFact*sqrt(press(q(i,j,k,:)))
                     IF (q(i,j,k,1) > rhoJeans) THEN !TrueLove threshold reached
                        ipos=(/i,j,k/)                                          
                        !                                                write(*,*) "cell at",(/i,j,k/), "violates truelove", rhoJeans,press(q(i,j,k,:)),q(i,j,k,1)                                                
                        !                           write(*,*) rhoJeans, q(i,j,k,1)
                        pos=(REAL(ipos)-half)*sink_dx + Info%xbounds(:,1)-offset                     
                        r=sqrt(sum((pos(1:nDim)-Particle%xloc(1:nDim))**2))
                        IF (r <= r_acc) THEN

                           !For this cell - check to see if it is most tightly bound to current particle
                           !Need Binding Energy to be negative otherwise cell is not bound to any particle
                           MinCellBindingEnergy=0d0
                           NULLIFY(MostBoundParticle, ClosestParticle)
                           minradius=huge(1d0)
                           TempParticleList=>SinkParticles
                           DO WHILE (ASSOCIATED(TempParticleList))
                              TempParticle=>TempParticleList%self                           
                              IF (TempParticle%iAccrete==FEDERRATH_ACCRETION .OR. TempParticle%iAccrete==KRUMHOLZ_ACCRETION) THEN
                                 DO ii=ioffset(1,1),ioffset(1,2)
                                    DO jj=ioffset(2,1),ioffset(2,2)
                                       DO kk=ioffset(3,1),ioffset(3,2)
                                          pOffSet=(/ii,jj,kk/)*(GxBounds(:,2)-GxBounds(:,1))
                                          r=sqrt(sum((pos(1:nDim)+pOffSet(1:nDim)-TempParticle%xloc(1:nDim))**2))
                                          CellBindingEnergy = GetBindingEnergy(Tempparticle, pos+pOffset, q(i,j,k,:),r)  
                                          IF (SUM((pos(1:nDim)+pOffset(1:nDim)-TempParticle%xloc(1:nDim))*(q(i,j,k,imom(1:nDim))/q(i,j,k,1)-TempParticle%Q(imom(1:nDim)))) <= 0) THEN                         
                                             IF (CellBindingEnergy < MinCellBindingEnergy) THEN
                                                MostBoundParticle=>TempParticle
                                                MinCellBindingEnergy=CellBindingEnergy
                                             END IF
                                          END IF
                                          IF (r < minradius) THEN
                                             ClosestParticle=>TempParticle
                                             minradius=r
                                          END IF
                                       END DO
                                    END DO
                                 END DO
                              END IF
                              TempParticleList=>TempParticleList%next
                           END DO
                           lAccrete=.false.
                           !                           write(*,*) minradius, r_inner_acc
                           IF (minradius < r_inner_acc) THEN !Don't check for most bound state - since this cell is too close to a particle
                              lAccrete = associated(ClosestParticle, target=Particle) !Automatically accrete from this particle
                           ELSE
                              IF (associated(MostBoundParticle, target=Particle)) THEN !this cell is most tightly bound to this particle
                                 !Check to see if it's radial velocity is toward the particle 
                                 !Already did this in determining whether it was bound
                                 lAccrete = .true. !(SUM((q(i,j,k,imom(1:nDim))/q(i,j,k,1)-Particle%vel(1:nDim)) * (pos(1:nDim)-Particle%xloc(1:nDim))) < 0)
                              END IF
                           END IF
                           IF (lAccrete) THEN
                              !                                                         write(*,*) "accreting from cell", i,j,k
                              IF (ALL(ipos(1:nDim) >= 1) .AND. ALL(ipos(1:nDim) <= Info%mX(1:nDim))) THEN
                                 Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)+Info%q(i,j,k,1:NrHydroVars)*sink_dv

                                 mass=sink_dv*(q(i,j,k,1)-rhoJeans)
                                 rmass(1:nDim)=mass*(pos(1:nDim)-Particle%xloc(1:nDim))
                                 Particle%drmass=Particle%drmass+rmass
                                 IF (nDim == 2) THEN
                                    Particle%dJ=Particle%dJ+Cross2D(rmass(1:2),q(i,j,k,imom(1:2))/q(i,j,k,1))
                                 ELSEIF (nDim == 3) THEN
                                    Particle%dJ(1:nDim)=Particle%dJ+Cross3D(rmass,q(i,j,k,imom(1:3))/q(i,j,k,1))
                                 END IF
                              END IF

                              IF (iE /= 0) THEN
                                 IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,iBx:iBz)**2) !Subtract off magnetic energy                                 
                                 q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Subtract off kinetic energy                              
                              END IF
                              q(i,j,k,1:NrHydroVars)=q(i,j,k,1:NrHydroVars)*rhoJeans/q(i,j,k,1)  !Everything else should be accreted proportionally to density...
                              IF (lMHD) THEN
                                 q(i,j,k,iBx)=.5*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
                                 q(i,j,k,iBy)=.5*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
                                 IF (nDim == 3) q(i,j,k,iBz)=.5*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
                              END IF
                              IF (iE /= 0) THEN
                                 q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Add in kinetic energy
                                 IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,iBx:iBz)**2) !Add magnetic energy back in.
                              END IF
                              IF (ALL(ipos(1:nDim) >= 1) .AND. ALL(ipos(1:nDim) <= Info%mX(1:nDim)))  &
                                   Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)-Info%q(i,j,k,1:NrHydroVars)*sink_dv
                              
                           END IF

                        END IF
                     END IF
                  END DO
               END DO
            END DO
         END DO
         IF (nOverlaps > 0) THEN
            DEALLOCATE(mSs, offsets)
            NULLIFY(mSs, offsets)
         END IF

   END SUBROUTINE SelfGravAccretion


   SUBROUTINE BondiAccretion(Info,Particle)
      TYPE(ParticleDef) :: Particle
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC) :: GM, q_fact, sKE, r, r2, esp, jsp2, sie, r_surf, dx, dz, w, drho, mass
      REAL(KIND=qPREC), DIMENSION(3) :: xpos, pos, v, v_rel, v_r, v_phi, sample_fact, rmass
      INTEGER :: i, j, k, ii, jj, kk,  iOverlap, nOverlaps
      INTEGER, DIMENSION(3) :: ipos, sample_res, iipos
      INTEGER, DIMENSION(3,2) :: mB, mS
      INTEGER, DIMENSION(3) :: offset
      INTEGER, DIMENSION(:,:), POINTER :: offsets
      INTEGER, DIMENSION(:,:,:), POINTER :: mSs, count
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: q
      ipos = PosCell(Info, Particle%xloc(1), Particle%xloc(2), Particle%xloc(3))
      mB=spread(Info%mGlobal(:,1)-1+ipos,2,2)
      mB(:,1)=Info%mGlobal(:,1)-1+ipos(:)-IR_ACC
      mb(:,2)=Info%mGlobal(:,1)-1+ipos(:)+IR_ACC

      CALL CalcCellOverlaps(Info, mB, mSs, nOverlaps, offsets, iEVERYWHERE, lHydroPeriodic) 
      IF (nOverlaps > 0) THEN
         q=>Info%q
         GM=Particle%q(1)*ScaleGrav
         dx=sink_dx
         dz=merge(dx, 0d0, nDim == 3)
         sample_res=1
         sample_fact=0d0
         sample_res(1:nDim)=8
         sample_fact(1:nDim)=1d0/REAL(sample_res(1:nDim),8)
         q_fact=product(sample_fact(1:nDim))
         r_surf=.25*sink_dx
         DO iOverlap=1,nOverlaps
            offset=offsets(iOverlap,:)
            mS=mSs(iOverlap,:,:) !overlap bounds for this particle
            IF(particle%q(1).eq.0d0) THEN
               DO k=mS(3,1),mS(3,2)
                  xpos(3)=Info%xBounds(3,1)+(real(k+offset(3),8)-half)*dz
                  iipos(3)=k
                  DO j=mS(2,1),mS(2,2)
                     xpos(2)=Info%xBounds(2,1)+(real(j+offset(2),8)-half)*dx
                     iipos(2)=j
                     DO i=mS(1,1),mS(1,2)
                        xpos(1)=Info%xBounds(1,1)+(real(i+offset(1),8)-half)*dx
                        iipos(1)=i
                        IF (all(xpos(1:nDim)+half*dx>Particle%xloc(1:nDim)).and.all(xpos(1:nDim)-half*dx<=Particle%xloc(1:nDim))) THEN
                           drho = Info%q(i,j,k,1)-0.25**2*pi*(soundspeed(Info%q(i,j,k,:)))**2/(ScaleGrav*dx**2)
                        END IF
                        !PRINT*, "drho = ", drho
                        IF (drho > 0d0) THEN
                           IF (ALL(iipos(1:nDim) >= 1) .AND. ALL(iipos(1:nDim) <= Info%mX(1:nDim))) THEN
                              IF (iCylindrical == NOCYL) THEN
                                 Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)+q(i,j,k,1:NrHydroVars)*sink_dv
                                 mass=sink_dv*(drho)
                                 rmass(1:nDim)=mass*(xpos(1:nDim)-Particle%xloc(1:nDim))
                                 Particle%drmass=Particle%drmass+rmass
                                 IF (nDim == 2) THEN
                                    Particle%dJ=Particle%dJ+Cross2D(rmass(1:2),q(i,j,k,imom(1:2))/q(i,j,k,1))
                                 ELSEIF (nDim == 3) THEN
                                    Particle%dJ(1:nDim)=Particle%dJ+Cross3D(rmass,q(i,j,k,imom(1:3))/q(i,j,k,1))
                                 END IF
                              ELSE
                                 Particle%dQ(1)=Particle%dQ(1)+q(i,j,k,1)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iE /= 0) Particle%dQ(iE)=Particle%dQ(iE)+q(i,j,k,iE)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iAngMom /= 0) Particle%dQ(iAngMom)=Particle%dQ(iAngMom)+q(i,j,k,iAngMom)*sink_dv*2d0*Pi*xpos(1)
                              END IF
                           END IF

                           IF (iE /= 0) THEN
                              IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,iBx:iBz)**2) !Subtract off magnetic energy                                 
                              q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Subtract off kinetic energy                              
                           END IF
                           q(i,j,k,1:NrHydroVars)=q(i,j,k,1:NrHydroVars)*(q(i,j,k,1)-drho)/q(i,j,k,1)  !Everything else should be accreted proportionally to density...
                           IF (lMHD) THEN
                              q(i,j,k,iBx)=.5*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
                              q(i,j,k,iBy)=.5*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
                              IF (nDim == 3) q(i,j,k,iBz)=.5*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
                           END IF
                           IF (iE /= 0) THEN
                              q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Add in kinetic energy
                              IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,iBx:iBz)**2) !Add magnetic energy back in.
                           END IF
                           IF (ALL(iipos(1:nDim) >= 1) .AND. ALL(iipos(1:nDim) <= Info%mX(1:nDim))) THEN
                              IF (iCylindrical == NOCYL) THEN
                                 Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)-q(i,j,k,1:NrHydroVars)*sink_dv
                              ELSE
                                 Particle%dQ(1)=Particle%dQ(1)-q(i,j,k,1)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iE /= 0) Particle%dQ(iE)=Particle%dQ(iE)-q(i,j,k,iE)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iAngMom /= 0) Particle%dQ(iAngMom)=Particle%dQ(iAngMom)-q(i,j,k,iAngMom)*sink_dv*2d0*Pi*xpos(1)
                              END IF
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
            ELSE
               !                print*, "particle%q(1).neq.0d0 called"
               ALLOCATE(count(ms(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2)))
               count=0   
               DO k=mS(3,1),mS(3,2)
                  xpos(3)=Info%xBounds(3,1)+(k-1+offset(3))*dz
                  DO j=mS(2,1),mS(2,2)
                     xpos(2)=Info%xBounds(2,1)+(j-1+offset(2))*dx
                     DO i=mS(1,1),mS(1,2)
                        xpos(1)=Info%xBounds(1,1)+(i-1+offset(1))*dx
                        v(1:nDim)=q(i,j,k,imom(1:nDim))/q(i,j,k,1)  !cell velocity
                        v_rel(1:nDim)=v(1:nDim) - Particle%q(imom(1:nDim)) !velocity w/respect to sink source
                        sKE=half*DOT_PRODUCT(v_rel(1:nDim),v_rel(1:nDim))
                        IF (iCylindrical == 2) sKE=half*(q(i,j,k,iAngMom)/q(i,j,k,1))**2 !Add angular momentum to energy
                        IF (sqrt(SUM((xpos(:)+half*(/dx,dx,dz/)-Particle%xloc(:))**2)) > IR_ACC*sink_dx) THEN
                           count(i,j,k)=-1
                           CYCLE
                        END IF
                        IF (q(i,j,k,1) <= MinDensity) CYCLE
                        DO kk=1,sample_res(3)
                           pos(3)=xpos(3)+(REAL(kk, 8)-half)*dz*sample_fact(3)
                           DO jj=1,sample_res(2)
                              pos(2)=xpos(2)+(REAL(jj, 8)-half)*dx*sample_fact(2)
                              DO ii=1,sample_res(1)
                                 pos(1)=xpos(1)+(REAL(ii, 8)-half)*dx*sample_fact(1)
                                 r2=sum((pos(1:nDim)-Particle%xloc(1:nDim))**2)
                                 r=sqrt(r2)
                                 esp=-GM/r+sKE
                                 IF (esp < 0d0) THEN !gas parcel is bound to sink particle
                                    v_r(1:nDim)=DOT_PRODUCT(v_rel(1:nDim), pos(1:nDim)-Particle%xloc(1:nDim))*(pos(1:nDim)-Particle%xloc(1:nDim))/r2
                                    v_phi(1:nDim)=v_rel(1:nDim)-v_r(1:nDim)
                                    jsp2=SUM(v_phi(1:nDim)**2)*r2
                                    IF (iCylindrical == WITHANGMOM) jsp2=jsp2+(q(i,j,k,iAngMom)/q(i,j,k,1))**2*r2
                                    IF (jsp2 < 2d0*r_surf*(esp*r_surf+GM)) THEN !gas parcel will come within "surface" of sink particle
                                       count(i,j,k)=count(i,j,k)+1 !fraction of cell that is accumulated
                                    END IF
                                 END IF
                              END DO
                           END DO
                        END DO
                     END DO
                  END DO
               END DO
               !               write(*,'(9I5)') count
               DO k=mS(3,1),mS(3,2)
                  xpos(3)=Info%xBounds(3,1)+(real(k+offset(3),8)-half)*dz
                  iipos(3)=k
                  DO j=mS(2,1),mS(2,2)
                     xpos(2)=Info%xBounds(2,1)+(real(j+offset(2),8)-half)*dx
                     iipos(2)=j
                     DO i=mS(1,1),mS(1,2)
                        xpos(1)=Info%xBounds(1,1)+(real(i+offset(1),8)-half)*dx
                        iipos(1)=i
                        IF (count(i,j,k) <= 0) CYCLE
                        r2=sum((xpos(1:nDim)-Particle%xloc(1:nDim))**2)
                        w=exp(-r2/Particle%Bondi_Kernel**2)
                        drho=min(.25d0, Particle%AccretionRate*w*REAL(count(i,j,k),KIND=qPREC)*q_fact*levels(Info%level)%dt)*Info%q(i,j,k,1) !fraction of cell to accumulate
                        IF (drho > 0d0) THEN
                           IF (ALL(iipos(1:nDim) >= 1) .AND. ALL(iipos(1:nDim) <= Info%mX(1:nDim))) THEN
                              IF (iCylindrical == NOCYL) THEN
                                 Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)+q(i,j,k,1:NrHydroVars)*sink_dv
                                 mass=sink_dv*(drho)
                                 rmass(1:nDim)=mass*(xpos(1:nDim)-Particle%xloc(1:nDim))
                                 Particle%drmass=Particle%drmass+rmass
                                 IF (nDim == 2) THEN
                                    Particle%dJ=Particle%dJ+Cross2D(rmass(1:2),q(i,j,k,imom(1:2))/q(i,j,k,1))
                                 ELSEIF (nDim == 3) THEN
                                    Particle%dJ(1:nDim)=Particle%dJ+Cross3D(rmass,q(i,j,k,imom(1:3))/q(i,j,k,1))
                                 END IF
                              ELSE
                                 Particle%dQ(1)=Particle%dQ(1)+q(i,j,k,1)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iE /= 0) Particle%dQ(iE)=Particle%dQ(iE)+q(i,j,k,iE)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iAngMom /= 0) Particle%dQ(iAngMom)=Particle%dQ(iAngMom)+q(i,j,k,iAngMom)*sink_dv*2d0*Pi*xpos(1)
                              END IF
                           END IF

                           IF (iE /= 0) THEN
                              IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,iBx:iBz)**2) !Subtract off magnetic energy                                 
                              q(i,j,k,iE)=q(i,j,k,iE)-half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Subtract off kinetic energy                              
                           END IF
                           q(i,j,k,1:NrHydroVars)=q(i,j,k,1:NrHydroVars)*(q(i,j,k,1)-drho)/q(i,j,k,1)  !Everything else should be accreted proportionally to density...
                           IF (lMHD) THEN
                              q(i,j,k,iBx)=.5*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
                              q(i,j,k,iBy)=.5*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
                              IF (nDim == 3) q(i,j,k,iBz)=.5*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
                           END IF
                           IF (iE /= 0) THEN
                              q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1) !Add in kinetic energy
                              IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,iBx:iBz)**2) !Add magnetic energy back in.
                           END IF
                           IF (ALL(iipos(1:nDim) >= 1) .AND. ALL(iipos(1:nDim) <= Info%mX(1:nDim))) THEN
                              IF (iCylindrical == NOCYL) THEN
                                 Particle%dQ(1:NrHydroVars)=Particle%dQ(1:NrHydrovars)-q(i,j,k,1:NrHydroVars)*sink_dv
                              ELSE
                                 Particle%dQ(1)=Particle%dQ(1)-q(i,j,k,1)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iE /= 0) Particle%dQ(iE)=Particle%dQ(iE)-q(i,j,k,iE)*sink_dv*2d0*Pi*xpos(1)
                                 IF (iAngMom /= 0) Particle%dQ(iAngMom)=Particle%dQ(iAngMom)-q(i,j,k,iAngMom)*sink_dv*2d0*Pi*xpos(1)
                              END IF
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
               DEALLOCATE(count)
            END IF
         END DO
         DEALLOCATE(mSs, offsets)
      END IF
   END SUBROUTINE BondiAccretion



  


   !> Calculates the gas force and it's first derivatives at the particles position using the gas potential within the grid
   !! @details Need to populate Particle%gas_accel(:) with (/Phi_x, Phi_y, Phi_z, Phi_xx, Phi_yy, Phi_zz, Phi_xy, Phi_yz, Phi_xz/)
   !! All evaluated at the Particle's position.
   !! @param Particle Particle to calculate potential of
   !! @param Info Info structure containing particle (and gas potential)
   !! @param ipos Index within Info of host cell
   SUBROUTINE GetGasForce(Particle, Info)
      INTEGER :: ir(3), il(3), ipos(3)
      TYPE(ParticleDef) :: Particle
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: phi
      !We can do a third order reconstruction of phi at the cell center and use the coefficients 
      !to shift the reconstruction to the particle's position and have a 2nd order reconstruction.
      REAL(KIND=qPREC) :: phi_x, phi_y, phi_z, phi_xx, phi_xy, phi_xz, phi_yy, phi_yz, phi_zz, &
           phi_xxx, phi_xxy, phi_xxz, phi_xyy, phi_xyz, phi_xzz, phi_yyy, phi_yyz, phi_yzz, phi_zzz
      REAL(KIND=qPREC), PARAMETER, DIMENSION(5,4) :: DC5s = RESHAPE((/&
           0.0833333333333333d0, -0.666666666666667d0,  0.00000000000000d0, 0.666666666666667d0, -0.0833333333333333d0, &
           -0.0416666666666666d0,  0.666666666666666d0, -1.25000000000000d0, 0.666666666666667d0, -0.0416666666666667d0, &
           -0.0416666666666666d0,  0.666666666666666d0, -1.25000000000000d0, 0.666666666666667d0, -0.0416666666666667d0, &
           -0.0416666666666666d0,  0.666666666666666d0, -1.25000000000000d0, 0.666666666666667d0, -0.0416666666666667d0/), (/5,4/))

      REAL(KIND=qPREC), PARAMETER, DIMENSION(3,2) :: DC3s = RESHAPE((/&
           -.5d0,  0d0, .5d0,  & 
           .5d0, -1d0, .5d0/),(/3,2/))

      REAL(KIND=qPREC), DIMENSION(3) :: dpos,f_grav,pos
      INTEGER :: i,ip(3,2),j,k,m
      REAL(KIND=qPREC) :: weight, total_weight
      IF (ParticleInGrid(Particle, Info,ipos)) THEN

         !Use surrounding 2**ndim cells to get gas force weighted by proximity
         dpos=(Particle%xloc-Info%xBounds(:,1))/sink_dx + half  !
         ip=1
         ip(1:nDim,1)=floor(dpos)
         ip(1:nDim,2)=ip(1:nDim,1)+1
         f_grav=0d0
         total_weight=0
         DO i=ip(1,1),ip(1,2)
            DO j=ip(2,1),ip(2,2)
               DO k=ip(3,1),ip(3,2)           
                  pos=(/i,j,k/)
                  ir=pos
                  il=pos
                  weight=1d0/(sqrt(sum((real(pos(1:nDim))-dpos(1:nDim))**2))+1e-3)
                  DO m=1,nDim
                     ir(m)=pos(m)+1
                     il(m)=pos(m)-1
                     f_grav(m)=f_grav(m)+weight*((Info%q(ir(1),ir(2),ir(3),iPhiGas)-Info%q(il(1),il(2),il(3),iPhiGas)))
                     ir(m)=pos(m)
                     il(m)=pos(m)
                  END DO
                  total_weight=total_weight+weight
               END DO
            END DO
         END DO
         Particle%gas_accel=-f_grav/total_weight/sink_dx/2d0

      END IF
   END SUBROUTINE GetGasForce


   !> @}

   !> @name Auxilliary module routines
   !! @{

   !> Calculate contributions to particles moments for info structure from region defined by mS
   !! @param Info Info structure
   !! @param Particle Particle object
   !! @param mS index defining region
   SUBROUTINE CalcMomentContributions(Info, Particle, mS, offset)
      TYPE(InfoDef) :: Info
      TYPE(ParticleDef) :: Particle
      INTEGER, DIMENSION(3,2) :: mS
      INTEGER :: i,j,k,ipos(3)
      REAL(KIND=qpREC) :: r, pos(3), offset(3)
      IF (nMoments == 1) THEN
         DO i=mS(1,1),mS(1,2)
            DO j=mS(2,1),mS(2,2)
               DO k=mS(3,1),mS(3,2)
                  pos=CellPos(Info,i,j,k)+offset-Particle%xloc
                  r=sqrt(sum(pos(1:nDim)**2))                  
                  IF (r <= r_acc) THEN
                     !                     write(*,*) 'Info%q(',i,',',j,',',k,',1)=',Info%q(i,j,k,1)
                     Particle%moments(1)=Particle%moments(1)+Info%q(i,j,k,1)*sink_dV
                  END IF
               END DO
            END DO
         END DO
      ELSE
         PRINT*, 'error - only 1 moment is currently supported'
         STOP
      END IF
   END SUBROUTINE CalcMomentContributions


   !> Calculates overlaps between a grid and a particle
   !! @param Info Info structure
   !! @param Particle Particle object
   !! @param mTs Array of overlap indices
   !! @param nOverlaps Number of overlaps
   !! @param nGhost number of ghost cells to include in overlap calculations
   SUBROUTINE CalcGridParticleOverlaps(Info, Particle, mTs, nOverlaps, offsets, nGhost)
      ! Calculates overlap bounds of cell-centered quantities for
      !  overlapping grids.
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: SourcemGlobal
      INTEGER, DIMENSION(3,2) :: mO,mGlobal,iOffSet
      INTEGER, DIMENSION(3) :: pOffSet
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs
      REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
      REAL(KIND=qPREC), DIMENSION(27,3) :: MaxOffsets
      INTEGER, DIMENSION(27,3,2) :: MaxMTs
      INTEGER :: i,j,k,rmbc,nOverlaps,level
      INTEGER, OPTIONAL :: nGhost
      TYPE(ParticleDef) :: Particle
      INTEGER, DIMENSION(3) :: ipos
      level=Info%level
      ! Calculate particles position in index space
      ipos(1:nDim)=floor((Particle%xloc(1:nDim)-GxBounds(1:nDim,1))/levels(level)%dx)+1
      ipos(1:nDim)=max(min(ipos(1:nDim),levels(Info%level)%mX(1:nDim)),1)
!      write(*,*) "sink_dx=",sink_dx
!      write(*,*) "ipos=",ipos, info%level

      SourcemGlobal(nDim+1:3,:)=1
      SourcemGlobal(1:nDim,1)=ipos(1:nDim)-particle%buffer(level) 
      SourcemGlobal(1:nDim,2)=ipos(1:nDim)+particle%buffer(level)
!            write(*,*) "sourcemglobal=", sourcemglobal
      rmbc=nGhost
!      write(*,*) "rmbc=",rmbc
      NULLIFY(mTs)
      nOverlaps=0
      mO(nDim+1:3,:)=1
      ioffset=0
      WHERE(lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=1
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)
               pOffSet=(/i,j,k/)*(GxBounds(:,2)-GxBounds(:,1))
               mGlobal(:,:)=SourcemGlobal(:,:)+SPREAD(pOffSet,2,2)

               mO(1:nDim,1)=max(Info%mGlobal(1:nDim,1)-rmbc&
                    &,mGlobal(1:nDim,1)) 
               mO(1:nDim,2)=min(Info%mGlobal(1:nDim,2)+rmbc&
                    &,mGlobal(1:nDim,2))

               IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                  nOverlaps=nOverlaps+1
                  MaxMTs(nOverlaps,:,:)=mO-Spread(  Info%mGlobal(:,1)&
                       &,2,2)+1
                  MaxOffsets(nOverlaps,:)=pOffSet
               END IF

               !               write(*,*) "Considering", Info%mGlobal, SourcemGlobal, mO, nOverlaps
            END DO
         END DO
      END DO
      IF (nOverlaps > 0) THEN
         ALLOCATE(MTs(nOverlaps,3,2), offsets(nOverlaps,3))
         MTs=MaxMTs(1:nOverlaps,:,:)
         offsets=MaxOffsets(1:nOverlaps,:)
      END IF

   END SUBROUTINE CalcGridParticleOverlaps


   !> Determine if a particle is inside of a grid and return the index of the host cell
   !! @param Particle Particle object
   !! @param Info Info structure
   !! @param ipos index of host cell
   FUNCTION ParticleInGrid(Particle, Info, ipos)
      LOGICAL :: ParticleInGrid
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(:) :: ipos
      Type(ParticleDef) :: Particle
      ipos=1
      ipos=PosCell(Info, Particle%xloc(1), Particle%xloc(2), Particle%xloc(3))
      ParticleInGrid=ALL(ipos(1:nDim) >= 1) .AND. ALL(ipos(1:nDim) <= Info%mX(1:nDim))
!      write(*,*) 'processor', mpi_id, 'ipos=', ipos, Info%mX
!      IF (ParticleInGrid) write(*,*) MPI_ID, Info%xBounds, Particle%xloc
   END FUNCTION ParticleInGrid


   !> Determine if a particle is inside of a grid and return the index of the host cell
   !! @param Particle Particle object
   !! @param Info Info structure
   !! @param ipos index of host cell
!   FUNCTION ParticleCell(Particle, n, ipos)
!      LOGICAL :: ParticleInGrid
!      INTEGER :: n
!      INTEGER, DIMENSION(:) :: ipos
!      Type(ParticleDef) :: Particle
!      ipos=1
!      ipos(1:nDim)=floor((Particle%xloc(1:nDim)-GxBounds(1:nDim,1))/levels(n)%dx)+1
!      ipos(1:nDim)=max(min(ipos(1:nDim),levels(n)%Gmx(1:nDim)),1)
!   END FUNCTION ParticleCell


   !> @}

END MODULE ParticleInfoOps
