!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    Bondi.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Bondi
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE DataDeclarations
  IMPLICIT NONE
  PUBLIC :: BH_alpha, fill_alpha_table, Bondi_lambda_critical, InitializeBondi
  SAVE
  REAL(KIND=qPREC), PUBLIC :: Bondi_lambda, Bondi_lambda2

  PRIVATE
  INTEGER, PARAMETER :: NrAlphaBins=513
  REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: Bondi_alpha_table, init_alpha_table
  REAL(KIND=qPREC) :: Bondi_gamma
!  REAL(KIND=qPREC) :: GM, mass, xloc(3), vel(3), rho_inf, c_inf, radius, r_BH, Bondi_lambda, Bondi_lambda2,p0, c_inf2, half_angle=1.570796327d0, rho, v_r, temperature, r_circle, radial_refinements(0:MaxDepth)
!  LOGICAL :: lFixed, lCircular_Boundary, lSet_Boundary
!  INTEGER :: iBondi_Particle,iRoutine

CONTAINS


  SUBROUTINE InitializeBondi()
    IF (nDim == 1) RETURN
    Bondi_lambda=Bondi_lambda_critical()
    Bondi_lambda2=Bondi_lambda**2
    CALL fill_alpha_table()
  END SUBROUTINE InitializeBondi


  FUNCTION Bondi_lambda_critical()
    REAL(KIND=qPREC) :: Bondi_lambda_critical
    SELECT CASE(mydim)
    CASE(1)
       Bondi_lambda_critical = 1d0
    CASE(2)
       IF (gamma <= 1.000001) THEN
          Bondi_lambda_critical=1.6487212707001281468486507878142d0 ! exp(.5) = sqrt(e)
          Bondi_gamma=1.000001
       ELSE IF (gamma >= 2.99999) THEN
          Bondi_lambda_critical = 1d0
          Bondi_gamma = 2.99999
       ELSE
          Bondi_gamma=gamma
          Bondi_lambda_critical =sqrt(((3d0-Bondi_gamma)/2d0*(2d0/(3d0-Bondi_gamma))**(2d0/(Bondi_gamma-1d0))))
       END IF
    CASE(3)
       IF (gamma <= 1.000000000001) THEN
          Bondi_lambda_critical=1.12042226758452d0
          Bondi_gamma = 1.000000000001
       ELSE IF (gamma >= 1.666666666667) THEN
          Bondi_lambda_critical = .25000000000000000d0
          Bondi_gamma = 1.666666666667
       ELSE
          Bondi_gamma=gamma

           ! Original version does not group large cancelling terms... leads to infinity times 0
!          Bondi_lambda_critical = (.5d0)**((gamma+1d0)/(2d0*gamma1)) * (1.25d0-.75d0*gamma)**(-(5d0-3d0*gamma)/(2d0*gamma1))

          
           ! Grouping cancelling terms makes calculation more accurate
!          Calculation in log space seems to give same answer
!          Bondi_lambda_critical = exp(log(((.5d0)/(1.25d0-.75d0*Bondi_gamma)))*((Bondi_gamma+1d0)/(2d0*(Bondi_gamma-1d0))) + log(((1.25d0-.75d0*Bondi_gamma)))*(-(5d0-3d0*Bondi_gamma)/(2d0*(Bondi_gamma-1d0))+(Bondi_gamma+1d0)/(2d0*(Bondi_gamma-1d0))))

        
          Bondi_lambda_critical = ((.5d0)/(1.25d0-.75d0*Bondi_gamma))**((Bondi_gamma+1d0)/(2d0*(Bondi_gamma-1d0))) * ((1.25d0-.75d0*Bondi_gamma))**(-(5d0-3d0*Bondi_gamma)/(2d0*(Bondi_gamma-1d0))+(Bondi_gamma+1d0)/(2d0*(Bondi_gamma-1d0)))
       END IF
    END SELECT
  END FUNCTION Bondi_lambda_critical


  SUBROUTINE fill_alpha_table()
    complex(8) :: a,b,e,f, h, P, Q, R, U, W, x, y, z
    INTEGER :: i,j
    REAL(8) :: temp_new, temp_prev, c,d,g
    LOGICAL :: lDiverging
    CHARACTER(LEN=40) :: Filename
    LOGICAL :: lswitch=.true.
    ALLOCATE(Bondi_alpha_table(NrAlphaBins,2))
    i=0
    j=NrAlphaBins
    temp_new = 1000d0
    lDiverging=.false.
!    lswitch=.false.
    SELECT CASE (mydim)
    CASE(2)
       b=-1d0
       e=b**2
       DO WHILE (j >=1)
          i=i+1
          z=1.02d0**i
          a=(z**(Bondi_gamma-1d0)-1d0)/(Bondi_gamma-1d0)
          d=-b/(2d0*a)
          c=.5d0*Bondi_lambda2/z**2
          f=-4d0*a*c
          g=sqrt(abs(e+f))
          temp_prev=temp_new
          temp_new=min(g, 1000d0)
          IF (.NOT. lDiverging .AND. temp_new > temp_prev)  lDiverging=.true.
          IF (lDiverging .and. lswitch) THEN
             Bondi_alpha_table(j,1)=abs(d - g/(2d0*a))
          ELSE !diverging so switch solutions
             Bondi_alpha_table(j,1)=abs(d + g/(2d0*a))
          END IF
          Bondi_alpha_table(j,2)=z
          j=j-1
       END DO
    CASE(3)
       DO WHILE (j >=1)
          i=i+1
          z=1.02d0**i
          a=(z**(Bondi_gamma-1d0)-1d0)/(Bondi_gamma-1d0)
          b=-1d0
          e=.5d0*Bondi_lambda2/z**2
          f=-3d0/8d0*b**2/a**2
          g=b**3/(8d0*a**3)
          h=-3d0*b**4/(256d0*a**4)+e/a
          P=-f**2/12d0-h
          Q=-f**3/108d0+f*h/3d0-g**2/8d0
          R=-Q/2d0+(Q**2/4d0+P**3/27d0)**(.5d0)
          U=R**(1d0/3d0)
          IF (U==0) THEN
             y=-5d0/6d0*f-Q**(1d0/3d0)
          ELSE
             y=-5d0/6d0*f+U-P/(3d0*U)
          END IF
          W=sqrt(abs(f+2d0*y))
          IF (REAL(3d0*f+2d0*y+2d0*g/W) > 0d0) CYCLE
          temp_prev=temp_new
          temp_new=min(sqrt(abs(-(3d0*f+2d0*y+2d0*g/W))), 1000d0)
          IF (.NOT. lDiverging .AND. temp_new > temp_prev)  lDiverging=.true.
          IF (lDiverging .and. lswitch) THEN
             Bondi_alpha_table(j,1)=abs(-b/(4d0*a)+(W-sqrt(abs(-(3d0*f+2d0*y+2d0*g/W))))/2d0)
          ELSE !diverging so switch solutions
             Bondi_alpha_table(j,1)=abs(-b/(4d0*a)+(W+sqrt(abs(-(3d0*f+2d0*y+2d0*g/W))))/2d0)
          END IF
          Bondi_alpha_table(j,2)=z
          j=j-1
       END DO
    END SELECT
    
    IF (MPI_ID == 0) THEN
       WRITE(FileName, '(1A)') "Bondi_alpha.tab"
       OPEN(UNIT=BONDI_DATA_HANDLE, file=FileName, status="unknown")
       write(BONDI_DATA_HANDLE, '(2E16.5)') transpose(Bondi_alpha_table)
       close(BONDI_DATA_HANDLE)
    END IF
  END SUBROUTINE fill_alpha_table

  function BH_alpha(x)
    REAL(KIND=qPREC) :: BH_alpha, x
    INTEGER :: slot
    INTEGER :: i

    IF (x <= Bondi_alpha_table(1,1)) THEN
       BH_alpha=Bondi_alpha_table(1,2)+(x-Bondi_alpha_table(1,1))*(Bondi_alpha_table(2,2)-Bondi_alpha_table(1,2))/(Bondi_alpha_table(2,1)-Bondi_alpha_table(1,1))
    ELSE IF (x >= Bondi_alpha_table(NrAlphaBins,1)) THEN
       BH_alpha = 1d0
    ELSE
       slot=256
       DO i=1,8
          IF (x>Bondi_alpha_table(slot+1,1)) THEN
             slot=slot+2**(8-i)
          ELSE
             slot=slot-2**(8-i)
          END IF
       END DO
       IF (x>Bondi_alpha_table(slot+1,1)) THEN
          slot=slot+1
       END IF
       BH_alpha=Bondi_alpha_table(slot,2)+(x-Bondi_alpha_table(slot,1))*(Bondi_alpha_table(slot+1,2)-Bondi_alpha_table(slot,2))/(Bondi_alpha_table(slot+1,1)-Bondi_alpha_table(slot,1))
    END IF
  END function BH_alpha




END MODULE Bondi



