!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    EOS.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file EOS.f90
!! @brief Main file for module EOS

!> @defgroup EOS Equation of State
!! @brief Module for dealing with equation of state
!! @ingroup Physics

!> Module for dealing with equation of state
!! @ingroup EOS
MODULE EOS
   USE GlobalDeclarations
   USE PhysicsDeclarations
   USE TreeDeclarations
   USE DataDeclarations

   IMPLICIT NONE
   PUBLIC Press, PrimPress, Prim_To_Cons, Cons_To_Prim, ConvertTotalToInternalEnergy, ConvertInternalToTotalEnergy, Protectq, InternalEnergy, PrintQ, soundspeed, primsoundspeed, SetPress, GetInternalEnergy, GetMaxSpeed, calc_eigens

   INTERFACE ConvertTotalToInternalEnergy
      MODULE PROCEDURE ConvertTotalToInternalEnergy0D, ConvertTotalToInternalEnergy1D, ConvertTotalToInternalEnergy2D, ConvertTotalToInternalEnergy3D
   END INTERFACE ConvertTotalToInternalEnergy

   INTERFACE ConvertInternalToTotalEnergy
      MODULE PROCEDURE ConvertInternalToTotalEnergy0D, ConvertInternalToTotalEnergy1D, ConvertInternalToTotalEnergy2D, ConvertInternalToTotalEnergy3D
   END INTERFACE ConvertInternalToTotalEnergy

   INTERFACE Cons_to_prim
      MODULE PROCEDURE Cons_to_prim_info, cons_to_prim_q
   END INTERFACE Cons_to_prim

   INTERFACE prim_to_cons
      MODULE PROCEDURE prim_to_cons_info, prim_to_cons_q
   END INTERFACE prim_to_cons


CONTAINS


   !> Performs protection on Info object
   !! @param Info Info object
   SUBROUTINE Protectq(Info,ip,caller, lStopProtect)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j,k,ii,jj,kk
      INTEGER, DIMENSION(3,2) :: ip, mB
      INTEGER :: level, rmbc,l, n, m
      REAL(KIND=qPREC), PARAMETER :: Precision=1e-14
      REAL(KIND=qPREC) :: vel(3),temp
      CHARACTER(LEN=*) :: caller
      REAL(KIND=qPREC), DIMENSION(:), POINTER :: q
      REAL(KIND=qPREC), DIMENSION(:,:,:), POINTER :: pressure, rho
      LOGICAL, DIMENSIOn(:,:,:), ALLOCATABLE :: mask
      LOGICAL, OPTIONAL :: lStopProtect

      level=Info%level
      DO i=ip(1,1),ip(1,2)
         DO j=ip(2,1),ip(2,2)
            DO k=ip(3,1),ip(3,2)              
               q=>Info%q(i,j,k,:)
               IF(ANY(q(1:NrVars).ne.q(1:NrVars) .OR. abs(q(1:NrVars)).gt.huge(abs(q(1:NrVars)))) .AND. .NOT. lRequestRestart) THEN
                  write(*,*) 'protection found Nans in ', caller
                  CALL Printq(Info, q, levels(Info%level)%tnow, i,j,k)
                  PRINT*, 'Processor', MPI_ID, 'requesting restart'
                  lRequestRestart=.true.
                  IF (PRESENT(lStopProtect)) THEN
                     IF (lStopProtect) THEN
                        WRITE(*,*) "*** Simulation stopped due to protections in the initialization! ***"
                        STOP
                     END IF
                  END IF
                  RETURN
               END IF
            END DO
         END DO
      END DO

      ALLOCATE(rho(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2)))
      rho=Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1)
      IF (iE /= 0) THEN
         ALLOCATE(pressure(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2)))
         DO i=ip(1,1),ip(1,2)
            DO j=ip(2,1),ip(2,2)
               DO k=ip(3,1),ip(3,2)              
                  pressure(i,j,k)=Press(Info%q(i,j,k,:))
               END DO
            END DO
         END DO
      END IF

      DO i=ip(1,1),ip(1,2)
         DO j=ip(2,1),ip(2,2)
            DO k=ip(3,1),ip(3,2)              
               q=>Info%q(i,j,k,:)
               IF (q(1) < MinDensity) THEN
                  IF (lRestartOnDensityProtections) THEN
                     IF (.NOT. lRequestRestart) THEN
                        WRITE(*,'(A,I5,3A)') 'Processor', MPI_ID, ' encountered a density protection from ', caller, ' and lRestartOnDensityProtections is true so requesting restart.  Consider setting lRestartOnDensityProtections=.false. in physics.data if you wish to not restart'
                        CALL Printq(Info, q, levels(Info%level)%tnow, i,j,k)
                        lRequestRestart=.true.
                     END IF
                  ELSE
                     !First protect momentum
                     SELECT CASE(iMomentumProtect)
                     CASE(CONSERVE_MOMENTUM)
                     CASE(CLEAR_MOMENTUM)
                        q(m_low:m_high)=0
                     CASE(AVG_NEARBY_VEL)
                        DO l=1, 5
                           mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                           mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                           ALLOCATE(mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                           mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity
                           n=count(mask)
                           IF (n > 0) THEN
                              DO m=m_low,m_high
                                 q(m)=sum(Info%q(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),m)/rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)), mask)/REAL(n,8)
                              END DO
                           END IF
                           DEALLOCATE(mask)
                           IF (n > 0) EXIT
                        END DO
                        IF (n == 0) q(m_low:m_high)=0d0 
                     END SELECT
                     !Then protect density
                     SELECT CASE(iDensityProtect)
                     CASE(MIN_DENSITY)
                        q(1)=MinDensity
                     CASE(MIN_NEARBY_DENSITY)
                        DO l=1, 5
                           mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                           mB(:,2)=min((/i,j,k/)+l, ip(:,2)) 
                           ALLOCATE(mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                           mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity
                           n=count(mask)
                           IF (n > 0) q(1)=minval(rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)), mask)
                           DEALLOCATE(mask)
                           IF (n > 0) EXIT
                        END DO
                        IF (n == 0) q(1)=MinDensity
                     CASE(AVG_NEARBY_DENSITY)
                        DO l=1, 5
                           mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                           mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                           ALLOCATE(mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                           mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity
                           n=count(mask)
                           IF (n > 0) q(1)=sum(rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)), mask)/real(n,8)
                           DEALLOCATE(mask)
                           IF (n > 0) EXIT
                        END DO
                        IF (n == 0) q(1) = MinDensity
                     END SELECT
                     IF (iMomentumProtect==AVG_NEARBY_VEL) q(m_low:m_high)=q(m_low:m_high)*q(1)
                  END IF
               END IF
               DO l=nTracerLo, nTracerHI
                  q(l)=max(q(l),0d0)
               END DO
               IF (iE /= 0) THEN
                  temp=pressure(i,j,k)/rho(i,j,k)
                  IF (temp < MinTemp) THEN
                     IF (lRestartOnPressureProtections) THEN
                        IF (.NOT. lRequestRestart) THEN
                           PRINT*, 'Processor', MPI_ID, ' encountered a pressure protection from ', caller, ' and lRestartOnPressureProtections is true so requesting restart.  Consider setting lRestartOnPressureProtections=.false. in physics.data if you wish to not restart', temp, mintemp
                           CALL Printq(Info, q, levels(Info%level)%tnow, i,j,k)
                           lRequestRestart=.true.
                        END IF
                     ELSE
                        SELECT CASE (iPressureProtect)
                        CASE(MIN_TEMP)
                           CALL SetTemp(Info%q(i,j,k,:), MinTemp)
                        CASE(MIN_NEARBY_PRESS)
                           DO l=1, 5
                              mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                              mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                              ALLOCATE(Mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                              Mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity .AND. pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))*MinTemp
                              n=count(Mask)
                              IF (n > 0) CALL SetPress(q(:), minval(pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)),Mask))
                              DEALLOCATE(Mask)
                              IF (n > 0) EXIT
                           END DO
                           IF (n == 0) CALL SetPress(q(:), q(1)*MinTemp)
                        CASE(AVG_NEARBY_PRESS)
                           DO l=1, 5
                              mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                              mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                              ALLOCATE(Mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                              Mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity .AND. pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))*MinTemp
                              n=count(Mask)
                              IF (n > 0) CALL SetPress(Info%q(i,j,k,:), sum(pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)),Mask)/REAL(n,8))
                              DEALLOCATE(Mask)
                              IF (n > 0) EXIT
                           END DO
                           IF (n == 0) CALL SetPress(q(:), q(1)*MinTemp)
                        CASE(MIN_NEARBY_TEMP)
                           DO l=1, 5
                              mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                              mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                              ALLOCATE(Mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                              Mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity .AND. pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))*MinTemp
                              n=count(Mask)
                              IF (n > 0) CALL SetTemp(Info%q(i,j,k,:), minval(pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))/rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)),Mask))
                              DEALLOCATE(Mask)
                              IF (n > 0) EXIT
                           END DO
                           IF (n == 0) CALL SetPress(q(:), q(1)*MinTemp)
                        CASE(AVG_NEARBY_TEMP)
                           DO l=1, 5
                              mB(:,1)=max((/i,j,k/)-l, ip(:,1))
                              mB(:,2)=min((/i,j,k/)+l, ip(:,2))
                              ALLOCATE(Mask(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)))
                              Mask=rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > MinDensity .AND. pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)) > rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))*MinTemp
                              n=count(Mask)
                              IF (n > 0) CALL SetTemp(Info%q(i,j,k,:), sum(pressure(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2))/rho(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)),Mask)/REAL(n,8))
                              DEALLOCATE(Mask)
                              IF (n > 0) EXIT
                           END DO
                           IF (n == 0) CALL SetPress(q(:), q(1)*MinTemp)
                        END SELECT
                     END IF
                  END IF
               END IF
            END DO
         END DO
      END DO
      DEALLOCATE(rho)
      IF (iE /= 0) DEALLOCATE(pressure)
      IF (lRequestRestart .AND. PRESENT(lStopProtect)) THEN
         IF (lStopProtect) THEN
            WRITE(*,*) "*** Simulation stopped due to protections in the initialization! ***"
            STOP
         END IF
      END IF
   END SUBROUTINE Protectq

   !> @name ConvertTotalToInternalEnergy
   !! @{

   !> Converts energy component of a single cell from total energy to internal energy
   !! @param q conservative vector
   SUBROUTINE ConvertTotalToInternalEnergy0D(q)
      REAL(KIND=qPREC), DIMENSION(:) :: q
      IF (iE .ne. 0) THEN
         q(iE)=q(iE)-half*sum(q(m_low:m_high)**2)/q(1)
         IF (lMHD) q(iE)=q(iE)-half*sum(q(iBx:iBz)**2)
      END IF
   END SUBROUTINE ConvertTotalToInternalEnergy0D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector array
   SUBROUTINE ConvertTotalToInternalEnergy1D(q)
      REAL(KIND=qPREC), DIMENSION(:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,iE)=q(:,iE)-half*sum(q(:,m_low:m_high)**2, 2)/q(:,1)
         IF (lMHD) q(:,iE)=q(:,iE)-half*sum(q(:,iBx:iBz)**2,2)
      END IF
   END SUBROUTINE ConvertTotalToInternalEnergy1D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector 2D array
   SUBROUTINE ConvertTotalToInternalEnergy2D(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,:,iE)=q(:,:,iE)-half*sum(q(:,:,m_low:m_high)**2, 3)/q(:,:,1)
         IF (lMHD) q(:,:,iE)=q(:,:,iE)-half*sum(q(:,:,iBx:iBz)**2,3)
      END IF
   END SUBROUTINE ConvertTotalToInternalEnergy2D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector 3D array
   SUBROUTINE ConvertTotalToInternalEnergy3D(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,:,:,iE)=q(:,:,:,iE)-half*sum(q(:,:,:,m_low:m_high)**2, 4)/q(:,:,:,1)
         IF (lMHD) q(:,:,:,iE)=q(:,:,:,iE)-half*sum(q(:,:,:,iBx:iBz)**2,4)
      END IF
   END SUBROUTINE ConvertTotalToInternalEnergy3D

   !> @}


   !> @name ConvertInternalToTotalEnergy
   !! @{

   !> Converts energy component of a single cell from total energy to internal energy
   !! @param q conservative vector
   SUBROUTINE ConvertInternalToTotalEnergy0D(q)
      REAL(KIND=qPREC), DIMENSION(:) :: q
      IF (iE .ne. 0) THEN
         q(iE)=q(iE)+half*sum(q(m_low:m_high)**2)/q(1)
         IF (lMHD) q(iE)=q(iE)+half*sum(q(iBx:iBz)**2)
      END IF
   END SUBROUTINE ConvertInternalToTotalEnergy0D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector array
   SUBROUTINE ConvertInternalToTotalEnergy1D(q)
      REAL(KIND=qPREC), DIMENSION(:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,iE)=q(:,iE)+half*sum(q(:,m_low:m_high)**2, 2)/q(:,1)
         IF (lMHD) q(:,iE)=q(:,iE)+half*sum(q(:,iBx:iBz)**2,2)
      END IF
   END SUBROUTINE ConvertInternalToTotalEnergy1D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector 2D array
   SUBROUTINE ConvertInternalToTotalEnergy2D(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,:,iE)=q(:,:,iE)+half*sum(q(:,:,m_low:m_high)**2, 3)/q(:,:,1)
         IF (lMHD) q(:,:,iE)=q(:,:,iE)+half*sum(q(:,:,iBx:iBz)**2,3)
      END IF
   END SUBROUTINE ConvertInternalToTotalEnergy2D

   !> Converts energy component of an array of cells from total energy to internal energy
   !! @param q conservative vector 3D array
   SUBROUTINE ConvertInternalToTotalEnergy3D(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:,:) :: q
      IF (iE .ne. 0) THEN
         q(:,:,:,iE)=q(:,:,:,iE)+half*sum(q(:,:,:,m_low:m_high)**2, 4)/q(:,:,:,1)
         IF (lMHD) q(:,:,:,iE)=q(:,:,:,iE)+half*sum(q(:,:,:,iBx:iBz)**2,4)
      END IF
   END SUBROUTINE ConvertInternalToTotalEnergy3D

   !> @}


   !> Calculate pressure and return optional variables if present
   !! @param q fluid variables (conservative form)
   !! @param kappa [dP/drho]Eth
   !! @param chi [dP/dEth]rho
   !! @param DPDQ Derivative of pressure wrt conserved variables
   FUNCTION Press(q,kappa,chi,DPDQ)
      !    USE COOL
      !    USE TF
      !    USE VANDERWAALS
      REAL (KIND=qPrec), DIMENSION(:), INTENT(IN) :: q
      REAL (KIND=qPrec), OPTIONAL, INTENT(OUT) :: kappa,chi
      REAL (KIND=qPrec), DIMENSION(:), OPTIONAL, INTENT(OUT) :: DPDQ
      REAL (KIND=qPrec) :: Press
      ! Internal variables
      REAL (KIND=qPrec) :: kappa1,chi1,ke
      ! Caching certain frequently-used derived values.
      SELECT CASE(iEOS)
      CASE(EOS_ISOTHERMAL)
         kappa1 = 0.0d0
         chi1 = Iso_Speed2
         Press = Iso_Speed2*q(1)
         IF (present(dPdQ)) THEN !This is dPdQ keeping everything other conservative quantity constant
            !P=rho*IsoSpeed2 = chi1*rho
            dPdQ=0
            dPdQ(1)=chi1
         END IF
      CASE(EOS_IDEAL_GAS)
         ke=half*SUM(q(m_low:m_high)**2)/q(1)
         Press=q(iE)-ke
         IF (lMHD) Press=Press-half*SUM(q(iBx:iBz)**2)
         Press=Press*gamma1
         kappa1=gamma1 !d(P)/d(rho)
         chi1=0.d0
         IF(PRESENT(DPDQ)) THEN !This is dPdQ keeping everything other conservative quantity constant
            !P=kappa1*(E-half*q(m_low:m_high)**2/q(1)-half*q(iBx:iBz)**2)+chi1*rho
            DPDQ = zero
            DPDQ(1)=chi1+kappa1*ke/q(1)
            DPDQ(m_low:m_high)=-kappa1*q(m_low:m_high)/q(1)
            DPDQ(iE) = kappa1
            IF (lMHD) THEN
               DPDQ(iBx:iBz)= -kappa1*q(iBx:iBz)
            END IF
         END IF
      CASE DEFAULT
         PRINT*,'unimplemented EOS, iEOS=',iEOS
         STOP
      END SELECT
      ! if present, return chi and kappa to caller
      IF(Present(chi)) chi=chi1
      IF(Present(kappa)) kappa=kappa1
   END FUNCTION Press


   FUNCTION Temperature(q)
      REAL(KIND=qPREC) :: Temperature
      REAL(KIND=qPREC), DIMENSION(:) :: q
      Temperature=Press(q)/q(1)
   END FUNCTION Temperature

   !> Returns internal energy given pressure p and conservative vector q
   FUNCTION GetInternalEnergy(q,p)
      REAL(KIND=qPREC) :: GetInternalEnergy
      REAL(KIND=qPREC) :: p
      REAL(KIND=qPREC), DIMENSION(:) :: q
      SELECT CASE(iEOS)
      CASE(EOS_ISOTHERMAL)
         GetInternalEnergy=p
      CASE(EOS_IDEAL_GAS)
         GetInternalEnergy=p*gamma7
      CASE DEFAULT
         PRINT*,'unimplemented EOS, iEOS=',iEOS
         STOP
      END SELECT
   END FUNCTION  GetInternalEnergy

   !> Sets energy given pressure p and conservative vector q
   SUBROUTINE SetPress(q, p)
      REAL(KIND=qPREC), DIMENSION(:) :: q
      REAL(KIND=qPREC) :: p, E
      E=half*SUM(q(m_low:m_high)**2)/q(1)
      IF (lMHD) E=E+half*SUM(q(iBx:iBz)**2)
      q(iE)=GetInternalEnergy(q,p)+E
   END SUBROUTINE SetPress


   !> Sets energy given temperature t and conservative vector q
   SUBROUTINE SetTemp(q, t)
      REAL(KIND=qPREC), DIMENSION(:) :: q
      REAL(KIND=qPREC) :: t, E
      E=half*SUM(q(m_low:m_high)**2)/q(1)
      IF (lMHD) E=E+half*SUM(q(iBx:iBz)**2)
      q(iE)=GetInternalEnergy(q,q(1)*t)+E
   END SUBROUTINE SetTemp

   !> Calculate sound speed
   !! @param q fluid variables (conservative form)
   FUNCTION SoundSpeed(q)
      REAL (KIND=qPrec), DIMENSION(:), INTENT(IN) :: q
      REAL (KIND=qPrec) :: SoundSpeed, Eth
      IF (iEOS == EOS_ISOTHERMAL) THEN
         SoundSpeed = sqrt(Iso_Speed2)
      ELSE
         Eth=q(iE)- half*DOT_PRODUCT(q(m_low:m_high),q(m_low:m_high))/q(1)
         IF(lMHD) Eth=Eth - half*DOT_PRODUCT(q(iBx:iBz),q(iBx:iBz))
         SELECT CASE(iEOS)
         CASE(EOS_IDEAL_GAS) ! Ideal
            SoundSpeed = sqrt(gamma*gamma1*Eth/q(1))
         CASE DEFAULT
            PRINT*,'unimplemented EOS, iEOS=',iEOS
            STOP
         END SELECT
      END IF
   END FUNCTION SoundSpeed


   !> Calculate sound speed
   !! @param q fluid variables (conservative form)
   FUNCTION PrimSoundSpeed(q)
      REAL (KIND=qPrec), DIMENSION(:), INTENT(IN) :: q
      REAL (KIND=qPrec) :: PrimSoundSpeed
      IF (iEOS == EOS_ISOTHERMAL) THEN
         PrimSoundSpeed = sqrt(Iso_Speed2)
      ELSE
         SELECT CASE(iEOS)
         CASE(EOS_IDEAL_GAS) ! Ideal
            PrimSoundSpeed = sqrt(gamma*q(iE)/q(1))
         CASE DEFAULT
            PRINT*,'unimplemented EOS, iEOS=',iEOS
            STOP
         END SELECT
      END IF
   END FUNCTION PrimSoundSpeed


   !> Calculate internal kinetic energy 
   !! @param q fluid variables (conservative form)
   PURE FUNCTION InternalEnergy(q)
      REAL (KIND=qPrec), DIMENSION(:), INTENT(IN) :: q
      REAL (KIND=qPrec) :: InternalEnergy
      ! Internal variables
      REAL (KIND=qPrec) :: gam,kappa1,invkappa1,chi1,eth,ke,rho,invq1,Bsq,rho_cgs,eth_cgs
      ! Caching certain frequently-used derived values.
      IF (iEOS == EOS_ISOTHERMAL) THEN
         InternalEnergy=1.5d0*q(1)*Iso_Speed2
      ELSE
         InternalEnergy=q(iE)-half*DOT_PRODUCT(q(m_low:m_high),q(m_low:m_high))/q(1)
         IF(lMHD) InternalEnergy=InternalEnergy-half*SUM(q(iBx:iBz)**2)
      END IF
   END FUNCTION InternalEnergy


   !> Calculate pressure and return optional variables if present
   !! @param q fluid variables (primitive form - iE is internal energy not pressure)
   !! @param kappa [dP/drho]Eth
   !! @param chi [dP/dEth]rho
   !! @param DPDQ Derivative of pressure wrt conserved variables
   FUNCTION PrimPress(q,kappa,chi,DPDQ)
      !   USE COOL
      !   USE TF
      !   USE VANDERWAALS
      REAL (KIND=qPrec), DIMENSION(:), INTENT(IN) :: q
      REAL (KIND=qPrec), OPTIONAL, INTENT(OUT) :: kappa,chi
      REAL (KIND=qPrec), DIMENSION(:), OPTIONAL, INTENT(OUT) :: DPDQ
      REAL (KIND=qPrec) :: PrimPress
      ! Internal variables
      REAL (KIND=qPrec) :: gam,kappa1,invkappa1,chi1,eth,ke,rho,invq1,Bsq,rho_cgs,eth_cgs


      ! Caching certain frequently-used derived values.

      !    IF(iCooling/=2) THEN
      SELECT CASE(iEOS)
      CASE(EOS_IDEAL_GAS) ! Ideal
         kappa1=gamma1
         chi1=0.d0
         PrimPress = q(iE)
         !       CASE(1) ! analytic corrected-thomas-fermi
         !          print *, "Need to rewrite TFEOS for Primitive Variables"
         !            rho_cgs=rho*rScale
         !            eth_cgs=eth*pScale
         !            CALL TFEOS(rho_cgs,eth_cgs,Press,kappa1,chi1)
         !            Press=Press/pScale
         !            chi1=chi1*rScale/pScale
         !       CASE(2) !Sesame
         ! set pressure, kappa1, chi1 here.
         ! scaling factors (cgs): nScale (# density), rScale (density), pScale, velScale, lScale, runTimeSc
         !rho_cgs=rho*rScale
         !eth_cgs=eth*pScale
         !CALL SesameEOS(rho*rScale,Eth*pScale,Press,kappa1,chi1)
         !Press=Press/pScale
         !chi=chi1*rScale/pScale
         !kappa=kappa1*rScale/pScale
         !       CASE(3) ! van der Waals
         !          Call VDW_EOS(q(1),q(iE),PrimPress,chi1,kappa1)
      CASE(EOS_ISOTHERMAL)
         kappa1=0.0
         chi1=Iso_Speed2
         PrimPress = Iso_Speed2*q(1)
      CASE DEFAULT
         PRINT*,'unimplemented EOS, iEOS=',iEOS
         STOP
      END SELECT
      !    ELSE
      !       CALL EOS_vars(q,gamma=gam)
      !       kappa1=gam-1
      !       chi1=0.d0
      !       PrimPress = kappa1*q(iE)
      !    END IF

      ! Return derivatives of p w.r.t. the field variables in q.
      IF(PRESENT(DPDQ)) THEN
         DPDQ = zero
         DPDQ(1)        = chi1
         IF (iE .ne. 0) DPDQ(iE)       = 1d0
      END IF
      ! if present, return chi and kappa to caller
      IF(Present(chi)) chi=chi1
      IF(Present(kappa)) kappa=kappa1
   END FUNCTION PrimPress

   !> Converts region of info structure from primitive form to conservative
   !! @param Info Info structure
   !! @param mB indices defining region
   SUBROUTINE Prim_To_Cons_info(Info,mB)
      TYPE(InfoDef) ::  Info
      INTEGER, DIMENSION(:,:) :: mB
      INTEGER :: i,j,k
      REAL(KIND=qpREC), DIMENSION(:,:,:,:), POINTER :: q
      q=>Info%q
      DO i=mb(1,1),mb(1,2)
         DO j=mb(2,1),mb(2,2)
            DO k=mb(3,1),mb(3,2)
               q(i,j,k,m_low:m_high)=q(i,j,k,m_low:m_high)*q(i,j,k,1)
               IF (iE /= 0) THEN 
                  q(i,j,k,iE)=gamma7*q(i,j,k,iE)+half*(SUM(q(i,j,k,m_low:m_high)**2)/q(i,j,k,1))
                  IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)+half*SUM(q(i,j,k,iBx:iBz)**2)
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE Prim_To_Cons_info



   !> Converts region of info structure from primitive form to conservative
   !! @param Info Info structure
   !! @param mB indices defining region
   SUBROUTINE Cons_To_Prim_info(Info,mB)
      TYPE(InfoDef) ::  Info
      INTEGER, DIMENSION(:,:) :: mB
      INTEGER :: i,j,k
      REAL(KIND=qpREC), DIMENSION(:,:,:,:), POINTER :: q
      q=>Info%q
      DO i=mb(1,1),mb(1,2)
         DO j=mb(2,1),mb(2,2)
            DO k=mb(3,1),mb(3,2)
               q(i,j,k,m_low:m_high)=q(i,j,k,m_low:m_high)/q(i,j,k,1)
               IF (iE /= 0) THEN 
                  q(i,j,k,iE)=gamma1*(q(i,j,k,iE)-half*(SUM(q(i,j,k,m_low:m_high)**2)*q(i,j,k,1)))
                  IF (lMHD) q(i,j,k,iE)=q(i,j,k,iE)-gamma1*half*SUM(q(i,j,k,iBx:iBz)**2)
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE Cons_To_Prim_info

   !> Converts region of info structure from primitive form to conservative
   !! @param Info Info structure
   !! @param mB indices defining region
   SUBROUTINE Cons_To_Prim_q(q)
      TYPE(InfoDef) ::  Info
      REAL(KIND=qpREC), DIMENSION(:) :: q
      q(m_low:m_high)=q(m_low:m_high)/q(1)
      if (iE == 0 ) return
      q(iE)=gamma1*(q(iE)-half*(SUM(q(m_low:m_high)**2)*q(1)))
      IF (lMHD) q(iE)=q(iE)-gamma1*half*SUM(q(iBx:iBz)**2)
   END SUBROUTINE Cons_To_Prim_q


   !> Converts region of info structure from primitive form to conservative
   !! @param Info Info structure
   !! @param mB indices defining region
   SUBROUTINE prim_To_cons_q(q)
      REAL(KIND=qpREC), DIMENSION(:) :: q
      q(m_low:m_high)=q(1)*q(m_low:m_high)
      IF (iE == 0) RETURN  
      q(iE)=gamma7*q(iE)+half*(SUM(q(m_low:m_high)**2)/q(1))
      IF (lMHD) q(iE)=q(iE)+half*SUM(q(iBx:iBz)**2)

   END SUBROUTINE Prim_To_cons_q



  !> Convert primitive variables to conservative using an isothermal eos
  !! @param w primitive array
  !! @param u conservative array
  SUBROUTINE rprim_to_cons_Iso(w,u)
    REAL(KIND=qPrec), DIMENSION(:) :: w,u
    u=w
    u(m_low:m_high)=w(m_low:m_high)*w(1)
  END SUBROUTINE rprim_to_cons_Iso

  !> Calculate fluxes using primitive array
  !! @param w primitive array
  !! @param f conservative fluxes
  SUBROUTINE f_prim_Iso(w,f)
    REAL(KIND=qPrec), DIMENSION(:) :: w,f
    REAL(KIND=qPrec) :: p, pT, B2,vx    
    INTEGER :: n
    n=size(f)
    pT=w(1)*Iso_Speed2
    f(1)=w(1)*w(2)
    f(2)=f(1)*w(2)+pT
    f(3:m_high)=f(1)*w(3:m_high)
    f(m_high+1:n)=w(m_high+1:n)*w(2)
  END SUBROUTINE f_prim_Iso

  !> Calculate the fast wave speed for an isothermal gas
  !! @param w primitive array
  FUNCTION fast_speed_Iso(w)
    REAL(KIND=qPrec), DIMENSION(:) :: w
    REAL(KIND=qPrec) :: fast_speed_Iso, a2b2, b2, bx2
    bx2=w(5)**2/w(1);
    b2=DOT_PRODUCT(w(6:7),w(6:7))/w(1)+bx2     
    a2b2 = (Iso_Speed2+b2)/2d0
    fast_speed_Iso = sqrt((a2b2+sqrt(a2b2**2-Iso_Speed2*bx2)))
  END FUNCTION fast_speed_Iso

  !> Function to test if abs(x) is less then 1e-100
  FUNCTION miniscule(x)
    logical :: miniscule
    REAL(8) :: x
    miniscule = (abs(x)<=1d-100)
  END FUNCTION miniscule

  !> Routine to convert from primitive to conservative with an isothermal eos and magnetic fields
  !! @param w primitive array
  !! @param u conservative array
  SUBROUTINE rprim_to_cons_MHD_Iso(w,u)
    REAL(KIND=qPrec), DIMENSION(:) :: w,u
    u=w
    u(2:4)=w(2:4)*w(1)
  END SUBROUTINE rprim_to_cons_MHD_Iso

  !> Routine to calculate flux given primitive fields for an isothermal eos with magnetic fields
  !! @param w primitive array
  !! @param f conservative fluxes
  SUBROUTINE f_prim_MHD_Iso(w,f)
    REAL(KIND=qPrec), DIMENSION(:) :: w,f
    REAL(KIND=qPrec) :: p, pT, B2,vx    
    INTEGER :: n
    n=size(f)
    B2=DOT_PRODUCT(w(5:7),w(5:7))
    pT=w(1)*Iso_Speed2+half*B2
    f(1)=w(1)*w(2)
    f(2)=f(1)*w(2)+pT-w(5)**2
    f(3)=f(1)*w(3)-w(5)*w(6)
    f(4)=f(1)*w(4)-w(5)*w(7)
    f(5)=0d0
    f(6)=w(6)*w(2)-w(5)*w(3)
    f(7)=w(7)*w(2)-w(5)*w(4)
    f(8:n)=w(8:n)*w(2)
  END SUBROUTINE f_prim_MHD_Iso


  !> Calculate the sound_speed for an ideal EOS
  !! @param w primitive array
  FUNCTION sound_speed(w)
    REAL(KIND=qPrec), DIMENSION(:) :: w
    REAL(KIND=qPrec) :: sound_speed
    sound_speed=sqrt(gamma*w(2)/w(1))
  END FUNCTION sound_speed

  !> Calculate the sound speed for an ideal EOS
  !! @param w Roe form of primitive variables
  FUNCTION sound_speed_Roe(w)
    REAL(KIND=qPrec), DIMENSION(:) :: w
    REAL(KIND=qPrec) :: sound_speed_Roe
    sound_speed_Roe=sqrt(gamma1*(w(2)-half*SUM(w(3:m_high+1)**2)))
  END FUNCTION sound_speed_Roe

  !> Convert from primitive form to Roe form
  !! @param w primitive array
  !! @param r Roe array
  SUBROUTINE prim_to_roe(w,r)
    REAL(KIND=qPrec), DIMENSION(:) :: w, r
    r=w
    r(2)=half*(gamma10*w(2)/w(1)+SUM(w(3:m_high+1)**2)) !gamma10 = 2*gamma/(gamma-1)
  END SUBROUTINE prim_to_roe

  !> Convert from primitive form to Roe form with magnetic fields
  !! @param w primitive array
  !! @param r Roe array  
  SUBROUTINE prim_to_roe_MHD(w,r)
    REAL(KIND=qPrec), DIMENSION(:) :: w, r
    r=w
    r(2)=half*(gamma10*w(2)/w(1)+SUM(w(3:m_high+1)**2))+SUM(w(6:8)**2)/w(1) !gamma10 = 2*gamma/(gamma-1)
  END SUBROUTINE prim_to_roe_MHD

  !> Calculate the fast speed for MHD 
  !! @param w primitive array
  FUNCTION fast_speed(w)
    REAL(KIND=qPrec), DIMENSION(:) :: w
    REAL(KIND=qPrec) :: fast_speed, B, B2
    B2=DOT_PRODUCT(w(6:8),w(6:8))
    B = gamma*w(2)+B2
    fast_speed = sqrt(half* (B + sqrt(B**2-4d0*gamma*w(2)*w(6)**2) ) /w(1) )
  END FUNCTION fast_speed

  !> Calculate the fast speed for MHD
  !! @param w Roe array
  FUNCTION fast_speed_Roe(w)
    REAL(KIND=qPrec), DIMENSION(:) :: w
    REAL(KIND=qPrec) :: fast_speed_Roe, B, B2, P
    B2=DOT_PRODUCT(w(6:8),w(6:8))
    P=gamma14*(w(1)*(w(2)-half*SUM(w(3:m_high+1)**2))-B2) !gamma14 = (gamma-1)/gamma
    B = gamma*P+B2
    fast_speed_Roe = sqrt(half*(B+sqrt(B**2-4d0*gamma*P*w(6)**2))/w(1))
  END FUNCTION fast_speed_Roe

  !> Convert from primitive to conservative for ideal EOS
  !! @param w primitive array
  !! @param u conservative array
  SUBROUTINE rprim_to_cons(w,u)
    REAL(KIND=qPrec), DIMENSION(:) :: w,u
    u=w
    u(3:m_high+1)=w(3:m_high+1)*w(1)
    u(2)=gamma7*w(2)+half*DOT_PRODUCT(u(3:m_high+1),w(3:m_high+1))
  END SUBROUTINE rprim_to_cons

  !> Convert from primitive to conservative for ideal EOS with magnetic fields
  !! @param w primitive array
  !! @param u conservative array
  SUBROUTINE rprim_to_cons_MHD(w,u)
    REAL(KIND=qPrec), DIMENSION(:) :: w,u
    u=w
    u(3:5)=w(3:5)*w(1)
    u(2)=gamma7*w(2)+half*(DOT_PRODUCT(u(3:5),w(3:5))+DOT_PRODUCT(w(6:8),w(6:8)))
  END SUBROUTINE rprim_to_cons_MHD

  !> Calculate conservative fluxes for an ideal EOS
  !! @param w primitive array
  !! @param f conservative fluxes
  SUBROUTINE F_prim(w,f)
    REAL(KIND=qPrec), DIMENSION(:) :: w,f
    REAL(KIND=qPrec) :: p, Energy
    INTEGER :: n
    n=size(f)
    Energy=gamma7*w(2)+half*DOT_PRODUCT(w(3:m_high+1),w(3:m_high+1))*w(1)
    f(1)=w(1)*w(3)
    f(2)=(Energy+w(2))*w(3)
    f(3)=f(1)*w(3)+w(2)
    f(4:m_high+1)=f(1)*w(4:m_high+1)
    f(m_high+2:n)=w(m_high+2:n)*w(3)
  END SUBROUTINE f_prim

  !> Calculate conservative fluxes for an ideal EOS with magnetic fields
  !! @param w primitive array
  !! @param f conservative fluxes
  SUBROUTINE F_prim_MHD(w,f)
    REAL(KIND=qPrec), DIMENSION(:) :: w,f
    REAL(KIND=qPrec) :: p, pT, B2,Energy
    INTEGER :: n
    n=size(f)
    B2=DOT_PRODUCT(w(6:8),w(6:8))
    pT=w(2)+half*B2
    Energy=gamma7*w(2)+half*(DOT_PRODUCT(w(3:5),w(3:5))*w(1)+DOT_PRODUCT(w(6:8),w(6:8)))
    f(1)=w(1)*w(3)
    f(2)=(Energy+pT)*w(3)-w(6)*DOT_PRODUCT(w(3:5),w(6:8))
    f(3)=f(1)*w(3)+pT-w(6)**2
    f(4)=f(1)*w(4)-w(6)*w(7)
    f(5)=f(1)*w(5)-w(6)*w(8)
    f(6)=0d0
    f(7)=w(7)*w(3)-w(6)*w(4)
    f(8)=w(8)*w(3)-w(6)*w(5)
    f(9:n)=w(9:n)*w(3)
  END SUBROUTINE F_prim_MHD



   !> Calculate sound speed
   !! @param q fluid variables (conservative form)
   FUNCTION GetMaxSpeed(q)
      REAL (KIND=qPrec), DIMENSION(:,:,:,:), INTENT(IN) :: q
      REAL (KIND=qPREC) :: GetMaxSpeed, fast_speed(3), A2, B2, B
      INTEGER :: i,j,k
      GetMaxSpeed=0d0
      DO i=1, size(q,1)
         DO j=1, size(q,2)
            DO k=1, size(q,3)
               IF (lMHD) THEN
                  B2 = SUM(q(i,j,k,iBx:iBz)**2)
                  A2=SoundSpeed(q(i,j,k,:))*q(i,j,k,1)
                  B = A2+B2
                  fast_speed(1:nDim) = sqrt(half*(B+sqrt(B**2-4d0*A2*q(i,j,k,iB(1:nDim))**2))/q(i,j,k,1))
                  GetMaxSpeed=max(GetMaxSpeed, maxval(q(i,j,k,imom(1:nDim))/q(i,j,k,1)+fast_speed(1:nDim)))
               ELSE
                  GetMaxSpeed=max(GetMaxSpeed, sqrt(sum(q(i,j,k,imom(1:nDim))**2))/q(i,j,k,1)+SoundSpeed(q(i,j,k,:)))
               END IF
            END DO
         END DO
      END DO
   END FUNCTION GetMaxSpeed


   SUBROUTINE PrintQ(Info,q, tnow, i, j, k)
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC), DIMENSION(:) :: q
      REAL(KIND=qPREC) :: tnow
      INTEGER, OPTIONAL :: i,j,k
      write(*,'(A,20E13.2)') 'q= ',q(:)
      write(*,'(A,20E13.2)') 'Info%q= ',Info%q(i,j,k,:)
      IF (PRESENT(i) .AND. PRESENT(j) .AND. (Present(K))) THEN
         write(*,'(A,20E20.8)') 'pos= ', Info%xBounds(:,1)+(/i,j,k/)*levels(Info%level)%dx-half*levels(Info%level)%dx
         write(*,'(A,6I6)') 'i,j,k,mx,my,mz',i,j,k,Info%mX
         !          write(*,'(A,6I6)') 'Info%mGlobal= ',Info%mGlobal
      END IF
      write(*,'(A,6I6)') 'Info%level',Info%level
      write(*,'(A,20E20.8)') 'tnow= ', tnow
      !         write(*,'(A,6I6)') 'Info%Step',levels(Info%level)%step

      !          write(*,'(A,6I6)') 'mb=', mB
   END SUBROUTINE PrintQ

   !> Calculate eigen system
   !! @param request_eigens Logical flag that returns whether or not calc_eigens was successfull
   !! @param prim fluid variables
   !! @param req_dim Logical flag that specifies which dimensions to calculate eigen system for
   !! @param lambda wave speeds
   !! @param n number of independent waves for each system
   !! @param l left eigenvectors
   !! @param r right eigenvectors
   !! @param i cell x-index
   !! @param j cell y-index
   !! @param k cell z-index
   SUBROUTINE calc_eigens(request_eigens, prim, req_dim, lambda, n, l, r,i,j,k,level)
      REAL(KIND=qPrec), DIMENSION(:) :: prim
      REAL(KIND=qPrec), DIMENSION(:,:) :: lambda
      REAL(KIND=qPrec), DIMENSION(:,:,:)  :: r,l
      INTEGER, DIMENSION(:) :: n
      INTEGER :: i,j,k,m,dim,ii,jj,kk,level
      REAL(KIND=qPrec), DIMENSION(:,:,:), ALLOCATABLE :: A_matrix
      REAL(8) :: rho,u,Bx,By,Bz,rho_i,sqrt_rho,sqrt_rho_i,ca,cx,cy,cz,&
           C_,Cy_,Cz_,a2,a,rhoa2,cf2,cs2,beta_f,beta_s, srho, alphaf, alphas, betay, betaz, cff, css, qf, qs, af, as, nf, ns, cfmcs, bt2, &
           alpha,cf,cs,bT,ds,df,s,temp,temp1,temp2,A_,B_,D_,E_,F_,G_,hy,hz,hyr,hzr,b2, a_i, a2_i
      REAL(8), PARAMETER :: sqrt2_i=.7071067811865476d0, sqrt2=1.4142135623730950
      LOGICAL :: req_dim(:)
      LOGICAL :: request_eigens
      rho=prim(1)
      rho_i=1d0/prim(1)
      IF (iEOS==EOS_ISOTHERMAL) THEN
         a2=Iso_Speed2
         a=Iso_Speed
      ELSE
         a2=gamma*prim(iE)*rho_i
         a=sqrt(a2)
      END IF
      a_i=1d0/a
      a2_i=a_i**2
      lambda=0d0
      r=0d0
      l=0d0
      n=0
      rhoa2=rho*a2
      
      IF (.NOT. lMHD) THEN
         n(1:nDim)=NrWaves
         lambda(1:nDim,1)=prim(ivx:ivx+nDim-1)-a
         lambda(1:nDim,2:NrWaves-1) = SPREAD(prim(ivx:ivx+nDim-1),2,NrWaves-2)
         lambda(1:nDim,NrWaves)=prim(ivx:ivx+nDim-1)+a

         IF (iEOS==EOS_ISOTHERMAL) THEN
            r(1:nDim,1,1:2)=SPREAD((/1d0,-a*rho_i/),1,nDim)
            r(1:nDim,NrWaves,1)=r(1:nDim,1,1)
            r(1:nDim,NrWaves,2)=-r(1:nDim,1,2)

            l(1:nDim,1,1:2)=SPREAD((/half,-half*rho*a_i/),1,nDim)
            l(1:nDim,NrWaves,1)=l(1:nDim,1,1)
            l(1:nDim,NrWaves,2)=-l(1:nDim,1,2)

            DO dim=2,NrWaves-1
               r(1:nDim,dim,dim+1)=1d0
               l(1:nDim,dim,dim+1)=1d0
            END DO
         ELSE
            r(1:nDim,1,1:3)=SPREAD((/1d0,a2,-a*rho_i/),1,nDim)
            r(1:nDim,NrWaves,1:2)=r(1:nDim,1,1:2)
            r(1:nDim,NrWaves,3)=-r(1:nDim,1,3)
            r(1:nDim,2,1)=1d0

            l(1:nDim,1,2:3)=SPREAD((/half*a2_i,-half*rho*a_i/),1,nDim)
            l(1:nDim,NrWaves,2)=l(1:nDim,1,2)
            l(1:nDim,NrWaves,3)=-l(1:nDim,1,3)
            l(1:nDim,2,1:2)=SPREAD((/1d0,-a2_i/),1,nDim)

            DO dim=3,NrWaves-1
               r(1:nDim,dim,dim+1)=1d0
               l(1:nDim,dim,dim+1)=1d0
            END DO
         END IF
      ELSE
         sqrt_rho=sqrt(rho)
         sqrt_rho_i=1d0/sqrt_rho
         Bx=prim(iBx)
         By=prim(iBy)
         Bz=prim(iBz)
         cx=Bx*sqrt_rho_i
         cy=By*sqrt_rho_i
         cz=Bz*sqrt_rho_i
         C_=half*cx**2
         Cy_=half*cy**2
         Cz_=half*cz**2      

         A_=half*a2
         D_=A_+Cy_+Cz_+C_

         DO dim=1, nDim
            IF (.NOT. req_dim(dim)) THEN 
               temp=cx
               cx=cy
               cy=cz
               cz=temp
               temp=C_
               C_=Cy_
               Cy_=Cz_
               Cz_=temp
               temp=Bx
               Bx=By
               By=Bz
               Bz=temp
               CYCLE
            END IF

            IF (Bx == 0) THEN
               s=0d0
            ELSE
               s = sign(1d0,Bx)
            END IF

            B_=Cy_+Cz_
            E_=2d0*sqrt(A_*C_)
            F_ = A_+B_-C_
            G_ = 2d0*sqrt(B_*C_)
            bt=sqrt(2d0*B_)
            IF (my_small(E_,D_)) THEN
               cs2 = E_**2/(2d0*D_)
               cf2 = 2d0*D_
            ELSE
               cs2 = D_ - sqrt(D_**2 - E_**2)
               cf2 = D_ + sqrt(D_**2 - E_**2)
            END IF
            cs = sqrt(cs2)
            cf = sqrt(cf2)
!            write(*,*) 'cs = ', cs
!            write(*,*) 'cf = ', cf
!            write(*,*) A_, B_, C_, D_, E_, F_, G_
!            STOP
            u=prim(ivx+dim-1)     
            ca=abs(cx)
            cfmcs=(cf2-cs2)
            IF (cfmcs < tiny(cfmcs)) THEN
               alphaf=1
               alphas=0
            ELSE
               alphaf=sqrt((a**2-cs**2)/cfmcs)
               alphas=sqrt((cf**2-a**2)/cfmcs)
               IF (ISNAN(alphas) .OR. ISNAN(alphaf)) THEN
                  alphas=0
                  alphaf=1
               END IF
            END IF
            bt2=sqrt(by**2+bz**2)
            IF (bt2 < tiny(bt2)) THEN
               betay=0d0
               betaz=0d0
            ELSE
               betay=by/bt2
               betaz=bz/bt2
            END IF
            !                write(*,*) bt2, betay, betaz, alphaf, alphas, cfmcs, tiny(cfmcs), tiny(cfmcs)
            cff=cf*alphaf
            css=cs*alphas
            qf=cf*alphaf*s
            qs=cs*alphas*s
            af=a*alphaf*sqrt(rho)
            as=a*alphas*sqrt(rho)
            rhoa2=rho*a2
            nf=half*a2_i
            ns=nf

!            srho=sqrt_rho
            
!            write(*,*) 'rho=', rho
!            write(*,*) 'Bx=', Bx
!            write(*,*) 'By=', By
!            write(*,*) 'Bz=', Bz
!            write(*,*) 'a=', a
!            write(*,*) 'cf=', cf
!            write(*,*) 'cs=', cs
!            write(*,*) 'cff=',cff
!            write(*,*) 'css=',css
!            write(*,*) 'qf=', qf
!            write(*,*) 'qs=', qs
!            write(*,*) 'af=', af
!            write(*,*) 'as=', as
!            write(*,*) 'rhoa2=', rhoa2
!            write(*,*) 'sqrt_rho=', sqrt_rho
!            write(*,*) 'nf=', nf
!            write(*,*) 'ns=', ns

            IF (iEOS==EOS_ISOTHERMAL) THEN

               r(dim,1,1:6)=(/rho*alphaf, -cff, qs*betay, qs*betaz, as*betay, as*betaz/)
               r(dim,2,1:6)=(/zero, zero, -betaz, betay, -betaz*s*sqrt_rho, betay*s*sqrt_rho/)
               r(dim,3,1:6)=(/rho*alphas, -css, -qf*betay, -qf*betaz, -af*betay, -af*betaz/)
               r(dim,NrWaves-2,1:6)=(/r(dim,3,1),-r(dim,3,2:4),r(dim,3,5:6)/)
               r(dim,NrWaves-1,1:6)=(/r(dim,2,1),-r(dim,2,2:4),r(dim,2,5:6)/)
               r(dim,NrWaves,1:6)=(/r(dim,1,1),-r(dim,1,2:4),r(dim,1,5:6)/)
               
               l(dim,1,1:6)=(/ nf*alphaf*a2*rho_i, -nf*cff,  nf*qs*betay,  nf*qs*betaz,    nf*as*betay*rho_i,    nf*as*betaz*rho_i/)
               l(dim,2,1:6)=(/ zero,    zero   ,   -half*betaz,   half*betay,       -half*betaz*s*sqrt_rho_i,  half*betay*s*sqrt_rho_i/)
               l(dim,3,1:6)=(/ ns*alphas*a2*rho_i, -ns*css, -ns*qf*betay, -ns*qf*betaz,    -ns*af*betay*rho_i,   -ns*af*betaz*rho_i/)
               l(dim,NrWaves-2,1:6)=(/l(dim,3,1),-l(dim,3,2:4),l(dim,3,5:6)/)
               l(dim,NrWaves-1,1:6)=(/l(dim,2,1),-l(dim,2,2:4),l(dim,2,5:6)/)
               l(dim,NrWaves,1:6)=(/l(dim,1,1),-l(dim,1,2:4),l(dim,1,5:6)/)


               n(dim)=NrWaves
               lambda(dim,1) = u - cf
               lambda(dim,2) = u - ca
               lambda(dim,3) = u - cs
               lambda(dim,NrWaves-2) = u + cs
               lambda(dim,NrWaves-1) = u + ca
               lambda(dim,NrWaves) = u + cf

               lambda(dim,5:NrWaves-3) = u

               DO ii=5, NrWaves-3
                  l(dim,ii,NrCons-5+ii)=1d0
                  r(dim,ii,NrCons-5+ii)=1d0
               END DO

            ELSE

               r(dim,1,1:7)=(/rho*alphaf, rhoa2*alphaf, -cff, qs*betay, qs*betaz, as*betay, as*betaz/)

               r(dim,2,1:7)=(/zero, zero, zero, -betaz, betay, -betaz*s*sqrt_rho, betay*s*sqrt_rho/)
               r(dim,3,1:7)=(/rho*alphas, rhoa2*alphas, -css, -qf*betay, -qf*betaz, -af*betay, -af*betaz/)
               r(dim,4,1:7)=(/one, zero, zero, zero, zero, zero, zero/)
               r(dim,NrWaves-2,1:7)=(/r(dim,3,1:2),-r(dim,3,3:5),r(dim,3,6:7)/)
               r(dim,NrWaves-1,1:7)=(/r(dim,2,1:2),-r(dim,2,3:5),r(dim,2,6:7)/)
               r(dim,NrWaves,1:7)=(/r(dim,1,1:2),-r(dim,1,3:5),r(dim,1,6:7)/)
               
               l(dim,1,1:7)=(/ zero,    nf*alphaf*rho_i, -nf*cff,  nf*qs*betay,  nf*qs*betaz,    nf*as*betay*rho_i,    nf*as*betaz*rho_i/)
               l(dim,2,1:7)=(/ zero,    zero,      zero   ,   -half*betaz,   half*betay,       -half*betaz*s*sqrt_rho_i,  half*betay*s*sqrt_rho_i/)
               l(dim,3,1:7)=(/ zero, ns*alphas*rho_i, -ns*css, -ns*qf*betay, -ns*qf*betaz,    -ns*af*betay*rho_i,   -ns*af*betaz*rho_i/)
               l(dim,4,1:7)=(/ one,    -one/a**2,    zero,         zero,       zero,           zero,                 zero/)
               l(dim,NrWaves-2,1:7)=(/l(dim,3,1:2),-l(dim,3,3:5),l(dim,3,6:7)/)
               l(dim,NrWaves-1,1:7)=(/l(dim,2,1:2),-l(dim,2,3:5),l(dim,2,6:7)/)
               l(dim,NrWaves,1:7)=(/l(dim,1,1:2),-l(dim,1,3:5),l(dim,1,6:7)/)
               

               n(dim)=NrWaves
               lambda(dim,1) = u - cf
               lambda(dim,2) = u - ca
               lambda(dim,3) = u - cs
               lambda(dim,4) = u
               lambda(dim,NrWaves-2) = u + cs
               lambda(dim,NrWaves-1) = u + ca
               lambda(dim,NrWaves) = u + cf

               lambda(dim,5:NrWaves-3) = u

               DO ii=5, NrWaves-3
                  l(dim,ii,NrCons-5+ii)=1d0
                  r(dim,ii,NrCons-5+ii)=1d0
               END DO

            END IF
            temp=cx
            cx=cy
            cy=cz
            cz=temp
            temp=C_
            C_=Cy_
            Cy_=Cz_
            Cz_=temp
            temp=Bx
            Bx=By
            By=Bz
            Bz=temp

         END DO
      END IF
   END SUBROUTINE calc_eigens


      !> Logical function that determines whether a modified approximation should be used
      !! @param x value
      !! @param y scaling value
      FUNCTION my_small(x,y)
         REAL(8), PARAMETER:: cutoff=1d-4
         logical :: my_small
         REAL(8) :: x,y
         my_small = (y .ne. 0d0 .AND. abs(x/y) <=cutoff)
      END FUNCTION my_small

      !> Logical function that determines whether a modified approximation should be used
      !! @param x value
      !! @param y scaling value
      FUNCTION really_small(x,y)
         logical :: really_small
         REAL(8) :: x,y
         really_small = (y .ne. 0d0 .AND. .1d0*x/y  <= 1d-16)
      END FUNCTION really_small

END MODULE EOS


