!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    PFFT.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE PFFT
  USE globaldeclarations
  USE LayoutDeclarations
  USE LayoutComms
  USE TreeDeclarations
  USE DataDeclarations
  USE Fields
  IMPLICIT NONE
  INCLUDE 'fftw3.f'
  SAVE
  PRIVATE

  TYPE PFFTPlanDef
     TYPE(pLayoutDef), DIMENSION(:), ALLOCATABLE :: Layouts
     COMPLEX(8), DIMENSION(:,:,:,:), POINTER :: data !transforms done 'in place'     
     INTEGER, DIMENSION(3,2) :: mB !Global bounds
     INTEGER, DIMENSION(3,2) :: lmB !Local bounds
  END type PFFTPlanDef

  INTEGER, PUBLIC, PARAMETER :: FORWARD = FFTW_FORWARD, BACKWARD = FFTW_BACKWARD
  
  
  PUBLIC :: CreatePlan, DestroyPlan, PFFTPlanDef, LoadFieldIntoPFFT, UnLoadFieldFromPFFT, ExecutePlan, SpectralProlongation


CONTAINS

  !> Executes sequence of 1D FFTs interlaced with necessary communication
  !! @param plan parallel fft plan
  !! @param direction fft direction FFTW_FORWARD or FFTW_BACKWARD
  SUBROUTINE ExecutePlan(plan, direction)
    TYPE(PFFTPlanDef) :: plan
    INTEGER :: direction, p
    INTEGER :: i
    DO i=1,nDim
       CALL DoFFT(plan%data, i, direction)
       CALL TransferLayouts(plan%layouts(i)%p,plan%layouts(mod(i,nDim)+1)%p, plan%data)
    END DO
  END SUBROUTINE ExecutePlan

  SUBROUTINE DoFFT(data, dir, direction)
    COMPLEX(8), DIMENSION(:,:,:,:), POINTER :: data
    COMPLEX(8), DIMENSION(:,:,:), POINTER :: in, out
    INTEGER :: dir, mx(3), ip(3,2), i, j, k, direction
    INTEGER(8) :: p
    mx=1
    mx(dir)=size(data,dir)
    ALLOCATE(in(mx(1),mx(2),mx(3)), out(mx(1),mx(2),mx(3)))
    ip(dir,:)=(/lbound(data,dir),ubound(data,dir)/)    
    CALL dfftw_plan_dft_1d(p, mx(dir), in, out, direction, FFTW_ESTIMATE)
    DO i=lBound(data,mod(dir,3)+1), ubound(data,mod(dir,3)+1)
       DO j=lBound(data,mod(dir+1,3)+1), ubound(data,mod(dir+1,3)+1)
          DO k=1,size(data,4)
             ip(mod(dir,3)+1,:)=i
             ip(mod(dir+1,3)+1,:)=j
             in=data(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),k)
             CALL dfftw_execute(p)
             data(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),k)=out
          END DO
       END DO
    END DO
    CALL dfftw_destroy_plan(p)
    DEALLOCATE(in, out)
  END SUBROUTINE DoFFT

  SUBROUTINE CreatePlan(plan, level, mB, fields)
    INTEGER, DIMENSION(3,2) :: mB
    TYPE(PFFTPlanDef), POINTER :: plan
    INTEGER :: i, fields, level
    ALLOCATE(plan)
    plan%mB=mB
    ALLOCATE(plan%layouts(1:nDim))
    DO i=1,nDim
       mB(i,:)=plan%mB(i,1)  !Shrink the bounds along each direction so that the FFT can be done
       CALL CreateLayout(mB, plan%layouts(i)%p)
       plan%layouts(i)%p%mB(:,i,2)=plan%mB(i,2) !re-expand the layouts 
       plan%layouts(i)%p%level=level
       mB(i,:)=plan%mB(i,:)
    END DO
    plan%lmB=plan%layouts(1)%p%mB(MPI_ID,:,:)
    ALLOCATE(plan%data(plan%lmB(1,1):plan%lmB(1,2), plan%lmB(2,1):plan%lmB(2,2), plan%lmB(3,1):plan%lmB(3,2),fields))
  END SUBROUTINE CreatePlan

  SUBROUTINE DestroyPlan(plan)
    TYPE(PFFTPlanDef), POINTER :: plan
    INTEGER :: i
    DO i=1,nDim
       CALL DestroyLayout(plan%layouts(i)%p)
    END DO
    DEALLOCATE(plan%layouts)
    DEALLOCATE(plan%data)
    DEALLOCATE(plan)
    NULLIFY(plan)
  END SUBROUTINE DestroyPlan

  SUBROUTINE LoadFieldIntoPFFT(plan,  FieldID)
    TYPE(PFFTPlanDef), POINTER :: plan
    INTEGER :: FieldID(:)
    INTEGER, DIMENSION(3,2) :: mB
    CALL LoadFieldIntoLayout(plan%layouts(1)%p, plan%data, FieldID)
  END SUBROUTINE LoadFieldIntoPFFT


  SUBROUTINE UnLoadFieldFromPFFT(plan, FieldID, lPeriodic, rmbc)
    TYPE(PFFTPlanDef), POINTER :: plan
    INTEGER :: FieldID(:,:)
    INTEGER, DIMENSION(3,2) :: mB
    INTEGER :: rmbc
    LOGICAL, DIMENSION(:) :: lPeriodic
!    mB=plan%layouts(1)%p%mB(MPI_ID,:,:)
!    ALLOCATE(plan%data(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2)))
    CALL UnLoadFieldFromLayout(plan%layouts(1)%p, plan%data, FieldID, lPeriodic, rmbc)
  END SUBROUTINE UnLoadFieldFromPFFT
  


  SUBROUTINE SpectralProlongation(plan, newplan)
     TYPE(PFFTPlanDef), POINTER :: plan, newplan     
     INTEGER :: i, j, k, n, l(3), mB(3,2), mC(3,2), mD(3,2), splitpoints(3,2), mA(3,2)
     REAL(KIND=qPREC) :: dphi(3), phase, kvec(3)
     !First do shifting




!     mB=plan%lmB
!     DO i=mB(1,1),mB(1,2)
!        DO j=mB(2,1), mB(2,2)
!           DO k=mB(3,1), mB(3,2)
!              kvec=SpectraK((/i,j,k/),plan%mB)
!              phase=sum(dphi * kvec)
!              plan%data(i,j,k,:)=plan%data(i,j,k,:)*exp(cmplx(0d0,phase))
!           END DO
!        END DO
!     END DO
!     plan%data=plan%data/product(plan%mB(:,2)-plan%mB(:,1)+1)

     !consider an 8 point plan
     ! the wave vectors will be
     ! 0 1 2 3 4 -3 -2 -1
     ! and the split point is 5
     ! we want to divide the 5th entry in half and send 1:5 and 5:8
     
     !consider a 7 point plan
     ! 0 1 2 3 -3 -2 -1
     ! the split point is 4 
     ! and we want to send 1:4 and 5:7

     splitpoints(:,1)=(sum(plan%mB,2)+0)/2+1
     splitpoints(:,2)=(sum(plan%mB,2)+1)/2+1
    
     
     ! First cut duplicates reals in half.
     mA=plan%mB
     mB=plan%lmB
     DO i=1,nDim
        IF (splitpoints(i,1)==splitpoints(i,2) .AND. (splitpoints(i,1) >= mB(i,1) .AND. splitpoints(i,1) <= mB(i,2))) THEN
           mA(i,:)=splitpoints(i,1)
           plan%data(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2),:)=half*&
                plan%data(mA(1,1):mA(1,2), mA(2,1):mA(2,2), mA(3,1):mA(3,2),:)
           mA(i,:)=mB(i,:)
        END IF
     END DO

     ! Then stretch fourier components

     newplan%data=0d0
     l=0
     DO i=0,2**nDim-1
        mC=plan%mB
        mD=newplan%mB
        DO n=1,nDim
           l(n)=MOD(i/2**(n-1),2)
           IF (l(n) == 0) THEN
              mC(n,2)=splitpoints(n,1)
              mD(n,2)=mD(n,1)+(mC(n,2)-mC(n,1))
           ELSE
              mC(n,1)=splitpoints(n,2)+1
              mD(n,1)=mD(n,2)-(mC(n,2)-mC(n,1))
           END IF
        END DO
        CALL LayoutTransferC(mC, mD, plan%layouts(1)%p, newplan%layouts(1)%p, plan%data, newplan%data)
     END DO

     ! And finally adjust phases due to grid point locations.

     dphi=-Pi/REAL(newplan%mB(:,2)-newplan%mB(:,1)+1)
     mB=newplan%lmB
     DO i=mB(1,1),mB(1,2)
        DO j=mB(2,1), mB(2,2)
           DO k=mB(3,1), mB(3,2)
              kvec=SpectraK((/i,j,k/),newplan%mB)
              phase=sum(dphi * kvec)
              newplan%data(i,j,k,:)=newplan%data(i,j,k,:)*exp(cmplx(0d0,phase))
           END DO
        END DO
     END DO

     ! And divide final data by size of original FFT.
     newplan%data=newplan%data/product(plan%mB(:,2)-plan%mB(:,1)+1)

  END SUBROUTINE SpectralProlongation

END MODULE PFFT
