!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    common_functions.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file common_functions.f90
!! @brief Main file for module CommonFunctions

!> defgroup CommonFunctions
!! @brief Module containing many common miscellaneous functions
!! @ingroup Physics

!> Module containing many common miscellaneous functions
!! @ingroup CommonFunctions
MODULE CommonFunctions
   USE GlobalDeclarations
   USE PhysicsDeclarations
   IMPLICIT NONE

   PUBLIC

   INTERFACE AddCurl
      MODULE PROCEDURE AddCurl2D, AddCurl3D
   END INTERFACE


   INTEGER, PARAMETER :: INOSMOOTH = 0, ITANH = 1, IDBL_TANH=2, IDECAYING_EXP=3
   INTEGER, PUBLIC, PARAMETER :: NOSOFT = 0, & !g ~ r/r^2 for all r
        SPLINESOFT = 1, & !g ~ r/r^2 for r < r_soft and then goes to 0 as r-> 0 
        PLUMMERSOFT = 2 !g ~ (r^2+r_soft^2)^(-3/2) r

CONTAINS


   !> @name AddCurl Routines
   !! @{

   !> Takes curl of a and adds it to b
   !! @param b output vector field
   !! @param a input vector field
   !! @param dx grid spacing
   SUBROUTINE AddCurl2D(b,a,dx)
      REAL(KIND=qPREC), DIMENSION(:,:) :: a
      REAL(KIND=qPREC), DIMENSION(:,:,:) :: b
      REAL(KIND=qPREC) :: dx
      INTEGER :: mx, my
      mx=size(a,1)-1
      my=size(a,2)-1
      b(1:mx+1,1:my,1)=b(1:mx+1,1:my,1)+&
           (a(1:mx+1,2:my+1)-a(1:mx+1,1:my))/dx
      b(1:mx,1:my+1,2)=b(1:mx,1:my+1,2)-&
           (a(2:mx+1,1:my+1)-a(1:mx,1:my+1))/dx
   END SUBROUTINE AddCurl2D

   !> Takes curl of a and adds it to b
   !! @param b output vector field
   !! @param a input vector field
   !! @param dx grid spacing
   SUBROUTINE AddCurl3D(b,a,dx)
      REAL(KIND=qPREC), DIMENSION(:,:,:,:) :: a
      REAL(KIND=qPREC), DIMENSION(:,:,:,:) :: b
      REAL(KIND=qPREC) :: dx
      INTEGER :: mx, my, mz
      mx=size(a,1)-1
      my=size(a,2)-1
      mz=size(a,3)-1
      b(1:mx+1,1:my,1:mz,1)=b(1:mx+1,1:my,1:mz,1)+&
           ((a(1:mx+1,2:my+1,1:mz,3)-a(1:mx+1,1:my,1:mz,3)) &
           -(a(1:mx+1,1:my,2:mz+1,2)-a(1:mx+1,1:my,1:mz,2)))/dx
      b(1:mx,1:my+1,1:mz,2)=b(1:mx,1:my+1,1:mz,2)+&
           ((a(1:mx,1:my+1,2:mz+1,1)-a(1:mx,1:my+1,1:mz,1)) &
           -(a(2:mx+1,1:my+1,1:mz,3)-a(1:mx,1:my+1,1:mz,3)))/dx
      b(1:mx,1:my,1:mz+1,3)=b(1:mx,1:my,1:mz+1,3)+&
           ((a(2:mx+1,1:my,1:mz+1,2)-a(1:mx,1:my,1:mz+1,2)) &
           -(a(1:mx,2:my+1,1:mz+1,1)-a(1:mx,1:my,1:mz+1,1)))/dx           
   END SUBROUTINE AddCurl3D


   !> @}


   !> @name Smoothing Functions
   !! @{



   FUNCTION SmoothFunction(r, ifunc, smooth)
      REAL(KIND=qPREC) :: r, smooth, SmoothFunction
      INTEGER :: ifunc
      SELECT CASE(ifunc)
      CASE (INOSMOOTH)
         SmoothFunction=1d0
      CASE(ITANH)
         SmoothFunction=smooth_tanh(r, smooth)
      CASE(IDBL_TANH)
         SmoothFunction= smooth_dbl_tanh(r, smooth)
      CASE(IDECAYING_EXP)
         SmoothFunction=exp(-r**2/smooth**2)
      CASE DEFAULT
         PRINT*, 'Unrecognized smoothing option in common_functions.f90'
         STOP
      END SELECT
   END FUNCTION SmoothFunction

   !> Function that performs tangential smoothing
   !! @param r radial factor (ratio of distance from cell to clump center divided by clump radius)
   !! @param smooth smoothing width
   FUNCTION smooth_tanh(r,smooth)
      REAL(KIND=qPREC) :: smooth_tanh, r, smooth
      IF (r >= 1d0) THEN
         smooth_tanh=0
      ELSEIF (r < 1d0-smooth) THEN
         smooth_tanh=1
      ELSE
         smooth_tanh=tanh((1d0-r)/smooth)/tanh(1d0)
      END IF
   END FUNCTION smooth_tanh

   !> Function that performs tangential smoothing
   !! @param r radial factor (ratio of distance from cell to clump center divided by clump radius)
   !! @param smooth smoothing width
   FUNCTION smooth_dbl_tanh(r,smooth)
      REAL(KIND=qPREC) :: smooth_dbl_tanh, r, smooth
      IF (r >= 1) THEN
         smooth_dbl_tanh=0d0
      ELSEIF (r < 1d0-smooth) THEN
         smooth_dbl_tanh=1d0
      ELSE
         smooth_dbl_tanh=half*(1d0+tanh(((1d0-smooth/2d0)-r)/(smooth/2d0))/tanh(1d0))
      END IF
   END FUNCTION smooth_dbl_tanh


   !> @}



   FUNCTION SphericalRotation(x, theta, phi)
      REAL(KIND=qPREC), DIMENSION(3) :: x, SphericalRotation
      REAL(KIND=qPREC) :: theta, phi
      SphericalRotation=rotate_z(rotate_x(x,-phi), -theta)
   END FUNCTION SphericalRotation

   FUNCTION InvSphericalRotation(x, theta, phi)
      REAL(KIND=qPREC), DIMENSION(3) :: x, InvSphericalRotation
      REAL(KIND=qPREC) :: theta, phi
      InvSphericalRotation=rotate_x(rotate_z(x,theta), phi)
   END FUNCTION InvSphericalRotation

  !> Rotates a vector v a degrees around the x-axis
  !! @param v vector to rotate
  !! @param a angle
  FUNCTION rotate_x(v,a)
    REAL(KIND=qPrec) :: v(3), a, rotate_x(3)
    rotate_x = (/           v(1), &
         COS(a)*v(2) - SIN(a)*v(3), &
         SIN(a)*v(2) + COS(a)*v(3)/)
  END FUNCTION rotate_x

  !> Rotates a vector v a degrees around the y-axis
  !! @param v vector to rotate
  !! @param a angle
  FUNCTION rotate_y(v,a)
    REAL(KIND=qPrec) :: v(3), a, rotate_y(3)

    rotate_y = (/COS(a)*v(1) + SIN(a)*v(3), &
         v(2), &
         -SIN(a)*v(1) + COS(a)*v(3)/)
  END FUNCTION rotate_y

  !> Rotates a vector v a degrees around the z-axis
  !! @param v vector to rotate
  !! @param a angle
  FUNCTION rotate_z(v,a)
    REAL(KIND=qPrec) :: v(3), a, rotate_z(3)
    rotate_z = (/COS(a)*v(1) - SIN(a)*v(2), &
         SIN(a)*v(1) + COS(a)*v(2), &
         v(3)/)
  END FUNCTION rotate_z

  FUNCTION rotate_z2D(v,a)
    REAL(KIND=qPrec) :: v(2), a, rotate_z2D(2)
    rotate_z2D = (/COS(a)*v(1) - SIN(a)*v(2), &
         SIN(a)*v(1) + COS(a)*v(2)/)
  END FUNCTION rotate_z2D


  !Set up and solve the Least Square Fitting based on observed times
  !Times is the input data set, ncase is the number of input cases, nvar is the size of the solution vector. 
  !sls stores the solution vector in an order of (xy,x,y,1) or (xyz,xy,yz,zx,x,y,z,1)
  !errls stores the relative error for the fitting
  SUBROUTINE LSSolver(JLs,lsdata,sls,errls)
    IMPLICIT NONE
    REAL(KIND=qPREC), INTENT(IN), DIMENSION(:) :: lsdata
    REAL(KIND=qPREC), INTENT(IN), DIMENSION(:,:) :: JLs
    REAL(KIND=qPREC), INTENT(INOUT), DIMENSION(:) :: sls
    REAL(KIND=qPREC), OPTIONAL, DIMENSION(:) :: errls

    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: temp
    INTEGER :: i,j,nvar
    REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: Jlsnew

    nvar=size(sls,1)
    ALLOCATE(temp(nvar))
    ALLOCATE(Jlsnew(1:nvar,1:nvar))

    ! Jlsnew is transpose(Jls)*Jls
    DO j=1,nvar
       DO i=1,nvar
          Jlsnew(i,j)=dot_product(Jls(:,i),Jls(:,j))
       END DO
    END DO

    ! Invert Jlsnew
    CALL MatrixInvert(Jlsnew,nvar) 
    
    ! Find the solution by the explicit equation
    DO i=1,nvar
       temp(i)=dot_product(Jls(:,i),lsdata(:))
    END DO
    DO i=1,nvar
       sls(i) = dot_product(Jlsnew(i,:),temp(:))
    END DO

    IF (present(errls)) THEN
       ! compute the error vector of this predictor
       DO i=1,size(errls)
          errls(i)=abs(lsdata(i)-dot_product(Jls(i,:),sls(:)))/lsdata(i)
       END DO
    END IF
    DEALLOCATE(Jlsnew,temp)
  END SUBROUTINE LSSolver
  
  ! Invert any input matrix A using LU decomposition, return its inversion stored in A.
  ! Input n is the size of the matrix A. tiny works as the tolerance, which can
  ! be varied according to needs
  SUBROUTINE MatrixInvert(A,n)
    REAL(KIND=qPREC), INTENT(INOUT), DIMENSION(1:n,1:n) :: A
    INTEGER, INTENT(IN) :: n

    INTEGER :: i,j,k,imax,ii,ll
    INTEGER, DIMENSION(:), ALLOCATABLE :: indx
    REAL(KIND=qPREC), PARAMETER :: tiny = 1.0d-20
    REAL(KIND=qPREC) :: aamax,summ,dum,d
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: v
    REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: temp,temp2

    d=1d0;
    ALLOCATE(indx(1:n))
    ALLOCATE(v(1:n))
    ALLOCATE(temp(1:n,1:n))
    ALLOCATE(temp2(1:n,1:n))
    
    temp=0d0; temp2=A
    DO i = 1, n
       aamax = 0d0
       DO j = 1, n
          IF(abs(A(i,j)).gt.aamax)THEN
             aamax = abs(A(i,j))
          END IF
       END DO
       IF (aamax.eq.0d0)THEN
          PRINT*, 'Found singular matrix in LSMatrixInvert. STOP'
          STOP
       END IF
       v(i) = 1d0/aamax
       temp(i,i)=1d0
    END DO

    DO j = 1, n
       DO i = 1, j-1
          summ = A(i,j)
          summ = summ - dot_product(A(i,1:i-1),A(1:i-1,j))
          A(i,j) = summ
       END DO
       aamax = 0d0
       DO i = j, n
          summ = A(i,j)
          summ = summ - dot_product(A(i,1:j-1),A(1:j-1,j))
          A(i,j) = summ
          dum = v(i)*abs(summ)
          IF(dum.ge.aamax)THEN
             imax = i
             aamax = dum
          END IF
       END DO
       IF(j.ne.imax)THEN
          DO k = 1, n
             dum = A(imax,k)
             A(imax,k) = A(j,k)
             A(j,k) = dum
          END DO
          d = -d
          v(imax) = v(j)
       END IF
       indx(j) = imax
       IF(A(j,j).eq.0d0)THEN
          A(j,j) = tiny
       END IF
       IF(j.ne.n)THEN
          dum = 1d0/A(j,j)
          DO i = j+1, n
             A(i,j) = A(i,j)*dum
          END DO
       END IF
    END DO
    
    DO j=1,n
       ii = 0
       DO i=1,n
          ll=indx(i)
          summ=temp(ll,j)
          temp(ll,j)=temp(i,j)
          IF(ii.ne.0)THEN
             summ=summ-dot_product(A(i,ii:i-1),temp(ii:i-1,j))
          ELSE IF(summ.ne.0d0)THEN
             ii=i
          END IF
          temp(i,j)=summ
       END DO
       DO i=n,1,-1
          summ=temp(i,j)
          summ=summ-dot_product(A(i,i+1:n),temp(i+1:n,j))
          temp(i,j)=summ/A(i,i)
       END DO
    END DO

    A=temp

    DEALLOCATE(indx)
    DEALLOCATE(v)
    DEALLOCATE(temp)
    DEALLOCATE(temp2)

    RETURN
    
  END SUBROUTINE MatrixInvert

  FUNCTION GravityPotential(mass,pos,r_soft,soft_function)
     REAL(KIND=qPREC), DIMENSION(3) :: pos
     REAL(KIND=qPREC) :: GravityPotential
     INTEGER :: soft_function
     REAL(KIND=qPREC) :: mass, r_soft, r, r2,theta(3),kmin(3), k(3), fact
     INTEGER , PARAMETER :: NMax=32
     INTEGER :: i,j,l
     
!     IF (.NOT. lGravityPeriodic) THEN
        r2=sum(pos(1:nDim)**2)
        r=sqrt(r2)
        SELECT CASE(soft_function)
        CASE(NOSOFT)
           GravityPotential=-1d0/r
        CASE(SPLINESOFT)
           GravityPotential=SplinePotential(r,r_soft)
        CASE(PLUMMERSOFT)
           GravityPotential=PlummerPotential(r,r_soft)
        END SELECT
!     ELSE
  
!        rho = M*(delta_func)
!        rho^(k) = M*1d0
!        phi^(k) = 4*Pi*G*M*1d0/k^2
!        phi = int(phi^{k) e(2Pi ikx) dk) = sum(phi^(k) e^(2Pi ikx)) delta_k**nDim
!        = 4*Pi*G*M*sum(e^{2Pi ikx)/k^2) * delta_k**nDim
!        = 4*Pi*G*M*delta_k**nDim * (sum(cos/k^2)
! has units of G M l^(2-nDim)
! so for nDim == 3 has units of GM/l
! for nDim == 2 has units of GM log(l)
! for nDim == 1 has units of GM l

!        GravityPotential=0d0


!        kmin(1:nDim)=2d0*Pi/(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))
!        SELECT CASE(nDim)
!        CASE(1)
!           DO i=1,Nmax
!              k(1)=kmin(1)*i
!              theta(1)=k(1)*pos(1)
!              GravityPotential=GravityPotential+cos(theta(1))/k(1)**2
!           END DO
!        CASE(2)
!           DO i=0,Nmax
!              k(1)=kmin(1)*i
!              theta(1)=k(1)*pos(1)
!              DO j=0,Nmax
!                 IF (i == 0 .AND. j == 0) CYCLE
!                 IF (i == 0 .OR. j == 0) THEN
!                    fact=2d0
!                 ELSE
!                    fact=4d0
!                 END IF
!                 k(2)=kmin(2)*j
!                 theta(2)=k(2)*pos(2)
!                 GravityPotential=GravityPotential+fact*cos(theta(1))*cos(theta(2))/sum(k(1:2)**2)
!              END DO
!           END DO
!        CASE(3)
!           DO i=1,Nmax
!              k(1)=kmin(1)*i
!              theta(1)=k(1)*pos(1)
!              DO j=1,Nmax
!                 k(2)=kmin(2)*j
!                 theta(2)=k(2)*pos(2)
!                 DO l=1,Nmax
!                    k(3)=kmin(3)*l
!                    theta(3)=k(3)*pos(3)
!                    GravityPotential=GravityPotential+cos(sum(theta(1:3)))/sum(k(1:3)**2)
!                 END DO
!              END DO
!           END DO
!        END SELECT
!        GravityPotential=GravityPotential*4d0*Pi/product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1))
!     END IF
     GravityPotential=GravityPotential*mass*ScaleGrav
  END FUNCTION GravityPotential


  FUNCTION GravityForce(mass,pos,r_soft,soft_function)
     REAL(KIND=qPREC), DIMENSION(3) :: GravityForce, pos
     INTEGER :: soft_function
     REAL(KIND=qPREC) :: mass, r_soft, r, r2
     SELECT CASE(soft_function)
     CASE(NOSOFT)
        r2=sum(pos(1:nDim)**2)
        r=sqrt(r2)
        GravityForce=-pos/(r*r2)
     CASE(SPLINESOFT)
!        write(*,*) 'spline r_soft=', r_soft
        GravityForce=-SplineSoftening(pos,r_soft)
     CASE(PLUMMERSOFT)
!        write(*,*) 'plummer r_soft=', r_soft
        GravityForce=-PlummerSoftening(pos,r_soft)
     END SELECT
     GravityForce=GravityForce*mass*ScaleGrav
  END FUNCTION GravityForce

   !> Gravitational softening parameter
   FUNCTION SplineSoftening(pos,r_soft)
      REAL(KIND=qPREC) :: r,r_soft,y,pos(3),r2
      REAL(KIND=qPREC) :: SplineSoftening(3)
      REAL(KIND=qPREC), PARAMETER :: &
           GSC1=1.06666666666666667d+1, &!4*(8/3)
           GSC2=-38.4d0,& !4*(-48/5)
           GSC3=32d0,& !4*(8)
           GSC4=2.13333333333333333d+1,& !4*(16/3)
           GSC5=-48d0,& !4*(-12)
           GSC6=38.4d0,& !4*(48/5)
           GSC7=-1.06666666666666667d+1,& !4*(-8/3)
           GSC8=-6.66666666666666667d-2 !4*(-1/60)
      INTEGER :: i
      IF (GravityDim == 3) THEN
         r2=sum(pos(1:nDim)**2)
         r=sqrt(r2)     
         y=r/r_soft
         IF (r >= r_soft) THEN
            SplineSoftening=pos/(r2*r)
         ELSE
            y=r/r_soft
            IF (r_soft**2*r <= TINY(1d0)) THEN 
               SplineSoftening=0d0
            ELSE
               IF (r >= half*r_soft) THEN
                  SplineSoftening=pos * (y*(GSC4+y*(GSC5+y*(GSC6+y*GSC7)))+GSC8/y**2) / (r_soft**2*r)
               ELSE
                  SplineSoftening=pos * (y*(GSC1+y**2*(GSC2+y*GSC3))) / (r_soft**2*r)
               END IF
            END IF
         END IF
      ELSE
         SplineSoftening=PlummerSoftening2D(pos,r_soft)
      END IF
  END FUNCTION SplineSoftening

   !> Potential for SplineSoftening
   FUNCTION SplinePotential(r,r_soft)
      REAL(KIND=qPREC) :: r,r_soft,y,pos(3),r2
      REAL(KIND=qPREC) :: SplinePotential
      REAL(KIND=qPREC), PARAMETER :: &
           GSC0=-14d0/5d0, &
           GSC1=16d0/3d0, &
           GSC2=-48d0/5d0, &
           GSC3=32d0/5d0, &
           GSC4=32d0/3d0, &
           GSC5=-16d0, &
           GSC6=48d0/5d0, &
           GSC7=-32d0/15d0, &
           GSC8=1d0/15d0, &
           GSC9=-16d0/5d0
      y=r/r_soft

      IF (GravityDim == 3) THEN
         IF (r >= r_soft) THEN
            SplinePotential=-1d0/r
         ELSE
            y=r/r_soft
            IF (r >= half*r_soft) THEN
               SplinePotential=(GSC9+(y**2*(GSC4+y*(GSC5+y*(GSC6+y*GSC7)))+GSC8/y )) / r_soft
            ELSE
               SplinePotential=(GSC0 + (y**2*(GSC1+y**2*(GSC2+y*GSC3)))) / r_soft
            END IF
         END IF
      ELSE        
         SplinePotential=PlummerPotential2D(r,r_soft)
      END IF
   END FUNCTION SplinePotential

   !> Gravitational softening parameter
   FUNCTION PlummerSoftening(pos,r_soft)
      REAL(KIND=qPREC) :: r,y,pos(3),r_soft
      REAL(KIND=qPREC) :: PlummerSoftening(3)
      IF (GravityDim == 3) THEN
         PlummerSoftening=pos/((sum(pos**2)+r_soft**2)**(1.5d0))
      ELSE
         PlummerSoftening=PlummerSoftening2D(pos,r_soft)
      END IF
   END FUNCTION PlummerSoftening


   !> Gravitational softening parameter
   FUNCTION PlummerSoftening2D(pos,r_soft)
      REAL(KIND=qPREC) :: r,y,pos(3),r_soft
      REAL(KIND=qPREC) :: PlummerSoftening2D(3)
      PlummerSoftening2D=2d0*pos/(sum(pos(1:nDim)**2)+r_soft**2)
   END FUNCTION PlummerSoftening2D


   !> Potential for PlummerSoftening
   FUNCTION PlummerPotential(r,r_soft)
      REAL(KIND=qPREC) :: r,r_soft, PlummerPotential     
      IF (GravityDim == 3) THEN
         PlummerPotential=-1d0/sqrt(r**2+r_soft**2)
      ELSE
         PlummerPotential=PlummerPotential2D(r,r_soft)
      END IF
   END FUNCTION PlummerPotential


   !> Potential for PlummerSoftening
   FUNCTION PlummerPotential2D(r,r_soft)
      REAL(KIND=qPREC) :: r,r_soft, PlummerPotential2D
      PlummerPotential2D=2d0*log(sqrt(r**2+r_soft**2)/R2DEff)!     (*exp(-.5d0))) !)sqrt(r**2+r_soft**2))
   END FUNCTION PlummerPotential2D


   !> 2D cross product of a with b
   !! @param a 1st vector
   !! @param b 2nd vector
   FUNCTION Cross2D(a,b)
      REAL(KIND=qPREC) :: a(2),b(2),Cross2D
      Cross2D=(a(1)*b(2)-a(2)*b(1)) 
   END FUNCTION Cross2D

   !> 3D cross product of a with b
   !! @param a 1st vector
   !! @param b 2nd vector
   FUNCTION Cross3D(a,b)
      REAL(KIND=qPREC) :: a(3),b(3),Cross3D(3)
      Cross3D=(/a(2)*b(3)-a(3)*b(2),a(3)*b(1)-a(1)*b(3),a(1)*b(2)-a(2)*b(1)/)
   END FUNCTION Cross3D


   !> @brief Solves a x b = c for a where a is a unit vector
   FUNCTION SolveCrossEq(b,c)
     REAL(KIND=qPrec), DIMENSION(3) :: b,c,SolveCrossEq
     SolveCrossEq = (Cross3D(b,c)+sqrt(sum(b**2)-sum(Cross3D(b,c)**2)/sum(b**2))*b)/sum(b**2)
   END FUNCTION SolveCrossEq
   

  FUNCTION fade(r, thickness)
    REAL(KIND=qPrec) :: fade, r, thickness
    IF (r <= 0d0 .OR. thickness <= 0) THEN
       fade = 1d0
    ELSE IF (r < 1d0) THEN
       fade = tanh((1d0-r)/thickness)
    ELSE
       fade = 0d0
    END IF
!    fade=1d0
  END FUNCTION fade

! Returns the associated Legendre Polynomial at point x with orders l and m
 FUNCTION pLgndr(l,m,x)
  REAL(KIND=qPrec) :: pLgndr
  INTEGER :: l, m, ll
  REAL(KIND=qPrec) :: x, pll,pmm,pmmp1,somx2
  IF (.NOT.(m>=0.and.m<=l.and.ABS(x)<=1.0))THEN
     PRINT*, "wrong input in plgndr"
     STOP
  END IF
  pmm=1.0
  IF(m>0) THEN
    somx2=SQRT((1.0-x)*(1.0+x))
    pmm=profac(1d0,2d0,m)*somx2**m
  END IF
  IF(l==m) THEN
    plgndr=pmm
  ELSE
    pmmp1=x*(2*m+1)*pmm
    IF(l==m+1) THEN
      plgndr=pmmp1
    ELSE
      DO ll=m+2,l
        pll=(x*(2*ll-1)*pmmp1-(ll+m-1)*pmm)/(ll-m)
        pmm=pmmp1
        pmmp1=pll
      END DO
      plgndr=pll
    END IF
  END IF
  RETURN
END FUNCTION pLgndr

! Returns Gamma functions at point xx
FUNCTION GAMMAS(xx)
  INTEGER :: i
  LOGICAL :: gamma_neg
  REAL(KIND=qPrec)  :: xx, gammas, tmp, x, summ, temp
  REAL(KIND=qPrec) , PARAMETER :: stp=2.5066282746310005, pi=3.141592653589793238
  REAL(KIND=qPrec) , PARAMETER, DIMENSION(6) :: coef=(/76.18009172947146,&
      -86.50532032941677,24.0140982408391,-1.231739572450155,&
      0.120865097386617e-2,-0.5395239384953e-5/)
  IF(xx<0)THEN
    IF(xx-REAL(FLOOR(xx))==0.) PRINT*, "infinit gamma function value!"
    x=1.-xx
    gamma_neg=.true.
  ELSE
    x=xx
    gamma_neg=.false.
  END IF
  summ=0.0; tmp=x+5.5
  tmp=(x+0.5)*LOG(tmp)-tmp
  DO i=1,6
    summ=summ+coef(i)/(x+REAL(i))
  END DO
  temp=EXP(tmp+LOG(stp*(1.000000000190015+summ)/x))
  IF(gamma_neg)THEN
    gammas=pi/(Sin(pi*xx)*temp)
  ELSE
    gammas=temp
  END IF
  RETURN
END FUNCTION GAMMAS


! Returns Spherical Bessel j at point x with order n
FUNCTION SBesj(n,x)
 INTEGER :: i,ind,n
 REAL(KIND=qPrec) , PARAMETER :: TOL=1.0e-12, pi=3.141592653589793238
 REAL(KIND=qPrec)  :: SBesj,x,k,r
 IF(x==0.) x=TOL
 SBesj=0.
 ind=(n+1)/2;k=sqrt(2./(pi*x))
 DO i =0, ind
    SBesj=SBesj+(-1.)**(n-i)*(Sin(x)*(2./x)**(n-2*i)*(GAMMAS(REAL(n-i+1,qPrec))/GAMMAS(REAL(i+1,qPrec)))*BNomial(-0.5-REAL(i,qPrec),REAL(n-2*i,qPrec))&
    -Cos(x)*(2./x)**(n+1-2*i)*(GAMMAS(REAL(n-i+1,qPrec))/GAMMAS(REAL(i+1,qPrec)))*i*BNomial(-0.5-REAL(i,qPrec),REAL(n-2*i+1,qPrec)))
 END DO
 SBesj=SBesj*k*sqrt(pi/(2.*x))
 RETURN
END FUNCTION SBesj

! Returns Spherical Bessel y at point x with order n
FUNCTION SBesy(n,x)
 INTEGER :: n
 REAL(KIND=qPrec)  :: x,SBesy
 SBesy=-SBesj(-n-1,x)
 RETURN
END FUNCTION SBesy

! Returns the Binomial function with any real order
FUNCTION BNomial(n,k)
  REAL(KIND=qPrec)  :: n,k,BNomial
  BNomial=GAMMAS(n+1.)/(GAMMAS(k+1.)*GAMMAS(n-k+1.))
  RETURN
END FUNCTION BNomial

! Return the product of an arithmetic series.
FUNCTION profac(start,inc,num)
  INTEGER :: k,num
  REAL(KIND=qPrec)  :: profac, start, inc, temp
  profac=start; temp=start
  IF(num>1)THEN
    DO k=1,num-1
      temp=temp+inc
      profac=profac*temp
    END DO
  END IF
  RETURN
END FUNCTION profac

!> Converts from cartesian (x,y,z) to cylindrical (s, phi, z)
  FUNCTION ConvertCylindrical(pos)
     REAL(KIND=qPREC), DIMENSION(3) :: pos, ConvertCylindrical
     ConvertCylindrical=(/sqrt(sum(pos(1:2)**2)),GetPhi(pos(1), pos(2)),pos(3)/)

  END FUNCTION ConvertCylindrical

!> Converts from cartesian (x,y,z) to spherical (r, theta, phi)
  FUNCTION ConvertSpherical(pos)
     REAL(KIND=qPREC), DIMENSION(3) :: pos, ConvertSpherical
     ConvertSpherical(1)=sqrt(sum(pos(1:3)**2))
     ConvertSpherical(2)=acos(pos(3)/ConvertSpherical(1))
     ConvertSpherical(3)=GetPhi(pos(1), pos(2))
  END FUNCTION ConvertSpherical

!> Calculates phi given x and y
 FUNCTION GetPhi(x, y)
     REAL(8) :: GetPhi, x, y
     GetPhi=acos(x/sqrt(x**2+y**2))
     IF (y < 0) THEN
        GetPhi=2d0*Pi-GetPhi
     END IF
  END FUNCTION GetPhi

  !> rho and temp in cu
  !> Returns Jeans Length in cu
  FUNCTION JeansLength(rho, temp)
    REAL(KIND=qPREC) :: rho, temp, JeansLength
    JeansLength=sqrt(gamma*temp*Pi/ScaleGrav/rho)
  END FUNCTION JeansLength


  !> Calculate a vector on a random sphere
  SUBROUTINE random_sphere(A)
    REAL(8), DIMENSION(:) :: A
    REAL :: rand
    REAL(8) :: theta, phi
    call random_number(rand)
    theta=acos((1d0-2d0*rand))
    call random_number(rand)
    phi=rand*2d0*ACOS(-1d0)
    A=(/sin(theta)*cos(phi),sin(theta)*sin(phi),cos(theta)/)
  END SUBROUTINE random_sphere

  SUBROUTINE random_circle(A)
    REAL(8), DIMENSION(:) :: A
    REAL :: rand
    REAL(8) :: phi
    call random_number(rand)
    phi=rand*2d0*ACOS(-1d0)
    A(1:2)=(/cos(phi),sin(phi)/)
  END SUBROUTINE random_circle

   FUNCTION SpectraK(ipos, mB)
     REAL(KIND=qPREC) :: SpectraK(3)
     INTEGER :: ipos(:)
     INTEGER :: mB(:,:)
     INTEGER :: i
     SpectraK=0
     DO i=1,nDim
        IF (ipos(i)==mB(i,1)) THEN
           SpectraK(i)=0
        ELSEIF (ipos(i) < sum(mB(i,:))/2) THEN
           SpectraK(i)=(ipos(i)-mB(i,1))/REAL(mB(i,2)-mB(i,1)+1)*(maxval(mB(:,2)-mB(:,1)+1))
        ELSE
           SpectraK(i)=(ipos(i)-mB(i,2)-1)/REAL(mB(i,2)-mB(i,1)+1)*(maxval(mB(:,2)-mB(:,1)+1))
        END IF
     END DO
   END FUNCTION SpectraK


   FUNCTION Curl2D(v, dx)
      REAL(KIND=qPREC) :: Curl2D, dx
      REAL(KIND=qPREC), DIMENSION(3,3,2) :: v
      Curl2D=(v(3,2,2)-v(1,2,2)-v(2,3,1)+v(2,1,1))/(2d0*dx)
   END FUNCTION Curl2D


   FUNCTION Curl3D(v, dx)
      REAL(KIND=qPREC) :: Curl3D(3), dx
      REAL(KIND=qPREC), DIMENSION(3,3,3,3) :: v
      Curl3D(1)=(v(2,3,2,3)-v(2,1,2,3)-v(2,2,3,2)+v(2,2,1,2))/(2d0*dx)
      Curl3D(2)=(v(2,2,3,1)-v(2,2,1,1)-v(3,2,2,3)+v(1,2,2,3))/(2d0*dx)
      Curl3D(3)=(v(3,2,2,2)-v(1,2,2,2)-v(2,3,2,1)+v(2,1,2,1))/(2d0*dx)
   END FUNCTION Curl3D



END MODULE CommonFunctions
