!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    physics_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir physics
!! @brief Contains modules for setting up physical variables

!> @defgroup Physics Physics
!! @brief Group of modules for managing physical variables

!> @file physics_control.f90
!! @brief Main file for module PhysicsControl

!> @defgroup PhysicsControl Physics Control
!! @brief Routines required to initialize physics data in program.
!! @ingroup Physics

!> Routines required to initialize physics data in program.
!! @ingroup PhysicsControl
MODULE PhysicsControl

  USE GlobalDeclarations
  USE EllipticDeclarations
  USE PhysicsDeclarations
  USE SlopeLim
  USE SourceControl
  
  IMPLICIT NONE
  PRIVATE
  PUBLIC PhysicsInit, PhysicsFinalizeTracers, PhysicsFinalizeInit

CONTAINS

  !> Read in and initialize physical constants (i.e., the contents of the PhysicsDeclarations module.
  SUBROUTINE PhysicsInit()

    INTEGER i,iErr
	 INTEGER :: Correctmbc(3,2)
    ! Open and read in data from the physics data file.
    OPEN(UNIT=PHYSICS_DATA_HANDLE, FILE=PHYSICS_DATA_FILE, STATUS='old', FORM='formatted')
    READ(PHYSICS_DATA_HANDLE, NML=PhysicsData)

    ! Check the correctness of MANDATORY and OPTIONAL parameter initialization
    ! [BDS][20100707]: Commented out source-term version.

    IF (gamma <= zero) THEN
       IF (MPI_ID == 0) write(*,*) 'Gamma must be larger than zero'
       STOP
    END IF

    ! Initialize gamma-related constants for the exact Riemann solver
    gamma1 = gamma-1.0
    gamma2  = 2.0/gamma1
    gamma3  = 0.5*gamma1/gamma
    gamma4  = (gamma+1.)/gamma1
    gamma5  = 2.0/SQRT(2.0*gamma*gamma1)
    gamma6  = 1.0/gamma
    gamma7  = 1.0/gamma1
    gamma8  = (gamma+1)/(2.0*gamma)
    gamma9  = 2.0/(gamma+1.0)
    gamma10 = 2.0*gamma/gamma1
    gamma11 = gamma1/(gamma+1.0)
    gamma12 = 0.5*gamma1
    gamma13 = 2.0-gamma
    gamma14 = gamma1/gamma
    gamma15 = gamma/gamma1

    !Currently no support for source terms.
    IF(nDim > 2   .AND. iCylindrical /= NoCyl) THEN
       IF (MPI_ID.eq.0) PRINT *,'!!! physics_control ERROR: Cylindrical symmetry,', &
            'is not implemented for nDim > 2'
       STOP
    END IF

    IF (iCylindrical /= NoCyl) THEN
       IF (lMHD) THEN
			 IF (iCylindrical /= WithAngMom) THEN
            IF (MPI_ID.eq.0) PRINT *,'!!! setprob WARNING: Setting iCylindrical from &
	   	       NoAngMom to WithMom since lMHD == T'
             iCylindrical = WithAngMom
			 END IF
       END IF
		 IF (ANY(GxBounds(1,:).eq.(/0.0,0.0/) .AND. Gmthbc(1,:) /= (/REFLECT_CYLINDRICAL,REFLECT_CYLINDRICAL/))) THEN
		 	Correctmbc=Gmthbc
			WHERE(GxBounds(1,:) == 0.) CorrectMbc(1,:)=REFLECT_CYLINDRICAL
		   IF (MPI_ID.eq.0) print*,'Left or right boundary corresponds to the z axis. So the boundary conditions Gmthbc is being set to', Correctmbc, 'instead of ', Gmthbc
	      Gmthbc=Correctmbc
       END IF
		 IF (lMHD) THEN
		    IF (ANY(GxBounds(2,:).eq.0. .AND. Gmthbc(2,:) /= REFLECT_BPARALLEL)) THEN
		    	Correctmbc=Gmthbc
		   	WHERE(GxBounds(2,:) == 0) CorrectMbc(2,:)=REFLECT_BPARALLEL
		      IF (MPI_ID.eq.0) print*,'Top or bottom boundary corresponds to the r axis. Perhaps they should be', Correctmbc, 'instead of ', Gmthbc
          END IF
		 ELSE
		    IF (ANY(GxBounds(2,:).eq.0. .AND. (Gmthbc(2,:) /= REFLECT_BPARALLEL .OR. Gmthbc(2,:) /= REFLECT_WALL))) THEN
		    	Correctmbc=Gmthbc
		   	WHERE(GxBounds(2,:) == 0) CorrectMbc(2,:)=REFLECT_BPARALLEL
		      IF (MPI_ID.eq.0) print*,'Top or bottom boundary corresponds to the r axis. Perhaps they should be', Correctmbc, 'instead of ', Gmthbc
          END IF
		 END IF!lMHD
   END IF!icyl

   R2DEff=(sqrt(product(GxBounds(1:2,2)-GxBounds(1:2,1)))/sqrt(pi)*exp(-.5d0))
    ! Source Terms are not implemented in this iteration

    CALL SetConservedVariableIndices()

    IF (lSelfGravity) THEN
       IF (iThreaded /= NON_THREADED) THEN
          IF (MPI_ID == 0) THEN
             PRINT*, 'SelfGravity only supported with iThreading = -1'
             PRINT*, 'Setting iThreading to -1'
          END IF
          iThreaded = NON_THREADED
          LevelBalance=(/0d0,0d0/)
       END IF
       lSinkParticles=.true.
       lParticles=.true.
       lElliptic=.true.
    END IF
    lUniformGravity = UniformGravity /= 0d0

    lExplicit = lResistive.or.lConductive.or.lViscous

    CALL SetPhysicalScales()

    lCanAddTracers=.true.
  END SUBROUTINE PhysicsInit

!> Finalize additional tracers
  SUBROUTINE PhysicsFinalizeTracers()
    !Build Tracers onto q
    CALL SrcInitTracers
    IF (NrTracerVars > 0) THEN
       nTracerHI = NrCons + NrTracerVars
    END IF
    NrHydroVars=NrCons+NrTracerVars
    NrEllipticVars=0
    lCanAddTracers=.false.
  END SUBROUTINE PhysicsFinalizeTracers

 !> After problem modules have initialized and codes have had time to request tracer fields etc... we should be able to finish building the q-array
  SUBROUTINE PhysicsFinalizeInit()
     CALL PhysicsFinalizeTracers()
     CALL FinalizeVariableIndices
  END SUBROUTINE PhysicsFinalizeInit


  !> Assign indices of the q variable array to the codes for certain physical values.
  SUBROUTINE SetConservedVariableIndices
     INTEGER :: i, iEllipticFree, iDiagnostics

     CALL AddFieldVar(irho, "rho")
     CALL AddFieldVar(ivx, "px")
     IF (nDim >= 2 .OR. lMHD) CALL AddFieldVar(ivy, "py")
     IF (nDim == 3 .OR. lMHD .OR. iCylindrical == WithAngMom) CALL AddFieldVar(ivz, "pz")
     IF (iEOS /= EOS_ISOTHERMAL) CALL AddFieldVar(iE, "E")
     IF (lMHD) THEN
        CALL AddFieldVar(iBx, "Bx")
        CALL AddFieldVar(iBy, "By")
        CALL AddFieldVar(iBz, "Bz")
     END IF
     mydim = merge(nDim, 3, iCylindrical == NOCYL)
     GravityDim = mydim
     NrCons=NrFieldVars
     m_low=ivx
     IF (ivz /= 0) THEN
        m_high=ivz
     ELSEIF (ivy /= 0) THEN
        m_high=ivy
     ELSE
        m_high=ivx
     END IF

     iAngMom=ivz
     imom(1:3)=(/ivx,ivy,ivz/)
     iB(1:3)=(/iBx,iBy,iBz/)

    nTracerLO = NrCons+1
    nTracerHI = 0
    NrTracerVars = 0

    IF (lMHD .AND. nDim >= 2) THEN
       nAux=nDim
       IF (nDim == 2) THEN          
          nEmf=1
          iEx=0
          iEy=0
          iEz=1
          EmfLoc=(/0,0,1/)
          EmfDir=(/3,0,0/)
       ELSE
          nEmf=3
          iEx=1
          iEy=2
          iEz=3
          EmfLoc=(/1,2,3/)
          EmfDir=(/1,2,3/)
       END IF
       MaintainAuxArrays=.true.
    ELSE
       nAux=0
    END IF

 END SUBROUTINE SetConservedVariableIndices
 

 SUBROUTINE FinalizeVariableIndices()
    USE EllipticDeclarations
    INTEGER :: i
    ! include elliptics from source terms
!    CALL SrcInitElliptics(NrVars,NrEllipticVars)
    nEllipticLo=NrHydroVars+1
    nEllipticHi=0
    IF (lSelfGravity) THEN
       CALL AddElliptic(iPhiGas, 'Gas Phi')
       CALL AddElliptic(iPhiDot, 'PhiDot')
       IF (iE /= 0) lStoreMassFlux=.true.
    END IF
    NrPhysicalVars=NrHydroVars+NrEllipticVars
    
    IF (lCheckDivergence) THEN
       IF (.NOT. lMHD) THEN
          IF (MPI_ID == 0) write(*,*) 'MHD not true in physics.data so ignoring CheckDivergence'
          lCheckDivergence=.false.
       ELSE
       END IF
    END IF       

    !Build BackupVars
    NrBackupVars=0
    IF (lSelfGravity) THEN
!       irhoOld=NrPhyiscalVars+NrBackupVars+1
!       ivOld(1:nDim)=(/(NrPhysicalVars+NrBackupVars+1+i,i=1,nDim)/)
!       NrBackupVars=NrBackupVars+nDim+1       
    END IF
    NrVars=NrPhysicalVars+NrBackupVars

!  Stuff for the AMR Engine
    nEllipticTransfers=0
    elliptic_mbc=0
    IF (lSelfGravity) THEN
!       nEllipticTransfers=NrEllipticVars !For now transfer all elliptic variables
!       ALLOCATE(EllipticTransferFields(nEllipticTransfers))
!       EllipticTransferFields=(/(nEllipticLo+i-1, i=1, NrEllipticVars)/)
!       elliptic_mbc=1 !Need one more ghost zone so we can take gradient of phi
       EGVars=1!NrEllipticVars
       ALLOCATE(EGCopyFields(EGVars)) 
       EGCopyFields=(/iPhiGas/)!(/(GVars+i,i=1,EGVars)/)
       TDVars=1
       ALLOCATE(TimeDerivFields(1))
       TimeDerivFields=(/iPhiDot/)
    ELSE
       EGVars=0
    END IF

    afterstep_mbc=0
    IF(lResistive)THEN
       afterstep_mbc=afterstep_mbc+1
    END IF
    IF(lConductive)THEN
       afterstep_mbc=afterstep_mbc+1
    END IF
    IF(lViscous)THEN
       afterstep_mbc=afterstep_mbc+1
    END IF

    GVars=NrHydroVars
    ALLOCATE(GCopyFields(GVars))
    GCopyFields=(/(i,i=1,GVars)/)        

    
!    EGVars=0
!    EGCopyF
!    CopyFields=NrHydroVars+NrEllipticVars      
!    ALLOCATE(HydroCopyFields(NrHydroVars))   
    
!    CopyFields(1:nCopyFields)=(/(i,i=1,nCopyFields)/)
    IF (nAux > 0) THEN
       nFlux=NrHydroVars-nAux
       nProlongate=NrHydroVars+NrEllipticVars-nAux
       nRestrict=NrHydroVars-nAux
       ALLOCATE(AuxFields(nAux))
       ALLOCATE(ProlongateFields(nProlongate))
       ALLOCATE(RestrictFields(nRestrict))        
       ALLOCATE(FluxFields(nFlux))        
       AuxFields=(/(i,i=iBx,iBx-1+nAux)/)
       ProlongateFields(1:NrPhysicalVars-nAux)=(/(i,i=1,AuxFields(1)-1),(i,i=AuxFields(nAux)+1,NrPhysicalVars)/)  !Removes auxfields
       RestrictFields(1:NrHydroVars-nAux)=(/(i,i=1,AuxFields(1)-1),(i,i=AuxFields(nAux)+1,NrHydroVars)/)  !Removes auxfields
       FluxFields(1:NrHydroVars-nAux)=(/(i,i=1,AuxFields(1)-1),(i,i=AuxFields(nAux)+1,NrHydroVars)/)  !Removes auxfields
    ELSE
       nFlux=NrHydroVars
       nProlongate=NrPhysicalvars
       nRestrict=NrHydroVars
       ALLOCATE(ProlongateFields(nProlongate))
       ALLOCATE(RestrictFields(nRestrict))        
       ALLOCATE(FluxFields(nFlux))        
       ProlongateFields=(/(i,i=1,NrPhysicalVars)/)
       RestrictFields=(/(i,i=1,NrHydroVars)/)
       FluxFields=(/(i,i=1,NrHydroVars)/)
    END IF

!Setup reverse mapping of flux fields
    ALLOCATE(invFluxFields(1:NrHydroVars))
    invFluxFields=0
    DO i=1,nFlux
       invFluxFields(FluxFields(i))=i
    END DO

    ALLOCATE(InterpMethod(NrPhysicalVars))
    InterpMethod=InterpOpts(1:NrPhysicalVars)
    IF (MPI_ID == 0) THEN
       IF (nAux > 0) THEN
          IF (ANY(InterpMethod(AuxFields(1:nDim)) /= 0)) THEN
             InterpMethod(AuxFields(1:nDim))=0
             write(*,*) 'Warning: InterpMethod should be set to 0 for aux fields to guarantee truncation limits on divergence'
             write(*,'(A,20I3)') 'Consider setting InterpOpts in physics.data to ', InterpMethod
             InterpMethod=InterpOpts(1:NrPhysicalVars)
          END IF
       END IF
    END IF
    IF (iPhiGas > 0) InterpMethod(iPhiGas)=PARABOLIC_INTERP
    IF (iPhiDot > 0) InterpMethod(iPhiDot)=PARABOLIC_INTERP

 END SUBROUTINE FinalizeVariableIndices

  !> Use the data read in from physics.data to calculate the unit scaling of various physical quantities.
  SUBROUTINE SetPhysicalScales

     IF (nScale <= zero .AND. rScale <= zero) THEN
        IF (MPI_ID == 0) PRINT*, 'nScale or rScale must be greater than zero'
        STOP
     END IF
     IF (TempScale <= zero .AND. pScale <= zero) THEN
        IF (MPI_ID == 0) PRINT*, 'TempScale or pScale must be greater than zero'
        STOP
     END IF
     IF (lScale <= zero) THEN
        IF (MPI_ID == 0) PRINT*, 'lScale must be greater than zero'
        STOP
     END IF
     IF (nScale > zero .AND. rScale > zero) nScale=0d0 !revert to using rScale
     IF (TempScale > zero .AND. pScale > zero) TempScale = 0d0 !revert to using pScale
     IF (Xmu == 0d0 .AND. nScale == 0d0 .AND. TempScale == 0d0) THEN ! Dimensionless run.
        TempScale=1d0
        nScale=1d0 !This shouldn't be used if Xmu = 0
     ELSE
        IF (nScale == zero) nScale = rScale/(hMass * Xmu)
        IF (rScale == zero) rScale = nScale * hMass * Xmu
        IF (pScale == zero) pScale = ((rScale/hMass)/Xmu)* TempScale * Boltzmann
        IF (TempScale == zero) TempScale = pScale*Xmu*hMass/(rScale*Boltzmann) !TempScale
     END IF

     VelScale = SQRT(pScale/rScale)
     TimeScale = lScale/VelScale
     BScale=SQRT(4d0 * Pi * pScale)
     mScale=rScale*lScale**3
     ScaleGrav=G*rscale*TimeScale**2  !computational value of 'G'

     MinTemp=MinTemp/TempScale !Converts MinTemp to computational units

     IF (MinTemp <= 0) THEN
        IF (MPI_ID == 0) PRINT*, 'MinTemp must be positive in physics.data.  Setting to 1d-10'
        MinTemp=1d-10
     END IF
     IF (MinDensity <= 0) THEN
        IF (MPI_ID == 0) PRINT*, 'MinDensity must be positive in physics.data.  Setting to 1d-10'
        MinDensity=1d-10
     END IF
     Iso_Speed2=gamma * MinTemp 
     Iso_Speed=sqrt(Iso_Speed2)
     IF (MPI_ID == 0) THEN 
        open(UNIT=SCALES_DATA_HANDLE, file=SCALES_DATA_FILE, status="unknown")
        WRITE(SCALES_DATA_HANDLE, NML=ScalesData)
        CLOSE(SCALES_DATA_HANDLE)
     END IF
  END SUBROUTINE SetPhysicalScales

END MODULE PhysicsControl

