!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    physics_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file physics_declarations.f90
!! @brief Main file for module PhysicsDeclarations

!> @defgroup PhysicsDeclarations Physics Declarations
!! @brief Module containing physics related variables
!! @ingroup Physics

!> Module containing physics related variables
!! @ingroup PhysicsDeclarations
MODULE PhysicsDeclarations

  USE GlobalDeclarations

  IMPLICIT NONE
  PUBLIC
  SAVE
  ! Variables used by user modules and advance modules
  INTEGER, PARAMETER :: MaxVars=20 !

  ! Universal constants in CGS units
  REAL (KIND=qPrec), PARAMETER :: Boltzmann = 1.3806503d-16,                    &
       mSolar    = 1.99d33,                          &
       G         = 6.6731d-8,                        &
       hMass     = 1.67262158d-24,                   &
       au        = 1.496d13,                         &
       Pi        = 3.1415926535897932384626433832795d0, &
       yr        = 31556926 , &
       pc        = 3.08568025e18
  

  ! Cool module related global parameters.
  REAL(KIND=qprec), PUBLIC, PARAMETER :: ev=1.60217646263d-12, & ! ergs/eV
       IonH = 13.5984 * ev,  &
       IonHe = 24.5874 * ev, &
       BindH2 = 4.476 * ev,  & ! molecular binding energy 
       AMU = 1.66044d-24,    & ! g/AMU
       planck = 6.62606876d-27, &
       emass = 9.10938188d-28

  ! molecular weight mumbo jumbo, amu's
  REAL(KIND=qprec), PUBLIC, PARAMETER :: mue = 5.48612529209125d-04 ,       &
       muH = 1.00794d0,                   &
       muHII = half*muH,                  &
       muH2 = 2.d0*muH,                   &
       muHe = 4.002602d0,                 &
       gammaH = 5.d0/3.d0,                &
       gammaH2 = 7.d0/5.d0


  ! Variables specified by user at runtime

  LOGICAL :: lMHD=.false.
  LOGICAL :: lCheckDivergence=.false.
  LOGICAL :: lCanAddTracers=.false.
  LOGICAL :: lGravityPeriodic=.false.

  ! Protection related parameters and variables
  INTEGER, PARAMETER :: MIN_TEMP=0, MIN_NEARBY_PRESS=1, AVG_NEARBY_PRESS=2, MIN_NEARBY_TEMP=3, AVG_NEARBY_TEMP=4
  INTEGER, PARAMETER :: MIN_DENSITY=0, MIN_NEARBY_DENSITY=1, AVG_NEARBY_DENSITY=2
  INTEGER, PARAMETER :: CONSERVE_MOMENTUM=0, CLEAR_MOMENTUM=1, AVG_NEARBY_VEL=2 
  LOGICAL :: lRestartOnPressureProtections = .false.
  LOGICAL :: lRestartOnDensityProtections = .false.
  LOGICAL :: lRestartonHypreMaxIters = .true.
  INTEGER :: iDensityProtect = AVG_NEARBY_DENSITY
  INTEGER :: iMomentumProtect = AVG_NEARBY_VEL
  INTEGER :: iPressureProtect = AVG_NEARBY_TEMP
  REAL(KIND=qPrec) :: MinTemp=1d-10, MinDensity=1d-10
  REAL (KIND=qPrec) :: SrcPrecision = 1d-3 !Source precision

  REAL (KIND=qPrec) :: Xmu=1d0  ! mean molecular weight per particle 
  REAL (KIND=qPrec) :: rScale=0d0, nScale=0d0, TempScale=0d0, pScale=0d0, lScale=0d0, VelScale, TimeScale, BScale, mScale, ScaleGrav!, ScaleCool,  !Scales

  REAL(KIND=qPREC) :: mean_density=0d0

  INTEGER, PARAMETER :: MAXFIELDLENGTH = 20
  INTEGER, PARAMETER :: MAXFIELDVARS=40
  CHARACTER(LEN = MAXFIELDLENGTH), DIMENSION(0:MAXFIELDVARS) :: FieldNames
  ! Indices for mapping quantities to position in q array

  INTEGER :: irho = 0, iE = 0,ivx = 0, ivy = 0, ivz = 0, iBx = 0, iBy = 0, iBz = 0, iPhiDot, iPhiGas, iEntropy, iAngMom,imom(3),ivOld(3),irhoOld, iB(3)
  INTEGER :: nTracerLO,nTracerHI       ! Passive tracer indices
  INTEGER :: NrHydroVars=0
  INTEGER :: NrTracerVars=0
  INTEGER :: NrOutputVars=0
  INTEGER :: NrPhysicalVars=0
  INTEGER :: NrBackupVars=0

  INTEGER, DIMENSION(8) :: var_index

  REAL(KIND=qPREC) :: R2DEff
  LOGICAL :: lSelfGravity=.false., lSinkParticles=.false., lSourceTerms=.true., luniformgravity=.false.
  REAL(KIND=qpREC) :: UniformGravity=0d0
  REAL(KIND=qPREC) :: OmegaRot=0d0
  REAL(KIND=qPREC) :: CellsPerJeansLength=4d0
  REAL(KIND=qPREC) :: DefaultAccretionRoutine = 1

  LOGICAL :: lResistive = .false.
  INTEGER :: ResType
  REAL(KIND=qPREC) :: resistivity

  LOGICAL :: lConductive = .false.
  INTEGER :: ConType
  REAL(kIND=qPREC) :: conductivity
  LOGICAL :: lThermalFluxLimiter
  LOGICAL :: lIsoDiff

  LOGICAL :: lViscous = .false.
  INTEGER :: VisType
  REAL(KIND=qPREC) :: bviscosity

  INTEGER :: iCylindrical! , iCooling , iSymAxis   ! Source switches
  INTEGER,PARAMETER :: NoCyl=0, NoAngMom=1, WithAngMom=2
  INTEGER, PARAMETER:: irdir=1,izdir=2 !Cylindrical directions

  INTEGER :: myDim !physical dimension (=3 for Cylindrical systems)
  INTEGER :: GravityDim !physical dimension for Point Gravity Objects

  ! Other physical variables used globally
  REAL(KIND=qPrec) :: Iso_Speed2, Iso_Speed

  ! Various gamma-related constants for the exact Riemann solver
  REAL (KIND=qPrec) :: gamma=5d0/3d0, gamma1,gamma2,gamma3,gamma4,gamma5,gamma6,        &
       gamma7,gamma8,gamma9,gamma10,gamma11,gamma12,gamma13,gamma14, gamma15


  INTEGER, PARAMETER :: EOS_IDEAL_GAS = 0
  INTEGER, PARAMETER :: EOS_VANDERWAALS = 3
  INTEGER, PARAMETER :: EOS_ISOTHERMAL = 4
  INTEGER :: iEOS=EOS_IDEAL_GAS    ! Index for the desired equation of state.
  INTEGER, DIMENSION(maxVars) :: InterpOpts=0
  REAL(KIND=qPREC), DIMENSION(maxVars) :: refineVariableFactor=1d0

  ! General declarations
  ! File unit numbers
  INTEGER, PARAMETER :: OK=0  

  NAMELIST /PhysicsData/ gamma,Xmu,MinTemp,MinDensity,nScale,rScale,TempScale,lSelfGravity, DefaultAccretionRoutine, iCylindrical, &
       pScale,lScale,iEOS, NrTracerVars, lCheckDivergence, InterpOpts, lMHD, refineVariableFactor, UniformGravity, &
       lResistive, ResType, resistivity, &
       lConductive, ConType, conductivity, lThermalFluxLimiter, lIsoDiff, &
       lViscous, VisType, bviscosity, &
       lRestartOnDensityProtections, lRestartOnPressureProtections, lRestartOnHypreMaxIters, iPressureProtect, iDensityProtect, iMomentumProtect, SrcPrecision, OmegaRot

  NAMELIST /ScalesData/ TimeScale,lScale,mScale,rScale,VelScale,pScale,nScale,BScale,TempScale,ScaleGrav


  !Stuff needed by sweep method should be declared and initialized within sweep module
  INTEGER :: iEx,iEy, iEz, NrCons, m_low, m_high, NrFieldVars, NrWaves




  INTEGER, PARAMETER :: MAXTRACERLENGTH = 20
  INTEGER, PARAMETER :: MAXTRACERVARS=40

  CHARACTER(LEN = MAXTRACERLENGTH), DIMENSION(0:MAXTRACERVARS) :: TracerNames


  INTEGER, ALLOCATABLE, DIMENSION(:) :: oneDx_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: oneDy_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: oneDz_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: wDx_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: wDy_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: wDz_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: fDx_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: fDy_i
  INTEGER, ALLOCATABLE, DIMENSION(:) :: fDz_i


CONTAINS

  !> Adds a tracer field and assigns it to slot i
  !! @param i integer to assign next slot
  !! @param str optional string to define variable 
  SUBROUTINE AddTracer(i, str)
     INTEGER :: i
     CHARACTER(LEN=*), OPTIONAL :: str
     IF (lCanAddTracers) THEN
        i=GetNextTracerIndex()
        NrTracerVars=NrTracerVars+1     
        IF (NrTracerVars <= MaxTracerVars) THEN
           IF (Present(str)) THEN
              TracerNames(NrTracerVars)=str(1:min(len(str), len(TracerNames(NrTracerVars))))
           ELSE
              write(TracerNames(NrTracervars), '(A6I3.3)') 'Tracer', NrTracerVars
           END IF
        ELSE
           IF (MPI_ID == 0) THEN
              PRINT*, 'Warning: Unable to store name of tracer variable without increasing MaxTracers in processing_declarations.f90'
           END IF
        END IF
     ELSE
        IF (MPI_ID == 0) THEN
           PRINT*, 'Warning: Tracers should be added during problem module init or in source init'
        END IF
        i=0
     END IF
  END SUBROUTINE AddTracer

  FUNCTION GetNextTracerIndex()
    INTEGER :: GetNextTracerIndex
    GetNextTracerIndex = nTracerLo+NrTracerVars
  END FUNCTION GetNextTracerIndex

  !> Returns a string containing the name of the tracer
  !! @param i index of tracer
  FUNCTION TracerName(i)
     INTEGER :: i
     CHARACTER(LEN=MAXTRACERLENGTH) :: TracerName

     IF (i <= MaxTracerVars) THEN
        TracerName=TracerNames(i)
     ELSE
        write(TracerName, '(A10I3.3)') 'Tracer', NrTracerVars
     END IF
     
  END FUNCTION TracerName


 !> Adds a diagnostic field and assigns it to slot i
  !! @param i integer to assign next slot
  !! @param str optional string to define variable 
  SUBROUTINE AddFieldVar(i, str)
     INTEGER :: i
     CHARACTER(LEN=*), OPTIONAL :: str
     i=1+NrFieldVars
     NrFieldVars=NrFieldVars+1     
     IF (NrFieldVars <= MaxFieldVars) THEN
        IF (Present(str)) THEN
           FieldNames(NrFieldVars)=str(1:min(len(str), len(FieldNames(NrFieldVars))))
        ELSE
           write(FieldNames(NrFieldvars), '(A10I3.3)') 'Field', NrFieldVars
        END IF
     ELSE
        IF (MPI_ID == 0) THEN
           PRINT*, 'Warning: Unable to store name of diagnostic variable without increasing MaxFieldVars in processing_declarations.f90'
        END IF
     END IF
  END SUBROUTINE AddFieldVar

  !> Returns a string containing the name of the diagnostic
  !! @param i index of diagnostic
  FUNCTION FieldName(i)
     INTEGER :: i
     CHARACTER(LEN=MAXFIELDLENGTH) :: FieldName

     IF (i <= MaxFieldVars) THEN
        FieldName=FieldNames(i)
     ELSE
        write(FieldName, '(A10I3.3)') 'Field', NrFieldVars
     END IF
     
  END FUNCTION FieldName


END MODULE PhysicsDeclarations
!> @}
