!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    cameras.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Cameras
   USE GlobalDeclarations
   USE PhysicsDeclarations
   USE CommonFunctions
   IMPLICIT NONE

   TYPE CameraDef
      REAL(KIND=qPREC), DIMENSION(3) :: pos
      REAL(KIND=qPREC), DIMENSION(3) :: UpVector
      REAL(KIND=qPREC), DIMENSION(3) :: Focus
      REAL(KIND=qPREC) :: FOV(2)
      REAL(KIND=qPREC), DIMENSION(3,3) :: matrix
      INTEGER :: res
      INTEGER :: iD
   END type CameraDef
   INTEGER, SAVE :: nCameras=0


CONTAINS

   SUBROUTINE CreateCamera(Camera)
      TYPE(CameraDef), POINTER :: Camera
      ALLOCATE(Camera)
      Camera%Focus=half*(SUM(GxBounds,2))
      Camera%FOV=(/30d0*Pi/180d0,30d0*Pi/180d0/)
      Camera%UpVector=(/0d0,0d0,1d0/)
      Camera%pos=Camera%Focus
      Camera%pos(2)=Camera%Focus(2)-1d0*(maxval((GxBounds((/1,3/),2)-GxBounds((/1,3/),1))/tan(Camera%FOV)))
      CALL UpdateMatrix(Camera)
      Camera%res=levels(MaxLevel)%mX(1)
      nCameras=nCameras+1
      Camera%iD=nCameras
   END SUBROUTINE CreateCamera


   SUBROUTINE UpdateMatrix(Camera)
      TYPE(CameraDef) :: Camera
      REAL(KIND=qPREC), DIMENSION(3) :: camera_normal(3), ex(3), ey(3)
      camera_normal=Camera%focus-Camera%pos
      camera_normal=camera_normal/sqrt(sum(camera_normal**2))
      ey=Camera%UpVector-SUM(Camera_normal*Camera%UpVector)*Camera_normal
      ey=ey/sqrt(sum(ey**2))
      ex=Cross3D(camera_normal, ey)
      Camera%matrix(:,1)=ex
      Camera%matrix(:,2)=ey
      Camera%matrix(:,3)=camera_normal
!      write(*,*) 'camera_nomral = ', camera_normal
!      write(*,*) 'ey = ', ey
!      write(*,*) 'ex = ', ex
!      write(*,'(3E25.16)') camera%matrix
   END SUBROUTINE UpdateMatrix

   !> Returns the distance location of the projected image.  The values between 0,0 and 1,1 correspond to the image plane.

   FUNCTION GetPixel(Camera, pos)
      TYPE(CameraDef), POINTER :: Camera
      REAL(KIND=qPREC) :: pos(3), GetPixel(2), vec(3)
      INTEGER :: i
      DO i=1,3
         vec(i)=SUM(Camera%matrix(:,i)*(pos-Camera%pos))
      END DO
!      write(*,*) 'vec=', vec
      GetPixel(1)=atan(vec(1)/vec(3))/Camera%FOV(1)+half
      GetPixel(2)=atan(vec(2)/vec(3))/Camera%FOV(2)+half
!      write(*,*) 'atan=', atan(vec(1)/vec(3)), atan(vec(2)/vec(3))
   END FUNCTION GetPixel


   !> Inverse function of GetPixel (except it returns unit vector from camera along given ray)
   FUNCTION GetRay(Camera, pixel)
      TYPE(CameraDef), POINTER :: Camera
      REAL(KIND=qPREC) :: GetRay(3), camera_vec(3), pixel(2)
      INTEGER :: i
      camera_vec(3)=1d0
      camera_vec(1)=tan((pixel(1)-half)*Camera%FOV(1))*camera_vec(3)
      camera_vec(2)=tan((pixel(2)-half)*Camera%FOV(2))*camera_vec(3)
      
      DO i=1,3
         GetRay(i)=SUM(Camera%matrix(i,:)*camera_vec(:))
      END DO
      GetRay=GetRay/sqrt(sum(GetRay**2))

!      write(*,*) GetPixel(camera, Camera%pos+GetRay), pixel
!      write(*,*) Camera%matrix
!      STOP

   END FUNCTION GetRay


   SUBROUTINE BinCell(Camera, pos, dx, data, rho)
      TYPE(CameraDef), POINTER :: Camera
      REAL(KIND=qPREC) :: pos(3),xpos(3)
      REAL(KIND=qPREC) :: dx, ddx, xbounds(3,2), my_pixel(2), pixel(2)
      REAL(KIND=qPREC), DIMENSION(:,:) :: data
      REAL(KIND=qPREC) :: rho,a,intersection(6,3), ray(3), max_distance
      INTEGER :: ipos(2), sample_res, i, j, k, npoints, min_pixels(2), max_pixels(2), dim, odim(2), edge
      xbounds(:,1)=(pos-half*dx)
      xbounds(:,2)=(pos+half*dx)
      min_pixels=huge(min_pixels(1))
      max_pixels=0
      DO i=1,2
         DO j=1,2
            DO k=1,2
               my_pixel=GetPixel(Camera, (/xbounds(1,i), xbounds(2,j), xbounds(3,k)/))*REAL(shape(data), KIND=qPREC)+half
               min_pixels=max(1,min(min_pixels, floor(my_pixel)))
               max_pixels=min(shape(data),max(max_pixels, ceiling(my_pixel)))
            END DO
         END DO
      END DO
      DO i=min_pixels(1), max_pixels(1)
         DO j=min_pixels(2), max_pixels(2)
            pixel=(REAL((/i,j/),KIND=qPREC)-half)/REAL(shape(data), KIND=qPREC)
            Ray=GetRay(Camera, pixel)
            npoints=0
            DO dim=1,3
               DO edge=1,2
                  ! Camera%pos(dim)+a*ray(dim)=xbounds(dim,edge)
                  a=(xbounds(dim,edge)-Camera%pos(dim))/ray(dim)
                  xpos=Camera%pos+a*ray
                  odim=modulo((/dim,dim+1/),3)+1
                  IF (ALL(xpos(odim) >= xbounds(odim,1) .AND. xpos(odim) < xbounds(odim,2))) THEN
                     npoints=npoints+1
                     intersection(npoints,:)=xpos
                  END IF
               END DO
            END DO
            IF (npoints == 0) CYCLE
            max_distance=0d0
            DO k=2,npoints
               max_distance=max(max_distance, sqrt(sum((intersection(k,:)-intersection(1,:))**2)))
            END DO
            data(i,j)=data(i,j)+rho*max_distance
         END DO
      END DO
   END SUBROUTINE BinCell


! same thing but we want to bin cells distributed over the softening radius with the exponential tail...

   SUBROUTINE BinParticle(Camera, Particle, data, rho)
      USE ParticleDeclarations
      TYPE(ParticleDef), POINTER :: Particle
      TYPE(CameraDef), POINTER :: Camera
      REAL(KIND=qPREC) :: pos(3),xpos(3)
      REAL(KIND=qPREC) :: dx, ddx, xlower(3)
      REAL(KIND=qPREC), DIMENSION(:,:) :: data
      REAL(KIND=qPREC) :: rho, wsum, r2, volume
      REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:) :: w
      INTEGER :: ipos(2), sample_res, i, j, k
      dx=8d0*sink_dx !diameter of sphere
      xlower=Particle%xloc-half*dx
      ! angular resolution of image = FOV/res
      ! angular size of object = dx/distance
      ! If object is much smaller than 1 pixel - then we should subsample...
      ! sample_res = angular resolution / 
      ! sample res = 
      sample_res=8d0*ceiling( (dx/sqrt(sum((Particle%xloc-Camera%pos)**2)) / (camera%FOV(1)/camera%res)))
      write(*,*) sample_res
      ddx=dx/REAL(sample_res)
      xlower=xlower-half*ddx
      ALLOCATE(w(sample_res,sample_res,sample_res))
      DO i=1,sample_res
         xpos(1)=xlower(1)+ddx*i
         DO j=1,sample_res
            xpos(2)=xlower(2)+ddx*j
            DO k=1,sample_res
               xpos(3)=xlower(3)+ddx*k
               r2=sum((xpos(1:nDim)-Particle%xloc(1:nDim))**2)
               IF (r2 < (half*dx)**2) THEN
                  w(i,j,k)=1d0!exp(-r2/(Particle%radius*sink_dx)**2)
               ELSE
                  w(i,j,k)=0d0
               END IF
            END DO
         END DO
      END DO
      wsum=sum(w)
      volume=4d0/3d0*Pi*(dx/2d0)**3
      write(*,*) wsum, volume, sample_res
      DO i=1,sample_res
         xpos(1)=xlower(1)+ddx*i
         DO j=1,sample_res
            xpos(2)=xlower(2)+ddx*j
            DO k=1,sample_res
               xpos(3)=xlower(3)+ddx*k
 !              r2=sum((xpos(1:nDim)-Particle%xloc(1:nDim))**2)
               CALL BinCell(Camera, xpos, ddx, data, rho/volume*w(i,j,k)/wsum*1e6)
            END DO
         END DO
      END DO
      DEALLOCATE(w)
   END SUBROUTINE BinParticle



   SUBROUTINE BinCell_old(Camera, pos, dx, data, rho)
      TYPE(CameraDef), POINTER :: Camera
      REAL(KIND=qPREC) :: pos(3),xpos(3)
      REAL(KIND=qPREC) :: dx, ddx, xlower(3)
      REAL(KIND=qPREC), DIMENSION(:,:) :: data
      REAL(KIND=qPREC) :: rho
      INTEGER :: ipos(2), sample_res, i, j, k
      xlower=pos-half*dx
      sample_res=2d0*ceiling(camera%FOV(1)/camera%res / (dx/sqrt(sum(pos-Camera%pos)**2)))
!     write(*,*) sample_res
      ddx=dx/REAL(sample_res)
      xlower=xlower-half*ddx
      DO i=1,sample_res
         xpos(1)=xlower(1)+ddx*i
         DO j=1,sample_res
            xpos(2)=xlower(2)+ddx*j
            DO k=1,sample_res
               xpos(3)=xlower(3)+ddx*k
               ipos(1:2)=nint(GetPixel(Camera, xpos)*shape(data))
               IF (ALL(ipos(1:2) >= 1) .AND. ALL(ipos(1:2) <= shape(data))) THEN
                  data(ipos(1),ipos(2))=data(ipos(1),ipos(2))+ rho*ddx**3
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE BinCell_old



   SUBROUTINE InputCamera(filehandle, Camera)
      TYPE(CameraDef), POINTER :: Camera
      TYPE(CameraDef) :: MyCamera
      iNTEGER :: filehandle
      NAMELIST /CameraData/ MyCamera
      MyCamera=Camera
      READ(CAMERA_DATA_HANDLE, NML=CameraData) 
      write(*,*) 'read camera id=', MyCamera%iD
      Camera=MyCamera
      CALL UpdateMatrix(Camera)
   END SUBROUTINE InputCamera

END MODULE Cameras
