! Module for calculating emission of a given transition for a given species

MODULE Emissions
   USE GlobalDeclarations   
   IMPLICIT NONE
   PUBLIC CalcEmiss

   INTEGER, PUBLIC, PARAMETER :: iOI = 1, iNII = 2, iSII_6716 = 3, iSII_6731 = 4, iHalpha = 5
   INTEGER, PUBLIC, PARAMETER :: iSII = 99

CONTAINS

   ! Main control routine which calculates the emission for given species and transition
   SUBROUTINE CalcEmiss(ne,T0,x,nH,iSpecies,emiss)
      REAL(KIND=qPREC), INTENT(OUT) :: emiss
      REAL(KIND=qPREC), INTENT(IN) :: ne, x, T0, nH   ! electron dens, ion frac, temp, total H dens
      INTEGER, INTENT(IN) :: iSpecies                 ! species index

      ! Dimension of b matrix is 5x5 since there are 5 energy levels
      REAL(KIND=qPREC) :: b(5,5), g(5), e(5), a(5,5), o(5,5), den(5), indx(5)
      REAL(KIND=qPREC) :: levpops, T, emiss1
      INTEGER :: lev1, lev2               ! the initial, final transition levels

      ! Use separate routine for H-alpha emission
      IF(iSpecies == iHalpha) THEN
         CALL alpha_emiss(ne,T0,x,nH,emiss1)
         emiss = emiss1
         RETURN
      END IF 

      ! Put T in 10^4 K
      T = T0/(10d0**4d0)

      IF(ne /= 0.) THEN

         SELECT CASE(iSpecies)
            CASE(iOI)                    ! [O I]  6300
               CALL OI(ne,T,g,e,a,o)
               lev1 = 4 ; lev2 = 1
            CASE(iNII)                   ! [N II] 6583
               CALL NII(ne,T,g,e,a,o)
               lev1 = 4 ; lev2 = 3
            CASE(iSII_6716)              ! [S II] 6716
               CALL SII(ne,T,g,e,a,o)
               lev1 = 3 ; lev2 = 1
            CASE(iSII_6731)              ! [S II] 6731
               CALL SII(ne,T,g,e,a,o)
               lev1 = 2 ; lev2 = 1
         END SELECT

         CALL Loadb(b,T,ne,g,e,a,o)

         ! Do matrix inversion (values in b and den will be overwritten)
         ! Input: matrix b and r.h.s. is (0,0,0,0,1) stored in den
         ! Output: LU decomposed matrix in b and density of levels in den
         ! From numerical recipes in FORTRAN
         den = (/ 0, 0, 0, 0, 1 /)
         CALL ludcmp(b,indx)
         CALL lubksb(b,indx,den)

      ELSE
         den = (/ 1, 0, 0, 0, 0 /)
      END IF

      levpops = den(lev1)
      IF(levpops == -0.) levpops = 0d0   

      CALL emissivity(levpops,ne,T0,x,nH,iSpecies,lev1,lev2,emiss1)

      emiss = emiss1

   END SUBROUTINE CalcEmiss

   ! Gets g, e, a, and c for OI in preparation to form matrix b
   SUBROUTINE OI(ne,T,g,e,a,o)
      REAL(KIND=qPREC), INTENT(IN) :: ne, T
      REAL(KIND=qPREC), INTENT(OUT) :: g(5), e(5), a(5,5), o(5,5)
      REAL(KIND=qPREC) :: a1(5,5), erg(5,5)
      INTEGER :: i, j

      ! Statistical weights for OI
      g(1) = 5.
      g(2) = 3.
      g(3) = 1.
      g(4) = 5.
      g(5) = 1.

      ! Energy levels of OI (eV)
      CALL E_Values(e,erg,iOI)

      ! Get A-values, Baluja, K., and Zeippen, C. 1988, J. Phys B. Atom. Mol. Opt. Phys. 21, 1455.
      ! Subroutine A_Values gives a as a(fin,init) but we need a as
      ! a(init,fin) to get matrix b correct, so we take the transpose...
      CALL A_Values(a1,iOI)
      DO i=1, 5
         DO j=1, 5
            a(i,j) = a1(j,i)
         END DO
      END DO

      ! Collision strengths for OI, Berrington and Burke 1981, Plan. Sp. Sci. 29, 377.
      ! and Le Dourneuf, M. and Neabet, R. 1976, J Phys B. Atom. Molec. Phys. 9, L241.
      o(1,4) = (5d0/9d0)*0.266*(T**0.97)*(T**(LOG10(T**(-0.27))))
      o(2,4) = (3d0/9d0)*0.266*(T**0.97)*(T**(LOG10(T**(-0.27))))
      o(3,4) = (1d0/9d0)*0.266*(T**0.97)*(T**(LOG10(T**(-0.27))))
      o(1,5) = (5d0/9d0)*0.0324*T*(T**(LOG10(T**(-0.22))))
      o(2,5) = (3d0/9d0)*0.0324*T*(T**(LOG10(T**(-0.22))))
      o(3,5) = (1d0/9d0)*0.0324*T*(T**(LOG10(T**(-0.22))))
      o(1,2) = 0.0987*T**1.11
      o(2,3) = 0.0265*T**1.17
      o(1,3) = 0.0292*T*(T**(LOG10(T**(-0.13))))
      o(4,5) = 0.105*T**0.54

   END SUBROUTINE OI

   ! Gets g, e, a, and c for NII in preparation to form matrix b
   SUBROUTINE NII(ne,T,g,e,a,o)
      REAL(KIND=qPREC), INTENT(IN) :: ne, T
      REAL(KIND=qPREC), INTENT(OUT) :: g(5), e(5), a(5,5), o(5,5)
      REAL(KIND=qPREC) :: a1(5,5), erg(5,5)
      INTEGER :: i, j

      ! Statistical weights for NII
      g(1) = 1.
      g(2) = 3.
      g(3) = 5.
      g(4) = 5.
      g(5) = 1.

      ! Energy levels of NII (eV)
      CALL E_Values(e,erg,iNII)

      ! Get A-values, Bell, Hibbert, Stafford 1995, Phys. Scr. 52, 240.
      !               Storey and Zeippen 2000, MNRAS 312, 813.
      !               Froese Fischer, C. and Saha, H. 1985, Phys Scr. 32, 181.
      ! Subroutine A_Values gives a as a(fin,init) but we need a as
      ! a(init,fin) to get matrix b correct, so we take the transpose...
      CALL A_Values(a1,iNII)
      DO i=1, 5
         DO j=1, 5
            a(i,j) = a1(j,i)
         END DO
      END DO

      ! Collision strengths for NII, Hudson and Bell A&A 430, 725, 2005
      o(1,2) = 0.43
      o(1,3) = 0.27
      o(1,4) = 0.30
      o(1,5) = 0.035
      o(2,3) = 1.15
      o(2,4) = 0.91
      o(2,5) = 0.11
      o(3,4) = 1.51
      o(3,5) = 0.18
      o(4,5) = 0.81

   END SUBROUTINE NII


   ! Gets g, e, a, and c for SII in preparation to form matrix b
   SUBROUTINE SII(ne,T,g,e,a,o)
      REAL(KIND=qPREC), INTENT(IN) :: ne, T
      REAL(KIND=qPREC), INTENT(OUT) :: g(5), e(5), a(5,5), o(5,5)
      REAL(KIND=qPREC) :: a1(5,5), erg(5,5)
      INTEGER :: i, j

      ! Statistical weights for SII
      g(1) = 4.
      g(2) = 4.
      g(3) = 6.
      g(4) = 2.
      g(5) = 4.

      ! Energy levels of SII (eV)
      CALL E_Values(e,erg,iSII)

      ! Get A-values, Mandoza & Zeippen 1982 MNRAS 198, 111.
      ! Subroutine A_Values gives a as a(fin,init) but we need a as
      ! a(init,fin) to get matrix b correct, so we take the transpose...
      CALL A_Values(a1,iSII)
      DO i=1, 5
         DO j=1, 5
            a(i,j) = a1(j,i)
         END DO
      END DO

      ! Collision strengths for SII, Keenan et al 1996, MNRAS 281, 1073.
      o(1,2) = 2.76*(T**(-0.15))
      o(1,3) = 4.14*(T**(-0.15))
      o(1,4) = 1.17*(T**(-0.07))
      o(1,5) = 2.35*(T**(-0.07))
      o(2,3) = 7.47*(T**(-0.10))
      o(2,4) = 1.79*(T**(-0.17))
      o(2,5) = 3.00*(T**(-0.12))
      o(3,4) = 2.20*(T**(-0.11))
      o(3,5) = 4.99*(T**(-0.15))
      o(4,5) = 2.71*(T**(-0.14))

   END SUBROUTINE SII


   ! Forms the matrix b
   SUBROUTINE Loadb(b,T,ne,g,e,a,o)
      REAL(KIND=qPREC), INTENT(IN) :: T, ne, g(5), e(5), a(5,5), o(5,5)
      REAL(KIND=qPREC) :: q12,q13,q14,q15,q23,q24,q25,q34,q35,q45,q21,q31,q41,q51,q32,q42,q52,q43,q53,q54,q5,q4,q3,q2
      REAL(KIND=qPREC), INTENT(OUT) :: b(5,5)
         
      q12 = (8.63d-8)*o(1,2)*(EXP(-(e(2) - e(1))/(0.8614*T)))/(g(1)*(T**0.5))
      q13 = (8.63d-8)*o(1,3)*(EXP(-(e(3) - e(1))/(0.8614*T)))/(g(1)*(T**0.5))
      q14 = (8.63d-8)*o(1,4)*(EXP(-(e(4) - e(1))/(0.8614*T)))/(g(1)*(T**0.5))
      q15 = (8.63d-8)*o(1,5)*(EXP(-(e(5) - e(1))/(0.8614*T)))/(g(1)*(T**0.5))
      q23 = (8.63d-8)*o(2,3)*(EXP(-(e(3) - e(2))/(0.8614*T)))/(g(2)*(T**0.5))
      q24 = (8.63d-8)*o(2,4)*(EXP(-(e(4) - e(2))/(0.8614*T)))/(g(2)*(T**0.5))
      q25 = (8.63d-8)*o(2,5)*(EXP(-(e(5) - e(2))/(0.8614*T)))/(g(2)*(T**0.5))
      q34 = (8.63d-8)*o(3,4)*(EXP(-(e(4) - e(3))/(0.8614*T)))/(g(3)*(T**0.5))
      q35 = (8.63d-8)*o(3,5)*(EXP(-(e(5) - e(3))/(0.8614*T)))/(g(3)*(T**0.5))
      q45 = (8.63d-8)*o(4,5)*(EXP(-(e(5) - e(4))/(0.8614*T)))/(g(4)*(T**0.5))
      q21 = (8.63d-8)*o(1,2)/(g(2)*(T**0.5))
      q31 = (8.63d-8)*o(1,3)/(g(3)*(T**0.5))
      q41 = (8.63d-8)*o(1,4)/(g(4)*(T**0.5))
      q51 = (8.63d-8)*o(1,5)/(g(5)*(T**0.5))
      q32 = (8.63d-8)*o(2,3)/(g(3)*(T**0.5))
      q42 = (8.63d-8)*o(2,4)/(g(4)*(T**0.5))
      q52 = (8.63d-8)*o(2,5)/(g(5)*(T**0.5))
      q43 = (8.63d-8)*o(3,4)/(g(4)*(T**0.5))
      q53 = (8.63d-8)*o(3,5)/(g(5)*(T**0.5))
      q54 = (8.63d-8)*o(4,5)/(g(5)*(T**0.5))
      q5 = q51 + q52 + q53 + q54
      q4 = q41 + q42 + q43 + q45
      q3 = q31 + q32 + q34 + q35
      q2 = q21 + q23 + q24 + q25

      ! Matrix values before inversion (row,column)
      b(1,1) = ne*q15
      b(1,2) = ne*q25
      b(1,3) = ne*q35
      b(1,4) = ne*q45
      b(1,5) = -a(5,5) - ne*q5
      b(2,1) = ne*q14
      b(2,2) = ne*q24
      b(2,3) = ne*q34
      b(2,4) = -a(4,4) - ne*q4
      b(2,5) = a(5,4) + ne*q54
      b(3,1) = ne*q13
      b(3,2) = ne*q23
      b(3,3) = -a(3,3) - ne*q3
      b(3,4) = a(4,3) + ne*q43
      b(3,5) = a(5,3) + ne*q53
      b(4,1) = ne*q12
      b(4,2) = -a(2,1) - ne*q2
      b(4,3) = a(3,2) + ne*q32
      b(4,4) = a(4,2) + ne*q42
      b(4,5) = a(5,2) + ne*q52
      b(5,:) = 1.

   END SUBROUTINE Loadb



   SUBROUTINE ludcmp(b,indx)
      REAL(KIND=qPREC), PARAMETER :: tiny = 1d-28
      REAL(KIND=qPREC) :: b(5,5), indx(5), vv(5), aamax, dum, sum
      REAL(KIND=qPREC) :: s
      INTEGER :: i, j, k, imax

      s = 1.
      DO i=1, 5
         aamax = 0.
         DO j=1, 5
            IF(ABS(b(i,j)) > aamax) aamax = ABS(b(i,j))
         END DO
         IF(aamax == 0.) PAUSE 'singular matrix'
         vv(i) = 1./aamax
      END DO
      DO j=1, 5
         DO i=1, j-1
            sum = b(i,j)
            DO k=1, i-1
               sum = sum - b(i,k)*b(k,j)
            END DO
            b(i,j) = sum
         END DO
         aamax = 0.
         DO i=j, 5
            sum = b(i,j)
            DO k=1, j-1
               sum = sum-b(i,k)*b(k,j)
            END DO
            b(i,j) = sum
            dum = vv(i)*ABS(sum)
            IF(dum >= aamax) THEN
               imax = i
               aamax = dum
            END IF
         END DO
         IF(j /= imax) THEN
            DO k=1, 5
               dum = b(imax,k)
               b(imax,k) = b(j,k)
               b(j,k) = dum
            END DO
            s = -s
            vv(imax) = vv(j)
         END IF
         indx(j) = imax
         IF(b(j,j) == 0) b(j,j) = tiny
         IF(j /= 5) THEN
            dum = 1./b(j,j)
            DO i=j+1, 5
               b(i,j) = b(i,j)*dum
            END DO
         END IF
      END DO    
   END SUBROUTINE ludcmp



   SUBROUTINE lubksb(b,indx,den)
      REAL(KIND=qPREC) :: b(5,5), indx(5), den(5), sum
      INTEGER :: i, ii, j, ll

      ii = 0
      DO i=1, 5
         ll = indx(i)
         sum = den(ll)
         den(ll) = den(i)
         IF(ii /= 0) THEN
            DO j=ii, i-1
               sum = sum - b(i,j)*den(j)
            END DO
         ELSE IF(sum /= 0) THEN
            ii = i
         END IF
         den(i) = sum
      END DO
      DO i=5, 1, -1
         sum = den(i)
         DO j=i+1, 5
            sum = sum - b(i,j)*den(j)
         END DO
         den(i) = sum/b(i,i)
      END DO
   END SUBROUTINE lubksb


   !  Calculate the emissivity from formula (n*A*hv/(4Pi))
   !  Units are erg/cm^3/s/str
   SUBROUTINE emissivity(levpops,ne,T0,x,nH,iSpecies,lev1,lev2,emiss7)
      REAL(KIND=qPREC), PARAMETER :: Pi = 3.14159, k = 8.617d-5, &   ! k is in eV/K
                     
                                  ! Ionization energies of neutral atoms
                                  EionOI = 13.62, &
                                  EionNI = 14.5341, &
                                  EionHI = 13.60, &

                                  ! Solar abundances from Hartigan & Morse, APJ, 660, 2007.
                                  AbundO = 8.82, &
                                  AbundN = 7.96, &
                                  AbundS = 7.30

      REAL(KIND=qPREC), INTENT(IN) :: levpops, ne, x, T0, nH
      INTEGER, INTENT(IN) :: lev1, lev2, iSpecies
      REAL(KIND=qPREC), INTENT(OUT) :: emiss7
      REAL(KIND=qPREC) :: kT, Abund_rat, nHII_nHI, levdens, avals(5,5), e(5), hv_ergs(5,5), &
                          nSpecies, nSpec, gHI, gHII,  gOI, gOII, gNI, gNII, nIon_Neut

      ! Statistical Weights (ground states)
      gHI = 2 ; gHII = 1 ; gOI = 3 ; gOII = 4 ; gNI = 4 ; gNII = 9

      ! kT values in eV
      kT = k * T0

      ! Calculate ratio of nHII/nHI = Xh/(1-Xh)
      IF(1. - x == 0.0) THEN
         nHII_nHI = x/(1. - x + 1d-10)
      ELSE
         nHII_nHI = x/(1. - x)
      END IF

      SELECT CASE(iSpecies)
         CASE(iOI)
            Abund_rat = AbundO - 12.
            nIon_Neut = nHII_nHI * (gHI*gOII/(gHII*gOI)) * EXP((EionHI - EionOI)/kT)
         CASE(iNII)
            Abund_rat = AbundN - 12.
            nIon_Neut = (1d0/nHII_nHI) * (gHII*gNI/(gHI*gNII)) * EXP(-(EionHI - EionNI)/kT)
         CASE(iSII_6716,iSII_6731)
            Abund_rat = AbundS - 12.
            nIon_Neut = 0d0           ! All S is assumed to be SII
      END SELECT
      
      nSpecies = nH * (10d0**Abund_rat)
      nSpec = nSpecies / (1d0 + nIon_Neut)
      levdens = levpops*nSpec

      CALL A_Values(avals,iSpecies)
      CALL E_Values(e,hv_ergs,iSpecies)
      emiss7 = levdens * avals(lev2,lev1) * hv_ergs(lev2,lev1) / (4d0*Pi)

   END SUBROUTINE emissivity


   ! Gives A_Values as a(final level, initial level) for given species
   SUBROUTINE A_Values(a, iSpecies)
      REAL(KIND=qPREC), INTENT(OUT) :: a(5,5)
      INTEGER, INTENT(IN) :: iSpecies

      SELECT CASE(iSpecies)
         CASE(iOI)
            a(4,5) = 1.215
            a(3,5) = 0d0
            a(2,5) = 7.60d-2
            a(1,5) = 2.73d-4
            a(3,4) = 8.93d-7
            a(2,4) = 1.82d-3
            a(1,4) = 5.63d-3
            a(2,3) = 1.74d-5
            a(1,3) = 1.20d-10
            a(1,2) = 8.96d-5
         CASE(iNII)
            a(4,5) = 0.923
            a(3,5) = 1.435d-4
            a(2,5) = 4.588d-2
            a(1,5) = 0d0
            a(3,4) = 3.015d-3
            a(2,4) = 9.819d-4
            a(1,4) = 1.928d-6
            a(2,3) = 7.46d-6
            a(1,3) = 1.062d-12
            a(1,2) = 2.08d-6
         CASE(iSII,iSII_6716,iSII_6731)
            a(4,5) = 1.03d-6
            a(3,5) = 0.179
            a(2,5) = 0.133
            a(1,5) = 0.225
            a(3,4) = 7.79d-2
            a(2,4) = 0.163
            a(1,4) = 9.06d-2
            a(2,3) = 3.35d-7
            a(1,3) = 2.60d-4
            a(1,2) = 8.82d-4
      END SELECT           

      a(5,5) = a(4,5) + a(3,5) + a(2,5) + a(1,5)
      a(4,4) = a(3,4) + a(2,4) + a(1,4)
      a(3,3) = a(2,3) + a(1,3)
      a(2:5,2) = 0d0
      a(4:5,3) = 0d0
      a(5,4) = 0d0

      ! Value for transitions out of level 1 = 0.0
      a(:,1) = 0d0

   END SUBROUTINE A_Values
  

   ! Gives energy levels for given species in eV, and transition energies in erg
   SUBROUTINE E_Values(e,erg,iSpecies)
      REAL(KIND=qPREC), INTENT(OUT) :: e(5), erg(5,5)
      INTEGER, INTENT(IN) :: iSpecies
      INTEGER :: i, j

      ! Energy levels in eV
      SELECT CASE(iSpecies)
         CASE(iOI)
            e(2) = 0.0196
            e(3) = 0.0282
            e(4) = 1.9689
            e(5) = 4.1931
         CASE(iNII)
            e(2) = 0.00604
            e(3) = 0.0162
            e(4) = 1.900
            e(5) = 4.056
         CASE(iSII,iSII_6716,iSII_6731)
            e(2) = 1.843
            e(3) = 1.847
            e(4) = 3.043
            e(5) = 3.049
      END SELECT

      e(1) = 0d0
      
      DO i=2, 5     ! initial
         DO j=1, 5  ! final
            IF (j<i) THEN
               erg(j,i) = (e(i) - e(j)) * (1.602d-12)
            ELSE
               erg(j,i) = 0.0
            END IF
         END DO
      END DO

      erg(:,1) = 0d0      ! Can't transition down from level 1 

   END SUBROUTINE E_Values


   ! Calculates emissivity of H-alpha from recombination and collisional
   ! excitation from levels 1 --> 3,4,5   
   SUBROUTINE alpha_emiss(ne,T0,x,nH,emissAlpha)
      REAL(KIND=qPREC), PARAMETER :: Halpha_hv = 3.0263d-12, & ! ergs, = 1.88889 eV, for H n= 3->2 transition
                                     E_thr_n3 = 12.09d0, &       ! in eV, for n=1->3
                                     E_thr_n4 = 12.75d0, &       ! in eV, for n=1->4
                                     E_thr_n5 = 13.056d0, &      ! in eV, for n=1->5
                                     k_B = 8.617d-5, &         ! in eV/K
                                     vern_a = 7.982d-11, &     ! cm^3/s
                                     vern_b = 0.7480d0, &
                                     vern_T0 = 3.148d0, &      ! K
                                     vern_T1 = 7.036d5, &      ! K
                                     Pi = 3.14159

      CHARACTER(LEN = 80) :: filename

      INTEGER :: i,j,k                                ! counters

      REAL(KIND=qPREC), DIMENSION(4,25) :: coll_str   ! table of effective collision strengths (e-H)
 
      REAL(KIND=qPREC) :: recombcoef, &      ! recombination rate coefficient
                          collexcite,  &     ! collisional excitation rate coefficient
                          emissAlpha_rec, &  ! emissivity due to H-alpha recombination
                          emissAlpha_coll, & ! emissivity due to collisional excitation
                          TeLog              ! Log of Temperature (in K)

      REAL(KIND=qPREC), INTENT(OUT) :: emissAlpha ! Total H-alp emissivity

      REAL(KIND=qPREC), INTENT(IN) :: ne, T0, x, nH      ! electron dens, temp (K), ion frac, hydrogen dens
      REAL(KIND=qPREC) :: upsilon(3)         ! effective collision strengths

      !!! Recombination !!!
      recombcoef = vern_a / ( SQRT(T0/vern_T0) * (1d0 + SQRT(T0/vern_T0))**(1d0 - vern_b) * &
                   (1d0 + SQRT(T0/vern_T1))**(1d0 + vern_b) )

      emissAlpha_rec = Halpha_hv * ne * x*nH * recombcoef / (4d0*Pi)

      !!! Collisional Excitation !!!
      OPEN(49, file='TABLES/Anderson_Coll_Str.tab', ACTION='READ')    ! First column is T, columns 2,3,4 are
      READ(49, '(4(ES12.4))') coll_str                        ! log(collision strength) from level 1
      CLOSE(49)                                                ! to levels 3,4,5 respectively

      TeLog = LOG10(T0)
      upsilon = 0d0

      DO i=1, 24
         IF(TeLog >= coll_str(1,i) .AND. TeLog < coll_str(1,i+1)) THEN

            upsilon(1) = 10d0**( (TeLog - coll_str(1,i))/(coll_str(1,i+1) - coll_str(1,i)) * &
                         (coll_str(2,i+1) - coll_str(2,i)) + coll_str(2,i) )

            upsilon(2) = 10d0**( (TeLog - coll_str(1,i))/(coll_str(1,i+1) - coll_str(1,i)) * &
                         (coll_str(3,i+1) - coll_str(3,i)) + coll_str(3,i) )

            upsilon(3) = 10d0**( (TeLog - coll_str(1,i))/(coll_str(1,i+1) - coll_str(1,i)) * &
                         (coll_str(4,i+1) - coll_str(4,i)) + coll_str(4,i) )

         ELSE IF(TeLog < coll_str(1,i)) THEN       ! collision strengths are tiny at low T, so
            upsilon(1) = 10d0**coll_str(2,1)       ! if TeLog is below min T in the table, set
            upsilon(2) = 10d0**coll_str(3,1)       ! upsilon to corresponding min coll_str
            upsilon(3) = 10d0**coll_str(4,1)
         END IF
      END DO

      ! Calculate rate coefficient (cm^3/s)
      collexcite = 8.63d-6/(2d0*SQRT(T0)) * ( EXP(-E_thr_n3/(k_B*T0)) * upsilon(1) + &
                   EXP(-E_thr_n4/(k_B*T0)) * upsilon(2) + EXP(-E_thr_n5/(k_B*T0)) * upsilon(3) )

      emissAlpha_coll = Halpha_hv * ne * (nH - x*nH) * collexcite / (4d0*Pi)

      !!! Total Emission for H-alpha !!!
      emissAlpha = emissAlpha_rec + emissAlpha_coll

   END SUBROUTINE alpha_emiss

END MODULE Emissions

