!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    fields.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Fields
   USE PhysicsDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE EOS
   USE ParticleDeclarations
   USE TreeDeclarations
   USE DataDeclarations
   USE CoolingSrc
   USE NEQCoolingSrc
   USE Emissions
   USE SourceDeclarations
   USE Shapes
   IMPLICIT NONE
   PUBLIC
   INTEGER, PARAMETER :: MAXFIELDSLENGTH=20
   CHARACTER(LEN=MAXFIELDSLENGTH), PARAMETER :: UNDEFINEDFIELDNAME='undefined           '
   INTEGER, PARAMETER :: GASCOMP=1, PARTICLECOMP=2, BOTHCOMP=3
   TYPE FieldDef
      INTEGER :: ID
      CHARACTER(LEN=MAXFIELDSLENGTH) :: Name=UNDEFINEDFIELDNAME
      INTEGER :: component=GASCOMP
   END type FieldDef


   INTEGER, PARAMETER :: &
        Mass_Field=101, &
        Px_Field=102, &
        Py_Field=103, &
        Pz_Field=104, &
        E_Field=105, &
        Bx_Field=106, &
        By_Field=107, &
        Bz_Field=108, &
        KE_Field=109, &
        iE_Field=110, &
        BE_Field=111, &
        P_Field=112, &
        Temp_Field=113, &
        GravEnergy_Field=114, &
        MixingRatio12_Field=115, & !First two tracers   
        JeansLength_Field=116, &
        CoolingStrength_Field=117, &
        vx_Field=118, &
        vy_Field=119, &
        vz_Field=120, &
        VMag_Field=121, &
        Enstrophy_Field=122, &
        SoundSpeed_Field=123, &
        SqrtPress_Field=124, &
        SoundSpeed2_Field=125, &
        CellSize_Field=126, &
        GasPotential_Field=127, &
        RhoSoundSpeed_Field=128, &
        srhovx_Field=129, &
        srhovy_Field=130, &
        srhovz_Field=131, &
        OI_Field=132, &
        NII_Field=133, &
        SII_6716_Field=134, &
        SII_6731_Field=135, &
        Halpha_Field=136, &
        MPI_ID_FIELD=137, &
        ChildMask_Field=138, &
        ErrFlag_Field=139, &
        Divergence_Field=140

   INTERFACE GetName
      MODULE PROCEDURE GetNameByID, GetNameByField
   END INTERFACE GetName


CONTAINS



   FUNCTION GetField(Info,i,j,k,id,pos_opt,emissinfo)
      TYPE(InfoDef) :: Info
      REAL(KIND=qPREC), DIMENSION(:), POINTER :: q
      REAL(KIND=qPREC) :: GetField
      INTEGER :: i,j,k, id
      TYPE(ParticleListDef), POINTER :: ParticleList
      REAL(KIND=qPREC), OPTIONAL :: pos_opt(3)
      REAL(KIND=qPREC) :: pos(3)
      REAL(KIND=qPREC) :: P, T0, nH, ne, x, mu, emiss
      REAL(KIND=qPREC), DIMENSION(0:nSpeciesHi) :: nvec
      LOGICAL :: lform
      INTEGER, DIMENSION(1:3), OPTIONAL :: emissinfo
      IF (PRESENT(pos_opt)) THEN
         pos=pos_opt
      ELSE
         pos=CellPos(Info, i, j, k)
      END IF
      q=>Info%q(i,j,k,:)
      IF (id >= 1 .AND. id <= NrVars) THEN
         GetField=q(id)
      ELSE
         IF (id >= OI_Field .AND. id <= Halpha_Field) THEN                ! If desired field is an emission
!            P = MERGE(PrimPress(q), Press(q), lform .eqv. PRIMITIVE)     ! line then get the parameters that
            P = Press(q)                                                  ! are common to all the different
            CALL GetNEQvars(q, mu, nvec)                                  ! emission fields
            T0 = P/q(1) * TempScale * mu
            CALL GetZvars(nvec, ne, x, nH)
         END IF            
         SELECT CASE(id)
         CASE(Mass_Field)
            GetField=q(1)
         CASE(Px_Field)
            GetField=q(ivx)
         CASE(Py_Field)
            GetField=q(ivy)
         CASE(Pz_Field)
            GetField=q(ivz)
         CASE(E_Field)
            GetField=q(iE)
         CASE(Bx_Field)
            GetField=q(iBx)
         CASE(By_Field)
            GetField=q(iBy)
         CASE(Bz_Field)
            GetField=q(iBz)
         CASE(KE_Field)
            GetField=half*sum(q(m_low:m_high)**2)/q(1)
         CASE(iE_Field)
            GetField=InternalEnergy(q)
         CASE(BE_Field)
            GetField=half*sum(q(iBx:iBz)**2)
         CASE(P_Field)
            GetField=Press(q)
         CASE(Temp_Field)
            GetField=Press(q)/q(1)
         CASE(GravEnergy_Field)
            IF (iPhiGas /= 0) THEN
               GetField=half*q(iPhiGas)*q(1)
            ELSE
               GetField=0d0
            END IF
            ParticleList=>SinkParticles
            DO WHILE (ASSOCIATED(ParticleList))
               IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
                  GetField=GetField+q(1)*GravityPotential(ParticleList%self%q(1), pos-ParticleList%self%xloc, ParticleList%self%PointGravityObj%soft_length, ParticleList%self%PointGravityObj%soft_function)
               END IF
               ParticleList=>ParticleList%next
            END DO
         CASE(MixingRatio12_Field)
            GetField=2d0*min(q(nTracerLo), q(nTracerLo+1))/(max(q(1), q(nTracerLo)+q(nTracerLo+1)))
         CASE(JeansLength_Field)
            GetField=SoundSpeed(q)*sqrt(pi/(ScaleGrav*q(1)))
         CASE(CoolingStrength_Field)
            GetField=GetCoolingStrength(q,CONSERVATIVE)
         CASE(vx_Field)
            GetField=q(ivx)/q(1)
         CASE(vy_Field)
            GetField=q(ivy)/q(1)
         CASE(vz_Field)
            GetField=q(ivz)/q(1)
         CASE(VMag_Field)
            GetField=sqrt(sum(q(imom(1:nDim))**2))/q(1)
         CASE(Enstrophy_Field)
            IF (nDim == 2) THEN
               GetField=half*(Curl2D(Info%q(i-1:i+1,j-1:j+1,k,imom(1:2))/spread(Info%q(i-1:i+1,j-1:j+1,k,1),3,2), levels(Info%level)%dx))**2
            ELSE
               GetField=half*sum(Curl3D(Info%q(i-1:i+1,j-1:j+1,k-1:k+1,imom(1:3))/spread(Info%q(i-1:i+1,j-1:j+1,k-1:k+1,1),3,3), levels(Info%level)%dx)**2)
            END IF
         CASE(SoundSpeed_Field)
            GetField=SoundSpeed(q)
         CASE(SqrtPress_Field)
            GetField=sqrt(q(1))*SoundSpeed(q)
         CASE(SoundSpeed2_Field)
            GetField=SoundSpeed(q)**2
         CASE(RhoSoundSpeed_Field)
            GetField=SoundSpeed(q)*q(1)
         CASE(CellSize_Field)
            GetField=levels(Info%level)%dx
         CASE(srhovx_Field)
            GetField=q(ivx)/sqrt(q(1))!cos(half*Pi*sqrt((pos-50d0)**2)
         CASE(srhovy_Field)
            GetField=q(ivy)/sqrt(q(1))
         CASE(srhovz_Field)
            GetField=q(ivz)/sqrt(q(1))
         CASE(GasPotential_Field)
            IF (iPhiGas /= 0) THEN
               GetField=q(iPhiGas)
            ELSE
               GetField=0d0
            END IF
         CASE(OI_Field)
            CALL CalcEmiss(ne,T0,x,nH,iOI,emiss)
            GetField = emiss
         CASE(NII_Field)
            CALL CalcEmiss(ne,T0,x,nH,iNII,emiss)
            GetField = emiss
         CASE(SII_6716_Field)
            CALL CalcEmiss(ne,T0,x,nH,iSII_6716,emiss)
            GetField = emiss
         CASE(SII_6731_Field)
            CALL CalcEmiss(ne,T0,x,nH,iSII_6731,emiss)
            GetField = emiss
         CASE(Halpha_Field)
            CALL CalcEmiss(ne,T0,x,nH,iHalpha,emiss)
            GetField = emiss
         CASE(MPI_ID_FIELD)
            GetField=MPI_ID
         CASE(ChildMask_Field)
            GetField=Info%ChildMask(i,j,k)
         CASE(ErrFlag_Field)
            IF (Info%level < MaxLevel) THEN
               GetField=Info%ErrFlag(i,j,k)
            ELSE
               GetField=0d0
            END IF
         CASE (Divergence_Field)
            IF (nDim == 2) THEN
               GetField=(Info%aux(i+1,j,k,1)-Info%aux(i,j,k,1)+Info%aux(i,j+1,k,2)-Info%aux(i,j,k,2))/levels(Info%level)%dx
            ELSE
               GetField=(Info%aux(i+1,j,k,1)-Info%aux(i,j,k,1)+Info%aux(i,j+1,k,2)-Info%aux(i,j,k,2)+Info%aux(i,j,k+1,3)-Info%aux(i,j,k,3))/levels(Info%level)%dx
            END IF
         CASE DEFAULT
            IF (MPI_ID == 0) PRINT*, 'Warning, field not recognized:', i
            GetField=0d0
         END SELECT
      END IF
      
   END FUNCTION GetField
  


   FUNCTION GetParticleField(Particle, i)
      REAL(KIND=qPREC) :: GetParticleField, r, temp(3)
      TYPE(ParticleDef), TARGET :: Particle
      TYPE(ParticleListDef), POINTER :: ParticleList
      INTEGER :: i
      IF (i >= 1 .AND. i <= NrVars) THEN
         temp=Particle%q(m_low:m_high)
         Particle%q(m_low:m_high)=Particle%q(m_low:m_high)*particle%q(1)
         GetParticleField=Particle%q(i)
         Particle%q(m_low:m_high)=temp(1:m_high-m_low+1)
      ELSE
         SELECT CASE(i)
         CASE(Mass_Field)
            GetParticleField=Particle%q(1)
         CASE(Px_Field)
            GetParticleField=Particle%q(ivx)*Particle%q(1)
         CASE(Py_Field)
            GetParticleField=Particle%q(ivy)*Particle%q(1)
         CASE(Pz_Field)
            GetParticleField=Particle%q(ivz)*Particle%q(1)
         CASE(E_Field)
            IF (iE /= 0) THEN
               GetParticleField=Particle%q(iE)
            ELSE
               GetParticleField=Particle%q(1)*Iso_Speed2/(gamma-1d0)+half*sum(Particle%q(m_low:m_high)**2)*Particle%q(1)
            END IF
         CASE(KE_Field)
            GetParticleField=half*sum(Particle%q(m_low:m_high)**2)*Particle%q(1)
         CASE(iE_Field)
            IF (iE /= 0) THEN
               GetParticleField=Particle%q(iE)-half*sum(Particle%q(m_low:m_high)**2)*Particle%q(1)
            ELSE
               GetParticleField=Particle%q(1)*Iso_speed2
            END IF
         CASE(GravEnergy_Field) !Only particle particle potential.  Gas Particle potential can be found in GasGravEnergy as this uses Phi and not PhiGas
            GetParticleField=0d0
            ParticleList=>SinkParticles
            DO WHILE (ASSOCIATED(ParticleList))
               IF (.NOT. ASSOCIATED(ParticleList%self, Particle)) THEN
                  IF (ASSOCIATED(ParticleList%self%PointGravityObj)) THEN
                     GetParticleField=GetParticleField+half*Particle%q(1)*GravityPotential(ParticleList%self%q(1), Particle%xloc-ParticleList%self%xloc, ParticleList%self%PointGravityObj%soft_length, ParticleList%self%PointGravityObj%soft_function)
                  END IF
               END IF              
               ParticleList=>ParticleList%next
            END DO
         CASE(MixingRatio12_Field)
            GetParticleField=2d0*min(Particle%q(nTracerLo), Particle%q(nTracerLo+1))/(max(Particle%q(1), Particle%q(nTracerLo)+Particle%q(nTracerLo+1)))            
         CASE(vx_Field)
            GetParticleField=Particle%q(ivx)
         CASE(vy_Field)
            GetParticleField=Particle%q(ivy)
         CASE(vz_Field)
            GetParticleField=Particle%q(ivz)
         CASE(VMag_Field)
            GetParticleField=sqrt(sum(Particle%q(imom(1:nDim))**2))
         CASE DEFAULT
            IF (MPI_ID == 0) PRINT*, 'Warning, particle field not recognized', i, GetName(i)
            GetParticleField=0d0
         END SELECT
      END IF
   END FUNCTION GetParticleField


   FUNCTION GetNameByField(Field)
      TYPE(FieldDef) :: Field
      CHARACTER(LEN=MAXFIELDSLENGTH) :: GetNameByField
      IF (Field%Name == UNDEFINEDFIELDNAME) THEN
         GetNameByField=GetNameByID(Field%id)
      ELSE
         GetNameByField=Field%Name
      END IF
   END FUNCTION GetNameByField

   FUNCTION GetNameByID(i)
      CHARACTER(LEN=MAXFIELDSLENGTH) :: GetNameByID
      INTEGER :: i
      IF (i >= 1 .AND. i <= NrFieldVars) THEN
         GetNameByID=FieldName(i)
      ELSEIF (i >= NrFieldVars+1 .AND. i <= NrFieldVars+NrTracerVars) THEN
         GetNameByID=TracerName(i-NrFieldVars)
      ELSEIF (i >= NrHydroVars+1 .AND. i <= NrHydroVars+NrEllipticVars) THEN
         GetNameByID=EllipticName(i-NrHydroVars)
      ELSE
         SELECT CASE(i)
         CASE(Mass_Field) 
            GetNameByID='Mass'
         CASE(Px_Field)
            GetNameByID='Px'
         CASE(Py_Field)
            GetNameByID='Py'
         CASE(Pz_Field)
            GetNameByID='Pz'
         CASE(Bx_Field)
            GetNameByID='Bx'
         CASE(By_Field)
            GetNameByID='By'
         CASE(Bz_Field)
            GetNameByID='Bz'
         CASE(KE_Field)
            GetNameByID='Kinetic_Energy'
         CASE(iE_Field)
            GetNameByID='Internal_Energy'
         CASE(BE_Field)
            GetNameByID='Magnetic_Energy'
         CASE(P_Field)
            GetNameByID='Pressure'
         CASE(Temp_Field)
            GetNameByID='Temp'
         CASE(GravEnergy_Field)
            GetNameByID='Grav_Energy'
         CASE(MixingRatio12_Field)
            GetNameByID='Mixing_Ratio_12'
         CASE(JeansLength_Field)
            GetNameByID='Jeans_Length'
         CASE(CoolingStrength_Field)
            GetNameByID='Cooling_Strength'
         CASE(vx_Field)
            GetNameByID='vx'
         CASE(vy_Field)
            GetNameByID='vy'
         CASE(vz_Field)
            GetNameByID='vz'
         CASE(VMag_Field)
            GetNameByID='V_mag'
         CASE(Enstrophy_Field)
            GetNameByID='Enstrophy'
         CASE(srhovx_Field)
            GetNameByID='srho_vx'
         CASE(srhovy_Field)
            GetNameByID='srho_vy'
         CASE(srhovz_Field)
            GetNameByID='srho_vz'
         CASE(MPI_ID_Field)
            GetNameById='MPI_ID'
         CASE(ChildMask_Field)
            GetNameById='ChildMask'
         CASE(ErrFlag_Field)
            GetNameById='ErrFlag'
         CASE(Divergence_Field)
            GetNameById='Divergence'
         CASE DEFAULT
            IF (MPI_ID == 0) PRINT *, 'Warning - unrecognized field total requested'
            write(GetNameById,'(A5,I5.5)') 'Field', i
         END SELECT
      END IF
   END FUNCTION GetNameByID


   FUNCTION FindMin(Field, OptShape)
      TYPE(FieldDef) :: Field
      INTEGER :: n,i,j,k
      TYPE(ShapeDef), OPTIONAL, POINTER :: OptShape
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(InfoDef), POINTER :: Info
      REAL(KIND=qPREC) :: FindMin, rpos(3)
      FindMin=Huge(1d0)
      IF (Field%Component==GASCOMP .OR. Field%Component==BOTHCOMP) THEN
         DO n=0, MaxLevel
            Nodelist=>Nodes(n)%p
            DO WHILE (ASSOCIATED(NodeList))
               info=>nodelist%self%info
               DO i=1, info%mx(1)
                  DO j=1, info%mx(2)
                     DO k=1, info%mx(3)
                        IF (Present(OptShape)) THEN
                           IF (.NOT. IsInShape(OptShape, Cellpos(info, i, j, k), rpos, levels(Info%level)%tnow)) CYCLE
                        END IF
                        IF (info%level < MaxLevel) THEN
                           IF (info%Childmask(i,j,k) /= 0) CYCLE
                        END IF
                        FindMin=min(FindMin, GetField(info,i,j,k, Field%id))
                     END DO
                  END DO
               END DO
               Nodelist=>Nodelist%next
            END DO
         END DO
      END IF
      IF  (Field%Component==PARTICLECOMP .OR. Field%Component==BOTHCOMP) THEN
         ParticleList=>SinkParticles
         DO WHILE (ASSOCIATED(ParticleList))
            FindMin=min(FindMin, GetParticleField(ParticleList%self, Field%ID))
            ParticleList=>ParticleList%next
         END DO
      END IF
      

   END FUNCTION FindMin


   FUNCTION FindMax(Field, OptShape)
      TYPE(FieldDef) :: Field
      INTEGER :: n,i,j,k
      TYPE(ShapeDef), OPTIONAL, POINTER :: OptShape
      TYPE(ParticleListDef), POINTER :: ParticleList
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(InfoDef), POINTER :: Info
      REAL(KIND=qPREC) :: FindMax, rpos(3)
      FindMax=-Huge(1d0)
      IF (Field%Component==GASCOMP .OR. Field%Component==BOTHCOMP) THEN
         DO n=0, MaxLevel
            Nodelist=>Nodes(n)%p
            DO WHILE (ASSOCIATED(NodeList))
               Info=>nodelist%self%info
               DO i=1, info%mx(1)
                  DO j=1, info%mx(2)
                     DO k=1, info%mx(3)
                        IF (Present(OptShape)) THEN
                           IF (.NOT. IsInShape(OptShape, Cellpos(Info, i, j, k), rpos, levels(Info%level)%tnow)) CYCLE
                        END IF

                        IF (info%level < MaxLevel) THEN
                           IF (info%Childmask(i,j,k) /= 0) CYCLE
                        END IF
                        FindMax=max(FindMax, GetField(info,i,j,k, Field%id))
                     END DO
                  END DO
               END DO
               Nodelist=>Nodelist%next
            END DO
         END DO
      END IF
      IF  (Field%Component==PARTICLECOMP .OR. Field%Component==BOTHCOMP) THEN
         ParticleList=>SinkParticles
         DO WHILE (ASSOCIATED(ParticleList))
            FindMax=max(FindMax, GetParticleField(ParticleList%self, Field%ID))
            ParticleList=>ParticleList%next
         END DO
      END IF
      

   END FUNCTION FindMax


END MODULE Fields
