!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    histograms.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Histograms
   USE TreeDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE EOS
   USE ParticleDeclarations
   USE Fields
   USE ProcessingDeclarations
   USE Shapes
   IMPLICIT NONE

   TYPE HistogramDef
      TYPE(FieldDef) :: Field
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: Data
      INTEGER :: Scale=LINEARSCALE
      REAL(KIND=qPREC) :: minvalue=MINOVERALL
      REAL(KIND=qPREC) :: maxvalue=MAXOVERALL
      INTEGER :: nBins=100
      TYPE(ShapeDef), POINTER :: shape
      TYPE(HistogramDef), POINTER :: next      
      INTEGER :: WeightField=BINBYVOLUME
   END TYPE HistogramDef

   TYPE(HistogramDef), POINTER :: FirstHistogram, LastHistogram
   SAVE
CONTAINS
   SUBROUTINE HistogramInit()
      Nullify(FirstHistogram, LastHistogram)
   END SUBROUTINE HistogramInit

   SUBROUTINE CreateHistogram(Histogram)
      TYPE(HistogramDef), POINTER :: Histogram
      ALLOCATE(Histogram)
      CALL AddHistogramToList(Histogram)
      NULLIFY(Histogram%next, Histogram%shape)
   END SUBROUTINE CreateHistogram

   SUBROUTINE DestroyHistogram(Histogram)
      TYPE(HistogramDef), POINTER :: Histogram
      DEALLOCATE(Histogram)
      NULLIFY(Histogram)
   END SUBROUTINE DestroyHistogram

   SUBROUTINE AddHistogramToList(Histogram)
      TYPE(HistogramDef),POINTER :: Histogram
      IF(.NOT. ASSOCIATED(FirstHistogram)) THEN ! First Histogram Object only
         FirstHistogram=>Histogram
         LastHistogram=>Histogram
      ELSE
         LastHistogram%next=>Histogram
         LastHistogram=>Histogram
      END IF
   END SUBROUTINE AddHistogramToList

   SUBROUTINE ProcessHistograms
      TYPE(HistogramDef), POINTER :: Histogram
      INTEGER :: n
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(ParticleListDef), POINTER :: ParticleList
      REAL(KIND=qPREC) :: minvalue, maxvalue, dx, weight, rpos(3)
      CHARACTER(LEN=25) :: FileName

      IF (MPI_ID == 0 .AND. ASSOCIATED(FirstHistogram)) THEN
         WRITE(FileName, '(A14,I5.5,A6)') 'out/Histogram_', current_frame, '.curve'
         OPEN(UNIT=HISTOGRAM_DATA_HANDLE, FILE=FileName, STATUS='UNKNOWN')
      END IF
      Histogram=>FirstHistogram
      DO WHILE (ASSOCIATED(Histogram))        
         ALLOCATE(Histogram%Data(Histogram%nBins))
         Histogram%Data=0d0
         IF (Histogram%minvalue==MINOVERALL) THEN
            IF (ASSOCIATED(Histogram%shape)) THEN
               minvalue=FindMin(Histogram%Field, Histogram%Shape)
            ELSE
               minvalue=FindMin(Histogram%Field)
            END IF
         ELSE
            minvalue=Histogram%minvalue
         END IF
         IF (Histogram%maxvalue==MAXOVERALL) THEN
            IF (ASSOCIATED(Histogram%shape)) THEN
               maxvalue=FindMax(Histogram%Field, Histogram%shape)
            ELSE
               maxvalue=FindMax(Histogram%Field)
            END IF
         ELSE
            maxvalue=Histogram%maxvalue
         END IF
         IF (Histogram%Scale==LOGSCALE) THEN
            minvalue=log10(minvalue)
            maxvalue=log10(maxvalue)
         END IF
         dx=(maxvalue-minvalue)/REAL(Histogram%nbins)
         IF (Histogram%Field%Component==GASCOMP .OR. Histogram%Field%Component==BOTHCOMP) THEN
            DO n=0, MaxLevel
               Nodelist=>Nodes(n)%p
               DO WHILE (ASSOCIATED(NodeList))
                  CALL ProcessHistogram(Nodelist%self%info, Histogram, minvalue, maxvalue, dx) 
                  Nodelist=>Nodelist%next
               END DO
            END DO
         END IF
         IF (Histogram%Field%Component==PARTICLECOMP .OR. Histogram%Field%Component==BOTHCOMP) THEN
            ParticleList=>SinkParticles
            DO WHILE (ASSOCIATED(ParticleList))
               IF (ASSOCIATED(Histogram%shape)) THEN
                  IF (.NOT. IsInShape(Histogram%shape, ParticleList%self%xloc, rpos, levels(0)%tnow)) THEN
                     ParticleList=>ParticleList%next
                     CYCLE
                  END IF
               END IF
               IF (Histogram%WeightField == BINBYVOLUME) THEN
                  weight=1d0
               ELSEIF (Histogram%WeightField == BINBYMASS) THEN
                  weight=ParticleList%self%q(1)
               ELSE
                  weight=GetParticleField(ParticleList%self, Histogram%WeightField)
               END IF
               IF (Histogram%Scale==LOGSCALE) THEN
                  CALL BinData(log10(GetParticleField(ParticleList%self, Histogram%Field%ID)), weight, Histogram%Data, minvalue, maxvalue, dx)
               ELSE
                  CALL BinData(GetParticleField(ParticleList%self, Histogram%Field%ID), weight, Histogram%Data, minvalue, maxvalue, dx)
               END IF
               ParticleList=>ParticleList%next
            END DO
         END IF
         CALL CollectHistogram(Histogram)
         CALL OutputHistogram(Histogram, minvalue, maxvalue, dx)
         DEALLOCATE(Histogram%Data)
         Histogram=>Histogram%next            
      END DO
      IF (MPI_ID == 0 .AND. ASSOCIATED(FirstHistogram)) CLOSE(HISTOGRAM_DATA_HANDLE)
    END SUBROUTINE ProcessHistograms

   SUBROUTINE ProcessHistogram(Info, Histogram, minvalue, maxvalue, dx)
      TYPE(InfoDef) :: Info
      TYPE(HistogramDef) :: Histogram
      INTEGER :: i,j,k
      REAL(KIND=qPREC) :: dv, minvalue, maxvalue, dx, weight, pos(3), rpos(3)
      dv=levels(Info%level)%dx**nDim
      DO i=1, Info%mX(1)
         IF (iCylindrical /= NoCyl) dv=levels(Info%level)%dx**nDim*(Info%xBounds(1,1)+levels(Info%level)%dx*(REAL(i)-.5d0))
         DO j=1, Info%mX(2)
            DO k=1, Info%mX(3)
               IF (Info%level < MaxLevel) THEN
                  IF (Info%ChildMask(i,j,k) /= 0) THEN
!                     PRINT*, 'cycling'
                     CYCLE
                  END IF
               END IF
               pos=CellPos(Info, i, j, k)
               IF (ASSOCIATED(Histogram%shape)) THEN
                  IF (.NOT. IsInShape(Histogram%shape, pos, rpos, levels(Info%level)%tnow)) CYCLE
               END IF

               IF (Histogram%WeightField == BINBYVOLUME) THEN
                  weight=dv
               ELSEIF (Histogram%WeightField == BINBYMASS) THEN
                  weight=dv*Info%q(i,j,k,1)
               ELSE
                  weight=dv*GetField(Info,i,j,k, Histogram%WeightField)
               END IF
               IF (Histogram%Scale==LOGSCALE) THEN
                  CALL BinData(log10(GetField(Info,i,j,k, Histogram%Field%ID)), weight, Histogram%Data, minvalue, maxvalue, dx)
               ELSE
                  CALL BinData(GetField(Info,i,j,k, Histogram%Field%ID), weight, Histogram%Data, minvalue, maxvalue, dx)
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE ProcessHistogram

   SUBROUTINE BinData(data, weight, bins, minvalue, maxvalue, dx)
      REAL(KIND=qPREC) :: data, weight, minvalue, maxvalue, dx
      REAL(KIND=qPrec), DIMENSION(:) :: bins
      INTEGER :: iloc
      IF (isNAN(data)) RETURN
      iloc=max(1,min(size(bins), ceiling((data-minvalue)/dx)))
      bins(iloc)=bins(iloc)+weight
   END SUBROUTINE BinData


   SUBROUTINE CollectHistogram(Histogram)
      TYPE(HistogramDef), POINTER :: Histogram
      INTEGER :: iErr
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: totals
      IF (MPI_NP == 1) RETURN
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, Histogram%Data, Histogram%nbins, MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
   END SUBROUTINE CollectHistogram


   SUBROUTINE OutputHistogram(Histogram, minvalue, maxvalue, dx)
      TYPE(HistogramDef), POINTER :: Histogram
      INTEGER :: i
      CHARACTER(LEN=9), PARAMETER :: DATAFORMAT = '(2E25.16)'
      REAL(KIND=qPREC) :: minvalue, maxvalue, dx
      IF (MPI_ID == 0) THEN
         WRITE(HISTOGRAM_DATA_HANDLE, '(A2,3A)') '# ', TRIM(WeightName(Histogram%weightfield)), '_weighted_',TRIM(GetName(Histogram%Field))
         WRITE(HISTOGRAM_DATA_HANDLE, DATAFORMAT) minvalue, 0d0
         DO i=1,size(Histogram%Data)
            WRITE(HISTOGRAM_DATA_HANDLE, DATAFORMAT) minvalue+real(i-1)*dx, Histogram%Data(i)/dx
            WRITE(HISTOGRAM_DATA_HANDLE, DATAFORMAT) minvalue+real(i)*dx, Histogram%Data(i)/dx
         END DO
         WRITE(HISTOGRAM_DATA_HANDLE, DATAFORMAT) maxvalue, 0d0
      END IF
   END SUBROUTINE OutputHistogram


   FUNCTION WeightName(FieldID)
      CHARACTER(LEN=MAXFIELDSLENGTH) :: WeightName
      INTEGER :: FieldID
      IF (FieldID == BINBYVOLUME) THEN
         WeightName='volume'
      ELSEIF (FieldID == BINBYMASS) THEN
         WeightName='mass'
      ELSE
         WeightName=GetName(FieldID)
      END IF
   END FUNCTION WeightName


END MODULE Histograms
