!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    pdfs.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE PDFs
   USE TreeDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE EOS
   USE ParticleDeclarations
   USE Shapes
   USE Fields
   USE IOBOV
   USE ProcessingDeclarations
   IMPLICIT NONE
   TYPE PDFDef
      TYPE(FieldDef) :: Field(2)
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: Data
      INTEGER :: Scale(2)=LINEARSCALE
      REAL(KIND=qPREC) :: minvalue(2)=MINOVERALL
      REAL(KIND=qPREC) :: maxvalue(2)=MAXOVERALL
      INTEGER :: nBins(2)=100
      INTEGER :: WeightField=BINBYVOLUME
      TYPE(ShapeDef), POINTER :: Shape
      TYPE(PDFDef), POINTER :: next      
   END TYPE PDFDef


   TYPE(PDFDef), POINTER :: FirstPDF, LastPDF

CONTAINS
   SUBROUTINE PDFInit()
      Nullify(FirstPDF, LastPDF)
   END SUBROUTINE PDFInit

   SUBROUTINE CreatePDF(PDF)
      TYPE(PDFDef), POINTER :: PDF
      ALLOCATE(PDF)
      CALL AddPDFToList(PDF)
      NULLIFY(PDF%next)
      NULLIFY(PDF%shape)
   END SUBROUTINE CreatePDF

   SUBROUTINE DestroyPDF(PDF)
      TYPE(PDFDef), POINTER :: PDF
      DEALLOCATE(PDF)
      NULLIFY(PDF)
   END SUBROUTINE DestroyPDF

   SUBROUTINE AddPDFToList(PDF)
      TYPE(PDFDef),POINTER :: PDF
      IF(.NOT. ASSOCIATED(FirstPDF)) THEN ! First PDF Object only
         FirstPDF=>PDF
         LastPDF=>PDF
      ELSE
         LastPDF%next=>PDF
         LastPDF=>PDF
      END IF
   END SUBROUTINE AddPDFToList

   SUBROUTINE ProcessPDFs
      TYPE(PDFDef), POINTER :: PDF
      INTEGER :: n,l
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(ParticleListDef), POINTER :: ParticleList
      REAL(KIND=qPREC) :: minvalue(2), maxvalue(2), dx(2), weight, data(2),rpos(3)
      CHARACTER(LEN=25) :: FileName

      PDF=>FirstPDF
      DO WHILE (ASSOCIATED(PDF))        
         IF (PDF%Field(1)%Component /= PDF%Field(2)%Component) THEN
            IF (MPI_ID == 0) PRINT*, 'Error pdf fields must have same component'
         ELSE

            ALLOCATE(PDF%Data(PDF%nBins(1), PDF%nBins(2)))
            PDF%Data=0d0
            DO l=1,2
               IF (PDF%minvalue(l)==MINOVERALL) THEN
                  IF (ASSOCIATED(PDF%shape)) THEN
                     minvalue(l)=FindMin(PDF%Field(l), PDF%Shape)
                  ELSE
                     minvalue(l)=FindMin(PDF%Field(l))
                  END IF
               ELSE
                  minvalue(l)=PDF%minvalue(l)
               END IF
               IF (PDF%maxvalue(l)==MAXOVERALL) THEN
                  IF (ASSOCIATED(PDF%shape)) THEN
                     maxvalue(l)=FindMax(PDF%Field(l), PDF%Shape)
                  ELSE
                     maxvalue(l)=FindMax(PDF%Field(l))
                  END IF
               ELSE
                  maxvalue(l)=PDF%maxvalue(l)
               END IF
               IF (PDF%Scale(l)==LOGSCALE) THEN
                  minvalue(l)=log10(minvalue(l))
                  maxvalue(l)=log10(maxvalue(l))
               END IF
            END DO
            dx=(maxvalue-minvalue)/REAL(PDF%nbins)
            IF (PDF%Field(1)%Component==GASCOMP .OR. PDF%Field(1)%Component==BOTHCOMP) THEN
               DO n=0, MaxLevel
                  Nodelist=>Nodes(n)%p
                  DO WHILE (ASSOCIATED(NodeList))
                     CALL ProcessPDF(Nodelist%self%info, PDF, minvalue, maxvalue, dx) 
                     Nodelist=>Nodelist%next
                  END DO
               END DO
            END IF
            IF (PDF%Field(1)%Component==PARTICLECOMP .OR. PDF%Field(1)%Component==BOTHCOMP) THEN
               ParticleList=>SinkParticles
               DO WHILE (ASSOCIATED(ParticleList))
                  IF (ASSOCIATED(PDF%shape)) THEN
                     IF (.NOT. IsInShape(PDF%shape, ParticleList%self%xloc, rpos, levels(0)%tnow)) THEN
                        ParticleList=>ParticleList%next
                        CYCLE
                     END IF
                  END IF

                  IF (PDF%WeightField == BINBYVOLUME) THEN
                     weight=1d0
                  ELSEIF (PDF%WeightField == BINBYMASS) THEN
                     weight=ParticleList%self%q(1)
                  ELSE
                     weight=GetParticleField(ParticleList%self, PDF%WeightField)
                  END IF
                  DO l=1,2
                     data(l)=GetParticleField(ParticleList%self, PDF%Field(l)%ID)
                     IF (PDF%Scale(l)==LOGSCALE) data(l)=log10(data(l))
                  END DO
                  CALL BinData2D(data,weight,PDF%Data,minvalue,maxvalue,dx)
                  ParticleList=>ParticleList%next
               END DO
            END IF
            CALL CollectPDF(PDF)
            CALL OutputPDF(PDF, minvalue, maxvalue, dx)
            DEALLOCATE(PDF%Data)
         END IF
         PDF=>PDF%next            
      END DO
   END SUBROUTINE ProcessPDFs

   SUBROUTINE ProcessPDF(Info, PDF, minvalue, maxvalue, dx)
      TYPE(InfoDef) :: Info
      TYPE(PDFDef) :: PDF
      INTEGER :: i,j,k,l
      REAL(KIND=qPREC) :: dv, minvalue(2), maxvalue(2), dx(2), weight, data(2), pos(3), rpos(3)
      dv=levels(Info%level)%dx**nDim
      DO i=1, Info%mX(1)
         IF (iCylindrical /= NoCyl) dv=levels(Info%level)%dx**nDim*(Info%xBounds(1,1)+levels(Info%level)%dx*(REAL(i)-.5d0))
         DO j=1, Info%mX(2)
            DO k=1, Info%mX(3)
               IF (Info%level < MaxLevel) THEN
                  IF (Info%ChildMask(i,j,k) /= 0) THEN
!                     PRINT*, 'cycling'
                     CYCLE
                  END IF
               END IF
               pos=CellPos(Info, i, j, k)
               IF (ASSOCIATED(PDF%shape)) THEN
                  IF (.NOT. IsInShape(PDF%shape, pos, rpos, levels(Info%level)%tnow)) CYCLE
               END IF

               IF (PDF%WeightField == BINBYVOLUME) THEN
                  weight=dv
               ELSEIF (PDF%WeightField == BINBYMASS) THEN
                  weight=dv*Info%q(i,j,k,1)
               ELSE
                  weight=dv*GetField(Info,i,j,k, PDF%WeightField)
               END IF
               DO l=1,2
                  data(l)=GetField(Info,i,j,k, PDF%Field(l)%ID)
                  IF (PDF%Scale(l)==LOGSCALE) data(l)=log10(data(l))
               END DO
               CALL BinData2D(data,weight,PDF%Data,minvalue,maxvalue,dx)
            END DO
         END DO
      END DO
   END SUBROUTINE ProcessPDF

   SUBROUTINE BinData2D(data, weight, bins, minvalue, maxvalue, dx)
      REAL(KIND=qPREC) :: data(2), weight, minvalue(2), maxvalue(2), dx(2)
      REAL(KIND=qPrec), DIMENSION(:,:) :: bins
      INTEGER :: iloc(2),l
      IF (isNAN(data(1)) .OR. isNan(data(2))) RETURN
      DO l=1,2
         iloc(l)=max(1,min(size(bins,l), ceiling((data(l)-minvalue(l))/dx(l))))
      END DO
      bins(iloc(1),iloc(2))=bins(iloc(1),iloc(2))+weight
   END SUBROUTINE BinData2D


   SUBROUTINE CollectPDF(PDF)
      TYPE(PDFDef), POINTER :: PDF
      INTEGER :: iErr
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: totals
      IF (MPI_NP == 1) RETURN
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, PDF%Data, product(PDF%nbins), MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
   END SUBROUTINE CollectPDF


   SUBROUTINE OutputPDF(PDF, minvalue, maxvalue, dx)
      TYPE(PDFDef), POINTER :: PDF
      INTEGER :: i
      CHARACTER(LEN=9), PARAMETER :: DATAFORMAT = '(2E25.16)'
      REAL(KIND=qPREC) :: minvalue(2), maxvalue(2), dx(2)
      CHARACTER(LEN=80) :: Name      
      IF (MPI_ID == 0) THEN
         WRITE(Name, '(A,A10,A,A4,A,A1,I5.5)') TRIM(WeightName(PDF%weightfield)), '_weighted_',TRIM(GetName(PDF%Field(1))), '_vs_', TRIM(GetName(PDF%Field(2))),'_',current_frame
         CALL WriteBOV2DScalar(name, minvalue, maxvalue, levels(0)%tnow, PDF%data, 'pdf')
      END IF
   END SUBROUTINE OutputPDF


   FUNCTION WeightName(FieldID)
      CHARACTER(LEN=MAXFIELDSLENGTH) :: WeightName
      INTEGER :: FieldID
      IF (FieldID == BINBYVOLUME) THEN
         WeightName='volume'
      ELSEIF (FieldID == BINBYMASS) THEN
         WeightName='mass'
      ELSE
         WeightName=GetName(FieldID)
      END IF
   END FUNCTION WeightName


END MODULE PDFs
