!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    processing_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir processing
!! @brief Contains modules for managing processing algorithm

!> @file processing_control.f90
!! @brief Main file for module ProcessingControl

!> @defgroup Processing Processing
!! @brief Group of modules for managing Processing algorithm

!> @defgroup ProcessingControl ProcessingControl
!> @brief Main module for managing Processing algorithm
!! @ingroup Processing

!> Main module for managing Processing
!! @ingroup ProcessingControl

MODULE ProcessingControl
  USE DataDeclarations
  USE TreeDeclarations
  USE ProcessingDeclarations
  USE DataLevelOps
# if defined HYPRE
  USE EllipticControl
# endif
  USE Totals
  USE Histograms
  USE PDFs
  USE Projections
  USE Spectras
  USE Timing
  IMPLICIT NONE
  PRIVATE
  PUBLIC ProcessInit, ProcessData
  SAVE

CONTAINS
  !> @name DataProcessingRoutines
  !! @{

  !> Initialize Processing params
  SUBROUTINE ProcessInit()
     INTEGER :: iErr
     NrDiagnosticVars=0
     CALL TotalInit()
     CALL HistogramInit()
     CALL PDFInit()
     CALL ProjectionInit()
     CALL SpectraInit()
  END SUBROUTINE ProcessInit

  SUBROUTINE ProcessData
     INTEGER :: n
     TYPE(NodeDef), POINTER :: node
     TYPE(NodeDefList), POINTER :: nodelist
     TYPE(InfoDef), POINTER :: Info
     CALL StartTimer(iProcessData, -2)
     IF (Processing_mbc > 0) THEN !Need to ghost cells...
        DO n=0, MaxLevel
           CALL GenericTransfer(n, GCopyFields, Processing_mbc, lHydroPeriodic)
           IF (EGVars > 0) CALL GenericTransfer(n, EGCopyFields, Processing_mbc, lEllipticPeriodic)
           CALL ApplyPhysicalBCs(n)
# if defined HYPRE           
           CALL ApplyEllipticBC(n)
# endif
        END DO
     END IF

     IF (NrDiagnosticVars > 0) THEN        
        DO n=0, MaxLevel
           nodelist=>Nodes(n)%p
           DO WHILE (associated(nodelist))
              node=>nodelist%self
              info=>node%info
              ALLOCATE(Info%diagnostics(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),1:NrDiagnosticVars))
              CALL ProcessDiagnostics(Info)
              nodelist=>nodelist%next
           END DO
        END DO
     END IF

     CALL ProcessTotals
     CALL ProcessHistograms
     CALL ProcessPDFs
     CALL ProcessProjections
     CALL ProcessSpectras
     CALL StopTimer(iProcessData, -2)
  END SUBROUTINE ProcessData



  SUBROUTINE ProcessDiagnostics(Info)
     TYPE(InfoDef) :: Info
     TYPE(DiagnosticListDef), POINTER :: Diagnostic
     INTEGER :: i,j,k,m
     REAL(KIND=qPREC) :: x,y,z,dx
     Diagnostic=>FirstDiagnostic
     m=1
     dx=levels(Info%level)%dx
     DO WHILE (ASSOCIATED(Diagnostic))
        DO i=1,Info%mX(1)
           x=Info%xBounds(1,1)+(REAL(i)-half)*dx
           DO j=1,Info%mX(2)
              y=Info%xBounds(2,1)+(REAL(j)-half)*merge(dx,0d0,nDim >= 2)
              DO k=1,Info%mX(3)
                 z=Info%xBounds(2,1)+(REAL(k)-half)*merge(dx,0d0,nDim >= 3)
                 Info%diagnostics(i,j,k,m) = GetField(Info,i,j,k,Diagnostic%Field%id, (/x,y,z/))
              END DO
           END DO
        END DO
        m=m+1
        Diagnostic=>Diagnostic%next
     END DO
  END SUBROUTINE ProcessDiagnostics
END MODULE ProcessingControl

           
!        IF (lStoreOI) CALL ProcessEmissions(iOI)
!        IF (lStoreNII) CALL ProcessEmissions(iNII)
!        IF (lStoreSII_6716) CALL ProcessEmissions(iSII_6716)
!        IF (lStoreSII_6731) CALL ProcessEmissions(iSII_6731)
!        IF (lStoreHalpha) CALL ProcessEmissions(iHalpha)
