!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    projections.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Projections
   USE TreeDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE EOS
   USE ParticleDeclarations
   USE Fields
   USE IOBOV
   USE ProcessingDeclarations
   USE Images
   USE Shapes
   USE Cameras
   USE Movies
   IMPLICIT NONE
   INTEGER, PARAMETER :: MAXIMUMPLOTLEVEL=-1
   TYPE ProjectionDef
      TYPE(FieldDef) :: Field
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: Data
      REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: ParticleData
      INTEGER :: dim=3
      REAL(KIND=qPREC) :: pow=1d0
      LOGICAL :: lReadCameraList=.false.
      INTEGER :: PlotLevel=MAXIMUMPLOTLEVEL
      TYPE(ShapeDef), POINTER :: Shape => NULL()
      TYPE(ProjectionDef), POINTER :: next      
      TYPE(CameraDef), POINTER :: Camera => NULL()
      TYPE(ImageDef), POINTER :: Image => NULL()
      TYPE(MovieDef), POINTER :: Movie => NULL()
   END TYPE ProjectionDef
   
   TYPE(ProjectionDef), POINTER :: FirstProjection, LastProjection

CONTAINS
   SUBROUTINE ProjectionInit()
      Nullify(FirstProjection, LastProjection)
   END SUBROUTINE ProjectionInit

   SUBROUTINE CreateProjection(Projection)
      TYPE(ProjectionDef), POINTER :: Projection
      ALLOCATE(Projection)
      CALL AddProjectionToList(Projection)
      NULLIFY(Projection%next)
      NULLIFY(Projection%shape)
   END SUBROUTINE CreateProjection

   SUBROUTINE DestroyProjection(Projection)
      TYPE(ProjectionDef), POINTER :: Projection
      DEALLOCATE(Projection)
      NULLIFY(Projection)
   END SUBROUTINE DestroyProjection

   SUBROUTINE AddProjectionToList(Projection)
      TYPE(ProjectionDef),POINTER :: Projection
      IF(.NOT. ASSOCIATED(FirstProjection)) THEN ! First Projection Object only
         FirstProjection=>Projection
         LastProjection=>Projection
      ELSE
         LastProjection%next=>Projection
         LastProjection=>Projection
      END IF
   END SUBROUTINE AddProjectionToList

   SUBROUTINE ProcessProjections
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: n,l,mx(3), PlotLevel, axis(3),ipos(3), i, j
      TYPE(NodeDefList), POINTER :: NodeList
      TYPE(ParticleListDef), POINTER :: ParticleList
      REAL(KIND=qPREC) :: rpos(3)
!      IF (nDim == 1) RETURN
      Projection=>FirstProjection
      DO WHILE (ASSOCIATED(Projection))        
         IF (Projection%PlotLevel == MAXIMUMPLOTLEVEL) THEN
            PlotLevel=FinestLevel
         ELSE
            PlotLevel=min(Projection%PlotLevel,MaxLevel)
         END IF

         IF (Projection%lReadCameraList) THEN
            OPEN(UNIT=CAMERA_DATA_HANDLE, FILE='camera.data', STATUS='old')
            READ(CAMERA_DATA_HANDLE, '(I10)') nCameras
            CALL CreateCamera(Projection%Camera)
            nCameras=nCameras-1
         ELSE
            nCameras=1
            IF (ASSOCIATED(Projection%Movie)) THEN
               IF (.NOT. ASSOCIATED(Projection%Camera)) CALL CreateCamera(Projection%Camera)
               CALL UpdateMovieCamera(Projection%Movie, Projection%Camera)
            END IF
         END IF
         DO i=1,nCameras
            IF (Projection%lReadCameraList) THEN
               CALL InputCamera(CAMERA_DATA_HANDLE, Projection%Camera)
            END IF
            IF (Projection%PlotLevel == MAXIMUMPLOTLEVEL) THEN
               PlotLevel=FinestLevel
            ELSE
               PlotLevel=min(Projection%PlotLevel,MaxLevel)
            END IF
            IF (.NOT. ASSOCIATED(Projection%Camera)) THEN
               axis=modulo(Projection%dim+(/0,1,2/),3)+1
               ALLOCATE(Projection%Data(levels(PlotLevel)%mx(axis(1)), levels(PlotLevel)%mx(axis(2))))
            ELSE
               ALLOCATE(Projection%data(Projection%Camera%res, nint(Projection%Camera%res*Projection%Camera%FOV(2)/Projection%Camera%FOV(1))))
            END IF
            Projection%Data=0d0
            IF (Projection%Field%Component==GASCOMP .OR. Projection%Field%Component==BOTHCOMP) THEN
               DO n=0, MaxLevel
                  Nodelist=>Nodes(n)%p
                  DO WHILE (ASSOCIATED(NodeList))
                     CALL ProcessProjection(Nodelist%self%info, Projection, axis, PlotLevel)
                     Nodelist=>Nodelist%next
                  END DO
               END DO
            END IF
            IF (MPI_ID == 0) THEN
               IF (Projection%Field%Component==PARTICLECOMP .OR. Projection%Field%Component==BOTHCOMP) THEN
                  IF (NrSinkParticles > 0) THEN 
                     ALLOCATE(Projection%ParticleData(NrSInkParticles, 3))
                     Projection%ParticleData=0d0
                     ParticleList=>SinkParticles
                     j=0
                     DO WHILE (ASSOCIATED(ParticleList))
                        j=j+1
                        IF (ASSOCIATED(Projection%shape)) THEN
                           IF (.NOT. IsInShape(Projection%shape, ParticleList%self%xloc, rpos, levels(0)%tnow)) THEN
                              ParticleList=>ParticleList%next
                              CYCLE
                           END IF
                        END IF
                        IF (ASSOCIATED(Projection%Camera)) THEN
                           Projection%ParticleData(j,:)=(GetPixel(Projection%Camera, ParticleList%self%xloc)-half)*projection%camera%fov(:)
                           
!                           CALL BinParticle(Projection%Camera, ParticleList%self,Projection%ParticleData(i, :)) 
                           !Projection%data, GetParticleField(ParticleList%Self, Projection%Field%iD)**Projection%Pow)
                           
                           !                     ipos(1:2)=nint(GetPos(Projection%Camera, ParticleList%self%xloc)*shape(projection%data))
!                     IF (ALL(ipos(1:2) >= 1) .AND. ALL(ipos(1:2) <= shape(projection%data))) THEN
!                        Projection%data(ipos(1),ipos(2))=&
!                             Projection%data(ipos(1),ipos(2))+ &
!                             GetParticleField(ParticleList%Self, Projection%Field%iD)**Projection%Pow
                           !                     END IF
                        ELSE
                           Projection%ParticleData(j, :)=ParticleList%self%xloc(axis(1:2)) !(/ipos(axis(1)),ipos(axis(2))/)
!                           ipos=max(1,min(levels(PlotLevel)%mX,nint((ParticleList%self%xloc-GxBounds(:,1))/levels(PlotLevel)%dx+half)))
!                           Projection%data(ipos(axis(1)),ipos(axis(2)))=&
!                                Projection%data(ipos(axis(1)),ipos(axis(2)))+ &
!                                GetParticleField(ParticleList%Self, Projection%Field%iD)**Projection%Pow

                        END IF
                        Projection%ParticleData(j,3)=GetParticleField(ParticleList%Self, Projection%Field%iD)**Projection%Pow
                        ParticleList=>ParticleList%next
                     END DO
                     CALL OutputParticleData(Projection, axis, PlotLevel)
                     DEALLOCATE(Projection%ParticleData)
                  END IF
               END IF
            END IF
            CALL CollectProjection(Projection)
            CALL OutputProjection(Projection,axis,PlotLevel)
            DEALLOCATE(Projection%Data)
         END DO
         IF (Projection%lReadCameraList) THEN
            CLOSE(UNIT=CAMERA_DATA_HANDLE)
         END IF

         Projection=>Projection%next            
      END DO
   END SUBROUTINE ProcessProjections

   
   SUBROUTINE ProcessProjection(Info, Projection, axis, PlotLevel)
      TYPE(InfoDef) :: Info
      TYPE(ProjectionDef) :: Projection
      INTEGER :: i,j,k,l,axis(3), PlotLevel,ipos(3,2)
      REAL(KIND=qPREC) :: dv, pos(3), rpos(3)
      IF ((nDim == 1 .OR. nDim == 2) .AND. Projection%Dim == 3) THEN
         dv=1d0!levels(max(Info%level,PlotLevel))%dx
      ELSE
         dv=levels(Info%Level)%dx
      END IF
      ipos(3,:)=1
      DO i=1, Info%mx(1)
         ipos(1,:)=MapToLevel(Info%mGlobal(1,1)-1+i, Info%level, PlotLevel)
         DO j=1,Info%mx(2)
            ipos(2,:)=MapToLevel(Info%mGlobal(2,1)-1+j, Info%level, PlotLevel)
            DO k=1,Info%mx(3)
               IF (nDim == 3) ipos(3,:)=MapToLevel(Info%mGlobal(3,1)-1+k, Info%level, PlotLevel)
               IF (Info%level < MaxLevel) THEN
                  IF (Info%ChildMask(i,j,k) /= 0) THEN
                     CYCLE
                  END IF
               END IF
               pos=CellPos(Info, i, j, k)
               IF (ASSOCIATED(Projection%shape)) THEN
                  IF (.NOT. IsInShape(Projection%shape, pos, rpos, levels(Info%level)%tnow)) CYCLE
               END IF
               IF (ASSOCIATED(Projection%Camera)) THEN
                  CALL BinCell(Projection%Camera, pos, levels(Info%level)%dx, Projection%data, GetField(Info,i,j,k, Projection%Field%iD)**Projection%Pow)
               ELSE
                  Projection%data(ipos(axis(1),1):ipos(axis(1),2),ipos(axis(2),1):ipos(axis(2),2))=&
                       Projection%data(ipos(axis(1),1):ipos(axis(1),2),ipos(axis(2),1):ipos(axis(2),2))+ &
                       GetField(Info,i,j,k, Projection%Field%ID)**Projection%pow*dv
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE ProcessProjection


   SUBROUTINE CollectProjection(Projection)
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: iErr
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: totals
      IF (MPI_NP == 1) RETURN
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, Projection%Data, size(Projection%Data), MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
   END SUBROUTINE CollectProjection


   SUBROUTINE OutputProjection(Projection,axis,PlotLevel)
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: i,axis(3),PlotLevel
      CHARACTER(LEN=80) :: Name
      CHARACTER(LEN=15) :: powstring
      IF (MPI_ID == 0) THEN
         IF (Projection%pow /= 1) THEN
            WRITE(powstring,'(F5.2)') Projection%pow
            WRITE(powstring,'(2A)') '_to_the_', TRIM(ADJUSTL(powstring))
         ELSE
            WRITE(powstring,*) ''
         END IF
         IF (ASSOCIATED(Projection%Camera)) THEN
            IF (Projection%lReadCameraList) THEN
               WRITE(Name, '(A,A,A1,I5.5,A1,I5.5)') TRIM(GetName(Projection%Field)), TRIM(powstring), '_',current_frame, '_', Projection%Camera%iD
            ELSE
               WRITE(Name, '(A,A,A1,I5.5,A1,I5.5)') TRIM(GetName(Projection%Field)), TRIM(powstring), '_', Projection%Camera%iD, '_', current_frame
            END IF
         ELSE
            WRITE(Name, '(A,A,A7,I1.1,A1,I5.5)') TRIM(GetName(Projection%Field)), TRIM(powstring), '_along_', Projection%dim,'_',current_frame
         END IF
         IF (ASSOCIATED(Projection%Camera)) THEN
            CALL WriteBOV2DScalar(name, -Projection%Camera%FOV/2d0, Projection%Camera%FOV/2d0 , levels(0)%tnow, Projection%data, 'projection')
         ELSE
            CALL WriteBOV2DScalar(name, GxBounds(axis(1:2),1), max(GxBounds(axis(1:2),2),GxBounds(axis(1:2),1)+levels(PlotLevel)%dx), levels(0)%tnow, Projection%data, 'projection')
         END IF
         IF (ASSOCIATED(Projection%Image)) CALL OutputImage(Projection%Image, name, Projection%data)
      END IF
   END SUBROUTINE OutputProjection

   SUBROUTINE OutputParticleData(Projection, axis, PlotLevel)
      CHARACTER(LEN=80) :: Name
      CHARACTER(LEN=15) :: powstring
      TYPE(ProjectionDef), POINTER :: Projection
      INTEGER :: i,axis(3),PlotLevel
      IF (Projection%pow /= 1) THEN
         WRITE(powstring,'(F5.2)') Projection%pow
         WRITE(powstring,'(2A)') '_to_the_', TRIM(ADJUSTL(powstring))
      ELSE
         WRITE(powstring,*) ''
      END IF
      IF (ASSOCIATED(Projection%Camera)) THEN
         IF (Projection%lReadCameraList) THEN
            WRITE(Name, '(A4,A,A,A1,I5.5,A1,I5.5,A4)') 'out/', TRIM(GetName(Projection%Field)), TRIM(powstring), '_',current_frame, '_', Projection%Camera%iD, '.okc'
         ELSE
            WRITE(Name, '(A4,A,A,A1,I5.5,A1,I5.5,A4)') 'out/', TRIM(GetName(Projection%Field)), TRIM(powstring), '_', Projection%Camera%iD, '_', current_frame, '.okc'
         END IF
      ELSE
         WRITE(Name, '(A4,A,A,A7,I1.1,A1,I5.5,A4)') 'out/', TRIM(GetName(Projection%Field)), TRIM(powstring), '_along_', Projection%dim,'_',current_frame, '.okc'
      END IF
      
      
      OPEN (UNIT=11, file=Name, status="unknown")

      ! Always create 1 dummy particle to make visit happy
      write(11, '(3I6)') 4, NrSinkParticles, 4 !makes visit happy to have at least one particle when opening the database
      write(11, *) "x"
      write(11, *) "y"
      write(11, *) "z"
      write(11, *) trim(GetName(Projection%Field))
      
      DO i=1,2
         IF (ASSOCIATED(Projection%Camera)) THEN
            write(11,'(2E15.5)') -Projection%Camera%FOV(i)/2d0, Projection%Camera%FOV(i)/2d0 
         ELSE
            write(11,'(2E15.5)') GxBounds(axis(i),1), GxBounds(axis(i),2)
         END IF
      END DO
      IF (ASSOCIATED(Projection%Camera)) THEN 
         write(11,'(2E15.5)') Projection%Camera%FOV(1)/2d0/size(projection%data,1) * (/-1d0,1d0/)
      ELSE
         write(11,'(2E15.5)') GxBounds(axis(1),1)/2d0/size(projection%data,1)*(/-1d0,1d0/)
      END IF

      write(11,'(2E15.5)') minval(Projection%ParticleData(:,3)), maxval(Projection%ParticleData(:,3))
      DO i=1,NrSinkParticles
         write(11, '(100E24.16)') Projection%ParticleData(i,1:2), 0d0, Projection%ParticleData(i,3)
      END DO
      close(11)
   END SUBROUTINE OutputParticleData

END MODULE Projections
